;;; -*- Mode:Common-Lisp; Package:W; Base:10 -*-

(defconstant *memmon-default-width* 150)
(defconstant *memmon-default-height* 80)

(defvar *memmon-display-width* 70)
(defvar *memmon-time-between-updates* 30)

(defvar *maximum-read-per-minute* 500)

(defflavor memory-monitor-window
	   (
	    graph-height
	    graph-width
	    array
	    array-pointer
	    (old-page-read nil)
	    )
	   (w:process-mixin
	    w:graphics-mixin
	    w:minimum-window
	    )
  (:default-init-plist
    :process '(memory-monitor)
    :expose-p t
    :deexposed-typeout-action :permit
    :save-bits t
    :blinker-p nil
    :width *memmon-default-width*
    :height *memmon-default-height*))

(defmethod (memory-monitor-window :after :init) (&rest ignore)
  (send self :initialize))

(defmethod (memory-monitor-window :screen-manage-deexposed-visibility) ()
  t)

(defmethod (memory-monitor-window :mouse-click) (char x y)
  (ignore x y)
  (case char
    (#\mouse-l-1
     (send self :refresh)
     t)
    (t nil)))

(defmethod (memory-monitor-window :deexposed-mouse-buttons) (mask x y)
  (send self :mouse-buttons mask x y))

(defmethod (memory-monitor-window :who-line-documentation-string) ()
  `(:mouse-l-1 "Refresh memory monitor"
    ,@(when (boundp 'w:*dragging-enabled?*) '(:mouse-m-hold "Drag window"))))

(defmethod (memory-monitor-window :deexposed-who-line-documentation-string) ()
  (send self :who-line-documentation-string))

(defmethod (memory-monitor-window :verify-new-edges)
	   (new-left new-top new-width new-height)
  (ignore new-left new-top)
  (when (or (< new-width 80) (< new-height 50))
    "Memory monitor size too small"))

(defmethod (memory-monitor-window :after :change-of-size-or-margins) (&rest ignore)
  (send self :initialize)
  (send self :refresh))

(defun memory-monitor (window)
  (send (send window :process) :set-priority -5)
  (send window :refresh)
  (send window :run))

(defmethod (memory-monitor-window :initialize) ()
  (setq graph-width (- w:width *memmon-display-width* 6))
  (setq graph-height (- w:height 25))
  (let ((array-width (lsh (lsh (+ graph-width 31) -5) 5)))
    (setq array (make-array (list graph-height array-width)
			    :element-type '(mod 2))))
  (setq array-pointer 0))

(defmethod (memory-monitor-window :refresh) (&rest ignore)
  (send self :clear-screen)
  (send self :draw-rectangle 0 0 w:width w:height)
  (send self :draw-line *memmon-display-width* 0 *memmon-display-width* w:height)
  (send self :draw-line 0 20 w:width 20)
;  (send self :draw-filled-rectangle
;	3 3 (- *memmon-display-width* 6) (- w:height 6) tv:25%-gray-color)
;  (send self :draw-rectangle (+ *memmon-display-width* 3) 3 (+ graph-width 2) (+ graph-height 2))
  (send self :string-out-centered-explicit "MEMORY"
	0 6 *memmon-display-width* nil fonts:hl10b)
  (send self :string-out-centered-explicit "PAGING"
	*memmon-display-width* 6 w:width nil fonts:hl10b)
  (send self :display))

(defmethod (memory-monitor-window :display) ()
  (w:prepare-sheet (self)
    (bitblt w:normal graph-width graph-height array 0 0
	    w:screen-array (+ *memmon-display-width* 3) 22))
  (multiple-value-bind (total avail) (si:swap-status nil)
    (let ((string1 (format nil "~dMB" (round (* avail si:page-size) 250000)))
	  (string2 (format nil "~d%" (round (* 100 avail) total))))
      (let* ((ytop 25)
	     (yrange (- w:height ytop)))
	(w:prepare-sheet (self)
	  (tv:%draw-rectangle (- *memmon-display-width* 6) (- yrange 3) 3 ytop w:erase self))
	(send self :string-out-centered-explicit string1
	      0 (+ (floor yrange 3) ytop -10) *memmon-display-width* nil
	      fonts:hl12)
	(send self :string-out-centered-explicit string2
	      0 (+ (floor yrange 3/2) ytop -5) *memmon-display-width* nil
	      fonts:hl12)))))

(defmethod (memory-monitor-window :run) ()
  (loop
    (send self :update-array)
    (send self :display)
    (sleep *memmon-time-between-updates*)))

(defmethod (memory-monitor-window :update-array) ()
  (let* ((page-read (si:read-meter 'si:%count-disk-page-reads)
;		      (si:read-meter 'si:%count-disk-page-writes)))
		       )
	 (diff (- page-read (or old-page-read page-read))))
    (let ((h (round (* graph-height
		       (/ (* diff 60) *memmon-time-between-updates*))
		    *maximum-read-per-minute*)))
      (setq h (min (max h 1) (1- graph-height)))
      (tv:%draw-line array-pointer (- graph-height h)
		     array-pointer (1- graph-height)
		     w:normal t array))
    (setq old-page-read page-read))
  (if (< array-pointer (1- graph-width))
      (incf array-pointer)
      (progn
	(bitblt w:normal (- graph-width 5) graph-height array 5 0 array 0 0)
	(decf array-pointer 4)
	(tv:%draw-rectangle 5 graph-height array-pointer 0 w:erase array))))


