;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:(HYPER); Base:10. -*-


;;; Code to animate a hypercube rotating around any of four axes.
;;; Algorithm by Joe Keane and Fritz Mueller; Made to run real fast by Jamie Zawinski.
;;;
;;; "Who's to blame when macro expansion gets out of bounds?" - paraphrased B52's.
;;;


(export '(hyper hyper-screenhack))


(defparameter *observer-z* 5)
(defparameter *x-offset* 512)
(defparameter *y-offset* 400)
(defparameter *unit-pixels* 768)

(defparameter *key-increment* .001)
(defparameter *super-key-increment* .01)

(defmacro smart-multiply (x y)
  (list 'the 'number
	(cond ((= x '1) y)
	      ((= x '-1) (list '- y))
	      ((= x '0) '0)
	      (t (list '* x y)))))


(defstruct (point-state (:constructor make-point-state (&optional old-x old-y new-x new-y same-p)))
  (old-x 0 :type number)
  (old-y 0 :type number)
  (new-x 0 :type number)
  (new-y 0 :type number)
  (same-p nil :type (member T NIL))
  )


(defmacro compute (a b c d point-state)
  `(let* ((temp-mult (float (/ *unit-pixels*
			       (- (+ (smart-multiply ,a az)
				     (smart-multiply ,b bz)
				     (smart-multiply ,c cz)
				     (smart-multiply ,d dz)
				     (smart-multiply ,a aw)
				     (smart-multiply ,b bw)
				     (smart-multiply ,c cw)
				     (smart-multiply ,d dw))
				  *observer-z*)))))
     (declare (float temp-mult))
     (setf (point-state-old-x ,point-state) (point-state-new-x ,point-state))
     (setf (point-state-new-x ,point-state)
	   (+ (round (* (+ (smart-multiply ,a ax)
			   (smart-multiply ,b bx)
			   (smart-multiply ,c cx)
			   (smart-multiply ,d dx))
			temp-mult))
	      *x-offset*))
     (setf (point-state-old-y ,point-state) (point-state-new-y ,point-state))
     (setf (point-state-new-y ,point-state)
	   (+ (round (* (+ (smart-multiply ,a ay)
			   (smart-multiply ,b by)
			   (smart-multiply ,c cy)
			   (smart-multiply ,d dy))
		     temp-mult))
	      *y-offset*))
     (setf (point-state-same-p ,point-state)
	   (and (= (point-state-new-x ,point-state) (point-state-old-x ,point-state))
		(= (point-state-new-y ,point-state) (point-state-old-y ,point-state))))))


(defmacro draw-line (window from-x from-y to-x to-y)
  `(sys:%draw-line ,from-x ,from-y ,to-x ,to-y W:ALU-XOR t ,window))


(defmacro move-line (window point-state-0 point-state-1 color)
  (let ((color-change (logxor W:BLACK (symbol-value color))))
    `(tv:prepare-color (,window ,color-change)
       (tv:prepare-sheet (,window)
	 (unless (and (point-state-same-p ,point-state-0)
		      (point-state-same-p ,point-state-1))
	   (draw-line ,window
		      (point-state-old-x ,point-state-0)
		      (point-state-old-y ,point-state-0)
		      (point-state-old-x ,point-state-1)
		      (point-state-old-y ,point-state-1))
	   (draw-line ,window
		      (point-state-new-x ,point-state-0)
		      (point-state-new-y ,point-state-0)
		      (point-state-new-x ,point-state-1)
		      (point-state-new-y ,point-state-1)))))))

     
(defmacro rotate (vector-name dimension-0 dimension-1 cosine sine)
  (let ((component-0 (intern (string-append vector-name dimension-0)))
	(component-1 (intern (string-append vector-name dimension-1))))
    `(psetq ,component-0 (+ (* (the float ,component-0)
			       (the float ,cosine))
			    (* (the float ,component-1)
			       (the float ,sine)))
	    ,component-1 (- (* (the float ,component-1)
			       (the float ,cosine))
			    (* (the float ,component-0)
			       (the float ,sine))))))


(defmacro rotates (dimension-0 dimension-1)
  (let ((cosine (intern (string-append "COS-" dimension-0 dimension-1)))
	(sine (intern (string-append "SIN-" dimension-0 dimension-1))))
    `(unless (zerop ,sine)
       (rotate "A" ,dimension-0 ,dimension-1 ,cosine ,sine)
       (rotate "B" ,dimension-0 ,dimension-1 ,cosine ,sine)
       (rotate "C" ,dimension-0 ,dimension-1 ,cosine ,sine)
       (rotate "D" ,dimension-0 ,dimension-1 ,cosine ,sine))))


(defun hyper (&key (xy 0) (xz 0) (yz 0) (xw 0) (yw 0) (zw 0)
	           (window tv:selected-window)
		   (interactive t))
  (send window :expose)
  (send window :select)
  (let* ((cos-xy (cos xy)) (sin-xy (sin xy))
	 (cos-xz (cos xz)) (sin-xz (sin xz))
	 (cos-yz (cos yz)) (sin-yz (sin yz))
	 (cos-xw (cos xw)) (sin-xw (sin xw))
	 (cos-yw (cos yw)) (sin-yw (sin yw))
	 (cos-zw (cos zw)) (sin-zw (sin zw))
	 (ax 1.0) (ay 0.0) (az 0.0) (aw 0.0)
	 (bx 0.0) (by 1.0) (bz 0.0) (bw 0.0)
	 (cx 0.0) (cy 0.0) (cz 1.0) (cw 0.0)
	 (dx 0.0) (dy 0.0) (dz 0.0) (dw 1.0)

	 (---- (make-point-state))
	 (---+ (make-point-state))
	 (--++ (make-point-state))
	 (-+-- (make-point-state))
	 (-+-+ (make-point-state))
	 (-++- (make-point-state))
	 (-+++ (make-point-state))
	 (+--- (make-point-state))
	 (+--+ (make-point-state))
	 (+-+- (make-point-state))
	 (+-++ (make-point-state))
	 (++-- (make-point-state))
	 (++-+ (make-point-state))
	 (+++- (make-point-state))
	 (++++ (make-point-state))
	 (--+- (make-point-state))
	 )
    (declare (optimize (speed 3) (safety 0))
	     (type float ax ay az aw bx bz bw cx cz cw dx dy dz dw)
	     )
    (loop
      (compute -1 -1 -1 -1 ----)
      (compute -1 -1 -1 +1 ---+)
      (compute -1 -1 +1 -1 --+-)
      (compute -1 -1 +1 +1 --++)
      (compute -1 +1 -1 -1 -+--)
      (compute -1 +1 -1 +1 -+-+)
      (compute -1 +1 +1 -1 -++-)
      (compute -1 +1 +1 +1 -+++)
      (compute +1 -1 -1 -1 +---)
      (compute +1 -1 -1 +1 +--+)
      (compute +1 -1 +1 -1 +-+-)
      (compute +1 -1 +1 +1 +-++)
      (compute +1 +1 -1 -1 ++--)
      (compute +1 +1 -1 +1 ++-+)
      (compute +1 +1 +1 -1 +++-)
      (compute +1 +1 +1 +1 ++++)
      
      (move-line window ---- ---+ w:red)
      (move-line window ---- --+- w:red)
      (move-line window --+- --++ w:red)
      (move-line window ---+ --++ w:red)
      
      (move-line window +--- +--+ w:orange)
      (move-line window +--- +-+- w:orange)
      (move-line window +-+- +-++ w:orange)
      (move-line window +--+ +-++ w:orange)
      
      (move-line window -+-- -+-+ w:yellow)
      (move-line window -+-- -++- w:yellow)
      (move-line window -++- -+++ w:yellow)
      (move-line window -+-+ -+++ w:yellow)
      
      (move-line window --++ -+++ w:white)
      (move-line window --++ +-++ w:white)
      (move-line window +-++ ++++ w:white)
      (move-line window -+++ ++++ w:white)
      
      (move-line window ---- -+-- w:green)
      (Move-line window ---- +--- w:green)
      (move-line window -+-- ++-- w:green)
      (move-line window +--- ++-- w:green)
      
      (move-line window ---+ -+-+ w:cyan)
      (move-line window ---+ +--+ w:cyan)
      (move-line window +--+ ++-+ w:cyan)
      (move-line window -+-+ ++-+ w:cyan)
      
      (move-line window --+- -++- w:blue)
      (move-line window --+- +-+- w:blue)
      (move-line window +-+- +++- w:blue)
      (move-line window -++- +++- w:blue)
      
      (move-line window ++-- ++-+ w:magenta)
      (move-line window ++-- +++- w:magenta)
      (move-line window +++- ++++ w:magenta)
      (move-line window ++-+ ++++ w:magenta)

      (when interactive
	(let* ((char (read-char-no-hang window)))
	  (when char
	    (send window :clear-input)
	    (case char
	      (#\space (psetq ax bx bx cx cx ax)
		       (psetq ay by by cy cy ay)
		       (psetq az bz bz cz cz az)
		       (psetq aw bw bw cw cw aw))
	      
	      (#\= (setq xy 0.0 cos-xy 1.0 sin-xy 0.0
			 xz 0.0 cos-xz 1.0 sin-xz 0.0
			 yz 0.0 cos-yz 1.0 sin-yz 0.0
			 xw 0.0 cos-xw 1.0 sin-xw 0.0
			 yw 0.0 cos-yw 1.0 sin-yw 0.0
			 zw 0.0 cos-zw 1.0 sin-zw 0.0))
	      
	      (#\Hyper-Control-Left-Arrow (setq xy 0.0 cos-xy 1.0 sin-xy 0.0))
	      (#\Hyper-Control-Right-Arrow (setq xy 0.0 cos-xy 1.0 sin-xy 0.0))
	      (#\Control-Left-Arrow (setq xy (+ xy *key-increment*)
					  cos-xy (cos xy)
					  sin-xy (sin xy)))
	      (#\Control-Right-Arrow (setq xy (- xy *key-increment*)
					   cos-xy (cos xy)
					   sin-xy (sin xy)))
	      (#\Super-Control-Left-Arrow (setq xy (+ xy *super-key-increment*)
						cos-xy (cos xy)
						sin-xy (sin xy)))
	      (#\Super-Control-Right-Arrow (setq xy (- xy *super-key-increment*)
						 cos-xy (cos xy)
						 sin-xy (sin xy)))
	      
	      (#\Hyper-Left-Arrow (setq xz 0.0 cos-xz 1.0 sin-xz 0.0))
	      (#\Hyper-Right-Arrow (setq xz 0.0 cos-xz 1.0 sin-xz 0.0))
	      (#\Left-Arrow (setq xz (+ xz *key-increment*)
				  cos-xz (cos xz)
				  sin-xz (sin xz)))
	      (#\Right-Arrow (setq xz (- xz *key-increment*)
				   cos-xz (cos xz)
				   sin-xz (sin xz)))
	      (#\Super-Left-Arrow (setq xz (+ xz *super-key-increment*)
					cos-xz (cos xz)
					sin-xz (sin xz)))
	      (#\Super-Right-Arrow (setq xz (- xz *super-key-increment*)
					 cos-xz (cos xz)
					 sin-xz (sin xz)))
	      
	      (#\Hyper-Up-Arrow (setq yz 0.0 cos-yz 1.0 sin-yz 0.0))
	      (#\Hyper-Down-Arrow (setq yz 0.0 cos-yz 1.0 sin-yz 0.0))
	      (#\Up-Arrow (setq yz (+ yz *key-increment*)
				cos-yz (cos yz)
				sin-yz (sin yz)))
	      (#\Down-Arrow (setq yz (- yz *key-increment*)
				  cos-yz (cos yz)
				  sin-yz (sin yz)))
	      (#\Super-Up-Arrow (setq yz (+ yz *super-key-increment*)
				      cos-yz (cos yz)
				      sin-yz (sin yz)))
	      (#\Super-Down-Arrow (setq yz (- yz *super-key-increment*)
					cos-yz (cos yz)
					sin-yz (sin yz)))
	      
	      (#\Hyper-Meta-Left-Arrow (setq xw 0 cos-xw 1.0 sin-xw 0.0))
	      (#\Hyper-Meta-Right-Arrow (setq xw 0 cos-xw 1.0 sin-xw 0.0))
	      (#\Meta-Left-Arrow (setq xw (+ xw *key-increment*)
				       cos-xw (cos xw)
				       sin-xw (sin xw)))
	      (#\Meta-Right-Arrow (setq xw (- xw *key-increment*)
					cos-xw (cos xw)
					sin-xw (sin xw)))
	      (#\Super-Meta-Left-Arrow (setq xw (+ xw *super-key-increment*)
					     cos-xw (cos xw)
					     sin-xw (sin xw)))
	      (#\Super-Meta-Right-Arrow (setq xw (- xw *super-key-increment*)
					      cos-xw (cos xw)
					      sin-xw (sin xw)))
	      
	      (#\Hyper-Meta-Up-Arrow (setq yw 0 cos-yw 1.0 sin-yw 0.0))
	      (#\Hyper-Meta-Down-Arrow (setq yw 0 cos-yw 1.0 sin-yw 0.0))
	      (#\Meta-Up-Arrow (setq yw (+ yw *key-increment*)
				     cos-yw (cos yw)
				     sin-yw (sin yw)))
	      (#\Meta-Down-Arrow (setq yw (- yw *key-increment*)
				       cos-yw (cos yw)
				       sin-yw (sin yw)))
	      (#\Super-Meta-Up-Arrow (setq yw (+ yw *super-key-increment*)
					   cos-yw (cos yw)
					   sin-yw (sin yw)))
	      (#\Super-Meta-Down-Arrow (setq yw (- yw *super-key-increment*)
					     cos-yw (cos yw)
					     sin-yw (sin yw)))
	      
	      (#\Hyper-Control-Up-Arrow (setq zw 0 cos-zw 1.0 sin-zw 0.0))
	      (#\Hyper-Control-Down-Arrow (setq zw 0 cos-zw 1.0 sin-zw 0.0))
	      (#\Control-Up-Arrow (setq zw (+ zw *key-increment*)
					cos-zw (cos zw)
					sin-zw (sin zw)))
	      (#\Control-Down-Arrow (setq zw (- zw *key-increment*)
					  cos-zw (cos zw)
					  sin-zw (sin zw)))
	      (#\Super-Control-Up-Arrow (setq zw (+ zw *super-key-increment*)
					      cos-zw (cos zw)
					      sin-zw (sin zw)))
	      (#\Super-Control-Down-Arrow (setq zw (- zw *super-key-increment*)
						cos-zw (cos zw)
						sin-zw (sin zw)))
	      
	      (t (beep)))
	    (send window :set-label (format nil "xy=~,4F xz=~,4F yz=~,4F xw=~,4F yw=~,4F zw=~,4F"
					    xy xz yz xw yw zw))
	    )))
      
      (rotates "X" "Y")
      (rotates "X" "Z")
      (rotates "Y" "Z")
      (rotates "X" "W")
      (rotates "Y" "W")
      (rotates "Z" "W")
      )))



(defun hyper-screenhack (&optional (window tv:selected-window))
  (send window :clear-screen)
  (hyper :xy 0.01 :yw 0.01 :xz 0.005 :window window :interactive nil))

(when (boundp 'tv:*screen-saver-hacks-list*)
  (pushnew 'hyper-screenhack tv:*screen-saver-hacks-list*))
