;;; -*- Mode: Lisp; Syntax:Common-Lisp; Package: CONX -*-
;;;
;;; New concoction, with pre-links as well as post-links,
;;; and funky mind-blowing real-time concocting.
;;; Written by Skef Wholey, TI hacks added by Jamie Zawinski.
;;;
(in-package "CONX")

;;; DB is our database of words, a hash table mapping strings to Word
;;; structures.  DB-Index is just an array of the things, so we can choose one
;;; randomly.  Current-Index tells us how many words we've got in the database.

(defvar db (make-hash-table :test #'equal))
(defvar db-index (make-array 100 :fill-pointer 0 :adjustable t))
(defvar current-index)

;;; If *Last-Word* is non-Nil, it's the Word structure for the previous word
;;; processed.

(defvar *last-word*)

(defun init-db ()
  (clrhash db)
  (setf (fill-pointer db-index) 0)
  (setq current-index 0)
  (setq *last-word* nil))

(defstruct (word (:constructor make-word (string index))
		 (:print-function print-word))
  string
  (count 1)
  index
  (comma-count 0)
  (period-count 0)
  (quem-count 0)
  (bang-count 0)
  (successor-count 0)
  (successors '())
  (predecessor-count 0)
  (predecessors '()))

(defun print-word (word stream depth)
  depth
  (format stream "#<Word ~S>" (word-string word)))

;;; To add a word, we look for its Word structure, making a new one if
;;; necessary, and update all the right stuff.

(defmacro add-word-relation (word related-word count list)
  `(progn
     (incf (,count ,word))
     (let ((relation (assoc ,related-word (,list ,word))))
       (if relation
	   (incf (cdr relation))
	   (push (cons ,related-word 1) (,list ,word))))))
  
(defun add-word (string)
  (let ((word (gethash string db)))
    (cond (word
	   (incf (word-count word)))
	  (t
	   (setq word (make-word string current-index))
	   (vector-push-extend word db-index)
	   (incf current-index)
	   (setf (gethash string db) word)))
    (when *last-word*
      (add-word-relation *last-word* word word-successor-count word-successors)
      (add-word-relation word *last-word* word-predecessor-count word-predecessors))
    (setq *last-word* word)))

(defun add-punctuation (char)
  (when *last-word*
    (case char
      (#\, (incf (word-comma-count *last-word*)))
      (#\. (incf (word-period-count *last-word*))
	   (setq *last-word* nil))
      (#\? (incf (word-quem-count *last-word*))
	   (setq *last-word* nil))
      (#\! (incf (word-bang-count *last-word*))
	   (setq *last-word* nil)))))

;;; Dump-DB lets us look at what we've got, for debugging.

(defun dump-db ()
  (maphash #'(lambda (string word)
	       (format t "~%~A: Count = ~D, Successors:" string (word-count word))
	       (dolist (successor (word-successors word))
		 (format t "~%  ~A (~D times)" (word-string (car successor)) (cdr successor))))
	   db))

;;; Concoct file gobbles a file up, entering all its words in the database.

(defun concoct-file (name)
  (init-db)
  (with-open-file (file name :direction :input)
    (do ((line (read-line file nil nil) (read-line file nil nil)))
	((null line))
      (concoct-line line))))

(defun concoct-string (string)
  (init-db)
  (concoct-line string))

(defun concoct-line (line)
  (do ((index 0 (1+ index))
       (start nil)
       (length (length line)))
      ((= index length)
       (when start
	 (add-word (nstring-downcase (subseq line start length)))))
    (let ((char (schar line index)))
      (cond (start
	     (cond ((or (alpha-char-p char) (eql char #\')))
		   (t
		    (add-word (nstring-downcase (subseq line start index)))
		    (add-punctuation char)
		    (setq start nil))))
	    ((alpha-char-p char)
	     (setq start index))))))



;;; TI Explorer specific stuff.

#+EXPLORER
(defun concoct-mail-buffer (buf)
  (unless (boundp 'current-index) (init-db))
  (when (zwei:mail-summary-p buf) (setq buf (send buf :sequence-buffer)))
  (when (zwei:mail-file-buffer-p buf)
    (dolist (msg (zwei:node-inferiors buf))
      (do* ((line (zwei:bp-line (send msg :headers-end-bp))
		  (zwei:line-next line)))
	   ((null line))
	(when (plusp (length line)) (concoct-line line))))))

#+LISPM
(defun concoct-buffer (buf &optional start-bp)
  (unless (boundp 'current-index) (init-db))
  (do* ((line (zwei:bp-line (or start-bp (zwei:interval-first-bp buf)))
	      (zwei:line-next line)))
       ((null line))
    (when (plusp (length line)) (concoct-line line))))

#+LISPM
(defun concoct-lisp-buffer (buf)
  (unless (boundp 'current-index) (init-db))
  (do* ((line (zwei:line-next (zwei:bp-line (zwei:interval-first-bp buf)))
	      (zwei:line-next line)))
       ((null line))
    (let* ((semi (position #\; line :test #'char-equal))
	   (semi-end (and semi (position #\; line :start semi :test-not #'char-equal))))
      (and semi-end
	   (concoct-line (nsubstring line semi-end))))))

#+LISPM
(defun concoct-all-buffers ()
  (unless (boundp 'current-index) (init-db))
  (dolist (buf zwei:*zmacs-buffer-list*)
    (cond ((zwei:mail-file-buffer-p buf) (concoct-mail-buffer buf))
	  ((zwei:mail-summary-p buf) nil)
	  ((member (send buf :saved-major-mode)
		   '(zwei:common-lisp-mode zwei:zetalisp-mode) :test #'eq)
	   (concoct-lisp-buffer buf))
	  ((member (send buf :saved-major-mode)
		   '(zwei:dired-mode zwei:edit-buffers-mode) :test #'eq)
	   nil)
	  ((typep buf 'zwei:send-mail-buffer)
	   (concoct-buffer buf (let* ((zwei:*interval* buf))
				 (zwei:search (zwei:interval-first-bp buf)
					      #.(string-append #\Newline #\Newline)))))
	  (t (concoct-buffer buf)))))



(defvar *association-bounce* 0.1)

(defun random-related-word (count list)
  (let ((foll (random count)))
    (dolist (relation list)
      (if (< foll (cdr relation))
	  (return (car relation))
	  (decf foll (cdr relation))))))

(defun random-successor (word)
  (if (zerop (word-successor-count word))
      word
      (let ((next (random-related-word
		   (word-successor-count word)
		   (word-successors word))))
	(if (< (random 1.0) *association-bounce*)
	    (random-successor
	     (random-related-word
	      (word-predecessor-count next)
	      (word-predecessors next)))
	    next))))


(defun s ()
  (fresh-line)
  (do* ((word (aref db-index (random (length db-index))) (random-successor word))
	(string (word-string word) (word-string word))
	(beginning-of-sentence t nil)
	(max-width 75)
	(current-pos 0))
       (())
    (when (>= (+ current-pos (length string)) max-width)
      (terpri) (setq current-pos 0))
    (incf current-pos (1+ (length string)))
    (cond ((or beginning-of-sentence
	       (string= string "i")
	       (and (> (length string) 1)
		    (string= string "i'" :end1 2 :end2 2)))
	   (write-cap-string string))
	  (t
	   (write-string string)))
    (let ((punc (random (word-count word))))
      (cond ((< punc (word-comma-count word))
	     (write-string ", "))
	    ((< (decf punc (word-comma-count word)) (word-period-count word))
	     (write-string ".")
	     (return (values)))
	    ((< (decf punc (word-period-count word)) (word-quem-count word))
	     (write-string "?")
	     (return (values)))
	    ((< (decf punc (word-quem-count word)) (word-bang-count word))
	     (write-string "!")
	     (return (values)))
	    (t
	     (write-string " "))))
    (when (zerop (word-successor-count word))
      (return (values)))))

(defun write-cap-string (string)
  (write-char (char-upcase (schar string 0)))
  (write-string string *standard-output* :start 1))

(defvar *skip-prob* 0.5)

(defun transmorgify-string (s)
  (init-emit-word)
  (do ((index 0 (1+ index))
       (start nil)
       (length (length s)))
      ((= index length)
       (when start
	 (emit-word (nstring-downcase (subseq s start length)))))
    (let ((char (schar s index)))
      (cond (start
	     (cond ((or (alpha-char-p char) (eql char #\')))
		   (t
		    (emit-word (nstring-downcase (subseq s start index)))
		    (emit-char char)
		    (setq start nil))))
	    ((alpha-char-p char)
	     (setq start index))
	    (t
	     (emit-char char))))))

(defvar *last-word-emitted*)

(defun init-emit-word ()
  (setq *last-word-emitted* nil))

(defun emit-word (s)
  (let ((word (gethash s db)))
    (cond ((or (null word)
	       (< (random 1.0) *skip-prob*))
	   (if *last-word-emitted*
	       (write-string s)
	       (write-cap-string s)))
	  (t
	   (when *last-word-emitted*
	     (setq word (random-successor *last-word-emitted*)))
	   (if *last-word-emitted*
	       (write-string (word-string word))
	       (write-cap-string (word-string word)))))
    (when word
      (setq *last-word-emitted* word))))

(defun emit-char (c)
  (write-char c)
  (when (member c '(#\. #\! #\?))
    (init-emit-word)))

;;; Fancy window stuff.

#|

(defstruct (word-window (:constructor make-word-window (id function)))
  id
  function)

(defvar *windows* '())

(defparameter window-specs
  '((732 183 275 50 word-display-predecessor-string)
    (732 238 275 50 word-display-predecessor-string)
    (732 293 275 50 word-display-successor-string)
    (732 348 275 50 word-display-successor-string)
    (732 403 275 50 word-display-successor-string)
    (732 458 275 50 word-display-predsucc-string)
    (732 513 275 50 word-display-predsucc-string)
    (732 567 275 50 word-display-predsucc-string)
    (732 622 275 50 word-display-predsucc-string)))

(defun init-windows ()
  (unless *windows*
    (x:open-console)
    (init-font)
    (dolist (spec window-specs)
      (push (make-word-window
	     (x:xcreatewindow (x:rootwindow) (first spec) (second spec)
			      (third spec) (fourth spec)
			      2 (x:blackpixmap) (x:whitepixmap))
	     (fifth spec))
	    *windows*)
      (x:xmapwindow (word-window-id (car *windows*))))
    (setq *windows* (reverse *windows*))))

(defun deinit-windows ()
  (dolist (w *windows*)
    (x:xdestroywindow (word-window-id w)))
  (setq *windows* nil))

(defconstant null-clip-mask 0)

(defparameter font-name "nonb44")

(defvar *font* nil)
(defvar *font-info* nil)
(defvar *font-height* nil)

(defun init-font ()
  (unless *font*
    (setf *font-info* (x:xopenfont font-name))
    (setf *font* (system:alien-access (x:fontinfo-id *font-info*)))
    (setf *font-height* (system:alien-access (x:fontinfo-height *font-info*)))))

(defparameter left-border 2)
(defparameter top-border 4)

(defun display-word-funkily (string)
  (init-windows)
  (let ((word (gethash string db)))
    (dolist (w *windows*)
      (x:xpixfill (word-window-id w) 0 0 1024 *font-height* x:whitepixel
		  null-clip-mask x:gxcopy x:allplanes))
    (when word
      (dolist (w *windows*)
	(let ((string (funcall (word-window-function w) word)))
	  (x:xtext (word-window-id w) left-border top-border
		   string (length string)
		   *font* x:blackpixel x:whitepixel))))))

(defun word-display-string (word)
  (word-string word))

(defun word-display-predecessor-string (word)
  (if (zerop (word-predecessor-count word))
      ""
      (word-string
       (random-related-word (word-predecessor-count word) (word-predecessors word)))))

(defun word-display-successor-string (word)
  (if (zerop (word-successor-count word))
      ""
      (word-string
       (random-related-word (word-successor-count word) (word-successors word)))))

(defun word-display-predsucc-string (word)
  (if (zerop (word-successor-count word))
      ""
      (word-string (random-successor word))))

;;; Hemlock interface.

(in-package "HEMLOCK")

(defmode "Concoct" :transparent-p t :precedence 1.0)

(defcommand "Auto Concoction" (p)
  "Toggle \"Concoct\" mode in the current buffer."
  "Toggle \"Concoct\" mode in the current buffer."
  (declare (ignore p))
  (setf (buffer-minor-mode (current-buffer) "Concoct")
	(not (buffer-minor-mode (current-buffer) "Concoct"))))

(defcommand "Concoct Word" (p)
  "Banana." "Fish."
  (declare (ignore p))
  (unless (eq (last-command-type) :concoct)
    (let ((region (spell-previous-word (current-point) t)))
      (when region
	(conx::display-word-funkily (nstring-downcase (region-to-string region))))))
  (setf (last-command-type) :concoct))

(eval-when (eval load)
  (dolist (c (command-bindings (getstring "Self Insert" *command-names*)))
    (let ((ch (svref (car c) 0)))
      (unless (or (alpha-char-p ch) (char= ch #\'))
	(bind-key "Concoct Word" (car c) :mode "Concoct")))))

(defcommand "Reconcoct Word" (p)
  "Banana." "Fish."
  (declare (ignore p))
  (let ((region (spell-previous-word (current-point) t)))
    (when region
      (conx::display-word-funkily (nstring-downcase (region-to-string region)))))
  (setf (last-command-type) :concoct))

(bind-key "Reconcoct Word" #\Control-Return :mode "Concoct")

|#
