;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:TAPE -*-

;;; File "AUDIO-TAPE"
;;; Yow, am I hacking PostScript yet??
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 16 Jul 88  Jamie Zawinski   Created.
;;; 30 Jul 90  Jamie Zawinski   Re-implemented to use the new PostScript code.
;;;                             Added correct Adobe structuring comments.
;;;

(in-package "TAPE")

;;; What the casual user needs:
;;;
(export '(dump-tapes-to-file load-albums tapes-by *signature-string*))


;;; What the programmer needs:
;;;
(export '(album album-p make-album album-id album-artist album-name
	  album-year album-songs album-other-name double-album
	  double-album-p make-double-album album-songs2
	  tape tape-p make-tape tape-side-a tape-side-b
	  *album-table* *all-tapes*
	  lookup-album
	  ))


(defvar *tape-prolog-file*
  ;; Look for the highest numbered file of the form "audio-tape-N.N.ps"
  (let* ((dir (pathname-directory "/u/jwz/ps/"))
	 (template (make-pathname :directory dir :name :wild :type "ps"))
	 (all-files (directory template))
	 (candidates (delete-if-not
		       #'(lambda (x)
			   (setq x (pathname-name x))
			   (string= "audio-tape" x :end2 (min (length x) 10)))
		       all-files)))
    (car (sort candidates #'string> :key #'pathname-name)))
  "Where the PostScript program lives.")

(defvar *tape-prolog* nil
  "The PostScript support code; read in from *tape-prolog-file*.")


(defstruct (album (:print-function %print-album))
  (id nil :type (or null fixnum))
  (artist  "" :type string)
  (name    "" :type string)
  (year    0  :type (or string number))   ; have to do things like "1982-1986"
  (songs  '() :type list)
  (other-name nil :type (or null string))  ; sometimes 2 EPs on one side of a
  )					   ;  tape - this is the "other name".

(defstruct (double-album (:conc-name ALBUM-)
			 (:print-function %print-album)
			 (:include album))
  (songs2 '() :type list)
  )

(defstruct (tape (:print-function %print-tape))
  (side-a nil :type (or null fixnum))
  (side-b nil :type (or null fixnum))
  )

(defun %print-album (album stream ignore)
  (declare (ignore ignore))
  (princ #\< stream)
  (princ (type-of album) stream)
  (princ #\Space stream)
  (princ (album-artist album) stream)
  (princ #\Space stream)
  (prin1 (album-name album) stream)
  (when (album-other-name album)
    (princ ", " stream)
    (prin1 (album-other-name album) stream))
  (princ #\> stream)
  )

(defun %print-tape (tape stream ignore)
  (declare (ignore ignore))
  (let* ((a (tape-side-a tape))
	 (b (tape-side-b tape))
	 (titles '()))
    (setq a (lookup-album a))
    (when a
      (push (album-name a) titles)
      (push (album-other-name a) titles))
    (setq b (lookup-album b))
    (when b
      (push (album-name b) titles)
      (push (album-other-name b) titles))
    (setq titles (nreverse (delete-if #'null titles)))
    (format stream "#<~S ~S~{ ~S~}>" (type-of tape) (car titles)
	    (cdr titles))))


(defvar *signature-string* nil
  "One line that is printed on the back flap of the tape, so you can tell
 your tapes from other peoples.")



(defun dump-tape (tape &optional (stream *standard-output*)
		       &key band-font (album-font band-font)
		       name-only)
  (check-type tape tape)
  (dump-tape-1 tape stream band-font album-font name-only))


(defun dump-tape-1 (tape stream &optional band-font album-font name-only)
  (check-type tape tape)
  (let* ((a (lookup-album (tape-side-a tape)))
	 (b (lookup-album (tape-side-b tape)))
	 (double (cond ((double-album-p a)
			(assert (null b) ()
				"Double albums must be alone on a tape.")
			a)
		       ((double-album-p b)
			(assert (null a) ()
				"Double albums must be alone on a tape.")
			b)))
	 (same-artist (or double
			  (and a b
			       (string-equal (album-artist a)
					     (album-artist b)))))
	 (n-albums (or (and a (album-other-name a))
		       (and b (album-other-name b)))))
    (when double (setq a double b nil))
    (let* ((name (album-artist a)))
      (when same-artist
	(let ((c (assoc name *special-name-fonts* :test #'string-equal)))
	  (when c
	    (setq name (car c) ; get capitalization, too.
		  band-font (second c)))))
      (when name-only (setq name ""))

      (terpri stream)
      (when same-artist
	(when band-font
	  (format stream "band-font /band-font /~A def~%" band-font))
	(when album-font
	  (format stream "album-font /album-font /~A def~%" album-font)))

      (flet ((write-songs-of (album &optional songs2)
	       (format stream "[")
	       (let ((songs
		      (if songs2 (album-songs2 album) (album-songs album))))
		 (when (and album songs)
		   (format stream "[(~A) (~A)]~%"
			   (or (caar songs) "") (or (cdar songs) ""))
		   (dolist (song (cdr songs))
		     (format stream " [(~A) (~A)]~%"
			     (or (car song) "") (or (cdr song) ""))))
		 (format stream " ]~%")))
	     (write-name-etc (album &optional two)
	       (format stream "(~A) (~A) ~:[~*~;(~A) ~](~A)~%"
		       (if album
			   (if (eq album a) name (album-artist album))
			   "")
		       (if album (album-name album) "")
		       two
		       (or (and album (album-other-name album)) "")
		       (if album (album-year album) ""))))
	(cond (double
	       (write-name-etc a)
	       (write-songs-of a)
	       (write-songs-of a t)
	       (format stream "double-album~%"))

	      ((and same-artist n-albums)
	       (format stream "(~A)~%  (~A) (~A) (~A)~%  (~A) (~A) (~A)~%"
		       name (album-name a)
		       (or (album-other-name a) "") (album-year a)
		       (album-name b) (or (album-other-name b) "")
		       (album-year b))
	       (write-songs-of a)
	       (write-songs-of b)
	       (format stream "N-albums~%"))
	      
	      (n-albums
	       (write-name-etc a t)
	       (write-songs-of a)
	       (write-name-etc b t)
	       (write-songs-of b)
	       (format stream "two-bands-N-albums~%"))
	      
	      (same-artist
	       (format stream "(~A)~%  (~A) (~A)~%  (~A) (~A)~%"
		       name (album-name a) (album-year a)
		       (album-name b) (album-year b))
	       (write-songs-of a)
	       (write-songs-of b)
	       (format stream "two-albums~%"))
	      
	      (t
	       (write-name-etc a)
	       (write-songs-of a)
	       (write-name-etc b)
	       (write-songs-of b)
	       (format stream "two-bands~%"))))
      (when same-artist
	(when album-font (format stream "/album-font exch def~%"))
	(when band-font (format stream "/band-font exch def~%")))
      (terpri stream))))


(defun dump-postscript-prolog (stream &optional (no-comments-p t))
  (unless *tape-prolog*
    (let ((lines '())
	  (past-important-comments nil))
      (with-open-file (stream *tape-prolog-file* :direction :input)
	(loop
	  (let ((line (or (read-line stream nil nil) (return))))
	    (when (string= line "%%EndProlog")
	      (push line lines)
	      (return))
	    (when (and no-comments-p past-important-comments)
	      (let* ((L (length line))
		     (c0 (if (> L 0) (char line 0) #\Null))
		     (c1 (if (> L 1) (char line 1) #\Null))
		     (c2 (if (> L 2) (char line 2) #\Null)))
		(when (and (char= c0 #\%)
			   (not (and (char= c1 #\%)
				     (alpha-char-p c2))))
		  (setq line nil))
		(when (and line
			   (or (zerop L)
			       (string= line "%%BeginDocumentation")
			       (string= line "%%EndDocumentation")))
		  (setq line nil))))
	    (when line (push line lines))
	    (when (string= line "%%EndComments")
	      (setq past-important-comments t)))))
      (setq *tape-prolog* (nreverse lines))))
  (dolist (line *tape-prolog*)
    (write-line line stream))
  (dotimes (i 3)
    (format stream "%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~%"))
  )



(defun dump-tapes-to-file (file tapes &key band-font (album-font band-font))
  (unless (consp tapes) (setq tapes (list tapes)))
  (with-open-file (stream file :direction :output)
    (dump-postscript-prolog stream)
    (format stream "~2&/signature (~A) def~2%" (or *signature-string* ""))
    (do* ((rest tapes (cdr rest))
	  (count 0 (1+ count)))
	 ((null rest)
	  (format stream "~2%tick { showpage } if~2%%%Trailer~%%%Pages: ~D~%"
		  (1+ (floor count 2))))
      (when (evenp count)
	(format stream "%%Page: ~D ~:*~D~%" (1+ (floor count 2))))
      (let* ((tape (car rest)))
	(dump-tape tape stream :band-font band-font :album-font album-font)
	))
    (truename stream)))


;;; parsing the database file

(defvar *read-buffer* (make-array 100 :element-type 'string-char
				  :fill-pointer 0))

(defun read-token (stream &optional (squeeze-blanklines t)
			  (squeeze-leadingspace t))
  (setf (fill-pointer *read-buffer*) 0)
  (let* ((any-input nil)
	 (got-white nil)
	 (eol nil))
    (do* ((done nil)
	  (ch (read-char stream nil nil) (read-char stream nil nil)))
	 ((or (null ch) done))
      (setq eol nil)
      (cond ((char= ch #\Tab)
	     (when any-input (return)))
	    ((char= ch #\Newline)
	     (setq eol t)
	     (cond (squeeze-blanklines
		    (when any-input (return)))
		   (t (setq any-input t)
		      (return))))
	    ((char= ch #\Space)
	     (cond (got-white (if any-input
				  (return)
				  (vector-push-extend ch *read-buffer*)))
		   (t
		    (vector-push-extend ch *read-buffer*)
		    (setq got-white t))))
	    (t (vector-push-extend ch *read-buffer*)
	       (setq any-input t)
	       (setq got-white nil))))
    (if any-input
	(values (if squeeze-leadingspace
		    (string-trim '(#\Space #\Tab #\Return) *read-buffer*)
		    (string-right-trim '(#\Space #\Tab #\Return)
				       *read-buffer*))
		eol)
	(values nil eol))))

(defun read-tape-line (stream)
  (multiple-value-bind (song eol) (read-token stream nil nil)
    (if (string= "---" song :end2 (min 3 (length song)))
	nil
	(let* ((time (unless eol (read-token stream nil t))))
	  (values song time)))))


(defvar *album-table* (make-hash-table :test #'eql))


(defun read-album (&optional (stream *standard-input*))
  (let* ((id (read-token stream))
	 (artist (read-token stream nil))
	 (album (read-token stream nil))
	 other-album-title
	 (year (read-token stream nil)))
    (unless (every #'(lambda (x) (or (digit-char-p x) (char= x #\-))) year)
      (setq other-album-title year
	    year (read-token stream nil)))
    (let* ((rest-of-line (when id
			   (if (char= (peek-char nil stream) #\Newline)
			       ""
			       (string-trim '(#\Space #\Tab)
					    (read-line stream)))))
	   (double-p (and id
			  (string-equal "double" rest-of-line
					:end2 (min 6 (length rest-of-line)))))
	   (lines '())
	   (lines-2 '()))
      (when id
	(setq id (parse-integer id))
	(check-type id integer)
	(when (equal "-" artist) (setq artist ""))
	(when (equal "-" album) (setq album ""))
	(when (equal "-" year) (setq year ""))
	(let* ((got-one nil))
	  (loop
	    (multiple-value-bind (song time) (read-tape-line stream)
	      (unless song (return))
	      (when (or got-one (string/= song ""))
		(push (cons song time) lines)
		(setq got-one t)))))
	(when double-p
	  (let* ((got-one nil))
	    (loop
	      (multiple-value-bind (song time) (read-tape-line stream)
		(unless song (return))
		(when (or got-one (string/= song ""))
		  (push (cons song time) lines-2)
		  (setq got-one t))))))
	(let* ((album
		 (if (not double-p)
		     (make-album :id id  :artist artist :name album
				 :other-name other-album-title :year year
				 :songs (nreverse lines))
		     (make-double-album  :id id  :artist artist
					 :name album :year year
					 :songs (nreverse lines)
					 :songs2 (nreverse lines-2)))))
	  (setf (gethash id *album-table*) album)
	  album)))))


(defvar *all-tapes* '())


(defun read-tape-specs (stream)
  (loop
    (let* (a b eol)
      (multiple-value-setq (a eol) (read-token stream))
      (unless eol
	(setq b (read-token stream nil)))
      (when (or (null a) (string= "---" a :end2 (min (length a) 3)))
	(return))
      (when (string= b "") (setq b nil))
      (when a (setq a (parse-integer a)))
      (when b (setq b (parse-integer b)))
      (push (make-tape :side-a a :side-b b) *all-tapes*)))
  (setq *all-tapes* (nreverse *all-tapes*))
  nil)


(defun load-albums (file)
  (clrhash *album-table*)
  (setq *all-tapes* nil)
  (with-open-file (stream file :direction :input)
    (read-tape-specs stream)
    (loop
      (unless (read-album stream) (return))))
  nil)


(defun lookup-album (id)
  (gethash id *album-table*))


(defun tapes-by (substring)
  (let* ((result nil))
    (dolist (tape *all-tapes*)
      (let* ((a (lookup-album (tape-side-a tape)))
	     (b (lookup-album (tape-side-b tape))))
	(when (or (and a (search substring (album-artist a)
				 :test #'char-equal))
		  (and b (search substring (album-artist b)
				 :test #'char-equal)))
	  (push tape result))))
    (nreverse result)))


;;; Special fonts.

(defvar *special-name-fonts*
	'(("PROPAGANDA" "Times-Roman")
	  ("INXS" "Times-Roman")
	  ("the FIXX" "Times-Roman")
	  ("THE ART OF NOISE" "Times-Roman")
	  ("bauhaus" "Courier-Bold")
	  )
  "The band names that get printed normally, but in different fonts.")
