;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:(ANDREW); Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONTB HL12B HL12I) -*-

;1;; File 3"READ-ANDREW-DATASTREAM"**
;1;; Reading and displaying Andrew BE2 DataStream multimedia files.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;   28 Feb 89*	1Jamie Zawinski*	1Created.*
;1;;*  1 3 Nov 89*	1Jamie Zawinski*	1Made it read the whole datastream, not just the rasters.*
;1;; *  15 Nov 89*	1Jamie Zawinski *	1Added code for dumping Text with embedded Rasters into a Zmacs buffer.*
;1;; *  16 Nov 89*	1Jamie Zawinski *	1Added code for parsing a .MS_MsgDir file.*
;1;;*  128 Nov 89*	1Jamie Zawinski *	1Got the Date field of a MsgDir message parsing properly.  It's base 64, can you believe that?*
;1;;*   16 Mar 90*	1Jamie Zawinski *	1Added mouse commands.*
;1;;*  120 Mar 90*	1Jamie Zawinski *	1Lots of good stuff.  Profile saving, profile restoring, correctly marking last-message-read, expand-all,*
;1;;*				1 unsubscribe, etc.*
;1;;*  129 Mar 90*	1Jamie Zawinski *	1Added 5fad* objects (animations).*
;1;;*

(defpackage 4"ANDREW"*
  (:export read-datastream-file visit-datastream-file visit-messages-file
	   ))

#+KSL (eval-when (load eval compile)
	(shadow '("3DEFSTRUCT*") "3ANDREW*")
	(sys:fdefine (find-symbol "3DEFSTRUCT*" "3ANDREW*") 'sys:defstruct-safe t))



(defun 4read-datastream-file *(pathname &optional skip-message-headers-p)
  "2Reads and returns a top-level datastream object from the given file. 
   If SKIP-MESSAGE-HEADERS-P is T, then lines in the file before the first blank line will be discarded first.*"
  (sys:with-open-decompressing-stream (stream pathname :direction :input :characters t)
;  (with-open-file (stream pathname :direction :input :characters t)
    (when (= #xF1 (char-code (peek-char nil stream)))
      (warn "2This is a binary raster file, not a datastream file.*")
      (return-from READ-DATASTREAM-FILE (read-andrew2-bitmap pathname)))
    (when skip-message-headers-p
      (do* () ((string= "" (read-line stream)))))
    (let* ((ds (make-instance 'datastream :input-stream stream))
	   (object (send ds :read)))
      (send ds :set-input-stream nil) ;1 so it can be GCed.*
      object)))


(defun 4visit-datastream-file *(pathname &optional skip-message-headers-p)
  "2Reads a datastream file into a Zmacs buffer.  Returns the buffer, and the datastream object.*"
  (let* ((buffer-name (string-append (second (multiple-value-list (zwei:editor-file-name pathname))) "3 (Andrew)*"))
	 (object (read-datastream-file pathname skip-message-headers-p))
	 (buffer (zwei:find-buffer-named buffer-name t)))
    (zwei:delete-interval buffer)
    (zwei:with-bp (bp (zwei:interval-first-bp buffer) :moves)
      (send object :grind-to-buffer buffer bp))
    (values buffer object)))


(defun 4visit-messages-file *(pathname &optional into-buffer)
  "2Reads a Messages datastream file into a Zmacs buffer, fontifying its headers.  
  If buffer is not supplied, one is created and returned.*"
  (unless into-buffer
    (let* ((buffer-name (string-append (second (multiple-value-list (zwei:editor-file-name pathname))) "3 (Andrew)*")))
      (setq into-buffer (zwei:find-buffer-named buffer-name t))))
  (send into-buffer :set-attribute :fonts '(fonts:hl12b fonts:hl12i) nil)
  (zwei:set-buffer-fonts into-buffer)
  (send into-buffer :set-major-mode 'ZWEI:TEXT-MODE)
  (sys:with-open-decompressing-stream (stream pathname :direction :input :characters t)
;  (with-open-file (stream pathname :direction :input :characters t)
    (zwei:delete-interval into-buffer)
    (zwei:with-bp (bp (zwei:interval-first-bp into-buffer) :moves)
      (loop
	(let* ((line (read-line stream)))
	  (when (string= "" line) (return (zwei:insert-moving bp #\Newline)))
	  (let* ((cont-p (or (char= (char line 0) #\Space)
			     (char= (char line 0) #\Tab)))
		 (colon (and (not cont-p) (position #\: line :test #'char=))))
	    (unless colon (setq cont-p t))
	    (cond (cont-p
		   (zwei:insert-moving bp (zwei:in-current-font line 1)))
		  (t (zwei:insert-moving bp (subseq line 0 (1+ colon)))
		     (zwei:insert-moving bp (zwei:in-current-font (subseq line (1+ colon)) 1))))
	    (zwei:insert-moving bp #\Newline))))
      (let* ((ds (make-instance 'datastream :input-stream stream))
	     (object (send ds :read)))
	(send ds :set-input-stream nil) ;1 so it can be GCed.*
	(send object :grind-to-buffer into-buffer bp)
	(zwei:make-buffer-read-only into-buffer)
	(values into-buffer object)))))


(defflavor 4datastream*
	   ((table '())
	    (input-stream nil)
	    (current-object nil)
	    )
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defun 4read-datastream-keyword* (stream &optional delimiters)
  "2Assumes stream is positioned just after a backslash.  Returns a keyword.  
  Leaves stream just before the first space, tab, newline, or open-curly found.*"
  (multiple-value-bind (string ignore delim)
		       (read-delimited-string (or delimiters '(#\{ #\Space #\Tab #\Newline)) stream)
    (unread-char delim stream)
    (intern (nstring-upcase string) *keyword-package*)))

(defvar 4*datastream-readtable** (copy-readtable)
  "2A readtable which treats brackets as parens and commas as whitespace.*")

(defun 4\{-reader* (stream &optional ignore ignore) (read-delimited-list #\} stream t))
(defun 4\[-reader* (stream &optional ignore ignore) (read-delimited-list #\] stream t))

(set-macro-character #\{ '\{-reader nil *datastream-readtable*)
(set-macro-character #\[ '\[-reader nil *datastream-readtable*)
(set-macro-character #\} (get-macro-character #\)) nil *datastream-readtable*)
(set-macro-character #\] (get-macro-character #\)) nil *datastream-readtable*)
(set-syntax-from-char #\, #\Space *datastream-readtable*)


(defmacro 4datastream-read-context* (&body body)
  "2Bind *READTABLE* to a readtable which treats brackets as parens and commas as whitespace, and bind *PACKAGE* to KEYWORD.*"
  `(let* ((*readtable* *datastream-readtable*)
	  (*package* *keyword-package*))
     ,@body))

(defmacro 4datastream-read *(&rest read-args)
  `(datastream-read-context (read ,@read-args)))

(defmethod 4(datastream :read*) (&optional command)
  "2Parse and store one datastream object (and its sub-objects).  Returns NIL.*"
  (block EOF
    (unless command
      (peek-char t input-stream)
      (let* ((c (read-char input-stream))
	     (c2 (peek-char nil input-stream)))
	(assert (char= c #\\) () "2First nonwhite character should have been a backslash.*")
	(assert (char/= c2 #\\) () "2Encountered double-backslash when ``\\begindata'' expected.*"))
      (setq command (read-datastream-keyword input-stream)))
    (peek-char t input-stream)
    (assert (char= #\{ (peek-char t input-stream)) () "2The first nonwhite character after \\~A was ~S, instead of #\\{.*"
	    command (peek-char t input-stream))
    (ecase command
      (:BEGINDATA
       (let* ((begindata-args (datastream-read input-stream))
	      (data-name (first begindata-args))
	      (data-id (second begindata-args))
	      (old-obj current-object)
	      (object (send self :instantiate-object data-name data-id)))
	 (send object :read-self input-stream)
	 (setq current-object old-obj)
	 object))
      (:VIEW
       (let* ((view-args (datastream-read input-stream))
	      (object-name (first view-args))
	      (object-id (second view-args)) ;1 3 others...*
	      )
	 (format t "2~&Viewing ~A ~A*" object-name object-id)))
      (:ENDDATA
       (error "2Superfluous \\enddata encountered!*"))
      )))


(defvar 4*datastream-type-mappings**
  '((:text . text)
    (:text822 . text)
    (:raster . raster)
    (:fad . fad)
    ))

(defmethod 4(datastream :instantiate-object*) (name id)
  "2Create a new datastream obect of the flavor appropriate to NAME and ID.  Make it be the current object, and store it in the object table.*"
  (assert (not (member id table :key #'car :test #'=)) ()
	  "2There is already and object with ID ~D: ~S*" id (cdr (assoc id table :test #'=)))
  (let* ((type (or (cdr (assoc name *datastream-type-mappings* :test #'eq))
		   'UNKNOWN-DATASTREAM-OBJECT))
	 (new (make-instance type :name name :id id :superior current-object :datastream self)))
    (when current-object (push new (send current-object :inferiors)))
    (push (cons id new) table)
    (setq current-object new)
    new))


;1;; Objects within the datastream.*


(defflavor 4basic-datastream-object*
  ((id -99)
   (name :unknown)
   (datastream nil)
   (superior nil)
   (inferiors '()))
  ()
  (:required-methods :read-self)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(basic-datastream-object :print-self)* (stream ignore slashify)
  (if slashify
      (format stream "3#<~A ~(~S~) ~D>*" (type-of self) name id)
      (format stream "3~(~A~) ~D*" name id)))


(defmethod 4(basic-datastream-object :grind-to-buffer)* (interval bp &optional (current-x 0) viewtype)
  (declare (ignore interval current-x))
  (zwei:insert-moving bp (format nil "2<< A ~A object would have been here. >>*" viewtype))
  bp)


(defflavor 4unknown-datastream-object *() (basic-datastream-object)
  (:documentation :flavor "2A kind of datastream object that discards everything up to the next matching \\enddata.*")
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(unknown-datastream-object :read-self*) (stream)
  (warn "2Reading unknown form ~A ~A*" name id)
  (let* ((saw-backslash nil))
    (block DONE
     (loop
      (let* ((c (read-char stream)))
	(cond ((char= c #\\)
	       (if saw-backslash
		   (setq saw-backslash nil)
		   (setq saw-backslash t)))
	      (saw-backslash
	       (setq saw-backslash nil)
	       (unread-char c stream)
	       (do* ((target "3enddata*")
		     (l (length target))
		     (i 0 (1+ i)))
		   ((>= i l)
		    (let* ((args (datastream-read stream)))
		      (when (and (consp args) (eq name (first args)) (eql id (second args)))
			(peek-char t stream) ;1 ditch trailing whitespace.*
			(return-from DONE t))))
		 (let* ((c2 (read-char stream)))
		   (unless (char-equal c2 (char target i))
		     (unread-char c2 stream)
		     (return nil)))))))))))


;1;; Text datastream objects.*


(defflavor 4text*
  ((version 0)
   (template nil)
   (known-attributes '())
   (open-attributes '())
   (lines '())
   )
  (basic-datastream-object)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(text :print-self)* (stream ignore slashify)
  (if slashify
      (format stream "3#<~A ~(~A~), ~D line~:P ~D>*" (type-of self) template (length lines) id)
      (format stream "3~(~A~) ~(~A~), ~D line~:P ~D*" (type-of self) template (length lines) id)))


(defstruct 4(text-attr *(:print-function %print-text-attr))
  (name nil :type symbol)
  menu
  fontfamily fontsize fontface
  leftmargin rightmargin justification
  indent script
  flags)

(defun 4%print-text-attr *(struct stream ignore)
  (format stream "3#<~S ~A~:(~{ ~A~}~)>*"
	  (type-of struct) (text-attr-name struct) (text-attr-menu struct)))


(defvar 4*default-attributes**
  (macrolet ((ta (name &rest args)
	       `(make-text-attr :name ',name ,@(mapcar #'(lambda (x) (list 'quote x)) args))))
    (list
      ;1;*
      ;1; On the 5Font* menu...*
      ;1;*
      (TA :ITALIC :fontface (:Italic :int :set) :menu (:Font Italic))
      (TA :BOLD :fontface (:Bold :int :set) :menu (:Font :Italic))
      (TA :TYPEWRITER :fontface (:FixedFace :int :set) :fontfamily (:AndyType :int 0) :menu (:Font :TypeWriter))
      (TA :BIGGER :fontsize (:PreviousFontSize :point 2) :menu (:Font :Bigger))
      (TA :SMALLER :fontsize (:PreviousFontSize :point -2) :menu (:Font :Smaller))
      (TA :SUPERSCRIPT :script (:PreviousScriptMovement :point -393216) :fontsize (:PreviousFontSize :point -2)
	  :menu (:Font :SuperScript))
      (TA :SUBSCRIPT :script (:PreviousScriptMovement :point 131072) :fontsize (:PreviousFontSize :point -2)
	  :menu (:Font :SubScript))
      ;1;*
      ;1; On the 5Region* menu...*
      ;1;*
      (TA :INDENT :leftmargin (:LeftMargin :inch 32768) :rightmargin (:RightMargin :inch 32768) :menu (:Region :Indent))
      (TA :LEFTINDENT :leftmargin (:LeftMargin :inch 32768) :menu (:Region :LeftIndent))
      (TA :DISPLAY :leftmargin (:LeftMargin :inch 32768) :rightmargin (:RightMargin :inch 32768) 
	  :justification (:LeftJustified :point 0) :menu (:Region :Display))
      (TA :EXAMPLE :leftmargin (:LeftMargin :inch 32768) :justification (:LeftJustified :point 0) 
	  :fontface (:FixedFace :int :set) :fontfamily (:AndyType :int 0) :menu (:Region :Example))
      (TA :DESCRIPTION :leftmargin (:LeftMargin :inch 32768) :indent (:LeftEdge :inch -32768) :menu (:Region :Description))
      (TA :QUOTATION :leftmargin (:LeftMargin :inch 32768) :rightmargin (:RightMargin :inch 32768) 
	  :fontface (:Italic :int :set) :menu (:Region :Quotation))
      (TA :FORMATNOTE :flags (:PassThru :int :set) :menu (:Region FormatNote))
      ;1;*
      ;1; On the 5Justify* menu...*
      ;1;*
      (TA :FLUSHLEFT :justification (:LeftJustified :point 0) :menu (:Justify :FlushLeft))
      (TA :FLUSHRIGHT :justification (:RightJustified :point 0) :menu (:Justify :FlushRight))
      (TA :CENTER :justification (:Centered :point 0) :menu (:Justify :Center))
      ;1;*
      ;1; On the 5Title* menu...*
      ;1;*
      (TA :MAJORHEADING :justification (:Centered :point 0) :fontsize (:PreviousFontSize :point 4)
	  :menu (:Title :MajorHeading))
      (TA :HEADING :justification (:Centered :point 0) :Fontface (:Bold :int :set) :menu (:Title :Heading))
      (TA :SUBHEADING :justification (:LeftJustified :point 0) :fontface (:Bold :int :set) :menu (:Title :SubHeading))
      (TA :CHAPTER :justification (:LeftJustified :point 0) :fontsize (:PreviousFontSize :point 4) :menu (:Title :Chapter))
      (TA :SECTION :justification (:LeftJustified :point 0) :fontsize (:PreviousFontSize :point 2) :menu (:Title :Section))
      (TA :SUBSECTION :justification (:LeftJustified :point 0) :fontface (:Bold :int :set) :menu (:Title :SubSection))
      (TA :PARAGRAPH :justification (:LeftJustified :point 0) :fontface (:Italic :int :set) :menu (:Title :Paragraph))
      ))
  "2The defintion of the environments for the ``Default'' template; we probably don't need the others.*")



(defvar 4*datastream-text-reader-buffer** (make-array 4096 :element-type 'string-char :fill-pointer 0))

(defmethod 4(text :read-self*) (stream)
  (let* ((buffer *datastream-text-reader-buffer*)
	 (saw-backslash nil))
    (declare (string buffer))
    (setf (fill-pointer buffer) 0)
    (loop
     (let* ((c (read-char stream)))
       (cond ((char= c #\\)
	      (cond (saw-backslash (vector-push-extend #\\ buffer)
				   (setq saw-backslash nil))
		    (t (setq saw-backslash t))))
	     ((char= c #\})
	      (cond (saw-backslash (vector-push-extend #\} buffer)
				   (setq saw-backslash nil))
		    (t (cond ((pop open-attributes)
			      (unless (zerop (fill-pointer buffer))
				(push (copy-seq buffer) lines)
				(setf (fill-pointer buffer) 0))
			      (push :ENV-POP lines))
			     (t (vector-push-extend #\} buffer))))))
	     ((char= c #\{)
	      (vector-push-extend #\{ buffer)
	      (setq saw-backslash nil))
	     
	     ((char= c #\Newline)
	      (unless saw-backslash (vector-push-extend #\Newline buffer))
	      (setq saw-backslash nil)
	      (push (copy-seq buffer) lines)
	      (setf (fill-pointer buffer) 0))
	     (t
	      (cond (saw-backslash ;1 we are about to read a keyword.  Output what we have.*
		     (setq saw-backslash nil)
		     (unless (zerop (fill-pointer buffer))
		       (push (copy-seq buffer) lines)
		       (setf (fill-pointer buffer) 0))
		     (unread-char c stream)
		     (let* ((keyword (read-datastream-keyword stream)))
		       (case keyword
			 (:TEXTDSVERSION
			  (setf version (car (datastream-read stream))))
			 (:TEMPLATE
			  (setf template (car (datastream-read stream)))
			  (unless (eq template :DEFAULT) (warn "2Template ~A unknown; using DEFAULT.*" template)))
			 (:DEFINE
			  (peek-char t stream)
			  (assert (char= #\{ (read-char stream)) () "2First nonwhite character after \\define must be {*")
			  (let* ((defining-name (read-datastream-keyword stream)))
			    (send self :read-define defining-name stream)))
			 (:BEGINDATA
			  (send datastream :read keyword))
			 (:VIEW
			  (unless (zerop (fill-pointer buffer))
			    (push (copy-seq buffer) lines)
			    (setf (fill-pointer buffer) 0))
			  (let* ((view-args (datastream-read stream))
				 (view-name (first view-args))
				 (object-id (second view-args)) ; 3 others...
				 (object (cdr (assoc object-id (send datastream :table))))
				 (next-char (progn (peek-char t stream) (read-char stream)))
				 (char-after-that (peek-char nil stream))
				 (no-line-advance-p (and (char= next-char #\\) (char= char-after-that #\Newline))))
			    (unless no-line-advance-p (unread-char next-char stream))
			    (setq object (if no-line-advance-p
					     (list :view view-name object :no-line-advance)
					     (list :view view-name object)))
			    (push object lines)
			    (peek-char t stream) ;1 lose trailing whitespace.*
			    ))
			 (:ENDDATA
			  (let* ((list (datastream-read stream))
				 (end-name (first list))
				 (end-id (second list)))
			    (assert (and (eq end-name name) (eql end-id id)) ()
				    "2\\begindata{~A, ~A} closed by \\endata{~A, ~A}*" name id end-name end-id)
			    (push (copy-seq buffer) lines)
			    (setq lines (nreverse lines))
			    (return)))
			 (t
			  (let* ((attr (or (find keyword known-attributes :test #'eq :key #'text-attr-name)
					   (find keyword (the list *default-attributes*) :test #'eq :key #'text-attr-name))))
			    (assert attr (attr) "2Unknown attribute ~A*" keyword)
			    (push attr open-attributes)
			    (push attr lines) ;1 ## ack.*
			    (peek-char t stream)
			    (assert (char= #\{ (read-char stream)) () "2First nonwhite char after \\~A must be ~{*" keyword)
			    )))))
		    (t (vector-push-extend c buffer))))))
     )))


(defmethod 4(text :read-define*) (defining-name stream)
  "2Read a definition of a new text environment.*"
  (let* ((attr (make-text-attr :name defining-name)))
    (loop
     (when (char= (peek-char t stream) #\})
       (read-char stream)
       (peek-char t stream)
       (return))
     (let* ((def-param (read-datastream-keyword stream '(#\: #\Space #\Tab #\Newline #\{))))
       (peek-char t stream)
       (assert (char= #\: (read-char stream)) () "2Expected a colon after ~A*" def-param)
       (peek-char t stream)
       (assert (char= #\[ (read-char stream)) () "2Expected a [ after ~A:*" def-param)
       (let* ((param-list (datastream-read-context (read-delimited-list #\] stream))))
	 (case def-param
	   (:ATTR
	    (let* ((key (car param-list))
		   (val (cdr param-list)))
	      (case key
		(:FONTFAMILY    (setf (text-attr-fontfamily attr)    val))
		(:FONTSIZE      (setf (text-attr-fontsize attr)      val))
		(:FONTFACE      (setf (text-attr-fontface attr)      val))
		(:LEFTMARGIN    (setf (text-attr-leftmargin attr)    val))
		(:RIGHTMARGIN   (setf (text-attr-rightmargin attr)   val))
		(:JUSTIFICATION (setf (text-attr-justification attr) val))
		(:INDENT        (setf (text-attr-indent attr)        val))
		(:SCRIPT        (setf (text-attr-script attr)        val))
		(:FLAGS         (setf (text-attr-flags attr)         val))
		(t (warn "3Unknown parameter ~A with value ~A*" key val)))))
	   (:MENU (setf (text-attr-menu attr) param-list))
	   (t (warn "2Unknown \\define parameter:3~( ~A[~{~A ~}]~)**" def-param param-list))))))
    (push attr known-attributes)
    attr))


(defun 4atk-distance-to-points *(unit number)
  (ecase unit
    ((:rawdot :rawdots) number)
    ((:inch :inches)    (* 72 (ash number -15)))
    ((:CM :CMs)         (* 28 (ash number -15)))
    ((:point :points)   number)
    ((:EM :EMs)         0) ;1 * /* amt * 10? */
    ((:line :lines)     0) ;1 * /* amt * 10? */
    ))


(defun 4extract-leftmargin *(environment-list)
  (loop
    (let* ((env (pop environment-list)))
      (unless env (return 0))
      (let* ((val (text-attr-leftmargin env)))
	(when val
	  (ecase (car val)
	    (:LeftMargin
	     (return
	       (+ (atk-distance-to-points (cadr val) (caddr val))
		  (or (extract-leftmargin environment-list)
		      0))))))))))

(defun 4extract-rightmargin *(environment-list)
  (loop
    (let* ((env (pop environment-list)))
      (unless env (return 0))
      (let* ((val (text-attr-rightmargin env)))
	(when val
	  (ecase (car val)
	    (:RightMargin
	     (return
	       (+ (atk-distance-to-points (cadr val) (caddr val))
		  (or (extract-rightmargin environment-list)
		      0))))))))))

(defun 4extract-fontfamily *(environment-list)
  (let* ((attr (find-if-not #'null environment-list :key #'text-attr-fontfamily)))
    (if attr
	(car (text-attr-fontfamily attr))
	:AndyType)))

(defun 4extract-fontsize *(environment-list)
  (loop
    (let* ((env (pop environment-list)))
      (unless env (return 10))
      (let* ((val (text-attr-fontsize env)))
	(when val
	  (ecase (car val)
	    (:ConstantFontSize (return (atk-distance-to-points (second val) (third val))))
	    (:PreviousFontSize (return (+ (atk-distance-to-points (second val) (third val))
					  (extract-fontsize environment-list))))
	    ))))))

(defun 4extract-fontface *(environment-list)
  (loop
    (let* ((env (pop environment-list)))
      (unless env (return nil))
      (let* ((val (text-attr-fontface env))
	     (clearp (eq (third val) :CLEAR)))
	(when val
	  (let* ((others (extract-fontface environment-list)))
	    (if clearp
		(setq others (delete (car val) others :test #'eq))
		(pushnew (car val) others :test #'eq))
	    (return others)))))))


(defstruct 4fontfamily*
  name
  size-alist)


(defvar 4*fontfamilies**
	(list
	  (make-fontfamily :name :HELVETICA
			   :size-alist 'fonts:((6  hl6  hl6   hl6   hl6    tvfont  tvfont   tvfont   tvfont)
					       (7  hl7  hl7   hl7   hl7    tvfont  tvfont   tvfont   tvfont)
					       (12 hl12 hl12b hl12i hl12bi cptfont cptfontb cptfonti cptfontbi)
					       (14 hl12 hl12b hl12i hl12bi medfnt  medfntb  medfnt   medfntb)))
	  (make-fontfamily :name :TIMES
			   :size-alist 'fonts:((8  tr8  tr8   tr8i  tr8i   tvfont  tvfont   tvfont   tvfont)
					       (9  tr10 tr10b tr10i tr10bi cptfont cptfontb cptfonti cptfontbi)
					       (10 tr12 tr12b tr12i tr12bi cptfont cptfontb cptfonti cptfontbi)
					       (18 tr18 tr18b tr18  tr18b  medfnt  medfntb  medfnt   medfntb)))
	  (make-fontfamily :name :ANDY
			   :size-alist 'fonts:((8  tr8  tr8   tr8i  tr8i   tvfont  tvfont   tvfont   tvfont)
					       (9  tr10 tr10b tr10i tr10bi cptfont cptfontb cptfonti cptfontbi)
					       (10 tr12 tr12b tr12i tr12bi cptfont cptfontb cptfonti cptfontbi)
					       (18 tr18 tr18b tr18  tr18b  medfnt  medfntb  medfnt   medfntb)))
	  (make-fontfamily :name :ANDYTYPE
			   :size-alist 'fonts:((8  tr8  tr8   tr8i  tr8i   tvfont  tvfont   tvfont   tvfont)
					       (9 tr10 tr10b tr10i tr10bi cptfont cptfontb cptfonti cptfontbi)
					       (10 tr12 tr12b tr12i tr12bi cptfont cptfontb cptfonti cptfontbi)
					       (18 tr18 tr18b tr18  tr18b  medfnt  medfntb  medfnt   medfntb)))
	  ))


(defun 4fontfamily-lookup *(fontfamily-name size faces)
  (setq faces (sort (copy-list faces) #'string<))
  (let* ((fontfamily (or (find fontfamily-name *fontfamilies* :test #'eq :key #'fontfamily-name)
			 (car *fontfamilies*)))
	 (list (cdr (or (find-if #'(lambda (x) (<= size (car x)))
				 (fontfamily-size-alist fontfamily))
			(car (last (fontfamily-size-alist fontfamily)))
			)))
	 (font (cond ((null faces)				(first list))
		     ((equal faces '(:bold))			(second list))
		     ((equal faces '(:italic))			(third list))
		     ((equal faces '(:bold :italic))		(fourth list))
		     ((equal faces '(:fixedface))		(fifth list))
		     ((equal faces '(:fixedface :bold))		(sixth list))
		     ((equal faces '(:fixedface :italic))	(seventh list))
		     ((equal faces '(:bold :fixedface :italic)) (eighth list))
		     (t (error "3Invalid faces: ~s*" faces)))))
    (or font (find-if-not #'null list))))


(defun 4extract-font *(environment-list)
  (let* ((family-name (extract-fontfamily environment-list))
	 (size (extract-fontsize environment-list))
	 (face (extract-fontface environment-list)))
    (fontfamily-lookup family-name size face)))


(defun 4white-p* (char)
  (declare (character char))
  (or (char= char #\Space) (char= char #\Tab) (char= char #\Newline)))

(defmacro 4word-map* ((string start end next-nonwhite &optional final-form) &body body)
  "2Iterate over the words in string, binding START and END to the start and end of each word, and NEXT-NONWHITE to the 
  position in the string of the next nonwhite character after the word (or NIL if there are none).*"
  (let* ((s (gensym))
	 (l (gensym)))
    `(let* ((,s ,string)
	    (,l (length ,s))
	    (,start 0)
	    ,end ,next-nonwhite)
       (loop
	 (setq ,end (or (position-if #'white-p ,s :start (if (zerop ,start)
							     (or (position-if-not #'white-p ,s :start 0) 0)
							     ,start))
			,l)
	      ,next-nonwhite (position-if-not #'white-p ,s :start ,end))
	,@body
	(setq ,start (or ,next-nonwhite (return ,final-form)))))))


(defmethod 4(text :grind-to-buffer)* (interval bp &optional (current-x 0) ignore)
  (let* ((leftmargin 0)
	 (rightmargin 0)
	 (environments '())
	 (window (or (find interval zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))
		     tv:default-screen))
	 (winwidth (if (eq window tv:default-screen)
		       800
		       (- (tv:sheet-inside-width window) 20)))
	 
	 (last-pixels-from-left-margin current-x)
	 (queued-pixels-from-left-margin 0)
	 (last-chars-within-string 0)
	 (queued-chars-within-string 0)
	 (current-font :no-font-yet)
	 )
    (labels ((4dump-queued-words* (string break-p)
	       (unless (string= "" string)
		 (let* ((substring (subseq string last-chars-within-string queued-chars-within-string)))
		   (when (< last-pixels-from-left-margin leftmargin)
		     (zwei:move-bp bp (zwei:indent-to bp leftmargin)))
		   ; ## no no - the #\newline at the end of lines is ignored.  Just like Scribe.
		   ; ## (setq break-p (char= #\Newline (char string (1- (length string)))))
		   (setq substring (string-right-trim '(#\Newline) substring))
		   (let* ((fontnum (position current-font (send interval :get-attribute :fonts))))
		     (unless fontnum (error "3Font ~S not found in interval's fonts ~S*" current-font
					    (send interval :get-attribute :fonts)))
		     (let* ((fonted-string (zwei:in-current-font substring fontnum)))
		       ;1; Nuke the tabs back down to font #0.*
		       (dotimes (i (length fonted-string))
			 (cond ((char-equal #\Tab (char fonted-string i))
				(setf (char fonted-string i) #\Tab))
			       ;; ##
			       ((char-equal #\Newline (char fonted-string i))
				(setf (char fonted-string i) #\Space))
			       ))
		       (zwei:insert-moving bp fonted-string)))
		   (cond (break-p
			  (zwei:insert-moving bp #\Newline)
			  (setq last-pixels-from-left-margin 0
				queued-pixels-from-left-margin 0))
			 (t (setq last-pixels-from-left-margin queued-pixels-from-left-margin)))
		   (setq last-chars-within-string queued-chars-within-string)
		   )))
	     (4shuffle-environments* ()
	       (setq current-font (extract-font environments)
		     leftmargin (extract-leftmargin environments)
		     rightmargin (extract-rightmargin environments)
		     )
	       (let* ((cfs (send interval :get-attribute :fonts)))
		 (unless (member current-font cfs :test #'string=)
		   (send interval :set-attribute :fonts (append cfs (list current-font)) nil)
		   (zwei:set-buffer-fonts interval)
		   ))))
      (shuffle-environments)
      (dolist (string lines)
	(typecase string
	  (STRING
	   (setq last-chars-within-string 0
		 queued-chars-within-string 0)
	   (cond ;((and (= 1 (length string)) (char= #\Newline (char string 0)))
		 ; (dump-queued-words string t))
	         ; ##
	         ; ## all lines end in #\newline - if we have an empty line, insert vertical space.
	         ; ## the normal newline at the end of the line is ignored.
	         ((and (every #'white-p string) (position #\Newline string))
		  (dump-queued-words string t))
		 (t
		  (word-map (string start end next-nonwhite
			     (progn
			       (setq queued-chars-within-string (length string))
			       (dump-queued-words string nil)))
		    (let* ((word-len (tv:sheet-string-length window string start end nil current-font))
			   (word-len+ (or (and next-nonwhite
					       (+ word-len
						  (tv:sheet-string-length window string end next-nonwhite nil current-font)))
					  word-len)))
		      (cond ((>= (+ leftmargin queued-pixels-from-left-margin word-len)
				 (- winwidth rightmargin))  		;1 reached rightmargin*
			     (dump-queued-words string t)
			     ))
		      (incf queued-chars-within-string (- (or next-nonwhite end) start))
		      (incf queued-pixels-from-left-margin word-len+)
		      )))))
	  (TEXT-ATTR
	   (push string environments)
	   (shuffle-environments))
	  
	  (t
	   (cond ((eq string :ENV-POP)
		  (pop environments)
		  (shuffle-environments))
		 ((and (consp string) (eq :VIEW (car string)))
		  (let* ((viewtype (second string))
			 (object (third string))
			 (newline-p (neq (fourth string) :no-line-advance)))
		    (setq bp (send object :grind-to-buffer interval bp last-pixels-from-left-margin viewtype))
		    (cond (newline-p
			   (zwei:insert-moving bp #\Newline)
			   (setq last-pixels-from-left-margin 0))
			  (t
			   (setq last-pixels-from-left-margin
				 (zwei:string-pos-to-pixel (zwei:bp-line bp) (zwei:bp-index bp)))))))
		 (t (error "3What's this: ~s*" string))))
	  ))))
  bp)



;1;; bitmap datastream objects.*


(defflavor 4raster*
  ((version 0)
   (options nil)
   (bitmap nil)
   xscale yscale xoffset yoffset
   subwidth subheight
   width height)
  (basic-datastream-object)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(raster :print-self)* (stream ignore slashify)
  (if slashify
      (format stream "3#<~A ~Dx~D ~D>*" (type-of self) width height id)
      (format stream "3~(~A~) ~Dx~D ~D*" (type-of self) width height id)))


(defvar 4*bit-flippage* *(let* ((a (make-array 256 :element-type '(unsigned-byte 8))))
			   (dotimes (i 256)
			     (let* ((flip-byte 0))
			       (dotimes (j 8)
				 (setq flip-byte (dpb (ldb (byte 1 (- 7 j)) i) (byte 1 j) flip-byte)))
			       (setf (aref a i) flip-byte)))
			   a)
  "2A table for quickly reversing the order of bits in a byte.*")


(defmethod 4(raster :read-self*) (stream)
  (setf version (datastream-read stream))
  (assert (and (numberp version) (>= version 2)) () "2Not a version 2 or greater raster image.*")
  (macrolet ((read-number (var)
	       `(progn (setf ,var (datastream-read stream))
		       (check-type ,var integer))))
    (read-number options)
    (read-number xscale)
    (read-number yscale)
    (read-number xoffset)
    (read-number yoffset)
    (read-number subwidth)
    (read-number subheight)
    (assert (eq :BITS (datastream-read stream)) () "2Expected the word BITS...*")
    (let* (sub-id)
      (read-number sub-id)
      (assert (= id sub-id) () "2Expected ~D and ~D to be equal.*" id sub-id))
    (read-number width)
    (read-number height)
    (assert (< 0 width  1000000) () "2Width ~D is too big.*" width)
    (assert (< 0 height 1000000) () "2Height ~D is too big.*" height))
  
  (setf bitmap (make-array #+LISPM (list height (+ 32 (* 32 (floor (1- width) 32))))
			   #-LISPM (list width height)
			   :element-type 'BIT :initial-element 0))
  (let* (#+LISPM (bytemap (make-array (list (array-dimension bitmap 0) (round (array-dimension bitmap 1) 8))
				      :element-type '(unsigned-byte 8) :displaced-to bitmap)))
    (flet (#-LISPM
	   (setmap-byte (byte bitmap word-x bit-y)
	     (let* ((bit-x (* word-x 8)))
	       (dotimes (i 8)
		 (setf (aref bitmap bit-x bit-y) (ldb (byte 1 (- 7 i)) byte))
		 (incf bit-x))))
	   #+LISPM
	   (setmap-byte (byte bytemap word-x bit-y)
	     (declare (type (unsigned-byte 8) byte)
		      (fixnum word-x bit-y)
		      (optimize speed))
	     (setq byte (the (unsigned-byte 8) (aref (the vector *bit-flippage*) byte)))
	     (setf (aref bytemap bit-y word-x) byte))
	   
	   (read-hex (stream)
	     (let* ((c1 (read-char stream))
		    (c2 (read-char stream)))
	       (if (digit-char-p c1)
		   (setq c1 (- (char-int c1) #.(char-int #\0)))
		   (setq c1 (- (char-int (char-downcase c1)) #.(- (char-int #\a) 10))))
	       (if (digit-char-p c2)
		   (setq c2 (- (char-int c2) #.(char-int #\0)))
		   (setq c2 (- (char-int (char-downcase c2)) #.(- (char-int #\a) 10))))
	       (+ (* 16 c1) c2)))
	   )
      (dotimes (row height)
	(let* ((x 0))
	  (loop
	    (let* ((c (read-char stream))
		   (code (char-int c)))
	      (cond ((char= c #\|)
		     ;1; Legal end-of-row.*
		     (return))

		    ((member c '(#\Space #\Tab #\Newline) :test #'char=) nil)
		    ((or (<= 0 code 31)
			 (> code 126))
		     (warn "2Illegal control character ~D, found while reading raster ~D.*" code id)
		     nil)
		    ((member c '(#\@ #\[ #\] #\^ #\_ #\` #\} #\~) :test #'char=)
		     (warn "2Illegal character ~A, found while reading raster ~D.*" c id)
		     nil)
		    
		    ((member c '(#\{ #\\) :test #'char=)
		     ;1; These are (illegal) end-of-line characters.*
		     (warn "2Illegal EOL character, ~A, found while reading raster ~D.*" c id)
		     (return))
		    
		    ((<= 27 code 47)	;1 Repeat code.  Read a hex code and repeat it.*
		     (let* ((repeat-count (- code 31))
			    (value (read-hex stream)))
		       (dotimes (i repeat-count)
			 #-LISPM (setmap-byte value bitmap x row)
			 #+LISPM (setmap-byte value bytemap x row)
			 (incf x))))
		    
		    ((<= #.(char-code #\g) code #.(char-code #\z))		;1 Multiple white bytes.*
		     (let* ((repeat-count (- code #.(1- (char-code #\g)))))
		       (dotimes (i repeat-count)
			 #-LISPM (setmap-byte #x00 bitmap x row)
			 #+LISPM (setmap-byte #x00 bytemap x row)
			 (incf x))))
		    
		    ((<= #.(char-code #\G) code #.(char-code #\Z))		;1 Multiple black bytes.*
		     (let* ((repeat-count (- code #.(1- (char-code #\G)))))
		       (dotimes (i repeat-count)
			 #-LISPM (setmap-byte #xFF bitmap x row)
			 #+LISPM (setmap-byte #xFF bytemap x row)
			 (incf x))))
		    
		    (t (unread-char c stream)
		       #-LISPM (setmap-byte (read-hex stream) bitmap x row)
		       #+LISPM (setmap-byte (read-hex stream) bytemap x row)
		       (incf x))
		    )))))))
  ;1;*
  ;1; Now we have to take care of our \enddata.*
  (peek-char #\\ stream)
  (assert (and (char= #\\ (read-char stream))
	       (eq :ENDDATA (datastream-read stream)))
	  () "2Expected a \\enddata immediately after the raster data.*")
  (let* ((end-list (datastream-read stream)))
    (assert (and (consp end-list)
		 (eq (first end-list) :RASTER)
		 (eql (second end-list) id))
	    () "2Expected an \\enddata{raster, ~D} immediately after the raster data.  Got \\enddata{~(~{~A ~}~)} instead.*" id end-list))
  )


(defflavor 4raster-diagram-line*
	   ((raster nil)		;1 The 5andrew:raster* which this diagram line represents.*
	    (x-offset 0)		;1 The horizontal offset of this image from the global left margin.*
	    (first-line nil))		;1 A 5zwei:line* which is the first line on which this diagram appears.*
;	   (zwei:restorable-line-diagram-mixin)
	   (zwei:line-diagram-mixin)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(raster-diagram-line :draw)* (line-to-draw sheet)
  "2Invoked by Zmacs redisplay to draw one pline of the Diagram.*"
  (send self :update-lines)
  (let* ((i (getf (zwei:line-plist line-to-draw) :diagram-line-number))
	 (n (send self :height-in-lines))
	 (dest-x (- x-offset (tv:sheet-left-margin-size sheet)))
	 (dest-y (- (tv:sheet-cursor-y sheet) (tv:sheet-top-margin-size sheet)))
;	 (dest-w (tv:sheet-inside-width sheet))
	 (line-h (tv:sheet-line-height sheet))
	 (bitmap (send raster :bitmap))
	 (source-x 0)
	 (source-y (and i (* i line-h)))
	 (source-w (array-dimension (send raster :bitmap) 1))
	 (source-h (array-dimension (send raster :bitmap) 0))
	 (box-line-p t)
	 (draw-alu (cond ((w:color-system-p sheet) TV:ALU-TRANSP)	;1 If color, draw transparent.  All will be taken care of.*
			 (tv:*current-screen-color* TV:ALU-SETA)	;1 If black-on-white, draw normal.*
			 (t TV:ALU-SETCA)))				;1 If white-on-black, draw inverted (Andrew is W on B).*
	 )
    (when (> (* line-h (1+ i)) source-h)
      (setq line-h (1+ (rem source-h line-h))))
    (when (< source-y source-h)
      (send sheet :bitblt draw-alu source-w line-h bitmap source-x (max 0 source-y) dest-x dest-y)
      (when box-line-p
	(let* ((x1 (+ dest-x (tv:sheet-left-margin-size sheet)))
	       (y1 (+ dest-y (tv:sheet-top-margin-size sheet)))
	       (x2 (+ x1 source-w))
	       (y2 (+ y1 line-h -1)))
	  (cond ((eql i 0)      (send sheet :draw-line (1+ x1) y1 (1- x2) y1))
		((eql i (1- n)) (send sheet :draw-line (1+ x1) y2 (1- x2) y2)))
	  (send sheet :draw-line x1 y1 x1 y2)
	  (send sheet :draw-line x2 y1 x2 y2)
	  )))))


(defmethod 4(raster-diagram-line :height-in-pixels*) ()
  (array-dimension (send raster :bitmap) 0))


(defmethod 4(raster-diagram-line :height-in-lines*) ()
  (1+ (ceiling (send self :height-in-pixels)
	       (+ (send zwei:*window* :vsp) (tv:sheet-line-height zwei:*window*)))))


;1;; The next few methods are more-or-less copied from the like-named methods of* DOX-DIAGRAM1.*  1I could just inherit*
;1;; *DOX-DIAGRAM1 to get these,* 1but then, raster-diagram-lines would be SHEETs as well.*  1There's no need for that - I assert that*
;1;; these should not be on *DOX-DIAGRAM1, but rather* 1should be* 1on an intermediate flavor between *DOX-DIAGRAM1 and*
;1;; *RESTORABLE-LINE-DIAGRAM-MIXIN1 - say,* MULTI-LINE-DISPLAYING-DIAGRAM-MIXIN1....*

(defmethod 4(raster-diagram-line :first-line-p*) (line)
  (zerop (getf (zwei:line-plist line) :diagram-line-number)))

(defmethod 4(raster-diagram-line :last-line-p*) (line)
  (cond ((and line (eq (getf (zwei:line-plist line) :diagram) self))
	 (or (null (setq line (zwei:line-next line)))
	     (neq (getf (zwei:line-plist line) :diagram) self)))))

(defmethod 4(raster-diagram-line :update-lines*) ()
  (when first-line
    (do ((i    (1- (send self :height-in-lines)) (1- i))
	 (line first-line next)
	 (next (zwei:line-next first-line) (zwei:line-next next)))
	((not (plusp i))
	 (do ((end line (zwei:line-next end)))
	     ((send self :last-line-p end)
	      (setq end (zwei:line-next end))
	      (setf (zwei:line-next line) end)
	      (when end (setf (zwei:line-previous end) line)))))
      (when (send self :last-line-p line)
	(dotimes (j i)
	  (let ((new (zwei:copy-line line (zwei:line-node line))))
	    (setf (zwei:line-next line) new)
	    (setf (zwei:line-previous new) line)
	    (setf (zwei:line-plist new) (copy-list (zwei:line-plist line)))
	    (incf (getf (zwei:line-plist new) :diagram-line-number))
	    (setq line new)))
	(setf (zwei:line-next line) next)
	(setf (zwei:line-previous next) line)
	(return)))
    (send self :mung)))


(defmethod 4(raster-diagram-line :mung*) ()
  (when first-line
    (do ((line first-line (zwei:line-next line)))
	((send self :last-line-p line)
	 (zwei:mung-line line))
      (zwei:mung-line line))))


(defmethod 4(raster :grind-to-buffer)* (interval bp &optional (current-x 0) viewtype)
  (unless (or (null viewtype) (eq viewtype :rasterview)) (warn "2What?  A raster with a ~S viewtype?*" viewtype))
  (when (> current-x 1000) (warn "2Hey, ~D is an awfully big X offset...*" current-x))
  (let* ((line (zwei:bp-line bp)))
    (when (or (not (zerop (length line)))
	      (eq line (zwei:bp-line (zwei:interval-first-bp interval))))
      (zwei:insert-moving bp #\Newline)
      (setq line (zwei:bp-line bp)))
    
    (let* ((new-line (zwei:create-line 'art-string 0 (zwei:line-node line)))
	   (diag (make-instance 'raster-diagram-line :first-line new-line :raster self :x-offset current-x)))
      ;1;*
      ;1; Install the diagram info on the line's plist.*
      (setf (getf (zwei:line-plist new-line) :diagram) diag)
      (setf (getf (zwei:line-plist new-line) :diagram-line-number) 0)
      ;1;*
      ;1; Insert the line into the buffer.*
      (zwei:insert-line-with-leader new-line line)	;1 This inserts NEW-LINE 6before* LINE.*
      (zwei:move-bp bp line (length line))
      (zwei:insert-moving bp #\Newline)))
  bp)


;1;; Structured-image datastream objects.*

(defflavor 4zip*
  ()
  (basic-datastream-object)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(zip :print-self)* (stream ignore slashify)
  (if slashify
      (format stream "3#<~A ~D>*" (type-of self) id)
      (format stream "3~(~A~) ~D*" (type-of self) id)))

(defvar 4*zipcodes**
	'((#\A . Caption)	; "zipocapt"
	  (#\B . FlexCapt)	; "zipofcap"
	  (#\C . Line)		; "zipoline"
	  (#\D . PolyLine)	; "zipoplin"
	  (#\E . Polygon)	; "zipopoly"
	  (#\F . Trapezoid)	; "zipotrap"
	  (#\G . Rectangle)	; "ziporect"
	  (#\H . Path)		; "zipopath"
	  (#\I . Imbed)		; "zipoimbd"
	  (#\J . Circle)	; "zipocirc"
	  (#\K . Photo)		; "zipobj"
	  (#\L . Ellipse)	; "zipoelli"
	  (#\M . Arc)		; "zipoarc"
	  (#\N . RoundAngle)	; "ziporang"
	  (#\O . Arrow)		; "zipoarrw"
	  (#\P . Symbol)	; "ziposym"
	  ))

;1;; ## Gee, Zip is more complicated than I suspected....*


;1;; Animation objects.*

(defflavor 4fad*
  ((frame-time 30)
   (frames 0)
   (label-font nil)
   (icon-font nil)
   (dest-w 0)
   (dest-h 0)
   )
  (basic-datastream-object)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(fad :print-self)* (stream ignore slashify)
  (if slashify
      (format stream "3#<~A ~D frames ~D>*" (type-of self) (length frames) id)
      (format stream "3~(~A~) ~D frames ~D*" (type-of self) (length frames) id)))

;(defconstant 4FAD-ICONFLAG*	-10)
;(defconstant 4FAD-LABELFLAG*	-10000)
;(defconstant 4FAD-ANIMATEFLAG*	-9999)
;(defconstant 4FAD-MAXWIDTH*	20000)
;(defconstant 4FAD-MAXHEIGHT*	20000)
;(defconstant 4FAD-LINEMODE*	0)
;(defconstant 4FAD-BOXMODE*	1)
;(defconstant 4FAD-ANIMATEMODE*	2)


;1;; Correct font mapping...*
(unless (boundp 'FONTS:ANDY12) (setq FONTS:ANDY12 'FONTS:ADOBE-TR12))


(defmethod 4(fad :read-self)* (stream)
  (loop
    (let* ((line (read-line stream)))
      (when (and (> (length line) 1) (char= #\$ (char line 0)))
	(case (char line 1)
	  (#\T (setf frame-time (parse-integer line :start 2)))
	  (#\C (setf frames     (make-array (parse-integer line :start 2) :fill-pointer 0)))
	  (#\L (when label-font (warn "3setting LABEL-FONT again...*"))
	       (let* ((font-name (subseq line 3)))
		 (setf label-font
		       (or (ignore-errors (tv:font-evaluate (intern (string-upcase font-name) "3FONTS*")))
			   (ignore-errors (tv:font-evaluate (intern (format nil "3ANDY-~:@(~A~)*" font-name) "3FONTS*")))
			   (progn (warn "3Couldn't find icon font ~S*" font-name)
				  (tv:font-evaluate FONTS:HL12B))))))
	  (#\N (when icon-font (warn "3setting ICON-FONT again...*"))
	       (let* ((font-name (subseq line 3)))
		 (setf icon-font
		       (or (ignore-errors (tv:font-evaluate (intern (string-upcase font-name) "3FONTS*")))
			   (ignore-errors (tv:font-evaluate (intern (format nil "3ANDY-~:@(~A~)*" font-name) "3FONTS*")))
			   (progn (warn "3Couldn't find font ~S*" font-name)
				  (tv:font-evaluate FONTS:MEDFNTB))))))
	  (#\F (let* ((array (if (zerop (length frames))
				 (make-array 50 :fill-pointer 0 :adjustable t)
				 (make-array (length (aref frames 0)) :fill-pointer 0 :adjustable t))))
		 (vector-push array frames)))
	  (#\P (string-subst-char #\Space #\, line nil nil)
	       (let (n pos)
		 (multiple-value-setq (n pos) (parse-integer line :junk-allowed t :start 2))
		 (multiple-value-setq (n pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (dest-w pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (dest-h pos) (parse-integer line :junk-allowed t :start pos))
		 ))
	  (#\V (string-subst-char #\Space #\, line nil nil)
	       (let (x1 y1 x2 y2 pos)
		 (multiple-value-setq (x1 pos) (parse-integer line :junk-allowed t :start 2))
		 (multiple-value-setq (y1 pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (x2 pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (y2 pos) (parse-integer line :junk-allowed t :start pos))
		 (if (minusp x2)
		     (vector-push-extend (list :ICON x1 y1 y2) (aref frames (1- (length frames))))
		     (vector-push-extend (list :LINE x1 y1 x2 y2) (aref frames (1- (length frames))))
		     )))
	  (#\A (string-subst-char #\Space #\, line nil nil)
	       (let (x1 y1 ch pos)
		 (multiple-value-setq (x1 pos) (parse-integer line :junk-allowed t :start 2))
		 (multiple-value-setq (y1 pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (nil pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (ch pos) (parse-integer line :junk-allowed t :start pos))
		 (vector-push-extend (list :action x1 y1 ch) (aref frames (1- (length frames))))
		 ))
	  (#\B (string-subst-char #\Space #\, line nil nil)
	       (let (x1 y1 x2 y2 pos)
		 (multiple-value-setq (x1 pos) (parse-integer line :junk-allowed t :start 2))
		 (multiple-value-setq (y1 pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (x2 pos) (parse-integer line :junk-allowed t :start pos))
		 (multiple-value-setq (y2 pos) (parse-integer line :junk-allowed t :start pos))
		 (vector-push-extend (list :BOX x1 y1 x2 y2) (aref frames (1- (length frames))))
		 ))
	  (#\S (string-subst-char #\Space #\, line nil nil)
	       (let (x y string pos)
		 (multiple-value-setq (x pos) (parse-integer line :junk-allowed t :start 2))
		 (multiple-value-setq (y pos) (parse-integer line :junk-allowed t :start pos))
		 (setq string (read-line stream))
		 (vector-push-extend (list :string x y string) (aref frames (1- (length frames))))
		 ))
	  (#\$ (return))
	  (t (warn "3Unknown FAD directive, ~S*" line))
	  ))))
  (unless label-font (setq label-font fonts:hl12b))
  (unless icon-font  (setq icon-font  fonts:hl12i))
  ;1;*
  ;1; Now we have to take care of our \enddata.*
  (peek-char #\\ stream)
  (assert (and (char= #\\ (read-char stream))
	       (eq :ENDDATA (datastream-read stream)))
	  () "2Expected a \\enddata immediately after the fad data.*")
  (let* ((end-list (datastream-read stream)))
    (assert (and (consp end-list)
		 (eq (first end-list) :FAD)
		 (eql (second end-list) id))
	    () "2Expected an \\enddata{fad, ~D} immediately after the fad data.  Got \\enddata{~(~{~A ~}~)} instead.*" id end-list))
  )

(defmethod 4(fad :draw-item)* (window item &optional (alu w:alu-transp))
  (when item
    (ecase (car item)
      (:line (send window :draw-line (second item) (third item) (fourth item) (fifth item) 1 w:black alu))
      
      (:box  (send window :draw-line (second item) (third item) (second item) (fifth item) 1 w:black alu)
	     (send window :draw-line (fourth item) (third item) (fourth item) (fifth item) 1 w:black alu)
	     (send window :draw-line (second item) (third item) (fourth item) (third item) 1 w:black alu)
	     (send window :draw-line (second item) (fifth item) (fourth item) (fifth item) 1 w:black alu))
      
      (:string (send window :string-out-explicit (fourth item) (second item) (third item)
		     (tv:sheet-inside-right window) (tv:sheet-inside-bottom window)
		     label-font alu))
      
      ((:icon :action)
;       (print (list  (fourth item) (second item) (third item) alu))
;       (send window :draw-char icon-font (fourth item) (second item) (third item) alu)
       (unless (and (numberp (fourth item)) (minusp (fourth item)))
	 (tv:prepare-sheet (window)
	   (w:draw-char icon-font (fourth item) (second item) (- (third item) (tv:font-baseline icon-font))
			alu window)))
       )
      )))

(defmethod 4(fad :draw-frame)* (window n &optional (alu w:alu-transp))
  (let* ((frame (if (arrayp n) n (aref frames n))))
    (dotimes (i (length frame))
      (let* ((item (aref frame i)))
	(send self :Draw-item window item alu)))))


(defmethod 4(fad :tween)* (window frame-n frame-n+1 tweens tween-n &optional tween-frame)
;  (when tween-frame (send self :draw-frame window tween-frame w:alu-back))
  (cond ((null tween-frame)
	 (setq tween-frame (make-array (max (length frame-n) (length frame-n+1)) :fill-pointer (length frame-n))))
	((< (length tween-frame) (length frame-n))
	 (adjust-array tween-frame (max (length frame-n) (length frame-n+1)) :fill-pointer (length frame-n)))
	(t
	 (setf (fill-pointer tween-frame) (length frame-n))))
  (dotimes (i (length frame-n))
;  (let ((i 1))
;    (when (< i (length tween-frame))
    (let* ((item-n (aref frame-n i))
	   (disappears? (<= (length frame-n+1) i))
	   (item-n+1 (if disappears?
			 (aref frame-n i)
			 (aref frame-n+1 i)))
	   (last (aref tween-frame i)))
;      (when (= tween-n 1) (print (list item-n disappears?)))
      (cond ((not (eq (car item-n) (car item-n+1)))
	     (setf (aref tween-frame i) item-n)
	     (when (= tween-n (1- tweens))
	       (send self :draw-item window item-n w:alu-sub)
	       (setf (aref tween-frame i) nil)
	       ))
	    ((and disappears? (= tween-n (1- tweens)))
	     (send self :draw-item window (or last item-n) w:alu-sub)
	     (setf (aref tween-frame i) nil)
	     )
	    (t
	     (ecase (car item-n)
	       ((:line :box)
		(let* ((x1-n   (second item-n))   (y1-n   (third item-n))
		       (x2-n   (fourth item-n))   (y2-n   (fifth item-n))
		       (x1-n+1 (second item-n+1)) (y1-n+1 (third item-n+1))
		       (x2-n+1 (fourth item-n+1)) (y2-n+1 (fifth item-n+1))
		       (x1-t (+ x1-n (floor (* tween-n (float (/ (- x1-n+1 x1-n) tweens))))))
		       (y1-t (+ y1-n (floor (* tween-n (float (/ (- y1-n+1 y1-n) tweens))))))
		       (x2-t (+ x2-n (floor (* tween-n (float (/ (- x2-n+1 x2-n) tweens))))))
		       (y2-t (+ y2-n (floor (* tween-n (float (/ (- y2-n+1 y2-n) tweens)))))))
;		  (format t "3~&~s ~s = ~s  ;  ~s ~s = ~s  ;  ~s ~s = ~s  ;  ~s ~s = ~s*"
;			  x1-n x1-n+1 x1-t y1-n y1-n+1 y1-t
;			  x2-n x2-n+1 x2-t y2-n y2-n+1 y2-t )
		  (setf (aref tween-frame i) (list (car item-n) x1-t y1-t x2-t y2-t))))
	       
	       ((:string :icon :action)
		(let* ((x-n   (second item-n))   (y-n   (third item-n))
		       (x-n+1 (second item-n+1)) (y-n+1 (third item-n+1))
		       (x-t (+ x-n (floor (* tween-n (float (/ (- x-n+1 x-n) tweens))))))
		       (y-t (+ y-n (floor (* tween-n (float (/ (- y-n+1 y-n) tweens)))))))
;		  (format t "3~&~s ~s = ~s  ;  ~s ~s = ~s*"
;			  x-n x-n+1 x-t y-n y-n+1 y-t )
		  (setf (aref tween-frame i) (list (car item-n) x-t y-t (fourth item-n)))))
	       )
	     (unless (equal (aref tween-frame i) last)
	       (and last (send self :draw-item window last w:alu-sub))
	       (send self :draw-item window (aref tween-frame i) w:alu-add))
	     )))
;    )
    )
  tween-frame)


(defmethod 4(fad :animate)* (window &optional (tweens 30))
  (send window :clear-screen)
;  (send window :draw-line 0 0 0 dest-h)
;  (send window :draw-line 0 0 dest-w 0)
;  (send window :draw-line 0 dest-h dest-w dest-h)
;  (send window :draw-line dest-w 0 dest-w dest-h)
  (let (tween-frame)
    (dotimes (i (1- (length frames)))
      (sleep 0.05)
;      (format t "3~&~s ~s*" (aref (aref frames i) 1) (aref (aref frames (1+ i)) 1))
      (dotimes (tween-n tweens)
	(setq tween-frame (send self :tween window (aref frames i) (aref frames (1+ i)) tweens tween-n tween-frame))
;	(process-sleep (floor (* frame-time (/ 60 1000))))
	))
    (send window :clear-screen)
    (send self :draw-frame window (1- (length frames)))
    ))

(defun 4animate-fad-diagram *(diagram-line zwei-window &optional (tweens 5))
  (let* ((fad (send diagram-line :fad))
	 (frames (send fad :frames))
	 (window diagram-line)
	 (lh (send window :line-height))
	 (top-line (send diagram-line :first-line))
	 (top-y (* lh (zwei:line-pline zwei-window top-line)))
	 tween-frame)
    (send diagram-line :clear-screen)
    (flet ((refresh-diag ()
	     (do* ((line top-line (zwei:line-next line))
		   (offset top-y (+ offset lh)))
		  ((not (eq diagram-line (getf (zwei:line-plist line) :diagram))))
	       (tv:sheet-set-cursorpos zwei-window 0 offset)
	       (send diagram-line :draw line zwei-window)
	       )))
      (dotimes (i (1- (length frames)))
	(dotimes (tween-n tweens)
	  (setq tween-frame (send fad :tween window (aref frames i) (aref frames (1+ i)) tweens tween-n tween-frame))
	  (refresh-diag)))
      (send window :clear-screen)
      (send fad :draw-frame window (1- (length frames)))
      (refresh-diag)
      (unwind-protect
	  (process-wait "3Any Key*" #'(lambda (stream)
				       (or (not (zerop (tv:mouse-buttons)))
					   (listen stream)))
			*terminal-io*)
;      (send fad :draw-frame window 0)
	(send diagram-line :set-display-list (send diagram-line :display-list))
	(send diagram-line :clear-screen)
	(send diagram-line :draw-display-list diagram-line)
	(send diagram-line :deexpose :force)
	(refresh-diag))))
  nil)


(defflavor 4fad-diagram-line*
	   ((fad nil)			;1 The 5andrew:fad* which this diagram line represents.*
;	    (x-offset 0)		;1 The horizontal offset of this image from the global left margin.*
;	    (first-line nil)		;1 A 5zwei:line* which is the first line on which this diagram appears.*
	    )
	   (zwei:dox-diagram)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defwhopper 4(fad-diagram-line :draw)* (&rest args)
  (ignore-errors	;1 Diagram lines suck.*
    (lexpr-continue-whopper args)))

(defmethod 4(fad :grind-to-buffer)* (interval bp &optional (current-x 0) viewtype)
  (unless (or (null viewtype) (eq viewtype :fadview)) (warn "2What?  A fad with a ~S viewtype?*" viewtype))
  (when (> current-x 1000) (warn "2Hey, ~D is an awfully big X offset...*" current-x))
  (let* ((line (zwei:bp-line bp)))
    (when (or (not (zerop (length line)))
	      (eq line (zwei:bp-line (zwei:interval-first-bp interval))))
      (zwei:insert-moving bp #\Newline)
      (setq line (zwei:bp-line bp)))
    
    (let* ((new-line (zwei:create-line 'art-string 0 (zwei:line-node line)))
	   (diag (make-instance 'fad-diagram-line :number-of-lines 1 :fad self)))
      (send diag :add-line new-line nil)
      (send self :update-diagram diag (aref frames 0))
      ;1;*
      ;1; Install the diagram info on the line's plist.*
      (setf (getf (zwei:line-plist new-line) :diagram) diag)
      (setf (getf (zwei:line-plist new-line) :diagram-line-number) 0)
      ;1;*
      ;1; Insert the line into the buffer.*
      (zwei:insert-line-with-leader new-line line)	;1 This inserts NEW-LINE 6before* LINE.*
      (zwei:move-bp bp line (length line))
      (zwei:insert-moving bp #\Newline)
      ))
  bp)

(defmethod 4(fad :update-diagram)* (diag frame)
  (let* ((data `((:box  1 1 ,(- dest-w 2) ,(- dest-h 2))
		 (:open 0 0 ,dest-w ,dest-h))))
    (dotimes (i (length frame))
      (let* ((item (aref frame i)))
	(ecase (car item)
	  (:line (push `(:line ,@(cdr item) 1 1) data))
	  (:box  (let* ((left  (min (second item) (fourth item)))
			(top   (min (third item)  (fifth item)))
			(right (max (second item) (fourth item)))
			(bot   (max (third item)  (fifth item)))
			(w (- right left))
			(h (- bot top)))
		   (push `(:box ,left ,top ,w ,h 1 1) data)))
	  
	  (:string (push `(:text ,(tv:font-name label-font) ,@(cdr item)) data))
	  
;	  (:icon (push `(:text fonts:hl12b ,(second item) ,(third item) ,(string (int-char (fourth item)))) data))
	  ((:icon :action) (push `(:text ,(tv:font-name icon-font) ,(second item) ,(third item)
					 ,(string (int-char (max 0 (fourth item)))))
				 data))
	  )))
    (setf data (nreverse data))
    (send diag :set-display-list data)
    (send diag :clear-screen)
    (send diag :draw-display-list diag)
    (send diag :deexpose :force)
    ))



;1;; Reading the .MS_MsgDir file.*

(defconstant 4AMS_DATESIZE*	4 * 7	"2Fixed size of compacted date*")
(defconstant 4AMS_CAPTIONSIZE*	4 *89	"2Fixed size of caption line*")
(defconstant 4AMS_CHAINSIZE*	4 * 4	"2Fixed size of chain number*")
(defconstant 4AMS_MIDHASHSIZE*	4 * 4	"2Fixed size of message id hash*")
(defconstant 4AMS_REPLYHASHSIZE * 4	"2Fixed size of in-reply-to or references hash*")
(defconstant 4AMS_ATTRIBUTESIZE *21	"2Fixed size of msg attributes*")
(defconstant 4AMS_IDSIZE*	4 *19	"2Fixed size of unique ID name*")

(defconstant 4AMS_SNAPSHOTSIZE* (+ AMS_DATESIZE AMS_CAPTIONSIZE AMS_CHAINSIZE AMS_MIDHASHSIZE AMS_REPLYHASHSIZE
				   AMS_ATTRIBUTESIZE AMS_IDSIZE)
  "2Fixed size of a field in a .MS_MsgDir file.*")

(defconstant 4AMS_ATT_RRR*	4 * 0	"2Return Receipt Requested*")
(defconstant 4AMS_ATT_ENCLOSURE * 1	"2Parcel Post*")
(defconstant 4AMS_ATT_DELETED*	4 * 2	"2Marked for deletion*")
(defconstant 4AMS_ATT_NEWDIR*	4 * 3	"2Announcing a new message subdirectory*")
(defconstant 4AMS_ATT_FORMATTED*  4	"2Multimedia format file*")
(defconstant 4AMS_ATT_MAYMODIFY*  5	"2Message this user may alter*")
(defconstant 4AMS_ATT_UNSEEN*	  6	"2Message is marked as Unseen*")
(defconstant 4AMS_ATT_UNAUTH*	4 * 7	"2Message sender is unauthenticated*")
(defconstant 4AMS_ATT_FROMNET*	4 * 8	"2Message sender is from remote machine*")
(defconstant 4AMS_ATT_VOTE*	4 * 9	"2Message calls for a vote*")
(defconstant 4AMS_ATT_URGENT*	4 *10	"2User marked this message as Urgent*")
(defconstant 4AMS_ATT_CLOSED*	4 *11	"2User marked this messge as Closed*")
(defconstant 4AMS_ATT_REPLIEDTO *12	"2Message has been replied to*")

(defstruct 4(message *(:print-function %print-message))
  date
  caption
  chain-number
  hash-number
  reply-to-hash-number
  attributes
  id-string
  )

(defun 4%print-message *(struct stream ignore)
  (sys:printing-random-object (struct stream :typep)
    (when (message-caption struct)
      (let* ((one (position #\Tab (message-caption struct)))
	     (two (and one (position #\Tab (message-caption struct) :start (1+ one))))
	     (*print-case* :capitalize))
	(prin1 (subseq (message-caption struct) (if one (1+ one) 0) (or two (length (message-caption struct))))
	       stream)
	(princ #\Space stream)
	(princ (message-attributes struct) stream)))))

;(defun 4b *(n &optional (width 40))
;  (fresh-line)
;  (setq width (* 8 (ceiling width 8)))
;  (dotimes (i (ceiling width 8))
;    (format t "3~8,'0b *" (ldb (byte 8 (* (- (ceiling width 8) i 1) 8)) n)))
;  n)

;(let ((bad (ldb (byte 32 0) bad))) (b bad)
;         (b (+ bad mt:*unix-universal-time-modifier*))
;         (b (- bad mt:*unix-universal-time-modifier*))
;         (b (- mt:*unix-universal-time-modifier* bad))  (terpri) (terpri)
;         (b good)
;         (b (+ good mt:*unix-universal-time-modifier*))
;         (b (- good mt:*unix-universal-time-modifier*))
;         (b (- mt:*unix-universal-time-modifier* good)) (values))


(defconstant 4base64-mapping *"0123456789:=ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
  "2The characters (indexed by weight) in Andrew's base 64 representation.*")

(defun 4base64-string-to-int *(string)
  "2Boy this is a silly representation.*"
  (let* ((n 0)
	 (sys:alphabetic-case-affects-string-comparison t)) ;1 for the microcode.*
    (declare (integer n))
    (dotimes (i (length string))
      (setq n (+ (* n 64)
		 (sys:%string-search-char (char string i) BASE64-MAPPING 0 64))))
    n))

(defun 4int-to-base64-string *(int &optional stream)
  "2Boy this is a silly representation.  If stream is NIL, returns a string.*"
  (if stream
      (let ((n (truncate int 64)))
	(or (zerop n) (int-to-base64-string n stream))
	(write-char (aref BASE64-MAPPING (rem int 64)) stream))
      (with-output-to-string (stream)
	(int-to-base64-string int stream))))


(defun 4parse-msgdir *(pathname)
  (let* ((result '()))
    (with-open-file (stream pathname :direction :input :characters nil :byte-size 8)
      (unless (and (= 3 (read-byte stream))
		   (= 4 (read-byte stream))
		   (= 27 (read-byte stream))
		   (= 191 (read-byte stream)))
	(error "3Not a MsgDir file.*"))
      (dotimes (i 2) (loop (when (= 10 (read-byte stream)) (return))))
      
      (flet ((4read-string-of-length* (n &optional (trimp t))
	       (let* ((s (make-string n)))
		 (dotimes (i n) (setf (char s i) (make-char (read-byte stream))))
		 (if trimp
		     (string-right-trim '(#.(make-char 0)) s)
		     s)))
	     
	     (4read-n-byte-int* (n)
	       (let* ((i 0))
		 (dotimes (j n) (setq i (dpb (read-byte stream) (byte 8 (* 8 j)) i)))
		 i)))
	
	;1; Parse out some stuff from the header.*
	(let* ((major (read-string-of-length 11 nil))
	       (minor (parse-integer (read-string-of-length 11 nil) :junk-allowed t))
	       (kill  (parse-integer (read-string-of-length 11 nil) :junk-allowed t))
	       (attr  (parse-integer (read-string-of-length 512 nil) :junk-allowed t)))
	  (push (list :HEADER major minor kill attr) result)
	  ;1; Lose the numerous pad bytes...*
	  (dotimes (i 535) (read-byte stream)))
	(do ()
	    ((not (peek-char nil stream nil nil)))
	  (let* ((message (make-message)))
	    (setf (message-date message) (+ mt:*unix-universal-time-modifier* ;1 convert Unix time to Common Lisp time.*
					    (base64-string-to-int (read-string-of-length AMS_DATESIZE)))
		  (message-caption message) (read-string-of-length AMS_CAPTIONSIZE)
		  (message-chain-number message) (read-n-byte-int AMS_CHAINSIZE)
		  (message-hash-number message)	(read-n-byte-int AMS_MIDHASHSIZE)
		  (message-reply-to-hash-number message) (read-n-byte-int AMS_REPLYHASHSIZE))
	    (string-subst-char #\Tab #.(make-char 9) (message-caption message) nil nil)
	    (let* ((attr-num (read-n-byte-int AMS_ATTRIBUTESIZE))
		   (attributes '()))
	      (when (logbitp AMS_ATT_RRR	attr-num) (push :RETURN-RECIPT	 attributes))
	      (when (logbitp AMS_ATT_ENCLOSURE	attr-num) (push :PARCEL-POST	 attributes))
	      (when (logbitp AMS_ATT_DELETED	attr-num) (push :DELETED 	 attributes))
	      (when (logbitp AMS_ATT_NEWDIR	attr-num) (push :NEW-BB-ANNOUNCE attributes))
	      (when (logbitp AMS_ATT_FORMATTED	attr-num) (push :MULTIMEDIA	 attributes))
	      (when (logbitp AMS_ATT_MAYMODIFY	attr-num) (push :ALTERABLE	 attributes))
	      (when (logbitp AMS_ATT_UNSEEN	attr-num) (push :UNSEEN		 attributes))
	      (when (logbitp AMS_ATT_UNAUTH	attr-num) (push :UNAUTHENTICATED attributes))
	      (when (logbitp AMS_ATT_FROMNET	attr-num) (push :REMOTE		 attributes))
	      (when (logbitp AMS_ATT_VOTE	attr-num) (push :VOTE		 attributes))
	      (when (logbitp AMS_ATT_URGENT	attr-num) (push :URGENT		 attributes))
	      (when (logbitp AMS_ATT_CLOSED	attr-num) (push :CLOSED		 attributes))
	      (when (logbitp AMS_ATT_REPLIEDTO	attr-num) (push :REPLIED 	 attributes))
	      (setf (message-attributes message) attributes))
	    (setf (message-id-string message) (read-string-of-length AMS_IDSIZE))
	    (push message result)))))
    result))


;    for (i=0; i<NumSubsInUse; ++i) {
;	if (SubsInUserOrder[i].sname != NULL) {
;	    fprintf(fp, "%s %s %d %s %d\n", SubsInUserOrder[i].sname, SubsInUserOrder[i].key, SubsInUserOrder[i].status, SubsInUserOrder[i].time64, SubsInUserOrder[i].filedate);
;	}

(defstruct 4(message-profile *(:print-function %print-msg-prof) (:conc-name PROF-))
  name
  pathname
  status	; /* subscription status code */
  time		; /* From SetAssociatedTime */
  filedate	; /* Ditto */
  )

(defun 4%print-msg-prof *(struct stream ignore)
  (format stream "3#<~S ~{~A~^ ~} ~D>*" (type-of struct) (prof-name struct) (sys:%pointer struct)))

(defvar 4*profile* *(make-hash-table :test #'equal :size 1000) "2Current message profiles structures.*")

(defun 4prof-newsgroup *(prof)
  (find (prof-name prof) (send *newsgroups-buffer* :newsgroups) :test #'equalp :key #'newsgroup-name))

(defun 4newsgroup-prof *(newsgroup)
  (gethash (newsgroup-name newsgroup) *profile*))


(defun 4dot-string-to-list *(string &optional (dot #\.))
  "2Given a string with periods in it, returns a list of the strings between the periods.*"
  (declare (string string) (string-char dot))
  (do* ((result '())
	(last -1 start)
	(start (position dot string :test #'char-equal)
	       (and start (position dot string :test #'char-equal :start (1+ start)))))
       ((null last)
	(nreverse result))
    (push (subseq string (1+ last) start) result)))


(defvar 4*andrew-host* *"2spice*")
(defvar 4*andrew-bb-root* *'("3afs*" "3andrew*" "3usr*" "3bb*" "3.MESSAGES*"))
(defvar 4*ams_prof-file* *nil)

(defun 4parse-ams.prof *(pathname)
  "2Parse a .AMS.prof file, and put MESSAGE-PROFILE structures into *PROFILE*.  Returns newsgroups.*"
  (with-open-file (stream pathname :direction :input :characters t)
    (let* ((ngs '()))
      (loop
	(let* ((line (read-line stream nil nil)))
	  (unless line (return (setf ngs (nreverse ngs))))
	  (unless (or (zerop (length line))
		      (char= #\$ (char line 0)))
	    (let* ((name-end (position #\Space line :test #'char=))
		   (path-end (and name-end (position #\Space line :test #'char= :start (1+ name-end))))
		   (stat-end (and path-end (position #\Space line :test #'char= :start (1+ path-end))))
		   (time-end (and stat-end (position #\Space line :test #'char= :start (1+ stat-end))))
		   (name (subseq line 0 name-end))
		   (path (and name-end (subseq line (1+ name-end) path-end)))
		   (stat (or (and path-end (parse-integer line :start (1+ path-end) :junk-allowed t)) 0))
		   (time (and stat-end (+ mt:*unix-universal-time-modifier*	;1 convert Unix time to Common Lisp time.*
					  (base64-string-to-int (subseq line (1+ stat-end) time-end)))))
		   (file (and time-end (+ mt:*unix-universal-time-modifier*
					  (parse-integer line :start (1+ time-end) :junk-allowed t))))
		   (namelist (dot-string-to-list name))
		   (newsgroup (make-newsgroup :name namelist
					      :msgdir-pathname (pathname (string-append (string *andrew-host*) "3: *" path
											"3/.MS_MsgDir*"))))
		   (prof (newsgroup-prof newsgroup)))
	      (push newsgroup ngs)
	      (when path
		(unless (char= #\/ (char path (1- (length path))))
		  (setq path (string-append path #\/)))
		(setq path (parse-namestring path *andrew-host*)))
	      (unless prof
		(setq prof (make-message-profile :name namelist))
		(setf (gethash namelist *profile*) prof))
	      (setf (prof-pathname prof) path
		    (prof-status prof) stat
		    (prof-time prof) time
		    (prof-filedate prof) file))))))))

(defun 4write-ams.prof *(entries pathname)
  "2Write an .AMS.prof file, given a list of MESSAGE-PROFILE structures.*"
  (with-open-file (stream pathname :direction :output :characters t)
    (dolist (prof entries)
      (when (newsgroup-p prof) (setq prof (newsgroup-prof prof)))
      (format stream "3~{~A~^.~} *" (prof-name prof))
      (write-string (string-right-trim #\/ (directory-namestring (prof-pathname prof))) stream)
      (format stream "3 ~A *" (or (prof-status prof) 0))
      (let* ((string (if (prof-time prof)
			 (int-to-base64-string (- (prof-time prof) mt:*unix-universal-time-modifier*))
			 "3000000*")))
	(dotimes (i (- 6 (length string))) (write-char #\0 stream))
	(if (prof-filedate prof)
	    (format stream "3~A ~D~%*" string (- (prof-filedate prof) mt:*unix-universal-time-modifier*))
	    (format stream "3~A 0~%*" string))))
    (truename stream)))


(defun 4parse-subscriptionmap *(pathname)
  (setq pathname (pathname pathname))
  (with-open-file (stream pathname :direction :input :characters t)
    (let* ((result '()))
      (loop
	(let* ((line (or (read-line stream nil nil) (return)))
	       (colon (position #\: line :test #'char=)))
	  (when colon
	    (let* ((name (dot-string-to-list (subseq line 0 colon)))
		   (path (make-pathname :host (pathname-host pathname)
					:raw-directory (dot-string-to-list (subseq line (+ colon 2)) #\/)
					:raw-name "3.MS_MsgDir*"
					:type :UNSPECIFIC))
		   (ng (make-newsgroup :name name :msgdir-pathname path))
		   (prof (newsgroup-prof ng)))
	      (unless prof
		(setq prof (make-message-profile :name name :pathname path :status 0))
		;(push prof *profile*)
		(setf (gethash name *profile*) prof)
		)
	      (push ng result)))))
      (nreverse result))))


(defun 4fontify-string-portion *(string font-number &optional (start 0) end)
  (unless end (setq end (length string)))
  (do* ((i start (1+ i)))
       ((>= i end))
    (unless (char= (char string i) #\Tab)
      (setf (char string i) (make-char (char string i) 0 font-number))))
  string)

;1;; ## this needs to deal better with variable-width fonts in multiple columns.*
;1;;*
(defun 4grind-headers-buffer *(interval msgdir-list &optional (last-time-read 0))
  (let* ((line (zwei:bp-line (zwei:interval-last-bp interval)))
	 (unread-line nil))
    (dolist (msg msgdir-list)
      (unless (consp msg)
	(let* ((string (message-caption msg))
	       (date (message-date msg))
	       (date-end (position #\Tab string :test #'char=))
	       (subj-end (position #\Tab string :test #'char= :start (1+ date-end)))
	       (new-line (zwei:create-line 'art-fat-string (+ 5 (length string)) (zwei:line-node line)))
	       (unread-p (> date last-time-read))
	       (prefix (if unread-p
			   "3 * *"
			   "3   *"))
	       (pl (length prefix)))
	  (when unread-p (setq unread-line new-line))
	  (setf (fill-pointer new-line) 0)
	  (with-output-to-string (stream new-line)
	    (write-string prefix stream)
	    (write-string string stream :end subj-end)
	    (setf (char new-line (+ pl date-end)) #\Space)
	    (fontify-string-portion new-line 1 (+ pl date-end 1) (+ pl subj-end))
	    (format stream "3~vt*" (+ 41 pl))
	    (write-char #\Tab stream)
	    (write-string string stream :start (1+ subj-end))
	    (fontify-string-portion new-line 2 (+ pl subj-end))
	    )
	  (setf (getf (zwei:line-plist new-line) :message) msg)
	  (zwei:insert-line-with-leader new-line line)	;1 This inserts NEW-LINE 6before* LINE.*
	  (setq line new-line))))
    (or unread-line
	(zwei:bp-line (zwei:interval-last-bp interval)))))


(defflavor 4Basic-AMS-Buffer *() (zwei:zmacs-buffer))

(defflavor 4AMS-Newsgroups-Buffer*
	   ((newsgroups nil)
	    (summary-buffer nil))
	   (basic-AMS-buffer)
  (:documentation :flavor "2The kind of buffer used for displaying the list of all subscribed newsgroups.*")
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defflavor 4AMS-Summary-Buffer*
	   ((newsgroup nil)
	    (newsgroups-buffer nil)
	    (message-buffer nil))
	   (basic-AMS-buffer)
  (:documentation :flavor "2The kind of buffer used for displaying the list of messages in an AMS newsgroups.*")
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defflavor 4AMS-Message-Buffer*
	   ((message nil)
	    (summary-buffer nil))
	   (basic-AMS-buffer)
  (:documentation :flavor "2The kind of buffer used for displaying an AMS newsgroup message.*")
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)


(defmethod 4(basic-ams-buffer :after :init)* (ignore)
  (setq zwei:undo-status :dont)
  (setq zwei:saved-mode-list nil)
  (setf (get self :dont-sectionize) t)
  (setf (get self 'zwei:inhibit-kill-buffer-cleanup) t))

(defmethod 4(AMS-Newsgroups-Buffer :after :init)* (ignore)
  (send self :set-attribute :mode 'andrew-newsgroups-mode nil))

(defmethod 4(AMS-Summary-Buffer :after :init)* (ignore)
  (send self :set-attribute :mode 'andrew-summary-mode nil))

(defmethod 4(AMS-Message-Buffer :after :init)* (ignore)
  (send self :set-attribute :mode 'andrew-message-mode nil))


;(defmethod 4(AMS-Newsgroups-Buffer :after :select)* (&rest ignore)
;  (send self :set-major-mode 'andrew-newsgroups-mode))

;(defmethod 4(AMS-Summary-Buffer :after :select)* (&rest ignore)
;  (send self :set-major-mode 'andrew-summary-mode))

;(defmethod 4(AMS-Message-Buffer :after :select)* (&rest ignore)
;  (send self :set-major-mode 'andrew-message-mode))


(defstruct 4(newsgroup *(:print-function %print-newsgroup))
  (name nil :type list)
  (msgdir-pathname nil :type (or null pathname))
  (summary nil :type list)
  (summary-time nil :type (or null integer))
  )

(defun 4%print-newsgroup *(struct stream ignore)
  (sys:printing-random-object (struct stream :typep)
    (format stream "3~{~A~^ ~}*" (newsgroup-name struct))
    (when (newsgroup-summary struct) (princ "3, parsed*" stream))))


#+Comment
(defun 4check-newsgroup *(ng)
  "2debug: verify that the files listed in the newsgroup's msgdir file exist.*"
  (unless (newsgroup-summary ng)
    (setf (newsgroup-summary ng) (parse-msgdir (newsgroup-msgdir-pathname ng))))
  (format t "3~&Newsgroup ~A~{.~A~}~%in directory ~A*"
	  (car (newsgroup-name ng)) (cdr (newsgroup-name ng)) (directory-namestring (newsgroup-msgdir-pathname ng)))
  (dolist (message (newsgroup-summary ng))
    (format t "3~&  Message ~S ~A ~A*" (message-id-string message) (message-caption message) (message-attributes message))
    (let* ((file (make-pathname :raw-name (string-append #\+ (message-id-string message)) :type :UNSPECIFIC
				:defaults (newsgroup-msgdir-pathname ng))))
      (format t "3~&  File ~A ~:[does not exist!~;exists.~]*" file (probe-file file)))))


(defmethod 4(AMS-newsgroups-buffer :grind-newsgroups)* ()
  (zwei:with-read-only-suppressed (self)
    (zwei:delete-interval self)
    (let* ((line (zwei:bp-line zwei:first-bp)))
      (dolist (newsgroup newsgroups)
	(let* (;(newsgroup (prof-newsgroup prof))
;	       (prof (or (newsgroup-prof newsgroup)
;			 (error "3No profile for ~A.*" newsgroup)))
	       (string (format nil "3   ~A~{.~A~}*" (car (newsgroup-name newsgroup)) (cdr (newsgroup-name newsgroup))))
	       (new-line (zwei:create-line 'art-string (length string) self)))
	  (replace new-line string)
	  (setf (char new-line 2) #\Tab)
;	  (let* ((date (or (prof-filedate prof) 0)))
;	    (update-newsgroup newsgroup :noparse)
;	    (when (< date (newsgroup-summary-time newsgroup))
;	      (setf (char new-line 1) #\*))
;	    (setf (prof-filedate prof) (newsgroup-summary-time newsgroup)))
;	  (setf (getf (zwei:line-plist new-line) :newsgroup) prof)
	  (setf (getf (zwei:line-plist new-line) :newsgroup) newsgroup)
	  (zwei:insert-line-with-leader new-line line)	;1 This inserts NEW-LINE 6before* LINE.*
	  )))))


(defmethod 4(AMS-Newsgroups-Buffer :select-newsgroup)* (bp-or-line-or-newsgroup)
  (when (consp bp-or-line-or-newsgroup)
    (setq bp-or-line-or-newsgroup (zwei:bp-line bp-or-line-or-newsgroup)))
  (when (stringp bp-or-line-or-newsgroup)
    (setf (char bp-or-line-or-newsgroup 1) #\Space)
    (let* ((w  (find self zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval))))
	   (w2 (find (send self :summary-buffer) zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
      (when w
	(zwei:must-redisplay w ZWEI:DIS-LINE bp-or-line-or-newsgroup 0)
	(zwei:redisplay w :point nil nil t))
      (setq bp-or-line-or-newsgroup (getf (zwei:line-plist bp-or-line-or-newsgroup) :newsgroup))
      (check-type bp-or-line-or-newsgroup newsgroup)
      (let* ((line (send summary-buffer :set-newsgroup bp-or-line-or-newsgroup)))
	(when w2
	  (zwei:move-bp (zwei:window-point w2) (zwei:create-bp line 0)))))
    ))

(defun 4fast-file-write-date *(pathname)
  "2file-write-date is implemented... poorly.*"
  (getf (cdr (second (fs:directory-list pathname))) :creation-date))

(defun 4update-newsgroup *(newsgroup &optional noparse)
  (when (message-profile-p newsgroup) (setq newsgroup (prof-newsgroup newsgroup)))
  (let* ((time nil))
    (when (or (null (newsgroup-summary newsgroup))
	      (null (newsgroup-summary-time newsgroup))
	      (/= (newsgroup-summary-time newsgroup)
		  (setq time (fast-file-write-date (newsgroup-msgdir-pathname newsgroup)))))
      (if noparse
	  (zwei:typein-line "3~&Probing ~A...*" (newsgroup-msgdir-pathname newsgroup))
	  (if (newsgroup-summary newsgroup)
	      (zwei:typein-line "3~&File has changed.  Re-parsing ~A...*" (newsgroup-msgdir-pathname newsgroup))
	      (zwei:typein-line "3~&Parsing ~A...*" (newsgroup-msgdir-pathname newsgroup))))
      (unless noparse
	(setf (newsgroup-summary newsgroup) (parse-msgdir (newsgroup-msgdir-pathname newsgroup))))
      (unless time (setq time (fast-file-write-date (newsgroup-msgdir-pathname newsgroup))))
      (setf (newsgroup-summary-time newsgroup) time)
      newsgroup)))

(defmethod 4(AMS-Summary-Buffer :set-newsgroup)* (new-newsgroup)
  (update-newsgroup new-newsgroup)
  (zwei:with-read-only-suppressed (self)
    (zwei:delete-interval self)
    (let* (time ng line)
      (cond ((message-profile-p new-newsgroup)
	     (setq time (or (prof-time new-newsgroup) 0)
		   ng (prof-newsgroup new-newsgroup)))
	    (t (setq ng new-newsgroup
		     time (prof-time (newsgroup-prof ng)))))
      (setq line (grind-headers-buffer self (newsgroup-summary ng) time))
      (setq newsgroup new-newsgroup)
      line)))


(defmethod 4(AMS-Summary-Buffer :select-message)* (bp-or-line-or-message)
  (zwei:com-beginning-of-line)
  (when (consp bp-or-line-or-message) (setq bp-or-line-or-message (zwei:bp-line bp-or-line-or-message)))
  (when (stringp bp-or-line-or-message)
    (setf (char bp-or-line-or-message 1) #\.)
    (let* ((w (find self zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
      (when w
	(zwei:must-redisplay w ZWEI:DIS-LINE bp-or-line-or-message 0)
	(zwei:redisplay w :point nil nil t)))
    (setq bp-or-line-or-message (getf (zwei:line-plist bp-or-line-or-message) :message)))
  (send message-buffer :grind-message bp-or-line-or-message newsgroup))


(defun 4font-names-to-font-alist *(names &optional nobarf)
  "2Given a list of font names, return an alist of the font symbols and the font objects.
  If NOBARF is true, then an undefined font will be silently replaced with some default.*"
  (mapcar #'(lambda (name)
	      (let* ((symbol (intern (string-upcase (string name)) "3FONTS*"))
		     (value (if (boundp symbol)
				(tv:font-evaluate symbol)
				(if (load (format nil "3SYS:FONTS;~A*" name) :package "3FONTS*" :if-does-not-exist nil)
				    (or (tv:font-evaluate symbol)
					(error "3SYS:FONTS;~A* did not contain the font ~A." name name))
				    (cond (nobarf
					   (warn "2Message wanted unknown font ~A; using ~A instead.*" symbol ucl:*default-font*)
					   (setq symbol (tv:font-name (tv:font-evaluate ucl:*default-font*)))
					   (tv:font-evaluate ucl:*default-font*))
					  (t (zwei:barf "3~A is not a font.*" symbol)))))))
		(cons symbol value)))
	  names))


(defmethod 4(AMS-Message-Buffer :grind-message)* (msg newsgroup)
  (when (message-profile-p newsgroup)
    (setq newsgroup (prof-newsgroup newsgroup)))
  (let* ((path (make-pathname :raw-name (string-append #\+ (message-id-string msg)) :type :UNSPECIFIC
			      :defaults (newsgroup-msgdir-pathname newsgroup)))
	 (datastream-message-p (member :MULTIMEDIA (message-attributes msg) :test #'eq))
	 (headers-lines nil))
    (zwei:with-read-only-suppressed (self)
      (zwei:delete-interval self)
      (send self :set-attribute :fonts '(FONTS:CPTFONT FONTS:HL12B FONTS:HL12I) nil)
      (zwei:set-buffer-fonts self)
      (let* ((w (find self zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
	(zwei:redefine-fonts w (font-names-to-font-alist (send self :get-attribute :fonts))))
      (with-open-file (input path :direction :input :characters t)
	;1;*
	;1; Read in all of the headers, fontify them, and store them away.*
	;1;*
	(loop
	  (let* ((line (read-line input))
		 (next-char (peek-char nil input)))
	    (when (string= line "") (return))
	    (do* ()
		 ((and (char/= #\Tab next-char) (char/= #\Space next-char)))
	      (setq line (string-append line #\Newline (read-line input))
		    next-char (peek-char nil input)))
	    ;1; Fontify the headers, and find the colon.*
	    (setq line (zwei:in-current-font line 1))
	    (let* ((colon-pos nil))
	      (dotimes (i (length line))
		(let* ((c (char line i)))
		  (cond ((char-equal c #\Tab) (setf (char line i) #\Tab))	;1 no fonts for tab.*
			(colon-pos (setf (char line i) (make-char c 0 2)))	;1 after colon = font 2.*
			((char-equal c #\:) (setq colon-pos i)))))
	      (when colon-pos
		(let* ((name (intern (string-upcase (subseq line 0 colon-pos)) "3KEYWORD*")))
		  (when (eq name :X-Andrew-ScribeFormat) (setq datastream-message-p nil))
		  (push (cons name line) headers-lines))))))
	(setf (get self :headers) headers-lines)
	(send self :show-formatted-headers)
	(zwei:with-bp (bp (zwei:interval-last-bp self) :moves)
	  (zwei:insert-moving bp #\Newline)
	  (if datastream-message-p
	      (condition-call-if (not (boundp 'debug)) (c)
		  (let* ((ds (make-instance 'datastream :input-stream input))
			 (object (send ds :read)))
		    (send ds :set-input-stream nil)	;1 so it can be GCed.*
		    (send object :grind-to-buffer self bp))
		((errorp c)
		 ;1; If we get a parse error, dump the rest as text.  Sub-optimal, but it lets us know what file position died.*
		 (format *error-output* "3~&Error: ~A*" c)
		 (with-open-stream (output (zwei:interval-stream-into-bp bp nil))
		   (sys:stream-copy-until-eof input output ZWEI:LINE-LEADER-SIZE))))
	      
	      (with-open-stream (output (zwei:interval-stream-into-bp bp nil))
		(sys:stream-copy-until-eof input output ZWEI:LINE-LEADER-SIZE)))))))
  (setq message msg) ;1 do this last, so that if there is an error we don't get an imcomplete message cached.*
  )


(defun 4bp-to-empty-line *(interval)
  "2Returns a bp to the front of the first blank line in interval.*"
  (do* ((line (zwei:bp-line (zwei:interval-first-bp interval))
	      (zwei:line-next line)))
       ((null line) nil)
    (when (string= line "") (return (zwei:create-bp line 0)))))


(pushnew :If-Type-Unsupported	zwei:*reformat-headers-exclude-list*)
(pushnew :X-Andrew-Message-Size	zwei:*reformat-headers-exclude-list*)
(pushnew :X-Added		zwei:*reformat-headers-exclude-list*)


(defmethod 4(AMS-Message-Buffer :show-formatted-headers)* ()
  (let* ((end-bp (bp-to-empty-line self))
	 (headers-lines (get self :headers))
	 (reformat-headers-lines (remove-if #'(lambda (x)
						(member (car x) zwei:*reformat-headers-exclude-list* :test #'eq))
					    headers-lines)))
    (sort reformat-headers-lines
	  #'(lambda (x y)
	      (< (or (position (car x) zwei:*reformat-headers-include-list* :test #'eq) 999)
		 (or (position (car y) zwei:*reformat-headers-include-list* :test #'eq) 999))))
    (when end-bp (zwei:delete-interval (zwei:interval-first-bp self) end-bp))
    (zwei:with-bp (bp (zwei:interval-first-bp self) :moves)
      (dolist (header reformat-headers-lines)
	(zwei:insert-moving bp (cdr header))
	(zwei:insert-moving bp #\Newline)))
    (setf (get self :headers-formatted) t)
    ))

(defmethod 4(AMS-Message-Buffer :show-unformatted-headers)* ()
  (let* ((end-bp (bp-to-empty-line self))
	 (headers-lines (get self :headers)))
    (when end-bp (zwei:delete-interval (zwei:interval-first-bp self) end-bp))
    (zwei:with-bp (bp (zwei:interval-first-bp self) :moves)
      (dolist (header headers-lines)
	(zwei:insert-moving bp (cdr header))
	(zwei:insert-moving bp #\Newline)))
    (setf (get self :headers-formatted) nil)
    ))


(defun 4three-windows-by-fraction* (top-fraction top-and-middle-fraction)
  "2Divide the Zmacs frame into three vertically-stacked windows.  
  The top window will consume TOP-FRACTION of the screen, and the top and middle windows will consume TOP-AND-MIDDLE-FRACTION.*"
  (declare (values top-window middle-window bottom-window))
  (let ((windows (zwei:frame-exposed-windows)))
    ;1; If 2 windows present, or more than 3 windows present, reduce to 1 and start from there.*
    (when (or (= 2 (length windows))
	      (> 3 (length windows)))
      (zwei:make-window-full-screen zwei:*window*)
      (setq windows (zwei:frame-exposed-windows)))
    (cond ((= 1 (length windows))
	   (let* ((list (send (zwei:window-frame zwei:*window*) :n-editor-windows 3))
		  (top-window (first list))
		  (middle-window (second list))
		  (bottom-window (third list))
		  (w1 (zwei:window-sheet top-window))
		  (w2 (zwei:window-sheet middle-window))
		  (w3 (zwei:window-sheet bottom-window))
		  (frame (zwei:window-frame top-window)))
	     (multiple-value-bind (left top right bottom) (send frame :inside-edges-without-mode-line-window)
	       (tv:preserve-substitute-status frame
		 (tv:delaying-screen-management
		   (send w1 :deexpose)
		   (send w2 :deexpose)
		   (send w3 :deexpose)
		   (let* ((w1-height (ceiling (* (- bottom top) top-fraction)))
			  (w1+2-height (ceiling (* (- bottom top) top-and-middle-fraction)))
			  ;(w2-height (- w1+2-height w1-height))
			  )
		     (send w1 :set-edges left top right (+ top w1-height))
		     (send w2 :set-edges left (+ top w1-height) right (+ top w1+2-height))
		     (send w3 :set-edges left (+ top w1+2-height) right bottom)
		     (send w1 :set-label nil)
		     (send w2 :set-label nil)
		     (send w3 :set-label nil)
		     (send w1 :expose nil :clean)	;1Make sure they are all there*
		     (send w2 :expose nil :clean)
		     (send w3 :expose nil :clean)
		     )))
	       (send frame :update-labels)
	       (values w1 w2 w3))))
	  ;1; Already have three windows... adjusting current ones is quicker than starting from scratch*
	  (t 
	   (let* ((w1 (zwei:window-sheet (first windows)))
		  (w2 (zwei:window-sheet (second windows)))
		  (w3 (zwei:window-sheet (third windows)))
		  (frame (zwei:window-frame (first windows))))
	     (multiple-value-bind (left top right bottom)
				  (send frame :inside-edges-without-mode-line-window)
	       (let* ((w1-height (ceiling (* (- bottom top) top-fraction)))
		      (w1+2-height (ceiling (* (- bottom top) top-and-middle-fraction)))
		      (w2-height (- w1+2-height w1-height)))
		 (multiple-value-bind (nil top1 nil bottom1) (send w1 :edges)
		   (multiple-value-bind (nil top2 nil bottom2) (send w2 :edges)
		     (unless (and (= w1-height (- bottom1 top1))
				  (= w2-height (- bottom2 top2)))
		       (send w1 :set-edges left top right (+ top w1-height))
		       (send w2 :set-edges left (+ top w1-height) right (+ top w1+2-height))
		       (send w3 :set-edges left (+ top w1+2-height) right bottom)
		       )))))
	     (send frame :update-labels)
	     (values w1 w2 w3))))))


(defvar 4*newsgroups-buffer* *nil "2The AMS Newsgroups Buffer.  Currently there can be only one.*")

(defun 4zmacs-andrew *()
  (unless *ams_prof-file*
    ;(setq *ams_prof-file* (make-pathname :name (user-name) :type "3AMS_prof*" :defaults (fs:user-homedir-pathname)))
    (setq *ams_prof-file* (pathname "3teak:/usr2/users/jwz/.AMS_prof.ti*"))
    )
  (clrhash *profile*)
  (let* ((newsgroups (parse-ams.prof *ams_prof-file*)))
    (multiple-value-bind (newsgroups-window summary-window message-window)
			 (three-windows-by-fraction 0.2 0.4)
      (let* ((ngs *newsgroups-buffer*)
	     sum msg)
	(cond ((null ngs)
	       (setq ngs (make-instance 'AMS-Newsgroups-Buffer :name "3Newsgroups*")
		     sum (make-instance 'AMS-Summary-Buffer :name "3Summary*" :newsgroups-buffer ngs)
		     msg (make-instance 'AMS-Message-Buffer :Name "3Message*" :summary-buffer sum))
	       (send ngs :set-summary-buffer sum)
	       (send sum :set-message-buffer msg)
	       (send ngs :set-saved-major-mode 'andrew-newsgroups-mode)
	       (send sum :set-saved-major-mode 'andrew-summary-mode)
	       (send msg :set-saved-major-mode 'andrew-message-mode)
	       (zwei:make-buffer-read-only ngs)
	       (zwei:make-buffer-read-only sum)
	       (zwei:make-buffer-read-only msg)
	       (setq *newsgroups-buffer* ngs))
	      (t
	       (setq sum (send ngs :summary-buffer)
		     msg (send sum :message-buffer))))
	(when newsgroups (send ngs :set-newsgroups newsgroups))
	(send ngs :grind-newsgroups)
	(send newsgroups-window :select)
	(setq zwei:*window* newsgroups-window)
	(send newsgroups-window :set-interval ngs)
	(send summary-window :set-interval sum)
	(send message-window :set-interval msg)
	(send ngs :select)
	))))

(zwei:defcom com-andrew "2Invoke the Andrew Message System bboard reader.*" ()
  (zmacs-andrew)
  zwei:dis-all)

(zwei:set-comtab zwei:*zmacs-comtab* () (zwei:make-command-alist '(com-andrew)))


(defmethod 4(zwei:zwei :after :mouse-select*) (&rest ignore)
  "2When mouse-selecting a Zmacs frame, make sure that the frame's *WINDOW*
  becomes the TV:SELECTED-WINDOW.  If we don't do this, then the cursor of the
  window in which the user clicked will be blinking, which is not always right,
  because that is not always the listening window.*"
  (let* ((selected-editor-window (funcall zwei:editor-closure 'symbol-value 'zwei:*window*)))
    (unless (eq self selected-editor-window)
      (send selected-editor-window :select))))


(defun 4(:property andrew-newsgroups-mode zwei:mouse-line-box-predicate)* (line)
  (getf (zwei:line-plist line) :newsgroup))

(defun 4(:property andrew-summary-mode zwei:mouse-line-box-predicate)* (line)
  (getf (zwei:line-plist line) :message))


(zwei:defmajor com-andrew-newsgroups-mode andrew-newsgroups-mode "2Andrew-Newsgroups*"
	       "2The major mode for being in the Newsgroups buffer.*" ()
  (setq zwei:*comtab* *ams-newsgroups-comtab*))

(zwei:defmajor com-andrew-summary-mode andrew-summary-mode "2Andrew-Summary*"
	       "2The major mode for being in the Summary buffer.*" ()
  (setq zwei:*comtab* *ams-summary-comtab*))

(zwei:defmajor com-andrew-message-mode andrew-message-mode "2Andrew-Message*"
	       "2The major mode for being in the Message buffer.*" ()
  (setq zwei:*comtab* *ams-message-comtab*))


(zwei:defcom 4com-read-andrew-newsgroup* "2Read the newsgroup which this line represents.*" ()
  (zwei:com-beginning-of-line)
  (send zwei:*interval* :select-newsgroup (zwei:point))
  (let* ((w (find (send zwei:*interval* :summary-buffer) zwei:*window-list*
		  :test #'eq :key #'(lambda (x) (send x :interval)))))
    (cond (w
	   (zwei:make-window-current w)
	   (zwei:must-redisplay w ZWEI:DIS-ALL)
	   (zwei:redisplay w :point nil nil t))
	  (t (send (send zwei:*interval* :summary-buffer) :select))))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-unsubscribe-newsgroup* "2Nuke the newsgroup which this line represents.*" ()
  (zwei:com-beginning-of-line)
  (let* ((ng (getf (zwei:line-plist (zwei:bp-line (zwei:point))) :newsgroup)))
    (unless ng (zwei:barf))
    (send self :set-newsgroups (delete ng (send self :newsgroups) :test #'eq))
    (zwei:with-read-only-suppressed (zwei:*interval*)
      (zwei:delete-interval (zwei:point) (zwei:beg-line (zwei:point) 1 t))))
  ZWEI:DIS-TEXT)


(zwei:defcom 4com-exit-andrew-newsgroup* "2Quit this newsgroup and reselect the newsgroups list.*" ()
  (let* ((w (find (send zwei:*interval* :newsgroups-buffer) zwei:*window-list*
		  :test #'eq :key #'(lambda (x) (send x :interval)))))
    (if w
	(zwei:make-window-current w)
	(send (send zwei:*interval* :newsgroups-buffer) :select)))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-exit-andrew-message* "2Go back to the summary buffer from the message buffer.*" ()
  (let* ((w (find (send zwei:*interval* :summary-buffer) zwei:*window-list*
		  :test #'eq :key #'(lambda (x) (send x :interval)))))
    (if w
	(zwei:make-window-current w)
	(send (send zwei:*interval* :summary-buffer) :select)))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-summary-message-next-page* "2Look at this message.*" ()
  (cond ((typep zwei:*interval* 'AMS-Message-Buffer)
	 ;1; If this is called from within a message buffer, just scroll by one page.*
	 (zwei:com-next-screen))
	(t
	 (let* ((msg (getf (zwei:line-plist (zwei:bp-line (zwei:point))) :message))
		(msgbuf (send zwei:*interval* :message-buffer))
		(w (find msgbuf zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
	   (cond (w
		  ;1; If the message buffer is visible on the screen...*
		  (cond ((eq msg (send msgbuf :message))
			 ;1; If the message is current, scroll one page.*
			 (zwei:recenter-window-relative w (- (zwei:window-n-plines w) zwei:*next-screen-context-lines*)))
			(t
			 ;1; If this is a new message, then display it.*
			 (send zwei:*interval* :select-message (zwei:point))
			 (zwei:redefine-fonts w (font-names-to-font-alist (send msgbuf :get-attribute :fonts)))
			 (zwei:must-redisplay w ZWEI:DIS-ALL)
			 (zwei:redisplay w :none)
			 (let* ((ng (send zwei:*interval* :newsgroup))
				(prof (newsgroup-prof ng))
				(msg (send msgbuf :message)))
			   (setf (prof-time prof) (message-date msg)))
			 )))
		 (t
		  ;1; If the message buffer is not visible, then select it and display this message.*
		  (send zwei:*interval* :select-message (zwei:point))
		  (send msgbuf :select))))))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-summary-message-previous-page* "2Look at this message.*" ()
  (cond ((typep zwei:*interval* 'AMS-Message-Buffer)
	 ;1; If this is called from within a message buffer, just scroll back one page.*
	 (zwei:com-previous-screen))
	(t
	 (let* ((msgbuf (send zwei:*interval* :message-buffer))
		(w (find msgbuf zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
	   (cond (w
		  ;1; If the message buffer is visible on the screen, scroll back one page.*
		  (zwei:recenter-window-relative w (- (- (zwei:window-n-plines w) zwei:*next-screen-context-lines*))))
		 (t
		  (zwei:barf))))))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-toggle-reformatted-headers* "2Look at real headers.*" ()
  (let* ((msgbuf (etypecase zwei:*interval*
		   (AMS-MESSAGE-BUFFER zwei:*interval*)
		    (AMS-SUMMARY-BUFFER (send zwei:*interval* :message-buffer))
		    (AMS-NEWSGROUPS-BUFFER (send (send zwei:*interval* :summary-buffer) :message-buffer))))
	 (w (find msgbuf zwei:*window-list* :test #'eq :key #'(lambda (x) (send x :interval)))))
    (unless msgbuf (zwei:barf))
    (zwei:with-read-only-suppressed (msgbuf)
      (if (get msgbuf :headers-formatted)
	  (send msgbuf :show-unformatted-headers)
	  (send msgbuf :show-formatted-headers)))
    (when w
      (zwei:must-redisplay w zwei:dis-text)
      (zwei:redisplay w)))
  ZWEI:DIS-TEXT)

(defprop 4com-mouse-select-or-mark-newsgroup* "2Select Newsgroup*" :mouse-short-documentation)
(zwei:defcom 4com-mouse-select-or-mark-newsgroup* "2Select newsgroup under mouse, or do mouse region marking.*"
	     ()
  (zwei:com-mouse-mark-region)
  (typecase zwei:*interval*
    (AMS-NEWSGROUPS-BUFFER (com-read-andrew-newsgroup))
    (AMS-SUMMARY-BUFFER    (com-summary-message-next-page)))
  ZWEI:DIS-BPS)


(zwei:defcom 4com-visit-message-as-text* "2Snarf the raw datastream of this message into a Zmacs buffer.*" ()
  (let* ((w (or (find (send *newsgroups-buffer* :summary-buffer) zwei:*window-list*
		      :test #'eq :key #'(lambda (x) (send x :interval)))
		(zwei:barf "3No summary window*")))
	 (msg (or (getf (zwei:line-plist (zwei:bp-line (zwei:window-point w))) :message)
		  (zwei:barf "3No message*")))
	 (newsgroup (send (send *newsgroups-buffer* :summary-buffer) :newsgroup))
	 (path (make-pathname :raw-name (string-append #\+ (message-id-string msg)) :type :UNSPECIFIC
			      :defaults (newsgroup-msgdir-pathname newsgroup))))
    (zwei:find-file path nil nil t t))
  ZWEI:DIS-NONE)


(zwei:defcom 4com-andrew-summary-menu *"2foo.*" ()
  (let* ((zwei:item-list (mapcar #'zwei:make-menu-command-1
				 (zwei:make-command-alist
				   '(;4com-andrew-summary-documentation*
				     ;4com-andrew-forward-message*
				     ;4com-reply-to-andrew-message*
				     com-save-AMS-profile
				     com-andrew-expand-all
				     com-unsubscribe-newsgroup
				     com-toggle-reformatted-headers
				     ;4com-andrew-summary-documentation*
				     com-exit-andrew-newsgroup
				     com-visit-message-as-text
				     ))))
	 (zwei:*label-computing-function*
	   #'(lambda () '(:font fonts:hl12b :string "2Andrew Summary Commands*"))))
    (declare (special zwei:item-list zwei:*label-computing-function*))

    (let* ((w (tv:window-under-mouse :interval))
	   (bp (and w (zwei:mouse-bp w)))
	   (line (and bp (zwei:bp-line bp)))
	   (diagram (and line (getf (zwei:line-plist line) :diagram))))
      (when diagram
	(typecase diagram
	  (RASTER-DIAGRAM-LINE
	   (setq zwei:item-list
		 (append zwei:item-list
			 (mapcar #'zwei:make-menu-command-1
				 (zwei:make-command-alist
				   '(com-view-diagram
				      com-save-diagram
				      ))))))
	  (FAD-DIAGRAM-LINE
	   (setq zwei:item-list
		 (append zwei:item-list
			 (mapcar #'zwei:make-menu-command-1
				 (zwei:make-command-alist
				   '(com-animate-diagram
				     com-animate-diagram-slowly
				      ;1 *com-save-diagram
				      ))))))
	  )))
    
    (zwei:make-menu-command-driver)))


(zwei:defcom 4com-save-ams-profile* "2Save the current state to a .AMS.prof file.*" ()
  (let* ((ng-buf (etypecase zwei:*interval*
		   (AMS-NEWSGROUPS-BUFFER zwei:*interval*)
		   (AMS-SUMMARY-BUFFER (send zwei:*interval* :newsgroups-buffer))
		   (AMS-MESSAGE-BUFFER (send (send zwei:*interval* :summary-buffer) :newsgroups-buffer))))
	 (ngs (send ng-buf :newsgroups))
	 (pathname (pathname (zwei:read-defaulted-pathname "2Save profile where?*" *ams_prof-file* nil nil :write))))
    (write-ams.prof ngs pathname)
    (setq *ams_prof-file* pathname)
    (zwei:typein-line "3~:|~A written.~%*" (truename pathname))
    zwei:dis-none))

(zwei:defcom 4com-andrew-expand-all* "2Get all the newsgroups.*" ()
  (let* ((ng-buf (etypecase zwei:*interval*
		   (AMS-NEWSGROUPS-BUFFER zwei:*interval*)
		   (AMS-SUMMARY-BUFFER (send zwei:*interval* :newsgroups-buffer))
		   (AMS-MESSAGE-BUFFER (send (send zwei:*interval* :summary-buffer) :newsgroups-buffer)))))
    (send ng-buf :set-newsgroups
	  (parse-subscriptionmap (make-pathname :host *andrew-host* :raw-directory *andrew-bb-root*
						:name "3.SubscriptionMap*" :type :unspecific)))
    (send ng-buf :grind-newsgroups)
    )
  ZWEI:DIS-ALL)


(zwei:defcom 4com-view-diagram* "2Look at this image in a popup window.*" ()
  (let* ((w (tv:window-under-mouse :interval))
	 (bp (and w (zwei:mouse-bp w)))
	 (line (and bp (zwei:bp-line bp)))
	 (diagram (and line (getf (zwei:line-plist line) :diagram))))
    (cond ((null diagram) (beep))
	  (t (view-bitmap (send (send diagram :raster) :bitmap))))
    ZWEI:DIS-NONE))


(zwei:defcom 4com-save-diagram* "2Save this image in a prompted-for file.*" ()
  (let* ((w (tv:window-under-mouse :interval))
	 (bp (and w (zwei:mouse-bp w)))
	 (line (and bp (zwei:bp-line bp)))
	 (diagram (and line (getf (zwei:line-plist line) :diagram))))
    (cond ((null diagram) (beep))
	  (t (let* ((pathname (zwei:read-defaulted-pathname "3Save this image to what file?*" nil "3BITMAP*" nil :write)))
	       (when pathname
		 (tv:write-bit-array-file pathname (send (send diagram :raster) :bitmap))
		 (zwei:typein-line "3~:|~A written.~%*" (truename pathname))
		 ))))
    ZWEI:DIS-NONE))

(zwei:defcom 4com-animate-diagram* "2Animate this FAD diagram.*" ()
  (let* ((w (tv:window-under-mouse :interval))
	 (bp (and w (zwei:mouse-bp w)))
	 (line (and bp (zwei:bp-line bp)))
	 (diagram (and line (getf (zwei:line-plist line) :diagram))))
    (cond ((null diagram) (beep))
	  (t (animate-fad-diagram diagram w (if (and zwei:*numeric-arg-p* (plusp zwei:*numeric-arg*))
						zwei:*numeric-arg*
						10))))
    ZWEI:DIS-NONE))

(zwei:defcom 4com-animate-diagram-slowly* "2Animate this FAD diagram.*" ()
  (let* ((zwei:*numeric-arg-p* t)
	 (zwei:*numeric-arg* 50))
    (com-animate-diagram)))




(defparameter *4basic-AMS-comtab**
	      (let* ((c (zwei:set-comtab 'basic-AMS-comtab
			 '(#\1 zwei:com-numbers #\2 zwei:com-numbers #\3 zwei:com-numbers #\4 zwei:com-numbers
			   #\5 zwei:com-numbers #\6 zwei:com-numbers #\7 zwei:com-numbers #\8 zwei:com-numbers
			   #\9 zwei:com-numbers #\0 zwei:com-numbers #\- zwei:com-negate-numeric-arg
			   #\M zwei:com-mail
			   #\N zwei:com-down-real-line
			   #\P zwei:com-up-real-line
			   #\S com-save-AMS-profile
			   #\< zwei:com-goto-beginning
			   #\> zwei:com-goto-end
			   #\Rubout zwei:com-up-real-line
			   #\Space zwei:com-down-real-line
			   )
			 nil)))
		(zwei:set-comtab-indirection c zwei:*zmacs-comtab*)
		c))

(defparameter *4ams-newsgroups-comtab**
	      (let* ((c (zwei:set-comtab 'AMS-newsgroups-comtab
			  `(#\? 4com-andrew-newsgroups-documentation*
			    #\Help 4com-andrew-newsgroups-documentation*
			    #\Q 4com-exit-andrew*
			    #\U com-unsubscribe-newsgroup
			    #\space com-read-andrew-newsgroup
			    #\End 4com-exit-andrew*
			    #\Abort 4com-abort-andrew*
			    #\mouse-L-1 com-mouse-select-or-mark-newsgroup
			    #\Mouse-R-1 ,(zwei:make-menu-command
					   '(com-save-ams-profile
					     ;4com-andrew-newsgroups-documentation*
					     ;4com-andrew-newsgroups-documentation*
					     ;4com-exit-andrew*
					     com-unsubscribe-newsgroup
					     com-read-andrew-newsgroup
					     com-andrew-expand-all
					     ;4com-exit-andrew*
					     ;4com-abort-andrew*
					     com-mouse-select-or-mark-newsgroup
					     )
					   #'(lambda (&rest ignore)
					       '(:font fonts:hl12b :string "2Andrew Newsgroups Commands*")))
			    )
			  nil)))
		(zwei:set-comtab-indirection c *basic-ams-comtab*)
		c))

(defparameter *4ams-summary-comtab**
	      (let* ((c (zwei:set-comtab 'AMS-newsgroups-comtab
			  `(#\? 4com-andrew-summary-documentation*
			    #\A com-andrew-expand-all
			    #\F 4com-andrew-forward-message*
			    #\Q com-exit-andrew-newsgroup
			    #\R 4com-reply-to-andrew-message*
			    #\S com-save-AMS-profile
			    #\U com-unsubscribe-newsgroup
			    #\H com-toggle-reformatted-headers
			    #\Space com-summary-message-next-page
			    #\Rubout com-summary-message-previous-page
			    #\Help 4com-andrew-summary-documentation*
			    #\End com-exit-andrew-newsgroup
			    #\Abort com-exit-andrew-newsgroup
			    #\mouse-L-1 com-mouse-select-or-mark-newsgroup
			    #\Mouse-R-1 com-andrew-summary-menu
			    ))))
		(zwei:set-comtab-indirection c *basic-ams-comtab*)
		c))

(defparameter *4ams-message-comtab**
	      (let* ((c (zwei:set-comtab 'AMS-newsgroups-comtab
			  '(#\? 4com-andrew-message-documentation*
			    #\Q com-exit-andrew-message
			    #\H com-toggle-reformatted-headers
			    #\Help 4com-andrew-message-documentation*
			    #\End com-exit-andrew-message
			    #\Abort com-exit-andrew-message
			    )
			  nil)))
		(zwei:set-comtab-indirection c *ams-summary-comtab*)
		c))



(defun 4view-bitmap *(bitmap &optional max-width max-height)
  (unless max-width  (setq max-width  (- (tv:sheet-inside-width  tv:default-screen) 10)))
  (unless max-height (setq max-height (- (tv:sheet-inside-height tv:default-screen) 20)))
  (let* ((width (array-dimension bitmap 1))
	 (height (array-dimension bitmap 0))
	 (window (make-instance 'w:window :inside-size (list (min width max-width) (min height max-height))
				:label '(:font fonts:hl12b :string "2View Bitmap*")))
	 (fromx 0)
	 (fromy 0)
	 (alu (if tv:*current-screen-color* tv:alu-seta tv:alu-setca)))
    (send window :expose-near '(:mouse))
    (send window :select)
    (unwind-protect
	(loop
	  (send window :bitblt alu (tv:sheet-inside-width window) (tv:sheet-inside-height window)
		bitmap fromx fromy 0 0)
	  (let* ((c (tv:read-any window)))
	    (case c
	      (#\ (incf fromx))
	      (#\ (decf fromx))
	      (#\ (incf fromy))
	      (#\ (decf fromy))
	      (#\Control- (incf fromx 10))
	      (#\Control- (decf fromx 10))
	      (#\Control- (incf fromy 10))
	      (#\Control- (decf fromy 10))
	      (#\Meta- (incf fromx 40))
	      (#\Meta- (decf fromx 40))
	      (#\Meta- (incf fromy 40))
	      (#\Meta- (decf fromy 40))
	      (#\Control-V (incf fromy (round (* (tv:sheet-inside-height window) 2/3))))
	      (#\Meta-V    (decf fromy (round (* (tv:sheet-inside-height window) 2/3))))
	      (#\Control-Sh-V (incf fromx (round (* (tv:sheet-inside-width window) 2/3))))
	      (#\Meta-Sh-V    (decf fromx (round (* (tv:sheet-inside-width window) 2/3))))
	      (#\Control-Space (setq alu (if (= alu tv:alu-seta) tv:alu-setca tv:alu-seta)))
	      (#\Space (return))
	      (#\End (return))
	      (#\Abort (return))
	      (#\Help
	       (send window :set-current-font fonts:hl12 t)
	       (format window "3~:|~2%  View Bitmap Commands:~2%~:{~6t~:C~30t~A~%~}~2%*"
		       '((#\ "2scroll right.*") (#\ "2scroll left.*") (#\ "2scroll down.*") (#\ "2scroll up.*") (#\C- "2scroll right faster.*")
			 (#\C- "2scroll left faster.*") (#\C- "2scroll down faster.*") (#\C- "2scroll up faster.*")
			 (#\C-V "2scroll down one page.*") (#\M-V "2scroll up one page.*") (#\C-Sh-V "2scroll left one page.*")
			 (#\M-Sh-V "2scroll right one page.*") (#\C-Space "2invert image.*") (#\Space "2done.*")))
	       (send window :set-current-font fonts:hl12i t)
	       (format window "3~6t(any key to flush)*")
	       (tv:read-any window)
	       (dolist (blinker (tv:sheet-blinker-list window))
		 (send blinker :set-visibility :off)))
	      (t
	       (if (consp c)
		   (return)
		   (beep))))
	    (setq fromx (max 0 (min fromx (- width (tv:sheet-inside-width window))))
		  fromy (max 0 (min fromy (- height (tv:sheet-inside-height window)))))
	    (send window :clear-input)
	    ))
      (send window :kill))))


#|

/* The current configuration of UATTRs (user attributes) means that the
    last predefined attribute would be number 135.  */

#define AMS_ATT_LAST_UATTR ((21 * 8) - 1)
#define AMS_ATT_UATTR(a) (AMS_ATT_LAST_UATTR - (a))
#define AMS_NUM_UATTRS 32
#define AMS_ATTRNAMEMAX 16

/* The following defines the subscription file, which lists the subscription
	information for the current user */

#define AMS_SUBSCRIPTIONMAPFILE ".SubscriptionMap"

#define AMS_UNSUBSCRIBED 0
#define AMS_ASKSUBSCRIBED 1
#define AMS_ALWAYSSUBSCRIBED 2
#define AMS_SHOWALLSUBSCRIBED 3
#define AMS_PRINTSUBSCRIBED 4

/* Definitions for formulation of a reply file */

#define AMS_REPLY_FRESH 1
#define AMS_REPLY_FORWARD 2 /* Forward stripping out formatting */
#define AMS_REPLY_WIDE 3
#define AMS_REPLY_SENDER 4
#define AMS_REPLY_WIDER 5
#define AMS_REPLY_REDRAFT 6
#define AMS_REPLY_ACK 7
#define AMS_REPLY_FORWARD_FMT 8 /* Forward with formatting */

3;; Writing out the .AMS.prof file.
;;*
    for (i=0; i<NumSubsInUse; ++i) {
	if (SubsInUserOrder[i].sname != NULL) {
	    fprintf(fp, "%s %s %d %s %d\n", SubsInUserOrder[i].sname, SubsInUserOrder[i].key, SubsInUserOrder[i].status, SubsInUserOrder[i].time64, SubsInUserOrder[i].filedate);
	}


/* The following is the structure used for a subscription/profile entry */

struct SubscriptionProfile {
    char *sname; /* short name */
    char *key; /* long name */
    int status; /* subscription status code */
    int pathelt; /* -1 if not on path */
    int priority; /* For subscription ordering */
    char time64[AMS_DATESIZE]; /* From SetAssociatedTime */
    long filedate; /* Ditto */
    int NeedsFreed; /* If zero, sname & key are part of one big malloc */
    int HasChanged; /* for use in NameChangedMap */
};

3 |*#



;1;;; Here's another kind of rasterfile that andrew uses.*

(defvar 4*bit-flippage* *(let* ((a (make-array 256 :element-type '(unsigned-byte 8))))
			   (dotimes (i 256)
			     (let* ((flip-byte 0))
			       (dotimes (j 8)
				 (setq flip-byte (dpb (ldb (byte 1 (- 7 j)) i) (byte 1 j) flip-byte)))
			       (setf (aref a i) flip-byte)))
			   a)
  "2A table for quickly reversing the order of bits in a byte.*")

(proclaim '(type (array (unsigned-byte 8) (256)) *bit-flippage*))

(defun 4extract-andrew-bitmap *(datastream)
  (typecase datastream
    ((ARRAY BIT 2) datastream)
    (RASTER (send datastream :bitmap))
    (NULL nil)
    (t
     (or (let* ((r (find-if #'(lambda (x) (typep x 'raster)) (send datastream :inferiors))))
	   (and r (send r :bitmap)))
	 (dolist (inf (send datastream :inferiors))
	   (let* ((i (extract-andrew-bitmap inf)))
	     (when i (return i))))))))
  
(defun 4read-andrew-bitmap* (file)
  "2Sucks a bitmap out of a datastream file.*"
  (extract-andrew-bitmap (read-datastream-file file nil)))

(defun 4read-andrew2-bitmap* (file)
  "2Some other random andrew format.*"
  (sys:with-open-decompressing-stream (stream file :direction :input :characters nil :byte-size 8)
    (labels ((read-short () (dpb (read-byte stream) (byte  8  8) (read-byte stream)))
	     (read-long  () (dpb (read-short)       (byte 16 16) (read-short))))
      (let* ((n1 (read-short))
	     (n2 (read-short))
	     (dn1 #xF100)
	     (dn2 #x40BB))
	
	(when (and (= (ldb (byte 8 8) n1) (char-code #\\))
		   (= (ldb (byte 8 0) n1) (char-code #\b))	;1; It's a datastream file if it starts*
		   (= (ldb (byte 8 8) n2) (char-code #\e))	;1; with 5\beg6indata{***
		   (= (ldb (byte 8 0) n2) (char-code #\g)))
	  (warn "2This is a datastream file, not a binary raster file.*")
	  (return-from READ-ANDREW2-BITMAP (read-andrew-bitmap file)))
	
	(unless (and (= n1 dn1) (= n2 dn2))
	  (cerror "2Try anyway3.**" "2Bad magic number.3~%~**
		3Desired magic number:      ~8,'0b ~8,'0b ~8,'0b ~8,'0b~%~*
		3Encountered magic number:  ~8,'0b ~8,'0b ~8,'0b ~8,'0b (~a ~a ~a ~a)~%*"
		  (ldb (byte 8 8) dn1) (ldb (byte 8 0) dn1) (ldb (byte 8 8) dn2) (ldb (byte 8 0) dn2)
		  (ldb (byte 8 8) n1) (ldb (byte 8 0) n1) (ldb (byte 8 8) n2) (ldb (byte 8 0) n2)
		  (make-char (ldb (byte 8 8) n1)) (make-char (ldb (byte 8 0) n1))
		  (make-char (ldb (byte 8 8) n2)) (make-char (ldb (byte 8 0) n2))
		  ))
	(let* ((width  (read-long))
	       (height (read-long))
	       (depth  (read-short)))
	  (assert (= 1 depth) () "2This image is more than one bit deep.*")
	  (let* ((width8  (+  8 (*  8 (floor (1- width)  8))))
		 (width32 (+ 32 (* 32 (floor (1- width) 32))))
		 (bitmap (make-array (list height width32) :element-type 'bit)))
	    (cond ((= width8 width32)
		   (let* ((indirect (make-array (* height (ceiling width32 8))
						:element-type '(unsigned-byte 8) :displaced-to bitmap)))
		     (catch 'EOF
		       (sys:condition-resume '(sys:end-of-file :bugout t ("3return what we've got*")
							       (lambda (ignore) (throw 'EOF nil)))
			 (dotimes (i (length indirect))
			   (setf (aref indirect i) (read-byte stream)))))
		     (sys:%buffer-char-map indirect 0 (length indirect) indirect 0 (length indirect)
					   *bit-flippage* #xFF 0)))
		  (t ;1;*
		   ;1; The image is stored in the file in scanlines which are a multiple of 8 instead of 32.*
		   ;1; This means we must read it a scanline at a time.  C'est la vie...*
		   (let* ((bitmap-byte-width (floor (array-dimension bitmap 1) 8))
			  (scanline-byte-width (floor width8 8))
			  (scanline (make-array scanline-byte-width :element-type '(unsigned-byte 8)
						:displaced-to bitmap :displaced-index-offset 0))
			  (translated-type (array-type scanline))
			  )
		     (declare (type (array (unsigned-byte 8) 1) scanline)
			      (fixnum bitmap-byte-width scanline-byte-width))
		     (catch 'EOF
		       (sys:condition-resume '(sys:end-of-file :bugout t ("3return what we've got*")
							       (lambda (ignore) (throw 'EOF nil)))
			 (dotimes (i height)
			   (sys:change-indirect-array scanline translated-type scanline-byte-width
						      bitmap (* i bitmap-byte-width))
			   (send stream :string-in t scanline))))
		     (let* ((a (make-array (* height (round (array-dimension bitmap 1) 8))
					   :element-type '(unsigned-byte 8) :displaced-to bitmap)))
		       (declare (type (array (unsigned-byte 8) 1) a))
		       (sys:%buffer-char-map a 0 (length a) a 0 (length a) *bit-flippage* #xFF 0))
		     )))
	    bitmap))))))
