/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: headers.c,v 1.18 1994/02/09 19:41:24 janssen Exp $
*/

#include "m3.h"
#include <time.h>

extern void declare_object_type(Type type, Context c);

#define IsHexDigit(x)	((((x)>='0')&&((x)<='9'))||(((x)>='a')&&((x)<='f'))||(((x)>='A')&&((x)<='F')))
#define HexValue(x)	(((x)<='9')?((x)-'0'):(((x)<='F')?((x)-'A'):(((x)<='f')?((x)-'a'):0)))

static boolean FirstItem;
static void sort_types_for_declaration (Type type,list sorted);

static void ConvertStringForm (char *buf, string str)
{
  register char *p, *q;

  for (p = str, q = buf;  *p != '\0';)
    {
      if (*p == '#')
	{
	  ++p;
	  if (*p == 'n')
	    *q++ = '\\', *q++ = 'n', p++;
	  else if (*p == 'r')
	    *q++ = '\\', *q++ = 'r', p++;
	  else if (IsHexDigit(*p))
	    {
	      unsigned int c = 0;
	      unsigned int i;

	      while (IsHexDigit(*p) && (i = c * 16 + HexValue(*p)) < 256)
		{
		  c = i;
		  ++p;
		}
	      sprintf (q, "\\%o", c);
	      q += strlen(q);
	    }
	  else if (*p == '\\' || *p == '"')
	    *q++ = '\\', *q++ = *p++;
	  else
	    *q++ = *p++;
	}
      else if (*p == '\\' || *p == '"')
	*q++ = '\\', *q++ = *p++;
      else
	*q++ = *p++;
    }
}

static void declare_constant (Constant c, Context context)
{
  enum PrimitiveTypes t = type_basic_type(c->type);

  if (c->interface != context->interface)
    return;

  if ((t == cardinal_Type || t == shortcardinal_Type || t == byte_Type)
      && (c->value->type == integer_Type))
    {
      fprintf (context->file, "CONST %s : %s = %u;\n",
	       iname(M3_CONSTANT_NAME(c), c->interface),
	       (t == cardinal_Type) ? "CARDINAL" : ((t == byte_Type) ? "[0..256]" : "[0..65535]"),
	       c->value->val.i.value);
    }
  else if ((t == integer_Type || t == shortinteger_Type)
      && (c->value->type == integer_Type))
    {
      fprintf (context->file, "CONST %s : %s = %s%u;\n",
	       iname(M3_CONSTANT_NAME(c), c->interface),
	       (t == integer_Type) ? "INTEGER" : "[-32768..32767]",
	       (c->value->val.i.sign < 0) ? "-" : "",
	       c->value->val.i.value);
    }
  else if ((t == real_Type || t == shortreal_Type)
      && (c->value->type == real_Type))
    {
      fprintf (context->file, "CONST %s : %s = %s%s.%se%d;\n",
	       iname(M3_CONSTANT_NAME(c), c->interface),
	       (t == real_Type) ? "LONGREAL" : "REAL",
	       (c->value->val.r.sign < 0) ? "-" : "",
	       c->value->val.r.value, (c->value->val.r.fraction == NULL) ? "0" : c->value->val.r.fraction,
	       c->value->val.r.exponent);
    }
  else if (t == boolean_Type)
    {
      fprintf (context->file, "CONST %s : BOOLEAN = %s;\n",
	       iname(M3_CONSTANT_NAME(c), c->interface),
	       (c->value->val.b) ? "TRUE" : "FALSE");
    }
  else if (t == shortcharacter_Type)
    {
      if (c->value->type == sequence_Type)
	{
	  char buf[1000];

	  ConvertStringForm (buf, c->value->val.s);
	  fprintf (context->file, "CONST %s : TEXT = \"%s\"\n",
		   iname(M3_CONSTANT_NAME(c), c->interface), buf);
	}
      else
	fprintf (context->file, "CONST %s : CHAR = '\\%o';\n",
		 iname(M3_CONSTANT_NAME(c), c->interface),
		 c->value->val.i.value);
    }
  else
    fprintf (stderr, "Invalid constant, %s, encountered.  Bad type for ILU constant.\n", name_base_name(c->name));
}

static boolean MatchPointer (refany p1, refany p2)
{
  return (p1 == p2);
}

static void SortArgTypes (Argument arg, list sorted)
{
  sort_types_for_declaration (arg->type, sorted);  
}

static void SortMethodTypes (Procedure m, list sorted)
{
  if (!m->returnOptional)
    sort_types_for_declaration (m->returnType, sorted);
  list_enumerate (m->arguments, (void (*) (void *, void *)) SortArgTypes, sorted);
}

static void sort_types_for_declaration (Type type, list sorted)
{
  enum PrimitiveTypes t;
  static list pending = NULL;

  if (type == NULL)
    return;

  if (pending == NULL)
    pending = new_list();

  t = type_basic_type(type);

  if (list_find (sorted, MatchPointer, type) != NULL || list_find (pending, MatchPointer, type) != NULL)
    return;
  else
    {
      list_insert (pending, type);
      if (type->importInterfaceName == NULL &&
	  (t == record_Type || t == array_Type || t == union_Type || t == sequence_Type || t == object_Type))
	{
	  switch (t)
	    {
	    case record_Type:
	      
	      list_enumerate(type_description(type)->structuredDes.record, (void (*) (void *, void *)) SortArgTypes, sorted);
	      break;
	      
	    case union_Type:
	      
	      list_enumerate(type_description(type)->structuredDes.uniond.types, (void (*) (void *, void *)) SortArgTypes, sorted);
	      break;
	      
	    case array_Type:
	      
	      if (!type_description(type)->structuredDes.array.optional)
		sort_types_for_declaration (type_description(type)->structuredDes.array.type, sorted);
	      break;

	    case sequence_Type:

	      sort_types_for_declaration (type_description(type)->structuredDes.sequence.type, sorted);
	      break;

	    case object_Type:

	      if (type->marked)
		break;
	      list_enumerate(class_object(type)->superclasses, (void (*) (void *, void *)) sort_types_for_declaration, sorted);
	      list_enumerate(class_object(type)->methods, (void (*) (void *, void *)) SortMethodTypes, sorted);
	      type->marked = TRUE;
	      break;

	    default:
	      break;

	    }
	}
      if (list_find(sorted, MatchPointer, type) == NULL)
	list_insert(sorted, type);
      list_remove (pending, type);
    }
}

static void GenerateRecordField (Argument a, Context context)
{
  fprintf (context->file, "   %s : %s;\n",
	   m3_argument_name(a), iname(M3_TYPE_NAME(a->type), a->type->interface));
}

static void GenerateRecordDeclaration (Type t, Context context)
{
  TypeDescription type_description();

  fprintf (context->file, "TYPE %s = RECORD\n", M3_SHORT_TYPE_NAME(t));
  list_enumerate (type_description(t)->structuredDes.record, (void (*) (void *, void *)) GenerateRecordField, context);
  fprintf (context->file, "END;\n");
}

struct double_s {
  Type t;
  Context c;
};

static void GenerateUnionField (Argument a, struct double_s *d)
{
  fprintf (d->c->file, "TYPE %s = %s BRANDED OBJECT v: %s END;\n",
	   UnionTypeName(d->t, a->type),
	   M3_SHORT_TYPE_NAME(d->t), iname(M3_TYPE_NAME(a->type), a->type->interface));
}

static void GenerateUnionDeclaration (Type t, Context context)
{
  list e = type_description(t)->structuredDes.uniond.types;
  struct double_s s;

  fprintf (context->file, "TYPE %s = BRANDED OBJECT END;  (* NIL not allowed *)\n", M3_SHORT_TYPE_NAME(t));
  s.c = context;
  s.t = t; 
  list_enumerate (e, (void (*) (void *, void *)) GenerateUnionField, &s);
}

static void PrintEnumField (EnumField e, Context context)
{
  if (FirstItem)
    FirstItem = FALSE;
  else
    fprintf (context->file, ",\n");

  fprintf (context->file, "  %s", m3_string(e->name));
}

static void GenerateEnumerationDeclaration (Type t, Context context)
{
  boolean oldFirst;

  fprintf (context->file, "TYPE %s = {\n", M3_SHORT_TYPE_NAME(t));
  context->class = t;
  oldFirst = FirstItem;  FirstItem = TRUE;
  list_enumerate (type_description(t)->structuredDes.enumeration, (EnumProc) PrintEnumField, context);
  FirstItem = oldFirst;
  fprintf (context->file, "};\n");
}

void ListArrayDimension (cardinal dim, Context context)
{
  fprintf (context->file, "ARRAY [0..%u] OF ", (dim < 1) ? 0 : dim - 1);
}

static void GenerateArrayDeclaration (Type array, Context context)
{
  Type atype = type_description(array)->structuredDes.array.type;

  fprintf (context->file, "TYPE %s = ", M3_SHORT_TYPE_NAME(array));
  context->class = array;
  list_enumerate (type_description(array)->structuredDes.array.dimensions, (EnumProc) ListArrayDimension, context);
  if (type_basic_type(atype) == byte_Type)
    fprintf (context->file, "Ilu.PackedByte;\n");
  else if (type_basic_type(atype) == shortcharacter_Type)
    fprintf (context->file, "Ilu.PackedShortChar;\n");
  else
    fprintf (context->file, "%s;\n", iname(M3_TYPE_NAME(atype), atype->interface));
}

static void GenerateOptionalDeclaration (Type t, Context context)
{
  enum PrimitiveTypes bt = type_basic_type (type_description(t)->structuredDes.optional);
  Type otype = type_description(t)->structuredDes.optional;
  if (bt == object_Type OR bt == sequence_Type OR bt == union_Type)
  fprintf (context->file, "TYPE %s = %s;  (* Union of %s and NULL, so NIL allowed *)\n",
	   M3_SHORT_TYPE_NAME(t), iname(M3_TYPE_NAME(otype), otype->interface),
	   M3_SHORT_TYPE_NAME(otype));
  else
    fprintf (context->file, "TYPE %s = REF %s;  (* Union of %s and NULL, so NIL allowed *)\n",
	     M3_SHORT_TYPE_NAME(t), iname(M3_TYPE_NAME(otype), otype->interface),
	     M3_SHORT_TYPE_NAME(otype));
}

static void GenerateSequenceDeclaration (Type s, Context context)
{ 
  Type d = type_description(s)->structuredDes.sequence.type;
  if (type_basic_type(d) == shortcharacter_Type)
    fprintf (context->file, "TYPE %s = TEXT;  (* NIL not allowed *)\n", M3_SHORT_TYPE_NAME(s));
  else if (type_basic_type(d) == byte_Type)
    fprintf (context->file, "TYPE %s = REF ARRAY OF BITS 8 FOR Ilu.Byte;  (* NIL not allowed *)\n",
	     M3_SHORT_TYPE_NAME(s));
  else
    fprintf (context->file, "TYPE %s = REF ARRAY OF %s;  (* NIL not allowed *)\n", M3_SHORT_TYPE_NAME(s),
	     iname(M3_TYPE_NAME(d), d->interface));
}

void declare_type (Type t, Context context)
{
/*
  printf ("%s %d%s  file is 0x%x\n", type_name(t), list_size(t->refs),
	  t->builtIn ? " builtin" : "", context->file);
*/
  if (t->builtIn)
    return;

  if (t->importInterfaceName != NULL)
    return;

  switch (type_basic_type(t))
    {
    case void_Type:
      break;

    case integer_Type:
      fprintf (context->file, "TYPE %s = INTEGER;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case cardinal_Type:
      fprintf (context->file, "TYPE %s = CARDINAL;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case shortinteger_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.ShortInteger;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case shortcardinal_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.ShortCardinal;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case longinteger_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.LongInteger;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case longcardinal_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.LongCardinal;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case real_Type:
      fprintf (context->file, "TYPE %s = LONGREAL;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case shortreal_Type:
      fprintf (context->file, "TYPE %s = REAL;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case longreal_Type:
      fprintf (context->file, "TYPE %s = LONGREAL;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case boolean_Type:
      fprintf (context->file, "TYPE %s = BOOLEAN;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case byte_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.Byte;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case character_Type:
      fprintf (context->file, "TYPE %s = IluRuntime.Character;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case shortcharacter_Type:
      fprintf (context->file, "TYPE %s = CHAR;\n", M3_SHORT_TYPE_NAME(t));
      break;

    case optional_Type:
      GenerateOptionalDeclaration(t, context);
      break;

    case array_Type:
      GenerateArrayDeclaration (t, context);
      return;

    case object_Type:
      declare_object_type (t, context);		/* see declare-object.c */
      break;

    case record_Type:
      GenerateRecordDeclaration (t, context);
      break;

    case union_Type:
      GenerateUnionDeclaration (t, context);
      break;

    case sequence_Type:
      GenerateSequenceDeclaration (t, context);
      break;

    case enumeration_Type:
      GenerateEnumerationDeclaration (t, context);
      break;

    default:
      fprintf (stderr, "Error:  Can't cope with declaration of type %s yet.\n", M3_TYPE_NAME(t));
      SEGFAULT;
    }
}

static void DeclareException (Exception e, Context context)
{
  if (e->importInterfaceName != NULL)
    return;

  fprintf (context->file, "EXCEPTION %s", M3_SHORT_EXCEPTION_NAME(e));
  if (e->type != NULL AND type_basic_type(e->type) != void_Type)
    fprintf (context->file, " (%s)", iname(M3_TYPE_NAME(e->type), e->type->interface));
  fprintf (context->file, ";\n");
}

static void generate_exception_table (Interface i, Context c)
{
  if (list_size(i->exceptions) > 0)
    {
      fprintf (c->file, "(* Exceptions *)\n\n");
      list_enumerate (i->exceptions, (EnumProc) DeclareException, c);
      fprintf (c->file, "\n");
    }
}

void generate_headers (Interface interface, FILE *file)
{
  struct context_s context;
  list sorted = new_list();

  context.file = file;
  context.interface = interface;
  context.class = NULL;
  context.module = GeneralInterface;

  generate_boilerplate(file, interface);

  fprintf (file, "INTERFACE %s;\n\n", m3_interface_name(interface));
  fprintf (file, "IMPORT Ilu, IluBasics, Thread;\n");
  list_enumerate (interface->imports, (EnumProc) ImportInterface, &context);
  fprintf (file, "\n");

  list_enumerate (interface->types, (EnumProc) UnmarkSupertypes, NULL);
  list_enumerate (interface->types, (EnumProc) sort_types_for_declaration, sorted);
  list_enumerate (sorted, (EnumProc) declare_type, &context);
  fprintf (file, "\n");

  list_enumerate (interface->constants, (EnumProc) declare_constant, &context);
  fprintf (file, "\n");

  generate_exception_table (interface, &context);

  fprintf (file, "END %s.\n", m3_interface_name(interface));
}
