(* IluRuntime.m3 *)
(* Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights
   Reserved.

   Unlimited use, reproduction, and distribution of this software is
   permitted.  Any copy of this software must include both the above
   copyright notice of Xerox Corporation and this paragraph.  Any
   distribution of this software must comply with all applicable United
   States export control laws.  This software is made available AS IS,
   and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
   NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY
   FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY
   DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE)
   OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGES. *)
(* Edited by Mike Spreitzer March 17, 1994 9:42 pm PST *)

UNSAFE MODULE IluRuntime;
IMPORT Cstdlib, Ctypes, Fmt, Ilu, IluF, IluKernel, M3asC, M3toC,
       MutexExtra, Stdio, Text, Thread, Time, Wr;
FROM IluBasics IMPORT Failed, Failure;

<*PRAGMA lL, Ll, Main*>

<*FATAL CantHappen*>
EXCEPTION CantHappen;

<*lL, Ll, Main unconstrained*>

VAR
  ocMu := NEW(MUTEX);
  <*lL >= {ocMu}*>
VAR objectCreators: ObjectCreatorList := NIL;

TYPE
  ObjectCreatorList = REF RECORD
                            <*lL >= {ocMu}*>

                            ot  : ObjectType;
                            oc  : ObjectCreator;
                            tail: ObjectCreatorList
                          END;

PROCEDURE RegisterObjectType (ot: ObjectType; creator: ObjectCreator)
  RAISES {Failed} =
  BEGIN
    IluKernel.ilu_RegisterClass(ot);
    LOCK ocMu DO
      objectCreators :=
        NEW(
          ObjectCreatorList, ot := ot, oc := creator,
          tail := objectCreators);
    END (*lock*);
  END RegisterObjectType;

<*Main Invariant holds*>
PROCEDURE M3ObjectFromSbh (sbh        : Ilu.SBH;
                           static_type: ObjectType;
                           mstid      : TEXT        ): Ilu.Object
  RAISES {Failed} =
  VAR ko: KernObj;
  PROCEDURE WithSbh (csbh: C_String) =
    PROCEDURE WithMstid (cmstid: C_String) =
      BEGIN
        ko := IluKernel.ilu_ObjectOfSBH(csbh, cmstid, static_type);
      END WithMstid;
    BEGIN
      M3asC.TasS(mstid, WithMstid);
    END WithSbh;
  BEGIN
    IF static_type = NIL THEN
      RAISE Failed(NEW(Failure, info := "NIL given as `static_type'"))
    END;
    IF mstid = NIL OR Text.Length(mstid) = 0 THEN
      RAISE
        Failed(
          NEW(Failure, info := "NIL or empty string given as `mstid'"))
    END;
    M3asC.TasS(sbh, WithSbh);
    IF ko = NIL THEN
      RAISE Failed(NEW(Failure, info := "sbh " & sbh & " is invalid"));
        (* and, ideally, subArg := something from
           IluKernel.ilu_ObjectOfSBH(sbh) *)
    END (*if*);
    RETURN M3ObjectFromKernObj(ko, static_type);
  END M3ObjectFromSbh;


<*Main otherwise unconstrained;
  before: Inside(ko's server, static_type),
          lL no higher than that requires;
  after:  Main Invariant holds;
  excpn:  Main Invariant holds*>
PROCEDURE M3ObjectFromKernObj (ko: KernObj; static_type: ObjectType):
  Ilu.Object RAISES {Failed} =
  VAR lso: ADDRESS := IluKernel.ilu_GetLanguageSpecificObject(ko);
  VAR ra: REFANY := IluF.FromStrongRef(lso);
  VAR ks := IluKernel.ilu_ServerOfObject(ko);
  VAR ot: ObjectType := NIL;
  VAR ocl: ObjectCreatorList;
  VAR oc: ObjectCreator := NIL;
  VAR server: Ilu.Server;
  VAR ans: Ilu.Object;
  VAR ans_server: Ilu.Server;
  VAR ans_type: Ilu.ObjectType;
  VAR ans_sid, ans_tn, ans_tid := "NIL";
  BEGIN
    TRY
      TYPECASE ra OF
      | NULL => EVAL 0;
      | Ilu.Object (x) => RETURN x;
      ELSE
        RAISE CantHappen
      END (*typecase ra*);
      ot := IluKernel.ilu_ClassOfObject(ko);
      server := IluF.SFromKs(ks);
      TYPECASE server OF
      | Ilu.TrueServer (ts) =>
          IF ts.objtab = NIL THEN
            RAISE
              Failed(
                NEW(Failure, info := "unable to re-create true object"))
          END;
          ans :=
            ts.objtab.HandleToObject(
              M3toC.CopyStoT(IluKernel.ilu_IhOfObject(ko)));
          IF ans = NIL THEN
            RAISE
              Failed(
                NEW(Failure, info := "unable to re-lookup true object"))
          END;
          RETURN ans;
      | Ilu.SurrogateServer (ss) =>
          LOCK ocMu DO
            ocl := objectCreators;
            WHILE ocl # NIL DO
              IF ocl.ot = ot THEN
                oc := ocl.oc;
                EXIT
              ELSE
                ocl := ocl.tail;
              END (*if match*);
            END (*while*);
          END (*lock ocMu*);
          IF oc # NIL THEN
            ans := oc.apply(ss);
            ans_server := ans.ILU_Get_Server();
            ans_type := ans.ILU_Get_Type();
            IF NOT (ans.ilu_is_surrogate AND ans_server = server
                      AND ans_type = ot AND ans.ILU_Qua_Type(ot) = ans) THEN
              IF ans_server # NIL THEN ans_sid := ans_server.id END;
              IF ans_type # NIL THEN
                ans_tn := M3toC.CopyStoT(ans_type.name);
                ans_tid := M3toC.CopyStoT(ans_type.unique_id)
              END;
              RAISE
                Failed(
                  NEW(
                    Failure,
                    info :=
                      Fmt.FN(
                        "surrogate object creator for type %s (id %s)"
                          & " fucked up: result .is_surrogate=%s,"
                          & " .Get_Server()=%s (should be %s),"
                          & " .Get_Type=%s(id %s), isLead=%s",
                        ARRAY OF
                          TEXT{
                          M3toC.CopyStoT(ot.name),
                          M3toC.CopyStoT(ot.unique_id),
                          FmtBool(ans.ilu_is_surrogate),
                          SafeText(ans_sid), SafeText(server.id),
                          SafeText(ans_tn), SafeText(ans_tid),
                          FmtBool(ans.ILU_Qua_Type(ot) = ans)})))
            END (*if*);
            ans.ko := ko;
            ans.sr := IluF.NewStrongRef(ans);
            IluKernel.ilu_RegisterLanguageSpecificObject(ko, ans.sr);
            ans.c_sbh := IluKernel.ilu_SBHOfObject(ko);
            IF ans.c_sbh = NIL THEN RAISE CantHappen END;
            ans.c_mstid := IluKernel.ilu_MstidOfObject(ko);
            IF ans.c_mstid = NIL THEN RAISE CantHappen END;
            ans.sbh := M3toC.CopyStoT(ans.c_sbh);
            ans.c_sbh := M3toC.TtoS(ans.sbh);
            ans.mstid := M3toC.CopyStoT(ans.c_mstid);
            ans.c_mstid := M3toC.TtoS(ans.mstid);
            RETURN ans;
          ELSE
            RAISE
              Failed(
                NEW(
                  Failure,
                  info := Fmt.F(
                            "attempt to input object of unregistered"
                              & " (with ILU/M3) static type %s (id %s)",
                            M3toC.CopyStoT(ot.name),
                            M3toC.CopyStoT(ot.unique_id))));
          END (*if oc#NIL*);
      ELSE
        RAISE CantHappen
      END (*typecase server*);
    FINALLY
      IluKernel.ilu_ExitServer(ks, static_type);
    END (*try*);
  END M3ObjectFromKernObj;

<*Main Invariant holds; Ll otherwise unconstrained*>

<*Ll >= {call's conn's callmu, iomu} afterward iff result non-NIL*>
PROCEDURE BeginCall (on: Ilu.Object): Call RAISES {Failed} =
  VAR server := on.ILU_Get_Server();
  VAR call: Call;
  BEGIN
    LOCK server DO
      IF server.ks = NIL THEN
        RAISE
          Failed(
            NEW(Failure, info := "given object with NIL server.ks"))
      END (*if*);
    END (*lock server*);
    call := IluKernel.ilu_BeginCall(server.ks);
    IF call = NIL THEN
      RAISE
        Failed(
          NEW(
            Ilu.ProtocolFailure,
            case := Ilu.ProtocolResultCode.LostConnection,
            info := "lost connection"))
    ELSE
      RETURN call
    END;
  END BeginCall;

<*Ll >= {call's conn's callmu, iomu}*>

PROCEDURE BeginRequest (call      : Call;
                        intro_type: ObjectType;
                        method    : IluKernel.Method;
                        argSize   : Cardinal          )
  RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_BeginRequest(call, intro_type, method, argSize) = 0 THEN
      RAISE Failed(NEW(Failure, info := "BeginRequest failed"))
    END;
  END BeginRequest;

PROCEDURE FinishRequest (call: Call) RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_FinishRequest(call) = 0 THEN
      RAISE Failed(NEW(Failure, info := "FinishRequest failed"))
    END;
  END FinishRequest;

PROCEDURE GetReply (call: Call): Cardinal RAISES {Failed} =
  VAR pexn, errorStatus: Cardinal;
  BEGIN
    pexn := IluKernel.ilu_GetReply(call, errorStatus);
    IF pexn = 0 THEN
      RETURN errorStatus
    ELSIF 1 <= pexn AND pexn < NUMBER(Ilu.ProtocolResultCode) THEN
      RAISE
        Failed(
          NEW(
            Ilu.ProtocolFailure,
            case := VAL(pexn, Ilu.ProtocolResultCode),
            info := IluKernel.ProtocolExceptionNames[pexn]))
    ELSE
      RAISE
        Failed(
          NEW(
            Failure,
            info := Fmt.F(
                      "InterpretReply returned protocol exception %s",
                      Fmt.Int(pexn))))
    END (*if*);
  END GetReply;

<*lL, Ll, Main unconstrained*>
PROCEDURE ExceptionOfMethod (method: Method; ecode: INTEGER):
  Exception =
  BEGIN
    RETURN IluKernel.ilu_ExceptionOfMethod(method, ecode);
  END ExceptionOfMethod;

<*Ll    >=    {call's conn's callmu, iomu} before,
 *Ll disjoint {call's conn's callmu, iomu} after*>
PROCEDURE FinishCall (call: Call) =
  BEGIN
    IluKernel.ilu_FinishCall(call);
  END FinishCall;

<*lL, Ll, Main unconstrained*>
PROCEDURE SetMethodStub (ot      : ObjectType;
                         idx, mid: INTEGER;
                         stub    : IluKernel.StubProc ) =
  VAR
    mi: IluKernel.Method := ot.methods
                              + idx * ADRSIZE(IluKernel.Method_Rec);
  BEGIN
    IF mi^.id # mid THEN RAISE CantHappen END;
    mi^.stubproc := LOOPHOLE(stub, Ctypes.void_star);
  END SetMethodStub;

<*Main Invariant holds*>

<*Ll          >= {call's conn's callmu, iomu} before;
after: Ll not >= {call's conn's         iomu},
after: Ll     >= {call's conn's callmu} iff protocol not concurrent*>

PROCEDURE RequestRead (call: Call) =
  BEGIN
    IluKernel.ilu_RequestRead(call);
  END RequestRead;

<*Ll      >=  {call's conn's callmu      } iff protocol not concurrent,
  Ll  not >=  {call's conn's         iomu} before;
  Ll disjoint {call's conn's callmu, iomu} excpn;
  Ll      >=  {call's conn's callmu, iomu} after. *>

PROCEDURE BeginReply (call      : Call;
                      exceptions: BOOLEAN;
                      replySize : CARDINAL ) RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_BeginReply(call, ORD(exceptions), replySize) = 0 THEN
      RAISE Failed(NEW(Failure, info := "ilu_BeginReply failed"))
    END (*if*);
  END BeginReply;

PROCEDURE BeginException (call: Call; evalue, argSize: CARDINAL)
  RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_BeginException(call, evalue, argSize) = 0 THEN
      RAISE Failed(NEW(Failure, info := "ilu_BeginException failed"))
    END (*if*);
  END BeginException;

<*Ll    >=    {call's conn's callmu, iomu} before,
 *Ll disjoint {call's conn's callmu, iomu} after/excpn*>

PROCEDURE FinishReply (call: Call) RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_FinishReply(call) = 0 THEN
      RAISE Failed(NEW(Failure, info := "ilu_FinishReply failed"))
    END (*if*);
  END FinishReply;

PROCEDURE FinishException (call: Call) RAISES {Failed} =
  BEGIN
    IF IluKernel.ilu_FinishException(call) = 0 THEN
      RAISE Failed(NEW(Failure, info := "ilu_FinishException failed"))
    END (*if*);
  END FinishException;

<*Ll  not >=  {call's conn's         iomu},
  Ll    >=    {call's conn's callmu} iff protocol not concurrent, before;
  Ll disjoint {call's conn's callmu, iomu} after/excpn*>

PROCEDURE FailReply (call: Call) =
  BEGIN
    TRY
      BeginException(call, 0, ORD(IluKernel.ProtocolException.Unknown));
      FinishException(call);
    EXCEPT
      Failed (f2) => HandleStubFailure2(call, f2)
    END (*try-except*);
  END FailReply;

PROCEDURE HandleStubFailure1 (call: Call; f: Failure) =
  BEGIN
    FailReply(call);
    HandleStubFailure2(call, f);
  END HandleStubFailure1;

PROCEDURE HandleStubAlert1 (call: Call) =
  BEGIN
    HandleServerProcAlert(call);
  END HandleStubAlert1;

PROCEDURE HandleServerProcFail (call: Call; f: Failure) =
  BEGIN
    FailReply(call);
    HandleStubFailure2(
      call, NEW(
              Failure, info := "server proc raised IluBasics.Failed",
              subArg := f));
  END HandleServerProcFail;

PROCEDURE HandleServerProcAlert (call: Call) =
  VAR
    cw := NARROW(LOOPHOLE(call.ls_data, REFANY), IluF.ConnectionWorker);
  BEGIN
    cw.work := FALSE;
    FailReply(call);
  END HandleServerProcAlert;

PROCEDURE NoReply (call: Call) =
  BEGIN
    IluKernel.ilu_NoReply(call);
  END NoReply;

<*Ll disjoint {call's conn's callmu, iomu}*>

PROCEDURE HandleStubFailure2 (call: Call; f: Failure) =
  VAR
    cw := NARROW(LOOPHOLE(call.ls_data, REFANY), IluF.ConnectionWorker);
  BEGIN
    IF cw.ts.HandleWorkerFailure(f) = Ilu.FailureAction.Quit THEN
      cw.work := FALSE
    ELSE
      Time.LongPause(3)
    END (*if*);
  END HandleStubFailure2;

PROCEDURE HandleStubAlert2 (call: Call) =
  VAR
    cw := NARROW(LOOPHOLE(call.ls_data, REFANY), IluF.ConnectionWorker);
  BEGIN
    cw.work := FALSE
  END HandleStubAlert2;

<*lL, Ll, Main unconstrained*>
PROCEDURE CheckedRuntimeError (what: TEXT) =
  <*FATAL Wr.Failure, Thread.Alerted*>
  BEGIN
    Wr.PutText(Stdio.stdout, what & "\n");
    Wr.Flush(Stdio.stdout);
    RAISE CantHappen;
  END CheckedRuntimeError;

<*Main Invariant holds*>
<*Ll >= {call's connection's callmu, iomu}*>

PROCEDURE EndSequence (call: Call) =
  BEGIN
    IluKernel.ilu_EndSequence(call)
  END EndSequence;

PROCEDURE EndUnion (call: Call) =
  BEGIN
    IluKernel.ilu_EndUnion(call)
  END EndUnion;

PROCEDURE EndArray (call: Call) =
  BEGIN
    IluKernel.ilu_EndArray(call)
  END EndArray;

PROCEDURE EndRecord (call: Call) =
  BEGIN
    IluKernel.ilu_EndRecord(call)
  END EndRecord;

PROCEDURE OutputOptional (call: Call; provided: BOOLEAN) =
  BEGIN
    IluKernel.ilu_OutputOptional(call, ORD(provided))
  END OutputOptional;

PROCEDURE OutputInteger (call: Call; i: Ilu.Integer) =
  BEGIN
    IluKernel.ilu_OutputInteger(call, i)
  END OutputInteger;

PROCEDURE OutputCardinal (call: Call; i: Ilu.Cardinal) =
  BEGIN
    IluKernel.ilu_OutputCardinal(call, i)
  END OutputCardinal;

PROCEDURE OutputCharacter (call: Call; i: Ilu.Character) =
  BEGIN
    IluKernel.ilu_OutputCharacter(call, i)
  END OutputCharacter;

PROCEDURE OutputEnum (call: Call; i: EnumOrd) =
  BEGIN
    IluKernel.ilu_OutputEnum(call, i)
  END OutputEnum;

PROCEDURE OutputShortInteger (call: Call; i: Ilu.ShortInt) =
  BEGIN
    IluKernel.ilu_OutputShortInteger(call, i)
  END OutputShortInteger;

PROCEDURE OutputShortCardinal (call: Call; i: Ilu.ShortCard) =
  BEGIN
    IluKernel.ilu_OutputShortCardinal(call, i)
  END OutputShortCardinal;

PROCEDURE OutputShortReal (call: Call; f: Ilu.ShortReal) =
  BEGIN
    IluKernel.ilu_shim_OutputShortReal(call, FLOAT(f, LONGREAL))
  END OutputShortReal;

PROCEDURE OutputLongInteger (call: Call; i: Ilu.LongInt) =
  BEGIN
    IluKernel.ilu_OutputLongInteger(call, i)
  END OutputLongInteger;

PROCEDURE OutputLongCardinal (call: Call; i: Ilu.LongCard) =
  BEGIN
    IluKernel.ilu_OutputLongCardinal(call, i)
  END OutputLongCardinal;

PROCEDURE OutputLongReal (call: Call; f: Ilu.LongReal) =
  BEGIN
    IluKernel.ilu_OutputLongReal(call, f)
  END OutputLongReal;

PROCEDURE OutputReal (call: Call; d: Ilu.Real) =
  BEGIN
    IluKernel.ilu_OutputReal(call, d)
  END OutputReal;

PROCEDURE OutputByte (call: Call; b: Ilu.Byte) =
  BEGIN
    IluKernel.ilu_OutputByte(call, b)
  END OutputByte;

PROCEDURE OutputBoolean (call: Call; x: BOOLEAN) =
  BEGIN
    IluKernel.ilu_OutputBoolean(call, ORD(x))
  END OutputBoolean;

PROCEDURE OutputSequence (call : Call;
                          len  : Cardinal;
                          limit: Cardinal := NormalLimit ) =
  BEGIN
    IluKernel.ilu_OutputSequence(call, len, limit, 0, 1);
  END OutputSequence;

PROCEDURE OutputUnion (call: Call; discriminator: ShortCard) =
  BEGIN
    IluKernel.ilu_OutputUnion(call, discriminator, 0, 1);
  END OutputUnion;

PROCEDURE OutputArray (call: Call) =
  BEGIN
    IluKernel.ilu_OutputArray(call, 0, 1);
  END OutputArray;

PROCEDURE OutputRecord (call: Call) =
  BEGIN
    IluKernel.ilu_OutputRecord(call, 0, 1);
  END OutputRecord;

PROCEDURE OutputString (call : Call;
                        x    : Ilu.String;
                        limit: Cardinal := NormalLimit ) =
  PROCEDURE WithString (cx: C_String) =
    BEGIN
      IluKernel.ilu_OutputString(call, cx, Text.Length(x), limit, 0);
    END WithString;
  BEGIN
    M3asC.TasS(x, WithString);
  END OutputString;

PROCEDURE OutputShortCharArray (call: Call;
                                READONLY x: ARRAY OF Ilu.PackedShortChar) =
  VAR a0: ADDRESS := NIL;
  BEGIN
    IF NUMBER(x) > 0 THEN a0 := ADR(x[0]) END;
    IluKernel.ilu_OutputStringVec(call, a0, NUMBER(x), 0);
  END OutputShortCharArray;

PROCEDURE OutputWString (call : Call;
                         x    : Ilu.WString;
                         limit: Cardinal := NormalLimit ) =
  VAR a0: ADDRESS := NIL;
  BEGIN
    IF NUMBER(x^) > 0 THEN a0 := ADR(x[0]) END;
    IluKernel.ilu_OutputWString(
      call, a0, NUMBER(x^), limit, NIL, NIL, 0);
  END OutputWString;

PROCEDURE OutputCharArray (         call: Call;
                           READONLY x   : ARRAY OF Ilu.Character ) =
  VAR a0: ADDRESS := NIL;
  BEGIN
    IF NUMBER(x) > 0 THEN a0 := ADR(x[0]) END;
    IluKernel.ilu_OutputWStringVec(call, a0, NUMBER(x), NIL, NIL, 0);
  END OutputCharArray;

PROCEDURE OutputBytes (call : Call;
                       x    : Ilu.Bytes;
                       limit: Cardinal := NormalLimit ) =
  VAR a0: ADDRESS := NIL;
  BEGIN
    IF NUMBER(x^) > 0 THEN a0 := ADR(x[0]) END;
    IluKernel.ilu_OutputBytes(call, a0, NUMBER(x^), limit, 0);
  END OutputBytes;

PROCEDURE OutputOpaque (call: Call; READONLY x: ARRAY OF Ilu.PackedByte) =
  VAR a0: ADDRESS := NIL;
  BEGIN
    IF NUMBER(x) > 0 THEN a0 := ADR(x[0]) END;
    IluKernel.ilu_OutputOpaque(call, a0, NUMBER(x), 0);
  END OutputOpaque;

PROCEDURE OutputM3Object (call         : Call;
                          obj          : Ilu.Object;
                          discriminator: BOOLEAN;
                          static_type  : ObjectType  ) RAISES {Failed} =
  VAR l := obj.ILU_Qua_Type(obj.ILU_Get_Type());
  VAR ko := IluF.KOfO(l);
  BEGIN
    IluKernel.ilu_OutputObjectID(
      call, ko, ORD(discriminator), static_type);
    RETURN;
  END OutputM3Object;


PROCEDURE InputOptional (call: Call): BOOLEAN =
  VAR optionalStatus: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputOptional(call, optionalStatus);
    RETURN optionalStatus # 0;
  END InputOptional;

PROCEDURE InputInteger (call: Call): Ilu.Integer =
  VAR i: IluKernel.Integer;
  BEGIN
    IluKernel.ilu_InputInteger(call, i);
    RETURN i;
  END InputInteger;

PROCEDURE InputCardinal (call: Call): Ilu.Cardinal =
  VAR i: IluKernel.Cardinal;
  BEGIN
    IluKernel.ilu_InputCardinal(call, i);
    RETURN i;
  END InputCardinal;

PROCEDURE InputShortInteger (call: Call): Ilu.ShortInt =
  VAR i: IluKernel.ShortInt;
  BEGIN
    IluKernel.ilu_InputShortInteger(call, i);
    RETURN i;
  END InputShortInteger;

PROCEDURE InputShortCardinal (call: Call): Ilu.ShortCard =
  VAR i: IluKernel.ShortCard;
  BEGIN
    IluKernel.ilu_InputShortCardinal(call, i);
    RETURN i;
  END InputShortCardinal;

PROCEDURE InputShortReal (call: Call): Ilu.ShortReal =
  VAR f: IluKernel.ShortReal;
  BEGIN
    IluKernel.ilu_InputShortReal(call, f);
    RETURN f;
  END InputShortReal;

PROCEDURE InputLongInteger (call: Call): Ilu.LongInt =
  VAR i: IluKernel.LongInt;
  BEGIN
    IluKernel.ilu_InputLongInteger(call, i);
    RETURN i;
  END InputLongInteger;

PROCEDURE InputLongCardinal (call: Call): Ilu.LongCard =
  VAR i: IluKernel.LongCard;
  BEGIN
    IluKernel.ilu_InputLongCardinal(call, i);
    RETURN i;
  END InputLongCardinal;

PROCEDURE InputLongReal (call: Call): Ilu.LongReal =
  VAR f: IluKernel.LongReal;
  BEGIN
    IluKernel.ilu_InputLongReal(call, f);
    RETURN f;
  END InputLongReal;

PROCEDURE InputReal (call: Call): Ilu.Real =
  VAR d: IluKernel.Real;
  BEGIN
    IluKernel.ilu_InputReal(call, d);
    RETURN d;
  END InputReal;

PROCEDURE InputEnum (call: Call): EnumOrd =
  VAR i: IluKernel.ShortCard;
  BEGIN
    IluKernel.ilu_InputEnum(call, i);
    RETURN i;
  END InputEnum;

PROCEDURE InputCharacter (call: Call): Ilu.Character =
  VAR i: IluKernel.Character;
  BEGIN
    IluKernel.ilu_InputCharacter(call, i);
    RETURN i;
  END InputCharacter;

PROCEDURE InputByte (call: Call): Ilu.Byte =
  VAR b: IluKernel.Byte;
  BEGIN
    IluKernel.ilu_InputByte(call, b);
    RETURN b;
  END InputByte;

PROCEDURE InputBoolean (call: Call): BOOLEAN =
  VAR b: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputBoolean(call, b);
    RETURN b # 0;
  END InputBoolean;

PROCEDURE InputString (call: Call; limit: Cardinal := NormalLimit):
  Ilu.String =
  VAR s: C_String := NIL;
  VAR len: IluKernel.Cardinal;
  VAR ans: Text.T;
  VAR sf: C_String;
  BEGIN
    IluKernel.ilu_InputString(call, s, len, limit, 0);
    IF len = 0 THEN RETURN "" END;
    sf := s;
    INC(sf, len * ADRSIZE(Ilu.PackedShortChar));
    IF sf^ # VAL(0, CHAR) THEN RAISE CantHappen END;
    ans := M3toC.StoT(s);
    IF Text.Length(ans) # len THEN RAISE CantHappen END;
    RETURN ans;
  END InputString;

PROCEDURE InputShortCharArray (    call: Call;
                               VAR x   : ARRAY OF Ilu.PackedShortChar ) =
  VAR s: C_String := NIL;
  VAR p: UNTRACED REF Ilu.PackedShortChar;
  BEGIN
    IluKernel.ilu_InputStringVec(call, s, NUMBER(x), 0);
    p := s;
    FOR i := 0 TO NUMBER(x) - 1 DO
      x[i] := p^;
      INC(p, ADRSIZE(Ilu.PackedShortChar));
    END (*for*);
    Cstdlib.free(s);
  END InputShortCharArray;

PROCEDURE InputWString (call: Call; limit: Cardinal := NormalLimit):
  Ilu.WString =
  VAR s: IluKernel.WString;
  VAR len: IluKernel.Cardinal;
  BEGIN
    IluKernel.ilu_InputWString(call, s, len, limit, 0);
    RETURN WideSLtoT(s, len);
  END InputWString;

PROCEDURE InputCharArray (call: Call; VAR x: ARRAY OF Ilu.Character) =
  VAR s: IluKernel.WString := NIL;
  VAR p: UNTRACED REF BITS 16 FOR Ilu.Character;
  BEGIN
    IluKernel.ilu_InputWStringVec(call, s, NUMBER(x), 0);
    p := s;
    FOR i := 0 TO NUMBER(x) - 1 DO
      x[i] := p^;
      INC(p, ADRSIZE(BITS 16 FOR Ilu.Character));
    END (*for*);
    Cstdlib.free(s);
  END InputCharArray;

PROCEDURE WideSLtoT (s: IluKernel.WString; l: Cardinal): Ilu.WString =
  VAR ans := NEW(Ilu.WString, l + 1);
  BEGIN
    FOR i := 0 TO l - 1 DO
      ans[i] := s^;
      INC(s, ADRSIZE(IluKernel.Character));
    END (*for*);
    ans[l] := VAL(0, IluKernel.Character);
    RETURN ans;
  END WideSLtoT;

PROCEDURE InputBytes (call: Call; limit: Cardinal := NormalLimit):
  Ilu.Bytes =
  VAR s: IluKernel.Bytes;
  VAR len: IluKernel.Cardinal;
  BEGIN
    IluKernel.ilu_InputBytes(call, s, len, limit, 0);
    RETURN ByteSLtoT(s, len);
  END InputBytes;

PROCEDURE InputOpaque (call: Call; VAR x: ARRAY OF Ilu.PackedByte) =
  VAR s: IluKernel.Bytes := NIL;
  VAR p: UNTRACED REF Ilu.PackedByte;
  BEGIN
    IluKernel.ilu_InputOpaque(call, s, NUMBER(x), 0);
    p := s;
    FOR i := 0 TO NUMBER(x) - 1 DO
      x[i] := p^;
      INC(p, ADRSIZE(Ilu.PackedByte));
    END (*for*);
    Cstdlib.free(s);
    RETURN;
  END InputOpaque;

PROCEDURE ByteSLtoT (s: IluKernel.Bytes; l: Cardinal): Ilu.Bytes =
  VAR p: UNTRACED REF Ilu.PackedByte := s;
  VAR ans := NEW(Ilu.Bytes, l + 1);
  BEGIN
    FOR i := 0 TO l - 1 DO
      ans[i] := p^;
      INC(p, ADRSIZE(Ilu.PackedByte));
    END (*for*);
    ans[l] := 0;
    RETURN ans;
  END ByteSLtoT;

PROCEDURE InputM3Object (call         : Call;
                         discriminator: BOOLEAN;
                         static_type  : ObjectType ): Ilu.Object
  RAISES {Failed} =
  VAR ko: KernObj;
  BEGIN
    IluKernel.ilu_InputObjectID(
      call, ko, ORD(discriminator), static_type);
    IF ko = NIL THEN
      RAISE Failed(NEW(Failure, info := "ilu_InputObjectID failed"))
    END;
    RETURN M3ObjectFromKernObj(ko, static_type);
  END InputM3Object;

PROCEDURE GetSingletonDiscriminator (call: Call): M3Obj
  RAISES {Failed} =
  VAR
    ko := IluKernel.ilu_GetServerSingleton(call.server, call.introType);
  BEGIN
    IF ko = NIL THEN
      RAISE Failed(NEW(Failure, info := "ilu_InputObjectID failed"))
    END;
    RETURN M3ObjectFromKernObj(ko, call.introType);
  END GetSingletonDiscriminator;

PROCEDURE InputSequence (call: Call; limit: Cardinal := NormalLimit):
  Cardinal =
  VAR len: IluKernel.Cardinal;
  VAR provided: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputSequence(call, len, limit, 0, provided);
    RETURN len;
  END InputSequence;

PROCEDURE InputUnion (call: Call): ShortCard =
  VAR disc: IluKernel.ShortCard;
  VAR provided: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputUnion(call, disc, 0, provided);
    RETURN disc;
  END InputUnion;

PROCEDURE InputArray (call: Call) =
  VAR provided: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputArray(call, 0, provided);
  END InputArray;

PROCEDURE InputRecord (call: Call) =
  VAR provided: IluKernel.Bool;
  BEGIN
    IluKernel.ilu_InputRecord(call, 0, provided);
  END InputRecord;


PROCEDURE SizeOptional (call: Call; optionalStatus: BOOLEAN): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfOptional(call, ORD(optionalStatus));
  END SizeOptional;

PROCEDURE SizeInteger (call: Call; i: Ilu.Integer): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfInteger(call, i);
  END SizeInteger;

PROCEDURE SizeCardinal (call: Call; i: Ilu.Cardinal): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfCardinal(call, i);
  END SizeCardinal;

PROCEDURE SizeCharacter (call: Call; i: Ilu.Character): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfCharacter(call, i);
  END SizeCharacter;

PROCEDURE SizeEnum (call: Call; i: Ilu.ShortCard): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfEnum(call, i);
  END SizeEnum;

PROCEDURE SizeShortInteger (call: Call; i: Ilu.ShortInt): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfShortInteger(call, i);
  END SizeShortInteger;

PROCEDURE SizeShortCardinal (call: Call; i: Ilu.ShortCard): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfShortCardinal(call, i);
  END SizeShortCardinal;

PROCEDURE SizeLongInteger (call: Call; i: Ilu.LongInt): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfLongInteger(call, i);
  END SizeLongInteger;

PROCEDURE SizeLongCardinal (call: Call; i: Ilu.LongCard): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfLongCardinal(call, i);
  END SizeLongCardinal;

PROCEDURE SizeLongReal (call: Call; d: Ilu.LongReal): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfLongReal(call, d);
  END SizeLongReal;

PROCEDURE SizeByte (call: Call; i: Ilu.Byte): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfByte(call, i);
  END SizeByte;

PROCEDURE SizeBoolean (call: Call; i: BOOLEAN): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfBoolean(call, ORD(i));
  END SizeBoolean;

PROCEDURE SizeReal (call: Call; d: Ilu.Real): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfReal(call, d);
  END SizeReal;

PROCEDURE SizeShortReal (call: Call; d: Ilu.ShortReal): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_shim_SizeOfShortReal(call, FLOAT(d, Ilu.Real));
  END SizeShortReal;

PROCEDURE SizeSequence (call : Call;
                        len  : Cardinal;
                        limit: Ilu.Cardinal := NormalLimit ): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfSequence(call, len, limit, 0, 1);
  END SizeSequence;

PROCEDURE SizeUnion (call: Call; discriminator: ShortCard): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfUnion(call, discriminator, 0, 1);
  END SizeUnion;

PROCEDURE SizeArray (call: Call): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfArray(call, 0, 1);
  END SizeArray;

PROCEDURE SizeRecord (call: Call): Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfRecord(call, 0, 1);
  END SizeRecord;

PROCEDURE SizeString (call : Call;
                      x    : Ilu.String;
                      limit: Cardinal := NormalLimit ): Cardinal =
  VAR ans: Cardinal;
  PROCEDURE WithString (cs: C_String) =
    BEGIN
      ans :=
        IluKernel.ilu_SizeOfString(call, cs, Text.Length(x), limit, 0);
    END WithString;
  BEGIN
    M3asC.TasS(x, WithString);
    RETURN ans;
  END SizeString;

PROCEDURE SizeShortCharArray (call: Call;
                              READONLY x: ARRAY OF Ilu.PackedShortChar):
  Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfStringVec(call, ADR(x[0]), NUMBER(x), 0);
  END SizeShortCharArray;

PROCEDURE SizeWString (call : Call;
                       x    : Ilu.WString;
                       limit: Cardinal := NormalLimit ): Cardinal =
  BEGIN
    RETURN
      IluKernel.ilu_SizeOfWString(
        call, ADR(x[0]), NUMBER(x^), limit, NIL, NIL, 0);
  END SizeWString;

PROCEDURE SizeCharArray (call: Call; READONLY x: ARRAY OF Ilu.Character):
  Cardinal =
  BEGIN
    RETURN
      IluKernel.ilu_SizeOfWStringVec(
        call, ADR(x[0]), NUMBER(x), NIL, NIL, 0);
  END SizeCharArray;

PROCEDURE SizeBytes (call : Call;
                     x    : Ilu.Bytes;
                     limit: Cardinal := NormalLimit ): Cardinal =
  BEGIN
    RETURN
      IluKernel.ilu_SizeOfBytes(call, ADR(x[0]), NUMBER(x^), limit, 0);
  END SizeBytes;

PROCEDURE SizeOpaque (call: Call; READONLY x: ARRAY OF Ilu.PackedByte):
  Cardinal =
  BEGIN
    RETURN IluKernel.ilu_SizeOfOpaque(call, ADR(x[0]), NUMBER(x), 0);
  END SizeOpaque;

PROCEDURE SizeM3Object (call         : Call;
                        obj          : Ilu.Object;
                        discriminator: BOOLEAN;
                        static_type  : ObjectType  ): Cardinal
  RAISES {} =
  VAR l := obj.ILU_Qua_Type(obj.ILU_Get_Type());
  VAR s := l.ILU_Get_Server();
  VAR ko: IluKernel.Object;
  VAR kot: ObjectType;
  BEGIN
    TRY ko := IluF.KOfO(l) EXCEPT Failed => RETURN 0 END (*try*);
    kot := IluKernel.ilu_ClassOfObject(ko);
    TRY
      RETURN
        IluKernel.ilu_SizeOfObjectID(
          call, ko, ORD(discriminator), static_type)
    FINALLY
      IluKernel.ilu_ExitServer(s.ks, kot);
    END (*try*);
  END SizeM3Object;


<*lL, Ll, Main unconstrained*>


PROCEDURE FmtBool (b: BOOLEAN): TEXT =
  BEGIN
    IF b THEN RETURN "TRUE" ELSE RETURN "FALSE" END;
  END FmtBool;

PROCEDURE SafeText (t: TEXT): TEXT =
  BEGIN
    IF t = NIL THEN RETURN "(NIL string)" ELSE RETURN t END;
  END SafeText;


TYPE Mutex = MUTEX OBJECT d1, d2: C_String END;

PROCEDURE CreateMutex (d1, d2: C_String): ADDRESS =
  VAR m := NEW(Mutex, d1 := d1, d2 := d2);
  BEGIN
    RETURN IluF.NewStrongRef(m);
  END CreateMutex;

PROCEDURE AcquireMutex (m: ADDRESS) =
  VAR ra := IluF.FromStrongRef(m);
  VAR self := Thread.Self();
  BEGIN
    TYPECASE ra OF
    | Mutex (mu) =>
        IF MutexExtra.Holds(self, mu) THEN RAISE CantHappen END;
        Thread.Acquire(mu);
    ELSE
      RAISE CantHappen
    END (*typecase*);
  END AcquireMutex;

PROCEDURE HoldMutex (m: ADDRESS) =
  VAR ra := IluF.FromStrongRef(m);
  VAR self := Thread.Self();
  BEGIN
    TYPECASE ra OF
    | Mutex (mu) =>
        IF NOT MutexExtra.Holds(self, mu) THEN RAISE CantHappen END;
    ELSE
      RAISE CantHappen
    END (*typecase*);
  END HoldMutex;

PROCEDURE ReleaseMutex (m: ADDRESS) =
  VAR ra := IluF.FromStrongRef(m);
  <*UNUSED*> VAR self := Thread.Self();
  BEGIN
    TYPECASE ra OF
    | Mutex (mu) => (* A long comment to force a line break *)
        Thread.Release(mu);
    ELSE
      RAISE CantHappen
    END (*typecase*);
  END ReleaseMutex;

PROCEDURE DestroyMutex (m: ADDRESS) =
  BEGIN
    ReleaseMutex(m);
  END DestroyMutex;

TYPE
  Condition = Thread.Condition OBJECT
                d1, d2    :  C_String;
                destroyed := FALSE;
              END;

PROCEDURE CreateCondition (d1, d2: C_String): ADDRESS =
  VAR c := NEW(Condition, d1 := d1, d2 := d2);
  BEGIN
    RETURN IluF.NewStrongRef(c);
  END CreateCondition;

PROCEDURE NotifyCondition (c: ADDRESS) =
  VAR ra := IluF.FromStrongRef(c);
  <*UNUSED*> VAR self := Thread.Self();
  BEGIN
    TYPECASE ra OF
    | Condition (cv) =>
        IF cv.destroyed THEN RAISE CantHappen END;
        Thread.Broadcast(cv);
    ELSE
      RAISE CantHappen
    END (*typecase*);
  END NotifyCondition;

PROCEDURE DestroyCondition (c: ADDRESS) =
  VAR ra := IluF.FromStrongRef(c);
  <*UNUSED*> VAR self := Thread.Self();
  BEGIN
    TYPECASE ra OF
    | Condition (cv) => cv.destroyed := TRUE; Thread.Broadcast(cv);
    ELSE
      RAISE CantHappen
    END (*typecase*);
  END DestroyCondition;

PROCEDURE WaitCondition (c, m: ADDRESS) =
  VAR
    cra := IluF.FromStrongRef(c);
    mra := IluF.FromStrongRef(m);
    cv  :  Condition;
    mu  :  Mutex;
  <*UNUSED*> VAR self := Thread.Self();
  BEGIN
    TYPECASE cra OF
    | Condition (x) => cv := x;
    ELSE
      RAISE CantHappen
    END (*typecase*);
    TYPECASE mra OF
    | Mutex (x) => mu := x;
    ELSE
      RAISE CantHappen
    END (*typecase*);
    IF cv.destroyed THEN RAISE CantHappen END;
    Thread.Wait(mu, cv);
    RETURN;
  END WaitCondition;

VAR
  lockTech := IluKernel.LockTech{
                CreateMutex, AcquireMutex, HoldMutex, ReleaseMutex,
                DestroyMutex, CreateCondition, NotifyCondition,
                DestroyCondition, WaitCondition};

VAR regLockTech := TRUE;

BEGIN
  IluF.StartIlu();
  IF regLockTech THEN
    IluKernel.ilu_RegisterLockTech(ADR(lockTech))
  ELSE
    IluKernel.ilu_RegisterLockTech(NIL)
  END (*if*);
  IluF.StartGcClientSide();
END IluRuntime.
