#|
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu-server.lisp,v 1.12 1994/05/10 19:42:53 janssen Exp $
|#

(in-package :ilu)		; in this package

(eval-when (compile load eval)
  (export '(signal-exception
	    rpc-server
	    server-unix-port
	    server-id
	    auto-serve
	    define-server-class
	    )))
	  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Actual code for dealing with server objects
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *default-server* nil)

(defclass rpc-server ()
  ((unix-port :reader server-unix-port :initarg :port :initform 0)
   (ilu-server :initform nil :reader server-ilu-server)
   (id :initarg :id :initform nil :reader server-id))
  )

(defclass rpc-stateless-server (rpc-server) ())

(defmethod has-state ((self rpc-server))
  t)

(defmethod has-state ((self rpc-stateless-server))
  nil)

(defun generate-server-id (has-state)
  (ensure-initialized)
  (let* ((id-pointer (ilu_generate-server-id (if has-state 1 0)))
	 (new-id (char*-to-string id-pointer)))
    (free id-pointer)
    new-id))

(defun create-true-server (server-id unix-ip-port)
  (ensure-initialized)
  (let* ((server (ilu_create-true-server (string-to-char* server-id) 0))
	 (port (open-port server unix-ip-port)))
    (set-server-default-port server port)
    server))

(defmethod initialize-instance ((self rpc-server) &key)
  (call-next-method)
  (unless (server-id self)
    (setf (slot-value self 'id)
      (generate-server-id (has-state self))))
  (unless (server-unix-port self)
    (error "No UNIX port specified for export of service ~s.~%" self))
  (setf (slot-value self 'ilu-server)
    (create-true-server (server-id self) (server-unix-port self)))
  (setf *default-server* self)
  )

(defmethod print-object ((self ilu:rpc-server) stream)
  (format stream "#<~s ~s>"
	  (type-of self)
	  (server-id self)))

(defun ensure-server ()
  (unless *default-server*
    (setf *default-server* (make-instance 'rpc-server :port 0)))
  *default-server*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Code for dealing with true objects
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; before: Main invariant holds
;; after:  Inside(server,pclass)
(defun find-or-create-true-kernel-object (ih server pclass lspo)
  (let ((kserver (server-ilu-server server)))
    (ilu_enter-server kserver pclass)
    (ilu_find-or-create-true-object
     ih kserver pclass
     (if lspo (register-lisp-object lspo) 0))))

(defclass ilu-true-object (ilu-object) ()); only appears in servers

(defparameter *object-id-counter* 1)

(defun create-default-id ()
  (setf *object-id-counter* (1+ *object-id-counter*))
  (string-to-char* (format nil "~a" *object-id-counter*)))

(defmethod object-write :before (call (self ilu-true-object) pclass)
  (declare (ignore pclass call))
  (when (null (ilu-cached-kernel-obj self))
    (setup-object-links
     self
     (find-or-create-true-kernel-object
      (create-default-id)
      (ensure-server)
      (find-ilu-class-record self) self))))

(defmethod object-size :before (call (self ilu-true-object) pclass)
  (declare (ignore pclass call))
  (when (null (ilu-cached-kernel-obj self))
    (setup-object-links
     self
     (find-or-create-true-kernel-object
      (create-default-id)
      (ensure-server)
      (find-ilu-class-record self) self))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Server functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *debug-uncaught-conditions* nil
  "If T, causes break to debugger.  Default to NIL.")

(defun handle-uncaught-condition (call cond lspo conn method-name sn)
  (declare (ignore conn))
  (if *debug-uncaught-conditions*
      (invoke-debugger cond)
    (progn
      (warn "Uncaught condition in call ~D: method ~A on ~A.~% Condition: ~A"
	    sn method-name lspo cond)
      (begin-exception call 0 +protocol-exception-unknown+)	;; send "unknown error"
      (finish-exception call))))

(defun get-discriminator (call pclass)
  (multiple-value-bind (ilu-obj class-record) (object-id-read call pclass 1)
    (if (and ilu-obj (not (zerop ilu-obj)))
	(ilu-object->instance
	 (gethash class-record *class-table*
		  (gethash pclass *class-table*))
	 ilu-obj)
      (error "Couldn't read kernel object~%"))))

(defun handle-input-from-connection (conn)
  ;; (format t "Handling input on conn ~x (fd ~d).~%" 
  ;;         conn (file-descriptor-of-connection conn))
  (let* ((call (make-array 1 :element-type '(unsigned-byte 32)))
	 (method-name-ptr (make-array 1 :element-type '(unsigned-byte 32)))
	 (type (make-array 1 :element-type '(unsigned-byte 32)))
	 (sn (ilulisp_get-request conn call type method-name-ptr)))
    (if (= sn -2)
	(progn
	  (warn "Problem reading or interpreting request.  Dropping connection 0x~x.~%" conn)
	  nil)
      (if (= sn -1)
	  (progn
	    (warn "Client on connection 0x~x dropped connection.~%" conn)
	    nil)
	(if (= sn 0)
	    ;; already handled internally
	    t
	  (let ((lspo (get-discriminator (aref call 0) (aref type 0)))
		(method-name (char*-to-string (aref method-name-ptr 0))))
	    (handler-bind ((serious-condition
			    #'(lambda (cond)
				(handle-uncaught-condition call cond lspo conn method-name sn)
				(return-from handle-input-from-connection t))))
;	      (format t "calling method ~s on object ~s with class precedence list ~s.~%"
;		      method-name lspo (clos:class-precedence-list (class-of lspo)))
	      (call-server-stub-for-method-name (aref call 0) (intern method-name :keyword) lspo))
	    t))))))	    
  
(defun watch-connection (conn fd)
  (loop
    (wait-for-input-available fd)
    (unless (handle-input-from-connection conn)
      (return))))

(defun handle-new-connection (port)
  ;;  (format t "New connection for ~s.~%" port)
  (let ((conn (handle-ilu-connection port)))
    (unless (zerop conn)
      (ilu-process:fork-process
       (format nil "ILU request processing thread for conn 0x~x (fd ~d)"
	       conn (file-descriptor-of-connection conn))
       #'watch-connection conn (file-descriptor-of-connection conn)))))

(defun watch-for-connections (port fd)
  (loop
    (wait-for-input-available fd)
    (unless (handle-new-connection port)
      (error "End of file detected on fd ~d, for port 0x~x~%" fd port))))

(defun open-port (server &optional (ip-port 0))
  (if (zerop server)
      (error "Can't open port on nil server~%"))
  (let* ((s-port (create-port server "sunrpc_" (format nil "tcp_localhost_~a" ip-port)))
	 (fd (and s-port (file-descriptor-of-mooring-of-port s-port))))
    (if (zerop s-port)
	(error "Can't open RPC port on UNIX port ~a.~%" ip-port)
      (if (or (null fd) (< fd 0))
	  (error "Can't obtain file descriptor of mooring of port ~s.~%" s-port)
	(progn
	  (set-server-default-port server s-port)
	  (ilu-process:fork-process
	   (format nil "ILU Listener thread for connections on port 0x~x (fd ~d)"
		   s-port (file-descriptor-of-mooring-of-port s-port))
	   #'watch-for-connections s-port (file-descriptor-of-mooring-of-port s-port))
	  s-port)))))
