/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: server.c,v 1.34 1994/04/04 22:29:46 spreitze Exp $ */
/* Last tweaked by Mike Spreitzer April 4, 1994 10:43 am PDT */

#define _POSIX_SOURCE

#include <stdlib.h>	/* for srand() and rand() */
#include <unistd.h>	/* for getpid() */
#include <time.h>	/* for time() */

#include <ilu.h>

#include "iluntrnl.h"
#include "server.h"
#include "object.h"
#include "type.h"
#include "connect.h"
#include "port.h"

/*L1 >= {smu}*/
/*L2, Main unconstrained*/

static ilu_private Servers = NULL;
static int need_seed = 1;

/*L1_sup < smu*/
ilu_string ilu_InventID ()
{
  char buf[1000];
  
  _ilu_AcquireMutex(ilu_smu);

  if (need_seed) {
    srand (OS_GETPID());
    need_seed = 0;
  }
  sprintf (buf, "%s.%x.%lx.%lx", _ilu_tcp_CurrentHostInetName(),
           OS_GETPID(), time(NULL), rand() );
  _ilu_ReleaseMutex(ilu_smu);
  return _ilu_Strdup(buf);
}

/*L1, L2, Main unconstrained*/

ilu_string ilu_IDOfServer ( ilu_Server s )
{
  return (s->sr_id);
}

ilu_boolean ilu_TrueServerP ( ilu_Server s )
{
  return (s->sr_true);
}

/*L1 >= {smu}*/
static void BeStarted(void)
{
  if (Servers == NULL)
    {
      Servers = _ilu_hash_MakeNewTable (SERVER_HASHTABLESIZE, NULL, NULL);
    }
  return;
}

/*L1_sup < smu*/

ilu_Server _ilu_FindServer (ilu_string serverID, ilu_boolean add,
			    ilu_string cinfo)
{
  ilu_Server s;
  ilu_string pinfo=NULL, tinfo=NULL;
  ilu_TransportClass tc;
  ilu_Protocol p;
  _ilu_AcquireMutex(ilu_smu);
  BeStarted();
  s = (ilu_Server) _ilu_hash_FindInTable (Servers, serverID);
  if ((s!=NULL) || !add)
      {}
  else if (!_ilu_ParseConnectInfo(cinfo, NULL, &pinfo, NULL, &tinfo)) {
      DEBUG(CONNECTION_DEBUG,
	    (stderr, "%s %s for server %s.\n",
	     "_ilu_FindServer:  error parsing contact info",
	     cinfo, serverID));
    }
  else if ((tc = _ilu_GetTransportClassByName(tinfo)) == NULL) {
      DEBUG(CONNECTION_DEBUG,
	    (stderr, "%s %s for server %s.\n",
	     "_ilu_FindServer:  invalid transport info",
	     tinfo, serverID));
    }
  else if ((p = _ilu_GetProtocolFromInfo(pinfo)) == NULL) {
      DEBUG(CONNECTION_DEBUG,
	    (stderr, "%s %s for server %s.\n",
	     "_ilu_FindServer:  invalid protocol info",
	     pinfo, serverID));
    }
  else {
      s = (ilu_Server) malloc (sizeof(struct _ilu_Server_s));
      s->sr_lock		= _ilu_CreateMutex("server ", serverID);
      s->sr_true		= FALSE;
      s->sr_id		= _ilu_Strdup(serverID);
      s->sr_tinfo		= tinfo;
      s->sr_trcls		= tc;
      s->sr_pinfo		= pinfo;
      s->sr_protocol	= p;
      s->sr_closing	= FALSE;
      s->sr_connHead.next	= NULL;
      s->sr_connHead.prev	= NULL;
      s->sr_ports		= NULL;
      s->sr_objs		= _ilu_hash_MakeNewTable(OBJECT_HASHTABLESIZE,
						 NULL, NULL);
      s->sr_singles	= _ilu_hash_MakeNewTable(3, _ilu_hash_HashPointer,
						_ilu_hash_PointerCompare);
      s->sr_objtab		= NULL;
      s->sr_default_port	= NULL;
      _ilu_hash_AddToTable (Servers, server_id(s), s);
      DEBUG(CONNECTION_DEBUG,
	    (stderr, "%s 0x%x via %s|%s for <%s>.\n",
	     "_ilu_FindServer:  Created new server", (unsigned long) s, s->sr_pinfo,
	     s->sr_tinfo, serverID));
    }
  /* FREETOKEN(tinfo); FREETOKEN(pinfo) when server closed */
  _ilu_ReleaseMutex(ilu_smu);
  return (s);
}

ilu_Server ilu_CreateTrueServer (
  ilu_string id,
  ilu_ObjectTable objtab)
{
  ilu_Server new;
  _ilu_AcquireMutex(ilu_smu);
  BeStarted();
  new = (ilu_Server) _ilu_hash_FindInTable (Servers, id);
  if (new != NULL) {
      DEBUG(0xFFFF, (stderr,
		     "ilu_CreateTrueServer:  given non-new id %s.\n",
		     id ));
      _ilu_ReleaseMutex(ilu_smu);
      return NULL;
    }
  new = (ilu_Server) malloc (sizeof(struct _ilu_Server_s));
  new->sr_lock	= _ilu_CreateMutex("server ", id);
  new->sr_true	= TRUE;
  new->sr_id	= id;
  new->sr_tinfo = new->sr_pinfo = NULL;
  new->sr_trcls	= NULL;
  new->sr_protocol	= NULL;
  new->sr_closing	= FALSE;
  new->sr_connHead.next = new->sr_connHead.prev = NULL;
  new->sr_ports	= NULL;
  new->sr_objs	= _ilu_hash_MakeNewTable(OBJECT_HASHTABLESIZE,
					 NULL, NULL);
  new->sr_singles	= _ilu_hash_MakeNewTable(3, _ilu_hash_HashPointer,
					    _ilu_hash_PointerCompare);
  new->sr_objtab	= objtab;
  new->sr_default_port = NULL;
  _ilu_hash_AddToTable (Servers, server_id(new), new);
  _ilu_ReleaseMutex(ilu_smu);
  return (new);
}

/*L1_sup < s*/
void ilu_SetServerDefaultPort(ilu_Server s, ilu_Port p)
{
  if (port_server(p) == s) {
      _ilu_AcquireMutex(server_lock(s));
      server_default_port(s) = p;
      _ilu_ReleaseMutex(server_lock(s));
    }
  else {
      /* Bitch to caller, when we get our error system. */
      return;
    }
}

/*L1 >= {s}*/
void _ilu_ClearConnectionFromServer (ilu_Connection c, ilu_Server s)
{
  _ilu_HoldMutex(server_lock(s));
  _ilu_Assert(connection_server(c) == s, "ClearConnectionFromServer");
  if (server_is_true(s)) {
      ilu_Port p = c->co_port;
      _ilu_UnlinkConnection(&p->po_connHead, c, ilu_psl);
      if (port_connections(p)==NULL && port_closed(p))
          _ilu_ClearPortFromServer(p, s);
      return;
    }
  else {
      _ilu_UnlinkConnection(&s->sr_connHead, c, ilu_psl);
      return;
    }
}

/*before: Main Invariant holds,
  after:  result!=NULL => Inside(s, intro_type);
  after:  result==NULL => Main Invariant holds*/
ilu_Object ilu_GetServerSingleton(ilu_Server s, ilu_Class intro_type)
{
  ilu_Object ans;
  ilu_EnterServer(s, intro_type);
  if (s->sr_singles == NULL) {
      ans = NULL;
      ilu_ExitServer(s, intro_type);
    }
  else {
      ans = (ilu_Object) _ilu_hash_FindInTable (server_singles(s),
						intro_type);
      if (ans == NULL)
          ilu_ExitServer(s, intro_type);
    }
  return ans;
}

/*L1 >= {cmu, s}; L2, Main unconstrained*/

static void CloseNonIoingConns(ilu_Server s, ilu_Connection first)
{
  ilu_Connection cur = first, next;
  while (cur != NULL) {
      next = cur->co_links[ilu_psl].next;
      if (cur->co_ioing != TRUE) {
          cur->co_ioing = TRUE;
          _ilu_CloseIoingConnection(cur);
          _ilu_ReleaseConnIO(cur);
        }
      cur = next;
    }
  return;
}

static void DisconnectServer(ilu_Server s)
{
  if (server_is_true(s)) {
      ilu_Port p = server_ports(s);
      while (p != NULL) {
          CloseNonIoingConns(s, p->po_connHead.next);
          p = p->po_next;
        }
    }
  else CloseNonIoingConns(s, server_connections(s));
}

void _ilu_ServerRemoveObject (ilu_Server s, ilu_Object obj)
{
  ASSERT(s->sr_objs != NULL, buf,
	 (buf, "ServerRemoveObject: (%s)->sr_objs == NULL", s->sr_id));
  _ilu_hash_RemoveFromTable(s->sr_objs, object_ih(obj));
  if (_ilu_hash_PairsInTable(s->sr_objs) > 0)
      {}
  else if (s->sr_closing == TRUE) {
      _ilu_Assert(_ilu_hash_PairsInTable(s->sr_singles) == 0,
                  "BankServer: singles not empty too");
      _ilu_hash_FreeHashTable(s->sr_objs, NULL, NULL);
      s->sr_objs = NULL;
      _ilu_hash_FreeHashTable(s->sr_singles, NULL, NULL);
      s->sr_singles = NULL;
    }
  else
      DisconnectServer(s);
  return;
}

/*L1 < cmu*/

void ilu_BankServer (ilu_Server s)
{
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(ilu_smu);
  _ilu_AcquireMutex(server_lock(s));
  if (s->sr_closing == TRUE) {
      _ilu_ReleaseMutex(server_lock(s));
      _ilu_ReleaseMutex(ilu_smu);
      _ilu_ReleaseMutex(ilu_cmu);
      return;
    }
  s->sr_closing = TRUE;
  if (Servers != NULL)
    {
      if (_ilu_hash_FindInTable (Servers, server_id(s)) != NULL)
	_ilu_hash_RemoveFromTable (Servers, server_id(s));
    }
  if (s->sr_objtab != NULL) {
      (*s->sr_objtab->ot_free_self)(s->sr_objtab);
      s->sr_objtab = NULL;
    }
  
  DisconnectServer(s);
  
  if (server_is_true(s)) {
      ilu_Port cur = server_ports(s), next;
      while (cur != NULL) {
          next = cur->po_next;
          _ilu_ClosePort(cur);
          cur = next;
        }
    }

  if (_ilu_hash_PairsInTable(s->sr_objs) == 0) {
      _ilu_Assert(_ilu_hash_PairsInTable(s->sr_singles) == 0,
                  "BankServer: singles not empty too");
      _ilu_hash_FreeHashTable(s->sr_objs, NULL, NULL);
      s->sr_objs = NULL;
      _ilu_hash_FreeHashTable(s->sr_singles, NULL, NULL);
      s->sr_singles = NULL;
    }

  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_smu);
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

/*before: 				       L1 disjoint {cmu, server};
  before: cl collectible		    => L1  not >=  {gcmu};
  before: cl collectible & server surrogate => Main Invariant holds;
  after:  Inside(server, cl)*/
void ilu_EnterServer(ilu_Server server, ilu_Class cl)
{
  if (class_collectible(cl) && server->sr_true)
      _ilu_AcquireMutex(ilu_gcmu);
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(server));
  return;
}

/*before: Inside(server, cl);
  after:				      L1 disjoint {cmu, server};
  after: cl collectible			   => L1  not >=  {gcmu};
  after: cl collectible & server surrogate => Main Invariant holds*/
void ilu_ExitServer(ilu_Server server, ilu_Class cl)
{
  _ilu_ReleaseMutex(server_lock(server));
  _ilu_ReleaseMutex(ilu_cmu);
  if (class_collectible(cl) && server->sr_true)
      _ilu_ReleaseMutex(ilu_gcmu);
  return;
}

/*Inside(server, result's type)*/
ilu_Object _ilu_FindObjectInServer(ilu_string ih, ilu_Server s)
{
  ilu_Object o = (ilu_Object) _ilu_hash_FindInTable(server_objs(s), ih);
  ilu_Object o2;
  if (o==NULL && server_is_true(s) && server_objtab(s)!=NULL) {
      o = (*server_objtab(s)->ot_object_of_ih) (server_objtab(s), ih);
      if (o == NULL)
          {}
      else if (!_ilu_Addable(s, object_class(o), &o2)) {
          DEBUG(SERVER_DEBUG,
                (stderr,
                 "%s %s of singleton type %s in server %s because %s %s",
                 "FindObjectInServer: won't cache", ih,
                 object_class(o)->cl_unique_id, s->sr_id, o2->ob_ih,
                 "already exists.\n"));
        }
      else {
          _ilu_Assert(_ilu_hash_AddToTable(server_objs(s), ih, o),
		      "FindObjectInServer: add to cache (1) failed");
          _ilu_AddSingleton(s, object_class(o), o);
        }
    }
  return (o);
}
