/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: courier.c,v 1.54 1994/04/28 23:44:29 spreitze Exp $ */
/* Last tweaked by Mike Spreitzer April 28, 1994 3:19 pm PDT */

#define _POSIX_SOURCE

#include <string.h>

#include "ilu.h"
#include "iluntrnl.h"
#include "packet.h"
#include "courier.h"

#include "call.h"
#include "protocol.h"
#include "connect.h"
#include "pipe.h"
#include "transprt.h"
#include "port.h"
#include "object.h"
#include "type.h"
#include "method.h"
#include "mooring.h"

#define ODD(x)		(((x)&0x1)!=0)
#define EVEN(x)		(((x)&0x1)==0)  
#define PADDED_SIZE(x)	((((unsigned) (x))+1) & (~0x1))
  
ilu_boolean _courier_InputSequence(ilu_Call c,
		ilu_cardinal *sequenceLength, ilu_cardinal limit);

ilu_cardinal _courier_SizeOfSequence(ilu_Call c, ilu_cardinal length,
				     ilu_cardinal limit);

/*************************************************************/
/*		Implement Courier PACKET		     */
/*************************************************************/

/*L1, Main unconstrained*/
/*L2 >= call's conn's iomu*/

static ilu_boolean _cp_put_u_short (PACKET p, ilu_shortcardinal l)
{
  if (packet_allocation(p) - packet_size(p) >= 2)
    {
      *p->next++ = (l >> 8) & 0xFF;
      *p->next++ = l & 0xFF;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _cp_put_long (PACKET p, ilu_integer l)
{
  return (_cp_put_u_short (p, (l >> 16) & 0xFFFF) && _cp_put_u_short (p, l & 0xFFFF));
}

static ilu_boolean _cp_put_short (PACKET p, ilu_shortinteger l)
{
  return (_cp_put_u_short (p, (ilu_shortcardinal) l));
}

static ilu_boolean _cp_put_byte (PACKET p, char l)
{
  return (_cp_put_u_short (p, (unsigned char) l));
}

static ilu_boolean _cp_put_u_long (PACKET p, ilu_cardinal l)
{
  return (_cp_put_u_short (p, (l >> 16) & 0xFFFF) && _cp_put_u_short (p, l & 0xFFFF));
}

static ilu_boolean _cp_put_u_byte (PACKET p, ilu_byte l)
{
  return (_cp_put_u_short (p, (ilu_shortcardinal) l));
}

static ilu_boolean _cp_put_float (PACKET p, float l)
{
  float l2 = l;

  return (_cp_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

static ilu_boolean _cp_put_double (PACKET p, double l)
{
  double l2 = l;

  return (_cp_put_u_long (p, ((ilu_cardinal *) &l2)[1]) && _cp_put_u_long (p, ((ilu_cardinal *) &l2)[0]));
}

static ilu_boolean _cp_put_bytes (PACKET p, ilu_bytes b, ilu_cardinal l)
{
  register ilu_bytes p1, p2;
  register ilu_cardinal c;

  DEBUG(COURIER_DEBUG,
        (stderr,
         "%s 0x%x, size 0x%x (diff %d), needed %u\n",
         "_cp_put_bytes:  allocation", packet_allocation(p),
         packet_size(p), packet_allocation(p) - packet_size(p),
         (2 + PADDED_SIZE(l))));
			
  if (packet_allocation(p) - packet_size(p) >= (2 + PADDED_SIZE(l)))
    {
      _cp_put_u_short (p, l);
      for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	*p1++ = *p2++;
      if (ODD(l))
	*p1++ = 0;
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _cp_get_u_short (PACKET p, ilu_shortcardinal *l)
{
  if (packet_allocation(p) - packet_size(p) < 2)
    return(FALSE);
  *l = (p->next[0] << 8) + p->next[1];
  p->next += 2;
  return (TRUE);
}

static ilu_boolean _cp_get_long (PACKET p, ilu_integer *l)
{
  return (_cp_get_u_short(p, ((ilu_shortcardinal *) l)) && _cp_get_u_short (p, ((ilu_shortcardinal *) l) + 1));
}

static ilu_boolean _cp_get_short (PACKET p, ilu_shortinteger *l)
{
  return (_cp_get_u_short(p, (ilu_shortcardinal *) l));
}

static ilu_boolean _cp_get_byte (PACKET p, char *l)
{
  ilu_shortcardinal l2;
  ilu_boolean stat;

  stat = _cp_get_u_short(p, &l2);
  if (stat)
    *l = (char) (unsigned char) (l2 & 0xFF);
  return (stat);
}

static ilu_boolean _cp_get_u_long (PACKET p, ilu_cardinal *l)
{
  ilu_boolean stat;

  stat = _cp_get_u_short(p, (ilu_shortcardinal *) l) && _cp_get_u_short (p, ((ilu_shortcardinal *) l) + 1);
  DEBUG(COURIER_DEBUG, (stderr, "_cp_get_u_long:  stat = %s, *l = 0x%x\n",
			stat ? "TRUE" : "FALSE", *l));
  return (stat);
}

static ilu_boolean _cp_get_u_byte (PACKET p, ilu_byte *l)
{
  ilu_shortcardinal l2;
  ilu_boolean stat;

  stat = _cp_get_u_short(p, &l2);
  if (stat)
    *l = (ilu_byte) (l2 & 0xFF);
  return (stat);
}

static ilu_boolean _cp_get_float (PACKET p, float *l)
{
  return (_cp_get_u_long (p, (ilu_cardinal *) l));
}

static ilu_boolean _cp_get_double (PACKET p, double *l)
{
  return (_cp_get_u_long (p, ((ilu_cardinal *) l) + 1) && _cp_get_u_long(p, ((ilu_cardinal *) l)));
}

static ilu_boolean _cp_get_bytes(PACKET p, ilu_bytes * b, ilu_cardinal *l, ilu_cardinal limit)
{
  ilu_shortcardinal l2;
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) < 2
      || (!_cp_get_u_short (p, &l2))
      || packet_allocation(p) - packet_size(p) < PADDED_SIZE(l2))
    return (FALSE);

  *l = 0;
  if (limit > 0 AND l2 > limit)
    {
      fprintf (stderr,
	       "%s %u, which exceeds call limit of %u.\n",
	       "Attempt to read string of length", l2,
	       limit);
      return (FALSE);
    }

  if (*b == NULL)
    *b = (ilu_byte *) malloc(l2 + 1);
  for (p1 = *b, p2 = p->next, c = l2;  c > 0;  c--)
    *p1++ = *p2++;
  if (ODD(l2))
    p2++;
  *p1 = '\0';
  p->next = p2;
  *l = l2;
  return (TRUE);
}

static ilu_boolean _cp_get_opaque (PACKET p, ilu_byte **b, ilu_cardinal len)
{
  return (_cp_get_opaque (p, b, len));
}

static void _cp_destroy (PACKET p)
{
  if (p->buf != NULL && p->count > 0)
    free (p->buf);
  free (p);
}

static PACKET _courier_CreatePacket (ilu_bytes buf, ilu_cardinal size)
{
  PACKET p = (PACKET) malloc (sizeof(struct ilu_packet_s));
  if (buf != NULL)
    p->buf = buf;
  else if (size > 0)
    p->buf = (ilu_byte *) malloc(size);
  else
    p->buf = NULL;
  p->count = size;
  p->next = p->buf;

  p->put_long = _cp_put_long;
  p->put_byte = _cp_put_byte;
  p->put_short = _cp_put_short;
  p->put_u_long = _cp_put_u_long;
  p->put_u_byte = _cp_put_u_byte;
  p->put_u_short = _cp_put_u_short;
  p->put_float = _cp_put_float;
  p->put_double = _cp_put_double;
  p->put_bytes = _cp_put_bytes;
  p->put_opaque = _cp_put_bytes;

  p->get_long = _cp_get_long;
  p->get_byte = _cp_get_byte;
  p->get_short = _cp_get_short;
  p->get_u_long = _cp_get_u_long;
  p->get_u_byte = _cp_get_u_byte;
  p->get_u_short = _cp_get_u_short;
  p->get_float = _cp_get_float;
  p->get_double = _cp_get_double;
  p->get_bytes = _cp_get_bytes;
  p->get_opaque = _cp_get_opaque;

  p->destroy = _cp_destroy;

  return (p);
}

/**********************************************************************
  End of packet implementation
***********************************************************************/

/*L1, L2, Main unconstrained*/

static ilu_refany _courier_CreateDataBlock (void)
{
  struct Courier *new = (struct Courier *) malloc(sizeof(struct Courier));

  new->cr_in = NULL;
  new->cr_out = NULL;
  new->cr_sequence_mode = None;
  new->cr_sequence_limit = 0;
  new->cr_sequence_length = 0;
  return ((ilu_refany) new);
}

/* AC: change this back!!! */
static void _courier_FreeDataBlock (struct Courier *d)
{
  if (d->cr_in != NULL)
    {
      packet_destroy (d->cr_in);
      d->cr_in = NULL;
    }
  if (d->cr_out != NULL)
    {
      packet_destroy (d->cr_out);
      d->cr_out= NULL;
    }
  free (d);
}

static void formRegistryName (ilu_string buf)
{
  ilu_string registryhome = (ilu_string ) getenv("ILUREGISTRY");
  ilu_string iluhome = (ilu_string ) getenv ("ILUHOME");

#ifdef MACOS
  if (registryhome != NULL)
    sprintf (buf, "%s:%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s:lib:%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s:%s", REGISTRY_LAST_RESORT, PROGRAM_NUMBER_REGISTRY_NAME);
#else
  if (registryhome != NULL)
    sprintf (buf, "%s/%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s/lib/%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s/%s", REGISTRY_LAST_RESORT, PROGRAM_NUMBER_REGISTRY_NAME);
#endif
}

/*L1 >= {prmu}*/

static HashTable ProgramNumberHashTable = NULL;
/* program number -> courierinfo* */

static HashTable RegistryHashTable = NULL;
/* type unique_id -> courierinfo* */

static void _courier_EnsureRegistries(void)
{
  courierinfo *s = NULL;

  if (RegistryHashTable == NULL)
    {
      char buf[1000], line[1000], name[1000], id[1000];
      FILE *f;
      ilu_cardinal lineno = 0;

      RegistryHashTable = _ilu_hash_MakeNewTable (137, NULL, NULL);
      ProgramNumberHashTable = _ilu_hash_MakeNewTable (137, _ilu_hash_HashPointer, _ilu_hash_PointerCompare);

      formRegistryName (buf);
      if ((f = fopen(buf, "r")) == NULL)
	{
	  fprintf (stderr, "_courier_EnsureRegistries:  Can't access program number registry file %s.\n",
		   buf);
	  return;
	}
      while (fgets(line, sizeof(buf), f) != NULL)
	{
	  lineno++;
     	  s = (courierinfo *) malloc(sizeof(courierinfo));
	  s->cri_class = NULL;
	  if (strncmp(line, ".LastAssignedProgramNumber ", 27) == 0)
	    ; /* skip this record */
	  else if (sscanf (line, "%1000[^:]:%1000s %x %u", name, id,
			   &s->cri_pnumber, &s->cri_version) != 4)
	    fprintf (stderr, "_courier_EnsureRegistries:  Invalid record at line %u in program number registry file \"%s\":  %s", lineno, buf, line);
	  else
	    {
	      if (_ilu_hash_FindInTable (RegistryHashTable, id)
		  OR _ilu_hash_FindInTable (ProgramNumberHashTable, (void *) s->cri_pnumber))
		{
		  fprintf (stderr, "_sunrpc_EnsureRegistries:  Duplicate type id or program number at line %u in program number registry \"%s\":  <%s> (ignoring)\n", lineno, buf, line);
		  continue;
		}
	      _ilu_hash_AddToTable (RegistryHashTable, s->cri_type_id = _ilu_Strdup(id), s);
	      _ilu_hash_AddToTable (ProgramNumberHashTable, (void *) s->cri_pnumber, s);
	    }
	}
      fclose (f);
    }
}

static courierinfo *_courier_FindCourierInformationForClass (ilu_Class class)
{
  courierinfo *s = NULL;

  _courier_EnsureRegistries();

  if ((s = (courierinfo *) _ilu_hash_FindInTable (RegistryHashTable, class->cl_unique_id)) != NULL)
    s->cri_class = class;

  DEBUG(COURIER_DEBUG,
	(stderr, "_courier_FindCourierInformationForClass:  Class \"%s:%s\", pnumber is 0x%x, version is %d.\n",
	 class->cl_name, class->cl_unique_id, s == NULL ? 0 : s->cri_pnumber, s == NULL ? 0 : s->cri_version));

  return (s);
}

static courierinfo *_courier_FindClassFromProgramNumber (ilu_cardinal pnumber, ilu_cardinal version)
{
  courierinfo *s = NULL;

  _courier_EnsureRegistries();

  s = (courierinfo *) _ilu_hash_FindInTable (ProgramNumberHashTable, (void *) pnumber);

  DEBUG(COURIER_DEBUG,
	(stderr, "_courier_FindClassFromProgramNumber:  pnumber is 0x%x, version is %u(%u), class is \"%s:%s\".\n",
	 pnumber, version, (s == NULL) ? 0 : s->cri_version,
	 (s == NULL) ? "" : s->cri_class->cl_name, (s == NULL) ? "" : s->cri_class->cl_unique_id));

  return (s);
}

/*======================================================================*/
/*======================== Courier XDR code ========================*/
/*======================================================================*/

#define INPUT_ERROR		1
#define OUTPUT_ERROR		2

/*L1, L2, Main unconstrained*/
static ilu_cardinal CourierMaxStringSize = 1000000;

/*L1, L2, Main unconstrained (this is only for calling from debugger)*/
static ilu_cardinal ilu_courier_SetMaxStringSize (ilu_cardinal size)
{
  ilu_cardinal old_size = CourierMaxStringSize;
  if (size > 0)
    CourierMaxStringSize = size;
  return (old_size);  
}

/*L2 >= {call's connection's iomu}*/
/*L1, Main unconstrained*/

/* ==================== integer ==================== */

static ilu_boolean _courier_OutputInteger (ilu_Call call, ilu_integer i)
{
  return (packet_put_long (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputInteger (ilu_Call call, ilu_integer *i)
{
  return (packet_get_long (COURIER(call_connection(call))->cr_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfInteger (ilu_Call call, ilu_integer i)
{
  return (4);
}

/* ==================== cardinal ==================== */

static ilu_boolean _courier_OutputCardinal (ilu_Call call, ilu_cardinal i)
{
  return (packet_put_u_long (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (packet_get_u_long (COURIER(call_connection(call))->cr_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfCardinal (ilu_Call call, ilu_cardinal i)
{
  return (4);
}

/* ==================== short integer ==================== */

static ilu_boolean _courier_OutputShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (packet_put_short (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputShortInteger (ilu_Call call, ilu_shortinteger *i)
{
  return (packet_get_short (COURIER(call_connection(call))->cr_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (2);
}

/* ==================== long integer ==================== */
#ifdef DEAD_CODE
static ilu_boolean _courier_OutputLongInteger (ilu_Call call, ilu_integer *i)
{
  return (packet_put_u_long (COURIER(call_connection(call))->cr_out, &i[1])
	  && packet_put_long (COURIER(call_connection(call))->cr_out, &i[0]));
}

static ilu_boolean _courier_InputLongInteger (ilu_Call call, ilu_longinteger (*i))
{
  return (packet_get_u_long (COURIER(call_connection(call))->cr_in, &(*i)[1])
	  && packet_get_long (COURIER(call_connection(call))->cr_in, &(*i)[0]));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfLongInteger (ilu_Call call, ilu_integer *i)
{
  return (8);
}
#endif
/* ==================== short cardinal ==================== */

static ilu_boolean _courier_OutputShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (packet_put_u_short (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputShortCardinal (ilu_Call call, ilu_shortcardinal *i)
{
  return (packet_get_u_short (COURIER(call_connection(call))->cr_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (2);
}

/* ==================== long cardinal ==================== */

#ifdef DEAD_CODE

static ilu_boolean _courier_OutputLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (packet_put_u_long (COURIER(call_connection(call))->cr_out, &i[1])
	  && packet_put_u_long (COURIER(call_connection(call))->cr_out, &i[0]));
}

static ilu_boolean _courier_InputLongCardinal (ilu_Call call, ilu_longcardinal (*i))
{
  return (packet_get_u_long (COURIER(call_connection(call))->cr_in, &(*i)[1])
	  && packet_get_u_long (COURIER(call_connection(call))->cr_in, &(*i)[0]));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (8);
}
#endif

/* ==================== optional ==================== */

static ilu_boolean _courier_OutputOptional (ilu_Call call, ilu_boolean i)
{
  return (packet_put_u_short (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputOptional (ilu_Call call, ilu_boolean *i)
{
  ilu_shortcardinal i2;
  ilu_boolean stat;

  stat = packet_get_u_short (COURIER(call_connection(call))->cr_in, &i2);
  *i = i2;
  return(stat);
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfOptional (ilu_Call call, ilu_boolean i)
{
  return (2);
}

/* ==================== enumeration ==================== */

static ilu_boolean _courier_OutputEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  return (packet_put_u_short (COURIER(call_connection(call))->cr_out, i));
}

static ilu_boolean _courier_InputEnumeration (ilu_Call call, ilu_shortcardinal *i)
{
  return (packet_get_u_short (COURIER(call_connection(call))->cr_in, i));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  return (2);
}

/* ==================== real ==================== */

static ilu_boolean _courier_OutputReal (ilu_Call call, double d)
{
  return (packet_put_double (COURIER(call_connection(call))->cr_out, d));
}

static ilu_boolean _courier_InputReal (ilu_Call call, double *d)
{
  return (packet_get_double (COURIER(call_connection(call))->cr_in, d));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfReal (ilu_Call call, double d)
{
  return (8);
}

/* ==================== short real ==================== */

static ilu_boolean _courier_OutputShortReal (ilu_Call call, float f)
{
  return (packet_put_float (COURIER(call_connection(call))->cr_out, f));
}

static ilu_boolean _courier_InputShortReal (ilu_Call call, float *f)
{
  return (packet_get_float (COURIER(call_connection(call))->cr_in, f));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfShortReal (ilu_Call call, float d)
{
  return (4);
}

/* ==================== string ==================== */

static ilu_boolean _courier_OutputString (ilu_Call call, ilu_string s, ilu_cardinal len, ilu_cardinal limit)
{
  DEBUG(COURIER_DEBUG, (stderr, "_courier_OutputString:  s = 0x%x, len = %u, limit = %u\n",
			(unsigned long) s, len, limit));
  if (limit > 0 && limit <= 0xFFFF)
    return (packet_put_bytes (COURIER(call_connection(call))->cr_out, (ilu_bytes) s, len));
  else
    {
      ilu_boolean stat = TRUE;
      ilu_cardinal segments, l2;
      ilu_string p;

      segments =  (len / 0xFFFF) + ((len % 0xFFFF) != 0);
      _courier_OutputCardinal (call, len);
      packet_put_bytes (COURIER(call_connection(call))->cr_out, (ilu_bytes) s, (len > 0xFFFF) ? 0xFFFF : len);
      packet_put_bytes (COURIER(call_connection(call))->cr_out, (ilu_bytes) s + 0xFFFF, (len > 0x1FFFE) ? 0xFFFF : ((len > 0xFFFF) ? (len - 0xFFFF) : 0));
      _courier_OutputShortCardinal (call, (segments > 2) ? (segments - 2) : 0);
      for (p = s + 0x1FFFE, l2 = len - 0x1FFFE;  segments > 2;  segments--, p += 0xFFFF, l2 -= 0xFFFF)
	{
	  stat = stat && packet_put_bytes (COURIER(call_connection(call))->cr_out, (ilu_bytes) p, (l2 > 0xFFFF) ? 0xFFFF : l2);
	}
      return (stat);
    }
}

static ilu_boolean _courier_InputString (ilu_Call call, ilu_string *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  if (limit > 0 && limit <= 0xFFFF)
    return (packet_get_bytes (COURIER(call_connection(call))->cr_in, 
		(ilu_byte **) s, len, limit));
  else
    {
      ilu_boolean stat = TRUE;
      ilu_cardinal slen = 0;
      ilu_string s2 = NULL;
      ilu_shortcardinal junk;
      ilu_cardinal s2len = 0, finallen = 0;

      _courier_InputCardinal (call, &slen);
      *s = s2 = (ilu_string) malloc(slen + 1);
      stat = packet_get_bytes (COURIER(call_connection(call))->cr_in, 
		(ilu_byte **) s, &finallen, limit);
      if (slen > 0xFFFF)
	{
	  s2 += 0xFFFF;
	  stat = stat && (packet_get_bytes (COURIER(call_connection(call))->cr_in,
		(ilu_byte **) &s2, &s2len, limit));
	  finallen += s2len;
	}
      else
	stat = stat && _courier_InputShortCardinal (call, &junk);
      if (slen > 0x1FFFE)
	{
	  ilu_shortcardinal count;

	  stat = stat && _courier_InputShortCardinal (call, &count);
	  for (s2 += 0xFFFF;  count > 0;  s2 += 0xFFFF, count -= 1)
	    {
	      if (!stat)
		return (FALSE);
	      stat = stat && packet_get_bytes (COURIER(call_connection(call))->cr_in, (ilu_byte **) &s2, &s2len, limit);
	      finallen += s2len;
	    }
	}
      else
	stat = stat && _courier_InputShortCardinal (call, &junk);
      
      *len = finallen;
      return (stat);
    }
}

  /*ARGSUSED*/
static ilu_cardinal _courier_SizeOfString (ilu_Call call, ilu_string s, ilu_cardinal len, ilu_cardinal limit)
{
  if (limit > 0 && limit <= 0xFFFF)
    return (2 + PADDED_SIZE(len));
  else
    {
      ilu_cardinal segments, l2, size, size2;
      ilu_string p;

      segments =  len / 0xFFFF + ((len % 0xFFFF) != 0);
      size = _courier_SizeOfCardinal (call, len)
	+ _courier_SizeOfSequence (call, (len <= 0xFFFF) ? len : 0xFFFF, 0xFFFF)
	  + (2 + PADDED_SIZE((len <= 0xFFFF) ? len : 0xFFFF))
	    + _courier_SizeOfSequence (call, (len > 0x1FFFE) ? 0xFFFF : ((len <= 0xFFFF) ? 0 : (len - 0xFFFF)), 0xFFFF)
	      + (2 + PADDED_SIZE((len > 0x1FFFE) ? 0xFFFF : ((len <= 0xFFFF) ? 0 : (len - 0xFFFF))))
		+ _courier_SizeOfSequence (call, segments - 2, 0xFFFF);
      for (p = s + 0x1FFFE, l2 = len - 0x1FFFE;  segments > 2;  segments--, p += 0xFFFF, l2 -= 0xFFFF)
	{
	  size2 = _courier_SizeOfString (call, p, (l2 > 0xFFFF) ? 0xFFFF : l2, 0xFFFF);
	  if (size2 == 0)
	    return (0);
	  else
	    size += size2;
	}
      return (size);
    }
}

/* ==================== bytes ==================== */

static ilu_boolean _courier_OutputBytes (ilu_Call call, ilu_bytes s, ilu_cardinal len, ilu_cardinal limit)
{
  return (_courier_OutputString (call, (ilu_string) s, len, limit));
}

static ilu_boolean _courier_InputBytes (ilu_Call call, ilu_bytes *s, ilu_cardinal *len, ilu_cardinal limit)
{
  return (_courier_InputBytes (call, s, len, limit));
}

  /*ARGSUSED*/
static ilu_cardinal _courier_SizeOfBytes (ilu_Call call, ilu_bytes s, ilu_cardinal len, ilu_cardinal limit)
{
  return (_courier_SizeOfString (call, (ilu_string) s, len, limit));
}

/* ==================== byte ==================== */

static ilu_boolean _courier_OutputByte (ilu_Call call, ilu_byte b)
{
  return (packet_put_byte (COURIER(call_connection(call))->cr_out, b));
}

static ilu_boolean _courier_InputByte (ilu_Call call, ilu_byte *b)
{
  return (packet_get_byte (COURIER(call_connection(call))->cr_in, (char *) b));
}

/*ARGSUSED*/
static ilu_cardinal _courier_SizeOfByte (ilu_Call call, ilu_byte i)
{
  return (2);
}

/* ==================== opaque ==================== */

static ilu_boolean _courier_OutputOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (packet_put_bytes (COURIER(call_connection(call))->cr_out, o, len));
}

static ilu_boolean _courier_InputOpaque (ilu_Call call, ilu_bytes *o, ilu_cardinal len)
{
  return (packet_get_opaque (COURIER(call_connection(call))->cr_in, o, len));
}

static ilu_cardinal _courier_SizeOfOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (PADDED_SIZE(len));
}

/*  Mapping long sequences

Problem:  map sequences of size 2^32-1 or less
to sequences of size 2^16-1 or less.

Using a sequence of sequences would yield

	(2^16-1)*(2^16-1)
      = 2^32 - 2^17 + 1

Adding two more short sequences (2^17 - 2) would bring us to

	2^32 - 1

which should work.  So an appropriate format would be

	CARDINAL		(length)
	SHORT SEQUENCE		(first 2^16-1 elements)
	SHORT SEQUENCE		(second 2^16-1 elements)
	SHORT SEQUENCE OF SHORT SEQUENCE
				(other 2^32 - 2^17 + 1 elements)
*/

static ilu_boolean _courier_OutputSequence (ilu_Call c, ilu_cardinal sequenceLength, ilu_cardinal limit)
{
  COURIER(call_connection(c))->cr_sequence_limit = limit;
  COURIER(call_connection(c))->cr_sequence_length = sequenceLength;
  COURIER(call_connection(c))->cr_sequence_mode = Output;

  if (limit > 0 && limit <= 0xFFFF)
    return (_courier_OutputShortCardinal (c, (ilu_shortcardinal) sequenceLength));
  else
    {
      ilu_boolean stat;

      stat = _courier_OutputCardinal (c, sequenceLength);
      stat = stat && _courier_OutputSequence (c, (sequenceLength < 0xFFFF) ? sequenceLength : 0xFFFF, 0xFFFF);
      return (stat);
    }
}

static ilu_boolean _courier_OutputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  ilu_cardinal length = COURIER(call_connection(c))->cr_sequence_length;

  if (extent == 0xFFFF && (COURIER(call_connection(c))->cr_sequence_length > 0xFFFF))
    {
      return (_courier_OutputSequence (c, (length < 0x1FFFE) ? (length - 0xFFFF) : 0xFFFF, 0xFFFF));
    }
  else if (extent == 0x1FFFE && (COURIER(call_connection(c))->cr_sequence_length > 0x1FFFE))
    {
      ilu_boolean stat;
      ilu_cardinal elements = (length / 0xFFFF) + ((length % 0xFFFF) != 0) - 2;

      stat = _courier_OutputSequence (c, elements, 0xFFFF);
      if (length > 0x1FFFE)
	stat = stat && _courier_OutputSequence (c, ((length - extent) < 0xFFFF) ? (length - extent) : 0xFFFF, 0xFFFF);
      return (stat);
    }
  else if (extent > 0x1FFFE && (COURIER(call_connection(c))->cr_sequence_length > extent))
    {
      return (_courier_OutputSequence (c, ((length - extent) < 0xFFFF) ? (length - extent) : 0xFFFF, 0xFFFF));
    }
  else
    return (FALSE);
}

static ilu_boolean _courier_InputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  ilu_cardinal length;
  if ((extent / 0xFFFF) <= 2)
    _courier_InputSequence (c, &length, 0xFFFF);
  if ((extent / 0xFFFF) >= 2 && COURIER(call_connection(c))->cr_sequence_length > extent)
    _courier_InputSequence (c, &length, 0xFFFF);
  return (TRUE);
}

ilu_boolean _courier_InputSequence (ilu_Call c, ilu_cardinal *sequenceLength, ilu_cardinal limit)
{
  ilu_boolean stat;
  ilu_shortcardinal junk;

  if (limit > 0 && limit <= 0xFFFF)
    {
      stat = (_courier_InputShortCardinal (c, &junk));
      *sequenceLength = junk;
    }
  else
    {
      stat = _courier_InputCardinal (c, sequenceLength);
      stat = stat && _courier_InputShortCardinal (c, &junk);
    }
  COURIER(call_connection(c))->cr_sequence_length = *sequenceLength;
  COURIER(call_connection(c))->cr_sequence_limit = limit;
  COURIER(call_connection(c))->cr_sequence_mode = Input;
  return(stat);
}

static ilu_boolean _courier_EndSequence (ilu_Call c)
{
  ilu_boolean stat = TRUE;

  if (COURIER(call_connection(c))->cr_sequence_mode == Output)
    {
      if (COURIER(call_connection(c))->cr_sequence_limit > 0xFFFF
	  || COURIER(call_connection(c))->cr_sequence_limit == 0)
	{
	  ilu_cardinal length = COURIER(call_connection(c))->cr_sequence_length;
	  
	  if (length <= 0x1FFFE)
	    stat = stat && _courier_OutputShortCardinal (c, 0);
	  if (length <= 0xFFFF)
	    stat = stat && _courier_OutputShortCardinal (c, 0);
	}
    }
  else if (COURIER(call_connection(c))->cr_sequence_mode == Input)
    {
      if (COURIER(call_connection(c))->cr_sequence_limit > 0xFFFF
	  || COURIER(call_connection(c))->cr_sequence_limit == 0)
	{
	  ilu_cardinal length = COURIER(call_connection(c))->cr_sequence_length;
	  ilu_shortcardinal junk;
	  
	  if (length <= 0x1FFFE)
	    stat = stat && _courier_InputShortCardinal (c, &junk);
	  if (length <= 0xFFFF)
	    stat = stat && _courier_InputShortCardinal (c, &junk);
	}
    }
  COURIER(call_connection(c))->cr_sequence_mode = None;
  return (stat);
}

ilu_cardinal _courier_SizeOfSequence (ilu_Call c, ilu_cardinal length, ilu_cardinal limit)
{
  COURIER(call_connection(c))->cr_sequence_mode = SizeOf;

  if (limit == 0 || limit > 0xFFFF)
    {
      ilu_cardinal size = 0;
      ilu_cardinal elements = (length / 0xFFFF) + ((length % 0xFFFF) != 0);

      size += (4		/* for the length */
	       + 2 + 2 + 2);	/* for the required subsequences */
      if (length > 0x1FFFE)
	size += (2 * (elements - 2));	/* for additional sub-sequences */
      return (size);
    }
  else
    return (_courier_SizeOfShortCardinal(c, (ilu_shortcardinal) length));
}

static ilu_boolean _courier_OutputUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_courier_OutputShortCardinal (c, typeIndex));
}

static ilu_boolean _courier_InputUnion (ilu_Call c, ilu_shortcardinal *typeIndex)
{
  return(_courier_InputShortCardinal (c, typeIndex));
}

static ilu_boolean _courier_EndUnion (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _courier_SizeOfUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_courier_SizeOfShortCardinal(c, typeIndex));
}

static ilu_boolean _courier_OutputArray (ilu_Call c, ilu_cardinal len)
{
  return(TRUE);
}

static ilu_boolean _courier_InputArray (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _courier_EndArray (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _courier_SizeOfArray (ilu_Call c, ilu_cardinal len)
{
  return(0);
}

static ilu_boolean _courier_OutputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _courier_InputRecord (ilu_Call c)
{
  return(TRUE);
}

static ilu_boolean _courier_EndRecord (ilu_Call c)
{
  return (TRUE);
}

static ilu_cardinal _courier_SizeOfRecord (ilu_Call c)
{
  return(0);
}

/*======================================================================
*=======================================================================
*=====================  Non-XDR code ===================================
*=======================================================================
*=====================================================================*/

/*L2, Main unconstrained*/
/*L1_sup < prmu*/
static ilu_string _courier_FormProtocolHandle (ilu_Object obj)
{
  char buf[100];
  courierinfo *s = NULL;
  ilu_Class class = object_class(obj);

  _ilu_AcquireMutex(ilu_prmu);
  if ((s = _courier_FindCourierInformationForClass (class)) == NULL)
    {
      fprintf (stderr, "_courier_FormProtocolHandle:  Can't figure program#/version for class %s.\n",
	       class->cl_name);
      _ilu_ReleaseMutex(ilu_prmu);
      return (NULL);
    }
  else
    {
      sprintf (buf, "courier_0x%x_%u", s->cri_pnumber, s->cri_version);
      _ilu_ReleaseMutex(ilu_prmu);
      return (_ilu_Strdup(buf));
    }
}
     
/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_bytes _courier_ReadPacket (ilu_Transport bs, ilu_cardinal *packetLength, ilu_PacketType *type, ilu_cardinal *sn)
{
  ilu_bytes packet;
  ilu_cardinal serialNumber, packetType, size;

  if (!transport_read_message(bs, &packet, &size)) {
    DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	  (stderr, "_courier_ReadPacket:  error on transport_read_message\n"));
    return (NULL);
  };

  packetType = (packet[0] << 8) | packet[1];
  serialNumber = (packet[2] << 8) | packet[3];

  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_ReadPacket:  read packet of length %d, SN %d, type %d (%s).\n",
	 size, serialNumber, packetType,
	 ((packetType == 0) ? "request"
	  : ((packetType == 2) ? "reply"
	     : ((packetType == 3 || packetType == 1) ? "exception"
		: "unknown type")))));

  *packetLength = size;
  *type = (packetType == 0) ? ilu_PacketType_Request : ilu_PacketType_Reply;
  *sn = serialNumber;
  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_ReadPacket:  returning packet SN %d\n", serialNumber));
  return (packet);  
}

static ilu_boolean _courier_InterpretRequest (ilu_Call call,
				ilu_bytes packet, ilu_cardinal len)
{
  struct Courier *c = COURIER(call_connection(call));
  ilu_cardinal programNumber;
  ilu_shortcardinal programVersion, methodID, junk;
  ilu_Class class;
  courierinfo *s;

  c->cr_in = _courier_CreatePacket (packet, len);

  _courier_InputShortCardinal (call, &junk);	/* packet type */
  _courier_InputShortCardinal (call, &junk);	/* serial no. of call */
  _courier_InputCardinal (call, &programNumber);
  _courier_InputShortCardinal (call, &programVersion);
  _courier_InputShortCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_InterpretRequest:  (call SN %d) class ID = 0x%x (version %u), method ID = %u\n",
	 call->ca_SN, programNumber, programVersion, methodID));

  _ilu_AcquireMutex(ilu_prmu);
  s = _courier_FindClassFromProgramNumber(programNumber, programVersion);
  class = (s == NULL) ? NULL
	: (s->cri_class == NULL) ? ilu_FindClassFromID(s->cri_type_id)
	: s->cri_class;
  call_intro_type(call) = class;
  _ilu_ReleaseMutex(ilu_prmu);
  if (NULL == s OR NULL == class)
    {
      DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	    (stderr, "_courier_InterpretRequest:  (call %d) Can't find ilu_Class with pn 0x%x, version %u\n",
	     call->ca_SN, programNumber, programVersion));
      return (FALSE);
    }

  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_InterpretRequest:  (call SN %d) intro_type is %s\n",
	 call->ca_SN, class->cl_name));

  call->ca_method = ilu_FindMethodByID (class, methodID);
  
  if (call->ca_method == NULL)
    {
      DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	    (stderr, "_courier_InterpretRequest:  (call %d) Can't find method on class %x with methodID = %u.\n",
	     call->ca_SN, programNumber, methodID));
      return (FALSE);
    }
  else
    {
      DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	    (stderr, "_courier_InterpretRequest:  record for method %d is 0x%x (%s).\n",
	     methodID, (unsigned long) call->ca_method, call->ca_method->me_name));
    }
  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_InterpretRequest:  returning TRUE\n"));
  return (TRUE);
}

/*L1, Main unconstrained*/
/*L2 >= {conn's iomu}*/

static ilu_ProtocolException _courier_InterpretReply (ilu_Call call, ilu_bytes packet, ilu_cardinal len, ilu_cardinal *estatus)
{
  struct Courier *c = COURIER(call->ca_connection);
  ilu_shortcardinal replyStatus, junk;
  static ilu_ProtocolException replyStatusExceptions[] = {
    ilu_ProtocolException_NoSuchClassAtServer,
    ilu_ProtocolException_ClassVersionMismatch,
    ilu_ProtocolException_NoSuchMethodOnClass,
    ilu_ProtocolException_GarbageArguments
    };

  c->cr_in = _courier_CreatePacket (packet, len);

  _courier_InputShortCardinal (call, &replyStatus);
  _courier_InputShortCardinal (call, &junk);		/* actually, serial number */

  DEBUG((INCOMING_DEBUG | COURIER_DEBUG),
	(stderr, "_courier_InterpretReply:  SN %d, %d bytes, replyStatus is %d\n",
	 call->ca_SN, len, replyStatus));

  if (replyStatus == 2)		/* MSG_ACCEPTED */
    {
      *estatus = 0;
      return (ilu_ProtocolException_Success);
    }
  else if (replyStatus == 3)
    {
      ilu_shortcardinal e;

      _courier_InputShortCardinal (call, &e);
      *estatus = e;
      return (ilu_ProtocolException_Success);
    }
  else if (replyStatus == 1)
    {
      _courier_InputShortCardinal (call, &replyStatus);
      switch (replyStatus)
	{
	case 1:
	  _courier_InputShortCardinal (call, &junk);	/* lowest version number */
	  _courier_InputShortCardinal (call, &junk);	/* highest version number */

	case 0:
	case 2:
	case 3:
	  return (replyStatusExceptions[replyStatus]);

	default:
	  return (ilu_ProtocolException_Unknown);
	}
    }
  return (ilu_ProtocolException_Unknown);
}

/*L1_sup < prmu*/

static ilu_boolean GetCourierProgramNumberAndVersion (ilu_Class pclass, ilu_cardinal *pnumber, ilu_cardinal *version)
{
  courierinfo *s = NULL;

  _ilu_AcquireMutex(ilu_prmu);
  if ((s = _courier_FindCourierInformationForClass (pclass)) != NULL)	/* probably consult external DB */
    {
      *pnumber = s->cri_pnumber;
      *version = s->cri_version;
    }
  _ilu_ReleaseMutex(ilu_prmu);
  return (s != NULL);
}

static ilu_boolean _courier_BeginRequest (ilu_Call call, ilu_Class pclass, ilu_Method method, ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  ilu_cardinal pnumber, version;

  call->ca_method = method;

  DEBUG(CALL_DEBUG, (stderr, "_courier_BeginRequest:  call 0x%x (sn %d), argSize %d, class %s (%s), method %s (%d)\n",
		     (unsigned long) call, call_serial_number(call), argSize,
		     class_name(pclass), class_unique_id(pclass), method_name(method), method_id(method)));

  if (!GetCourierProgramNumberAndVersion (pclass, &pnumber, &version))
    {
      DEBUG(CALL_DEBUG, (stderr, "_courier_BeginRequest:  Can't determine program-number/version of class \"%s\" on call 0x%x.\n", pclass->cl_name, (unsigned long) call));
      return (FALSE);
    }

  DEBUG(CALL_DEBUG, (stderr, "_courier_BeginRequest:  pnum/vers 0x%x/%d\n", pnumber, version));

  packetSize = argSize
    + 12;	/* for fields of header */

  COURIER(call_connection(call))->cr_out = _courier_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _courier_OutputShortCardinal (call, 0);	/* CallMessageBody */
  _courier_OutputShortCardinal (call, call->ca_SN);	/* transaction ID */
  _courier_OutputCardinal (call, pnumber);
  _courier_OutputShortCardinal (call, version);
  _courier_OutputShortCardinal (call, method->me_id);
  
  DEBUG(CALL_DEBUG, (stderr, "_courier_BeginRequest:  request %d begun (size %d).\n",
		       call->ca_SN, packetSize));
  return (TRUE);
}

/*L1, Main unconstrained*/
/*L2 >= {conn's iomu}*/

static void _courier_DropInput (ilu_Call call)
{
  struct Courier *s = COURIER(call_connection(call));
  if (s->cr_in != NULL)
      packet_destroy(s->cr_in);
  s->cr_in = NULL;
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static void _courier_FlushBuffer(ilu_Call call)
{
  transport_flush_output(call_transport(call));
}

static void _courier_DropRequest (ilu_Call call)
{
  struct Courier *s = COURIER(call_connection(call));
  if (s->cr_out != NULL)
      packet_destroy(s->cr_out);
  s->cr_out = NULL;
}

static ilu_boolean _courier_SendPacket(ilu_Call call, ilu_boolean isReply)
{
  ilu_Transport bs = connection_transport(call_connection(call));
  struct Courier *s = COURIER(call_connection(call));
  _ilu_Assert(packet_buffer(s->cr_out) != NULL, "_courier_SendPacket");
  transport_send_message (bs, packet_buffer(s->cr_out), packet_size(s->cr_out));
  if (!bs->tr_class->tc_timesout)
    packet_buffer(s->cr_out) = NULL;
  if (!method_asynchronous(call_method(call)))
    transport_flush_output(bs);
  if (isReply) {
      if (bs->tr_class->tc_timesout) {
          _ilu_CacheCall(call, packet_buffer(s->cr_out),
                         packet_size(s->cr_out));
          packet_buffer(s->cr_out) = NULL;
        }
      _courier_DropRequest(call);
    }
  return (TRUE);
}

static ilu_boolean _courier_FinishRequest (ilu_Call call)
{
    return (_courier_SendPacket(call, FALSE));
}

static ilu_boolean _courier_FinishReply (ilu_Call call)
{
  return (_courier_SendPacket (call, TRUE));
}

static ilu_boolean _courier_FinishException (ilu_Call call)
{
  return (_courier_SendPacket(call, TRUE));
}

/*L1, Main unconstrained*/
/*L2 >= {call's conn's iomu}*/

static ilu_boolean _courier_BeginReply (ilu_Call call, ilu_boolean exceptions, ilu_cardinal argSize)
{
  ilu_cardinal packetSize;

  DEBUG(PACKET_DEBUG, (stderr, "_courier_BeginReply:  SN %d, argSize %d, exceptions %s, fd %d.\n",
		       call->ca_SN, argSize, exceptions ? "TRUE" : "FALSE",
		       transport_file_descriptor(connection_transport(call_connection(call)))));

  packetSize = argSize
    + 4;			/* for the basic header fields */

  COURIER(call->ca_connection)->cr_out = _courier_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _courier_OutputShortCardinal (call, 2);		/* message type == REPLY */
  _courier_OutputShortCardinal (call, call->ca_SN);

  DEBUG(PACKET_DEBUG, (stderr, "_courier_BeginReply:  started reply %d (size %d).\n", call->ca_SN, packetSize));
  return (TRUE);
}

static ilu_boolean _courier_BeginException (ilu_Call call, ilu_cardinal evalue, ilu_cardinal argSize)
                   
                         	/* the exception value */
                          
{
  ilu_cardinal packetSize;

  if (evalue == 0)		/* send protocol error */
    packetSize = 6;		/* no values to send */
  else
    packetSize = argSize + 6;	/* for the fields of the header */

  COURIER(call->ca_connection)->cr_out = _courier_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  _courier_OutputShortCardinal (call, (evalue == 0) ? 1 : 3);		/* message type == REJECT or ABORT */
  _courier_OutputShortCardinal (call, call->ca_SN);
  _courier_OutputShortCardinal (call, (evalue == 0) ? argSize : evalue);	/* exception value */

  DEBUG(PACKET_DEBUG, (stderr, "_courier_BeginException:  exception started to peer %d:  SN %d, size %d, evalue %d.\n",
		       transport_file_descriptor(connection_transport(call_connection(call))),
		       call_serial_number(call), packetSize, (evalue == 0) ? argSize : evalue));
  return (TRUE);
}

/*L2, Main unconstrained*/
/*L1 >= {prmu}*/

static ilu_Protocol _courier_NewCourier (void)
{
  ilu_Protocol new = (ilu_Protocol) malloc(sizeof(struct _ilu_Protocol_s));

  new->pr_type = ilu_ProtocolType_Courier;
  new->pr_concurrent_requests = ilu_FALSE;

  new->pr_begin_request = _courier_BeginRequest;
  new->pr_finish_request = _courier_FinishRequest;
  new->pr_resend_request = _courier_FinishRequest;
  new->pr_drop_request = _courier_DropRequest;
  new->pr_begin_reply = _courier_BeginReply;
  new->pr_finish_reply = _courier_FinishReply;
  new->pr_begin_exception = _courier_BeginException;
  new->pr_finish_exception = _courier_FinishException;

  new->pr_output_integer = _courier_OutputInteger;
  new->pr_input_integer = _courier_InputInteger;
  new->pr_size_of_integer = _courier_SizeOfInteger;

  new->pr_output_shortinteger = _courier_OutputShortInteger;
  new->pr_input_shortinteger = _courier_InputShortInteger;
  new->pr_size_of_shortinteger = _courier_SizeOfShortInteger;

  new->pr_output_cardinal = _courier_OutputCardinal;
  new->pr_input_cardinal = _courier_InputCardinal;
  new->pr_size_of_cardinal = _courier_SizeOfCardinal;

  new->pr_output_shortcardinal = _courier_OutputShortCardinal;
  new->pr_input_shortcardinal = _courier_InputShortCardinal;
  new->pr_size_of_shortcardinal = _courier_SizeOfShortCardinal;

  new->pr_output_real = _courier_OutputReal;
  new->pr_input_real = _courier_InputReal;
  new->pr_size_of_real = _courier_SizeOfReal;

  new->pr_output_shortreal = _courier_OutputShortReal;
  new->pr_input_shortreal = _courier_InputShortReal;
  new->pr_size_of_shortreal = _courier_SizeOfShortReal;

  new->pr_output_optional = _courier_OutputOptional;
  new->pr_input_optional = _courier_InputOptional;
  new->pr_size_of_optional = _courier_SizeOfOptional;

  new->pr_output_enum_code = _courier_OutputEnumeration;
  new->pr_input_enum_code = _courier_InputEnumeration;
  new->pr_size_of_enum_code = _courier_SizeOfEnumeration;

  new->pr_output_byte = _courier_OutputByte;
  new->pr_input_byte = _courier_InputByte;
  new->pr_size_of_byte = _courier_SizeOfByte;

  new->pr_output_boolean = (ilu_boolean (*)(ilu_Call,ilu_boolean)) _courier_OutputShortCardinal;
  new->pr_input_boolean = (ilu_boolean (*)(ilu_Call,ilu_boolean *)) _courier_InputShortCardinal;
  new->pr_size_of_boolean = (ilu_cardinal (*)(ilu_Call,ilu_boolean)) _courier_SizeOfShortCardinal;

  new->pr_output_character = _courier_OutputShortCardinal;
  new->pr_input_character = _courier_InputShortCardinal;
  new->pr_size_of_character = _courier_SizeOfShortCardinal;

  new->pr_output_shortchar = _courier_OutputByte;
  new->pr_input_shortchar = _courier_InputByte;
  new->pr_size_of_shortchar = _courier_SizeOfByte;

  new->pr_output_string = _courier_OutputString;
  new->pr_input_string = _courier_InputString;
  new->pr_size_of_string = _courier_SizeOfString;

  new->pr_output_bytes = _courier_OutputBytes;
  new->pr_input_bytes = _courier_InputBytes;
  new->pr_size_of_bytes = _courier_SizeOfBytes;

  new->pr_output_opaque = _courier_OutputOpaque;
  new->pr_input_opaque = _courier_InputOpaque;
  new->pr_size_of_opaque = _courier_SizeOfOpaque;

  new->pr_output_stringvec = (ilu_boolean (*)(ilu_Call,ilu_string,ilu_cardinal)) _courier_OutputOpaque;
  new->pr_input_stringvec = (ilu_boolean (*)(ilu_Call,ilu_string *,ilu_cardinal)) _courier_InputOpaque;
  new->pr_size_of_stringvec = (ilu_cardinal (*)(ilu_Call,ilu_string,ilu_cardinal)) _courier_SizeOfOpaque;

  new->pr_output_pipe = NULL;
  new->pr_input_pipe = NULL;
  new->pr_size_of_pipe = NULL;

  new->pr_output_sequence = _courier_OutputSequence;
  new->pr_output_sequence_mark = _courier_OutputSequenceMark;
  new->pr_input_sequence = _courier_InputSequence;
  new->pr_input_sequence_mark = _courier_InputSequenceMark;
  new->pr_end_sequence = _courier_EndSequence;
  new->pr_size_of_sequence = _courier_SizeOfSequence;

  new->pr_output_record = _courier_OutputRecord;
  new->pr_input_record = _courier_InputRecord;
  new->pr_end_record = _courier_EndRecord;
  new->pr_size_of_record = _courier_SizeOfRecord;

  new->pr_output_array = _courier_OutputArray;
  new->pr_input_array = _courier_InputArray;
  new->pr_end_array = _courier_EndArray;
  new->pr_size_of_array = _courier_SizeOfArray;

  new->pr_output_union = _courier_OutputUnion;
  new->pr_input_union = _courier_InputUnion;
  new->pr_end_union = _courier_EndUnion;
  new->pr_size_of_union = _courier_SizeOfUnion;

  new->pr_form_handle = _courier_FormProtocolHandle;

  new->pr_read_packet = _courier_ReadPacket;
  new->pr_interpret_request = _courier_InterpretRequest;
  new->pr_request_read = _courier_DropInput;
  new->pr_interpret_reply = _courier_InterpretReply;
  new->pr_finish_call = _courier_DropInput;

  new->pr_create_data_block = _courier_CreateDataBlock;
  new->pr_free_data_block = (void (*)(void *)) _courier_FreeDataBlock;
  new->pr_flush = _courier_FlushBuffer;

  return (new);
}

/*L1_sup < prmu*/

ilu_Protocol _ilu_courier_Protocol(void)
{
  /*L1 >= {prmu}*/
  static ilu_Protocol StandardCourier = NULL;
  _ilu_AcquireMutex(ilu_prmu);
  if (StandardCourier == NULL)
    StandardCourier = _courier_NewCourier();
  _ilu_ReleaseMutex(ilu_prmu);
  return (StandardCourier);
}
