;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: User; Base: 10 -*-

(in-package :User)

;;;===========================================================================
;;; (C) 1992 Marty Hall. Permission is granted for any non-commercial use of
;;;     this code. DRAFT version 0.9, 11/92.
;;;===========================================================================

;;;===========================================================================
;;; This file is one of four files that define the Memoization facility:
;;;    - Load-Memoization [THIS FILE]: Defines Memoization package and loads 
;;;                                    other 3 files
;;;    - Memoization: Defines core memoization routines.
;;;    - Save-Memo-Table: Defines routines to save and load hash tables
;;;                       associated with a memoized function.
;;;    - Memoization-Examples: Simplistic version of memoization to illustrate
;;;                            the principle, and 2 example functions to which
;;;                            memoization can be applied.
;;;
;;; This file [Load-Memoization] defines the Memoization package, then loads
;;; the three other necessary files. The very first time memoization is added
;;; to your system, you should examine and possibly alter the five variables
;;; listed here and defined in the following code:
;;;   o *Memoization-Base-Directory* -- Directory this file is in
;;;   o *Source-File-Extension* -- ending for lisp source files
;;;   o *Compiled-File-Extension* -- ending for binary LISP files
;;;   o *Memo-Table-Base-Pathname* -- pathname to saved hash tables
;;;   o *Memoization-File-Names* -- Names of files that define this system
;;;
;;; Once they are defined, call (Memo:Compile-Memoization-System), which will
;;; create binary versions of the files in *Memoization-File-Names*. From 
;;; that time on, you can simply load this file to get the binary version
;;; of the automatic memoization facility. 
;;;
;;; Marty Hall
;;; Artificial Intelligence Lab
;;; AAI Corporation
;;; PO Box 126
;;; Hunt Valley, MD  21030
;;; hall@aplcen.apl.jhu.edu
;;; (410) 683-6455
;;;===========================================================================

;;;===========================================================================
;;; Defines the package Memoization is done in. Only the things that are 
;;; exported are needed by users of memoization. CLtL/2 users will want
;;; to use the more convenient "defpackage".

(unless (find-package :Memoization)
  (make-package :Memoization :nicknames '(:Memo)))

(in-package :Memoization)

;;;===========================================================================
;;; FIVE VARIABLE DEFINITIONS THAT MAY NEED TO BE ALTERED.

;;;--------------------------------------------------------------------------
;;; Home directory of the 4 memoization source files. THIS WILL DEFINITELY
;;; NEED TO BE CHANGED.

(defvar *Memoization-Base-Directory*
	#+:symbolics              "bugs:/devel/sms/Memoization/"
	#+:lucid                  "/devel/sms/Memoization/"
	#+:harlequin-common-lisp  "/u2/instructors/hall/lisp/Memoization/"
	#+dec                     "/usr/users/instruct/hall/lisp/Memoization/"
	#+:akcl                   "/home/grad/hall/lisp/Memoization/"
)

;;;--------------------------------------------------------------------------
;;; The source file extension, which is usually just .lisp. IF THIS IS NOT
;;; THE DEFAULT NAME FOR LISP SOURCE FILES, THEN CHANGE THIS TO .LSP (OR
;;; THE APPROPRIATE ENDING) **AND** RENAME THE FOUR MEMOIZATION SOURCE FILES
;;; FROM XXX.LISP TO XXX.LSP (OR WHATEVER ENDING).

(defvar *Source-File-Extension* ".lisp" )

;;;--------------------------------------------------------------------------
;;; Extension for binary files. ADD AN ENTRY IF THE APPROPRIATE ONE DOES NOT
;;; ALREADY APPEAR.

(defvar *Compiled-File-Extension*
	#+(and :lucid :sun)             ".sbin"
	#+:imach                        ".ibin"
	#+(and :symbolics (not :imach)) ".bin"
	#+:harlequin-common-lisp        ".wfasl"
	#+dec                           ".fas"
	#+:akcl                         ".o"
)

;;;--------------------------------------------------------------------------
;;; Names of other three files besides Load-Memoization. This doesn't 
;;; normally require changing.

(defvar *Memoization-File-Names* '("Save-Memo-Table"
				   "Memoization"
				   "Memoization-Examples"))

;;;--------------------------------------------------------------------------
;;; The directory that the saved memo-table files will sit in. This variable
;;; is only used by the routines in Save-Memo-Table.lisp that save/reload
;;; memo tables from disk. Thus, this only needs to be changed if those 
;;; capabilities are being used.
;;;
;;; The actual file name is assumed to be <Function Name>.<type> 
;;; where <type> is defined by *Source-File-Extension*, and <Function Name> 
;;; is the lowercase equivalent of the name of the LISP function. This may 
;;; need to be changed if you are using an OS that disallows filenames that 
;;; are legal LISP function names.
;;;   See the functions Memo-Table-Source-File and Memo-Table-Object-File.

(defvar *Memo-Table-Base-Pathname*
  #+:symbolics             "bugs:/devel/sms/Memoization/Memo-Tables/"
  #+:lucid                 "/devel/sms/Memoization/Memo-Tables/"
  #+:harlequin-common-lisp "/u2/instructors/hall/lisp/Memoization/Memo-Tables/"
  #+dec                "/usr/users/instruct/hall/lisp/Memoization/Memo-Tables/"
  #+:akcl                  "/home/grad/hall/lisp/Memoization/Memo-Tables/"
)

;;;===========================================================================
;;; Loads all of the memoization system. Binary if it exists, otherwise source.
;;; If you are using a package that does (use-package :Memoization), then this
;;; printout can be removed.

(defun Load-Memoization-System ()
  (let (Filename Binary Source)
    (dolist (File *Memoization-File-Names*)
      (setq Filename  (concatenate 'string *Memoization-Base-Directory* File))
      (setq Binary (concatenate 'string Filename *Compiled-File-Extension*)
	    Source (concatenate 'string Filename *Source-File-Extension*))
      (if 
        (probe-file Binary)
	(load Binary)
	(load Source)) )
  (format t "~2%Memoization routines are in the MEMOIZATION package.~%~
              Use (in-package :Memoization) or ~%~
              (use-package :Memoization <Package>) to use it.~2%")
))

;;;===========================================================================

(defun Compile-Memoization-System ()
  (dolist (File *Memoization-File-Names*)
    (locally
      (proclaim '(optimize (speed 3) (safety 2) (compilation-speed 0)))
      (compile-file (concatenate 'string
				 *Memoization-Base-Directory*
				 File
				 *Source-File-Extension*)) )))

;;;===========================================================================
;;; Once your own package is established, you may want to have an entry here 
;;; that has that package use-package :Memoization, or uncomment the line below
;;; that makes Memoization available in the :User package.

(let ((Symbols
	'(Define-Memo-Function
	  Define-Precalculated-Memo-Function
	  Memoize
	  Memoize-Functions
	  Unmemoize
	  Unmemoize-Functions
	  Unmemoize-All-Functions
	  Rememoize
	  Rememoize-Functions
	  Clear-Memo-Table
	  Clear-Memo-Tables
	  Save-Memo-Table
	  Memoized-Function-Call-Count
	  Memoized-Time
	  With-Memoization
	  Without-Memoization
	  *Memoized-Function-Names*
	  Compile-Memoization-System
	  Load-Memoization-System)))
  (export Symbols :Memoization)  )

;; possibly (use-package :Memoization :user)
(Load-Memoization-System)

;;;===========================================================================