/*
 * Functions for interpreting surface segments as walls
 *
 */

#include <stdio.h>
#include <math.h>

#include "constants.h"
#include "structures.h"
#include "parameters.h"
#include "constructs.h"
#include "display.h"


float wall_segment_min_length = 800.0; /* in mm */
rect wall_window = {-1800, 200, 500, 2500 }; /* wall-finding region, ll to rr */

wall left_wall;			/* current right and left walls */
wall right_wall;

sobj *
find_wall_segs(sobj_list *segl)	/* segl is a list of segments */
{
  int n = segl->n;
  sobj **s = segl->s;
  sobj *best = NULL;
  float l = wall_segment_min_length;
  float d = 0;
  float ds;
  surf *sf;

  while (n-- > 0) {
    sf = &((*s)->surface);	/* get surface parameters */
    ds = MAX(sf->x1,sf->x2);
    if (ds >= wall_window.x1 && ds <= wall_window.x2 &&
	sf->d > l && ABS(sf->cy) > d)  {
      best = *s;
      l = sf->d;
      d = ABS(sf->cy);
    }
    s++;
  }
  return(best);
}
      
	
draw_wall(wall *w)		/* draw it thickly */
{
  surf s;
  point_to_seg((point *)w, w->length, &s);
  draw_surf(&s,1);
}

void
test_wall_proc(boolean reset)	/* just take current best hypotheses */
{
  sobj *s;

  if (reset) {
    left_wall.type = PT_WALL;
    right_wall.type = PT_WALL;
    add_point((point *)&left_wall);
    add_point((point *)&right_wall);
    left_wall.viewable = false;
    right_wall.viewable = false;
  }
   
  else {
    s = find_wall_segs(&sobj_r);	/* right side segments */
    if (s != NULL) {
      seg_to_point(&s->surface,(point *)&right_wall);
      right_wall.length = s->surface.d;
      right_wall.viewable = true;
      right_wall.new = true;
    }

    s = find_wall_segs(&sobj_l);	/* left side segments */
    if (s != NULL) {
      seg_to_point(&s->surface,(point *)&left_wall);
      left_wall.length = s->surface.d;
      left_wall.viewable = true;
      left_wall.new = true;
    }
    if (left_wall.viewable) draw_wall(&left_wall);
    if (right_wall.viewable) draw_wall(&right_wall);
  }
}


