/* 
 * eval-rules.c -- evaluate fuzzy control rules
 * 
 */

#include <stdio.h>

#include "constants.h"
#include "structures.h"
#include "constructs.h"
#include "parameters.h"


float c_goal, c_progress, c_act_t, c_act_v;

c_ret *
eval_rules(beh_rule *rules, int n, beh_params params)
				/* assumes that the parameters have been updated */
{
  float accv = 0.0, turnv = 0.0;
  int i;
  float ant, cv;
  int cp;
  float totacc = 0.0, maxacc = 0.0, totturn = 0.0, maxturn = 0.0;
  c_ret ret = {0.0, 0.0, 0.0, 0.0};
  for (i=0; i<n; i++)
    {
      ant = params[rules->antecedent].f; /* get antecedent value */
      cv  = 0.0;
      if (ant >= 0.0)
	{
	  cv = params[rules->parameter].f;
	  cp = rules->consequent;
	  if (cv < 0.0) cv = 0.0;
	  else cv = cv * ant;
	}
      switch(cp)
	{
	case ACCEL:
	  totacc += ant;
	  if (ant > maxacc) maxacc = ant;
	  accv += cv;
	  break;
	case DECEL:
	  totacc += ant;
	  if (ant > maxacc) maxacc = ant;
	  accv -= cv;
	  break;
	case TURN_RIGHT:
	  totturn += ant;
	  if (ant > maxturn) maxturn = ant;
	  turnv -= cv;
	  break;
	case TURN_LEFT:
	  totturn += ant;
	  if (ant > maxturn) maxturn = ant;
	  turnv += cv;
	  break;
	}
      rules++;
    }
  
  if (maxacc > 0.0) 
    {
      ret.accv = accv / totacc;
      ret.acc = maxacc;
    }

  if (maxturn > 0.0) 
    {
      ret.turnv = turnv / totturn;
      ret.turn = maxturn;
    }

  return(&ret);
}



void
eval_bcl(beh_closure *b)	/* evaluates one behavior closure */
{
  (b->beh->update_fn)(b->params); /* evaluate update function */
  b->vals = *eval_rules(b->beh->rules,b->beh->n,b->params);
  b->act_t = c_act_t;
  b->act_v = c_act_v;
  b->goal = c_goal;
}


beh_closure *			/* sets up a behavior closure */
				/* usage: init_bcl(&bhvr,&beh_params,pri) */
init_bcl(behavior *b, beh_params p, int priority) 
{
  beh_closure *bcl = (beh_closure *)malloc(sizeof(beh_closure));
  bcl->priority = priority;
  bcl->beh = b;
  bcl->params = (b->setup_fn)(p); /* evaluate setup function */
  bcl->running = 1;
  return(bcl);
}


float exec_acc, exec_turn;	/* for reference */

execute(float acc, float turn)	/* execute these commands */
{
  exec_acc = acc; exec_turn = turn;
  if (ABS(acc) > 10.0) set_control(VEL,acc);
  if (turn > 0.0)
    { if (target_head() - turn > 6.0) turn = 0.0; }
  else 
    { if (target_head() - turn < -6.0) turn = 0.0; }
  if (ABS(turn) > 1.0) set_control(HEAD,turn*DEG_TO_RAD);
}


/*
 * Behavior list
 *
 * List of behavior closures, sorted by priority
 *
 */


#define MAX_BCS 20
beh_closure *behaviors[MAX_BCS]; 
int num_bcs = 0;		/* current number of behaviors */

void
add_behavior(beh_closure *b)	/* add this to the behavior list */
{
  int p = b->priority;
  int i, j;
  for (i=0; i<num_bcs; i++)	/* sort by priority */
    {
      if (behaviors[i]->priority > p) break;
    }
  for (j = num_bcs; j > i; j--)
    {
      behaviors[j] = behaviors[j-1];
    }
  behaviors[j] = b;
  num_bcs++;
}

void 
remove_behavior(beh_closure *b)	/* remove it from the list */
{
  int i,j;
  for (i=0; i<num_bcs; i++)
    { if (behaviors[i] == b) break; }
  for (; i<num_bcs-1; i++)
    { behaviors[i] = behaviors[i+1]; }
  num_bcs--;
}


void
execute_current_behaviors()	/* execute those on the list */
{
  int p = 0, pri = 0, i;
  float maxactt = 0.0, maxactv = 0.0, actt = 0.0, actv = 0.0, cxt = 0.0;
  float acc = 0.0, turn = 0.0;
  beh_closure *b;

  for (i=0; i < num_bcs; i++)
    {
      b = behaviors[i];
      if (b->running)
	{
	  eval_bcl(b);
	  if (p < b->priority)	/* change of priority */
	    {
	      maxactv = actv;
	      maxactt = actt;
	      p = b->priority;
	    }
	  cxt = 1.0;		/* context, to be used later */
	  b->act_v = MIN(cxt*b->act_v, 1.0-maxactv);
	  b->act_t = MIN(cxt*b->act_t, 1.0-maxactt);
	  b->vals.turn  = MIN(b->act_t,b->vals.turn);
	  b->vals.acc   = MIN(b->act_v,b->vals.acc);
	  actv = MAX(actv,b->act_v);
	  actt = MAX(actt,b->act_t);
	}
    }

  for (i=0; i < num_bcs; i++)	/* defuzzification */
    {
      b = behaviors[i];
      if (b->running)
	{
	  acc += b->vals.acc * b->vals.accv;
	  turn += b->vals.turn * b->vals.turnv;
	}
    }
  
  execute(acc,turn);
}


/*
 * test function
 */


extern behavior constant_velocity;
extern behavior avoid_collision;
extern behavior go_to_pos;

void
test_control_proc(boolean reset)
{
  static point * cp = NULL;
  static beh_closure *gp;
  beh_closure *p;
  beh_params par = beh_alloc(3);
  c_ret *ret;

  if (reset)
    {
      num_bcs = 0;
      par[0].f = 300.0;
      p = init_bcl(&constant_velocity, par, 2);
      add_behavior(p);
      p->running = 0;
      par[0].f = 1.5; par[1].f = 0.8; par[2].f = 4.0;
      p = init_bcl(&avoid_collision, par, 0);
      p->running = 1;
      add_behavior(p);
      p->running = 0;
      behavior_buttons();
      while ((cp = find_cp()) != NULL) rem_point(cp);
    }
  else
    {
      if (cp == NULL)
	{
	  if ((cp = find_cp()) != NULL) /* have a control point */
	    {
	      par[0].f = 300.0; /* speed */
	      par[1].p = cp; /* control point */
	      par[2].f = 250.0;	/* success radius */
	      gp = init_bcl(&go_to_pos, par, 1);
	      add_behavior(gp);
	      behavior_buttons();
	    }
	}
      else if (cp != NULL)
	{
	  if (gp->goal > 0.9)	/* achieved it */
	    {
	      rem_point(cp); /* pop previous point */
	      if ((cp = find_cp()) == NULL)
		{
		  remove_behavior(gp);
		  behavior_buttons();
		}
	      else
		{
		  (gp->params)[6].p = cp;
		}
	    }
	}
      execute_current_behaviors();
    }
}

