/* #########################################
 * world.c
 *
 * world segment definitions for
 *  the erratic simulator
 *#########################################
 */

#include <malloc.h>
#include <stdio.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

/* Segment definitions for the world */
/* world is a structure defining the world */

simworld world;

void reset_world()		/* reset the world, given width and height set */
{
  int i; buclist *p;
  world.local = 2500.0;		/* 2.5 meters is the local area */
  world.x = CEILING(world.w / world.local);
  world.y = CEILING(world.h / world.local);
  world.b = (buclist *)calloc(world.x*world.y,sizeof(buclist));
  p = world.b;
  for (i=0; i<world.x*world.y; i++)
     (p++)->max = 100;
  (world.ss).n = 0;
  (world.ss).max = 1000;	/* 1000 segments at most */
  (world.ss).arr = (char **)calloc(1000,sizeof(char*));
}


buclist *
bucket(i,j)			/* returns the bucket i,j */
{
  return((world.b) + i + j*world.x);
}

void add_to_buffer(s)		/* adds it to the buckets */
     seg *s;
{
  float xmax, ymax, xmin, ymin;
  int i,j;
  float l,l2;

  xmax = MAX(s->x1,s->x2);
  ymax = MAX(s->y1,s->y2);
  xmin = MIN(s->x1,s->x2);
  ymin = MIN(s->y1,s->y2);

  xmax = MIN(world.x,2+CEILING(xmax/world.local));
  ymax = MIN(world.y,2+CEILING(ymax/world.local));
  xmin = MAX(0,FLOOR(xmin/world.local)-1);
  ymin = MAX(0,FLOOR(ymin/world.local)-1);

  l = world.local;
  l2 = world.local * .5;

  for (i=xmin; i<xmax; i++)	/* loop over all relevant buckets */
    { 
      for (j=ymin; j<ymax; j++)
	{
	  if (within_dist_to_seg(l2+i*world.local,l2+j*world.local,s,l))
	    push_aptr(s,(buclist *)bucket(i,j));
	}
    }
}
         

seg *
add_to_world(x1,y1,x2,y2)	/* adds this segment to the current world */
     float x1,x2,y1,y2;
{
  seg *s;
  s = (seg *)malloc(sizeof(seg)); /* get new seg structure */
  s->x1 = x1; s->x2 = x2; s->y1 = y1; s->y2 = y2;
  set_seg_params(s);
  push_ptr(s,&world.ss);
  add_to_buffer(s);
  return(s);
}


buclist *cur_seg_list;		/* current segment list */

void
set_current_segs()		/* sets the current bucket list, based on robot pos */
{
  int i,j;
  i = FLOOR(erratic.x / world.local);
  j = FLOOR(erratic.y / world.local);
  if (i < 0 || j < 0 || i >= world.x || j >= world.y)
    { 
      cur_seg_list = NULL;	/* just return NULL pointer, no error here */
    }
  else cur_seg_list = bucket(i,j);
}

void
print_current_segs()		/* print them out... */
{
  int i; seg **p;
  if (cur_seg_list == NULL)
    printf("\nNo current segment list.");
  else
    {
      p = (seg **)&cur_seg_list->segs;
      for (i=0; i<cur_seg_list->n; i++)
	{
	  printf("\n%.0f,%.0f %.0f,%.0f",(*p)->x1,(*p)->y1,(*p)->x2,(*p)->y2);
	  p++;
	}
    }
}
  
void print_all_buckets()
{
  int i,j;
  for (i=0; i<world.x; i++)
    for (j=0; j<world.y; j++)
      { printf("\nBUCKET: %d,%d",i,j);
	cur_seg_list = bucket(i,j);
	print_current_segs();
      }
}

void
print_world_segs()		/* print them out... */
{
  int i; seg **p;
  p = (seg **)world.ss.arr;
  for (i=0; i<world.ss.n; i++)
    {
      printf("\n%.0f,%.0f %.0f,%.0f",(*p)->x1,(*p)->y1,(*p)->x2,(*p)->y2);
      p++;
    }
}



/*
 * Opening and reading a world file.
 *
 * Format:
 *   Comment lines begin with semicolon
 *   Segment lines begin with a number, are: x1 y1 x2 y2
 *   World size lines:  width nnnn  or height nnnn
 *   Initial placement: position xxxx yyyy aaaa
 *
 * All numbers are integers
 * Distances in mm, angles in degrees
 *
 */


char outbuf[100];
char namebuf[100];

shorten(char *name)		/* shorten the file name into outbuf */
{
  int n,i;
  char *p;
  p = namebuf;
  for (n = strlen(name); n > 0; n--)
    {
      if ( name[n-1] == '/' )
	{
	  for (i = n; i < strlen(name); i++)
	    {
	      *p++ = name[i];
	    }
	  n = 0;
	}
    }
  *p = 0;
}

static int count;		/* segment count */

load_file(name)			/* load the file named "name" */
     char *name;
{
  FILE *fp;
  char buf[100];
  int reset = 1;
  shorten(name);
  if ((fp = fopen(name, "r")) == NULL)
    { 
      sprintf(outbuf,"Can't open %s",namebuf);
      world_message(outbuf);
      return(-1);
    }
  else
    {
      count = 0;
      world.h = 0.0;  world.w = 0.0;
      while(fgets(buf,100,fp)!=NULL) /* read a line, process it */
	{ 
	  if (reset && world.h > 0.0 && world.w > 0.0) 
	    { reset = 0; reset_world(); }
	  proc_line(buf);
	}
      sprintf(outbuf,"World: %s  %d segments",namebuf,count);
      world_message(outbuf);
    }
  fclose(fp);
  scroll_to_erratic_position();
  window_refresh();
  return(0);
}



proc_line(l)			/* process one line */
     char *l;
{
  int d1, d2, d3, d4;
  int n;
  float w, h;
  n = 0;
  sscanf(l,";%n",&n);
  if (n>0)			/* comment line */
    return(0);
  n = 0;
  sscanf(l,"width %d%n",&d1,&n);
  if (n>0)
    { world.w = (float)d1;
/*      printf("\nWorld width: %d",d1); */
      return(0);
    }
  n = 0;
  sscanf(l,"height %d%n",&d1,&n);
  if (n>0)
    { world.h = (float)d1;
/*      printf("\nWorld height: %d",d1); */
      return(0);
    }
  n = 0;
  sscanf(l,"position %d %d %d%n",&d1,&d2,&d3,&n);
  if (n>0)
    { 
/*      printf("\nRobot position: %d %d %d\n",d1,d2,d3); */
      jump_robot((float)d1,(float)d2,deg_to_ang(d3));
      set_current_segs();
      return(0);
    }
  n = 0;
  sscanf(l,"%d %d %d %d%n",&d1,&d2,&d3,&d4,&n);
  if (n>0)			/* have a segment line here */
    {
/*      printf("\nSegment: %d,%d %d,%d",d1,d2,d3,d4); */
      add_to_world((float)d1,(float)d2,(float)d3,(float)d4);
      count++;
    }
}







