;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)

;; define confidence-set and confidence-set-generic-instance
;; as a bootstrap

(DEFCLASS CONFIDENCE-SET (TOOLBED::DATA-STRUCTURES) 
       
       ;; a value of NIL in the NEUTRAL-EQUIVALENT slot means
       ;; that there is no equivalent in the confidence set. 
       ;; The GT etc. comparative functions will still need to
       ;; handle NEUTRAL values. 
 ((type :type symbol :initarg type)
  (values-list :type list :initarg values-list)
  (max :initarg max)
  (min :initarg nil)
  (NEUTRAL-EQUIVALENT :INITFORM 'NEUTRAL :initarg neutral-equivalent)
  (GT :TYPE function :initarg gt)
  (LT :TYPE function :initarg lt)
  (GE :TYPE function :initarg ge)
  (LE :TYPE function :initarg le)
  (EQ :type function :initarg eq)
  (member-confidence-set :type function :initarg member-confidence)
  (aliases :type list :initform nil :initarg aliases)))

(defparameter confidence-set-generic-instance 
  (pcl::*make-instance 'confidence-set))

(defmacro define-confidence-set (set-name &body gt-code)
  (if (null gt-code)
	    (error "Error in the confidence set definition of ~S. ~
			Nothing was specified." set-name))
  (format *trace-output* "~%Defining confidence set ~S...~%" set-name)
  `(compile-confidence-set ',set-name ',gt-code))

(defun compile-confidence-set (set-name code-body)
  "Compiles a confidence set specification."
  (let ((missing
	 '(superclass= type= min= max= values-list= gt= lt= eq= 
		       member-confidence-set= ge= le= 
		       aliases= neutral-equivalent=))
	(slot-list nil)
	superclass
	(generic-instance-name (read-from-string
				(concatenate 'string
					     (string set-name)
					     "-GENERIC-INSTANCE"))))
    (format *trace-output* "Parsing the specifications. ~%")
    (multiple-value-setq (slot-list superclass)
			 (parse-confidence-spec
			  set-name code-body missing))
    
    (eval `(defclass ,set-name (,superclass) ,slot-list))
    
    (eval 
     `(defparameter ,generic-instance-name (pcl::*make-instance ',set-name)))

    (find-class set-name)))


(defun parse-confidence-spec (set-name body missing)
  (let ((slot-list nil) superclass type (min nil)
	(max nil) (values-list nil) ge le gt lt eq 
	member-confidence-set 
	(aliases nil) (neutral-equivalent nil) super-generic-inst)
    (dolist 
     (specifier body)
     (format *trace-output* "    ~S~%" (car specifier))
     (case (car specifier)
	   (superclass=
	    (if (not (find-class (cadr specifier) nil))
		(error "~S is not a defined CLOS class, so cannot be a superclass= specifier." (cadr specifier))
	        (if (not (subclassp (find-class (cadr specifier))
				    (find-class 'confidence-set)))
		    (error "~S is not a subclass of CONFIDENCE-SET, so is not a valid superclass= specifier." (cadr specifier))
		    (setf superclass (cadr specifier)))))
	   (type= (setf type (cadr specifier)))
	   (min= (setf min (cadr specifier)))
	   (max= (setf max (cadr specifier)))
	   (values-list= (setf values-list (cdr specifier)))
	   (ge= (setf ge (cadr specifier)))
	   (gt= (setf gt (cadr specifier)))
	   (le= (setf le (cadr specifier)))
	   (lt= (setf lt (cadr specifier)))
	   (eq= (setf eq (cadr specifier)))
	   (member-confidence-set= (setf member-confidence-set
					 (cadr specifier)))
	   (aliases= (setf aliases (cdr specifier)))
	   (neutral-equivalent= (setf neutral-equivalent
				      (cadr specifier))))
     (setf missing (remove (car specifier) missing)))
    ;; set up values or min/max
    (format *trace-output* "Noting superclass...~%")
    (if (member 'superclass= missing)
	(setf superclass 'confidence-set))
    (setf super-generic-inst (return-generic-instance superclass))
    ;; make sure it has values-list or min and max, or that we can get
    ;; them from the superclass

    (format *trace-output* "Noting type...~%")
    (if (member 'type= missing)
	(if (not (slot-empty-p super-generic-inst 'type))
	    (setf type (slot-value super-generic-inst 'type))
	    (setf type t)))

    (format *trace-output* "Checking values-list or max/min specification.~%")
    (if (and (member 'values-list= missing)
	     (or (member 'min= missing)
		 (member 'max= missing)))
	;; don't have either values-list= or max & min pair -- try to
	;; get them from superclass's generic-instance

	  (if (not (slot-empty-p super-generic-inst
				 'values-list))

	     ;; definitely have values-list
		(setf values-list
		      (slot-value super-generic-inst
				  'values-list)
		      missing
		      (remove 'values-list= missing))

	     ;; else try to get min and max
	      (if
		  (and
		   (if (and (member 'min= missing)
			    (not
			     (slot-empty-p super-generic-inst 'min)))
		       (setf min 
			     (slot-value super-generic-inst 'min))
		       min)
		   (if (and (member 'max= missing)
			    (not
			     (slot-empty-p super-generic-inst 'max)))
		       (setf max
			     (slot-value super-generic-inst 'max))
		       max))
		  ;; have min and max
		  (progn
		    (setf missing (remove 'min= missing))
		    (setf missing (remove 'max= missing)))
		
		  ;; couldn't get min and max or values-list

		  (error
		   "The specification of CONFIDENCE-SET ~S did not ~
 		include either a values-list= specifier, or min= and 
		max specifiers, and these could not be inherited from the
		superclass ~S." set-name superclass))))

    ;; Check type of values-list or min/max
    (format *trace-output* 
	    "Checking type of values-list or min/max specification.~%")
    (if values-list
	(dolist (val values-list)
		(if (not (typep val type))
		    (error "Value ~S from values-list is not of type ~S." 
			   val type)))
        (if (not (typep min type))
	    (error "MIN value ~S is not of type ~S." min type)
	    (if (not (typep max type))
		(error "MAX value ~S is not of type ~S." max type))))

    ;; now we have the values-list or the missing stuff, let's set up
    ;; the comparison and member fns

    (let ((present-fns 
	   (remove-if #'(lambda (x) (member x missing))
		      '(gt= lt= eq= member-confidence-set= ge= le=)))
	  (absent-fns
	   (remove-if-not #'(lambda (x) (member x missing))
			  '(gt= lt= eq= member-confidence-set= ge= le=)))
	  this-fn)


    ;; first, do the fns provided in the spec by the user:

      (format *trace-output* "Parsing function specifications.~%")
      (dolist (fn present-fns)
	      (case fn
		    (gt= (setf gt (transform-user-fn gt max min
						     values-list)))
		    (lt= (setf lt (transform-user-fn lt max min
						     values-list)))
		    (ge= (setf ge (transform-user-fn ge max min
						     values-list)))
		    (le= (setf le (transform-user-fn le max min
						     values-list)))
		    (eq= (setf eq (transform-user-fn eq max min
						     values-list)))
		    (member-confidence-set=
		     (setf member-confidence-set
			   (transform-user-fn member-confidence-set
					      max min values-list)))))
		    
      ;; now fill in the rest of the functions:
      ;;  if the function is present in the superclass, use that,
      ;; else create it from the defaults (member if it has a values-list,)
      ;; #'>, etc. if it has min/max

      (format *trace-output* "Obtaining or generating defnintions for unspecified functions.~%")
      (dolist (fn absent-fns)
	      ;; if the super has a binding for this-fn, set the value of
	      ;; this-fn to be the result of parsing that, else set
	      ;; the value of this-fn to be the system-generated fn
	      
	      (setf this-fn (read-from-string
			     (string-trim '(#\=) (string fn))))

	      (if (not (slot-empty-p super-generic-inst this-fn))
		  (case fn
			(gt= (setf gt (transform-noop-references
				       (slot-value super-generic-inst
						   'gt)
				       max min values-list)))
			(lt= (setf lt (transform-noop-references
				       (slot-value super-generic-inst
						   'lt)
				       max min values-list)))
			(ge= (setf ge (transform-noop-references
				       (slot-value super-generic-inst
						   'ge)
				       max min values-list)))
			(le= (setf le (transform-noop-references
				       (slot-value super-generic-inst
						   'le)
				       max min values-list)))
			(eq= (setf eq (transform-noop-references
				       (slot-value super-generic-inst
						   'eq)
				       max min values-list)))
			(member-confidence-set= 
			 (setf member-confidence-set
			       (transform-noop-references
				(slot-value super-generic-inst
					    'member-confidence-set)
				max min values-list))))
		  (case fn
			(gt= (setf gt (generate-standard-fn
				       'gt type max min values-list
				       gt lt eq ge le)))
			(lt= (setf lt (generate-standard-fn
				       'lt type max min values-list
				       gt lt eq ge le)))
			(ge= (setf ge (generate-standard-fn
				       'ge type max min values-list
				       gt lt eq ge le)))
			(le= (setf le (generate-standard-fn
				       'le type max min values-list
				       gt lt eq ge le)))
			(eq= (setf eq (generate-standard-fn
				       'eq type max min values-list
				       gt lt eq ge le)))
			(member-confidence-set= 
			 (setf member-confidence-set
			       (generate-standard-fn
				'member-confidence-set type
				max min values-list gt lt eq ge le)))))
	      (setf missing (remove fn missing))))

    ;; set up aliases
    (format *trace-output* "Setting up aliases.~%")
    (if (not (member 'aliases= missing))
	(setf aliases (parse-alias-list aliases member-confidence-set))
        ;; this isn't required, don't scream if it isn't there
        (setf missing (remove 'aliases= missing)))
    
    (format *trace-output* "Setting up NEUTRAL equivalent.~%")
    (if (not (member 'neutral-equivalent= missing))
	(if (not (and (typep 'neutral type)
		      (eval `(apply ,(if (and
					  (listp member-confidence-set)
					  (eq (car member-confidence-set)
					      'lambda))
					 `',member-confidence-set
				       member-confidence-set)
				    (list 'neutral)))))

	    ;; NEUTRAL is not a member of the confidence set.
	    ;; Try to set it up as an alias for neutral-equivalent.

	    (if 
		(not 
		 (eval `(apply ,(if (and
				     (listp member-confidence-set)
				     (eq (car member-confidence-set)
					 'lambda))
				    `',member-confidence-set
				  member-confidence-set)
			       (list ',neutral-equivalent))))


		;; The value specified as NEUTRAL-EQUIVALENT is not in
		;; the set. 

		(error "~S is not a member of the confidence set ~S, so cannot be its NEUTRAL-EQUIVALENT.~%" neutral-equivalent set-name)


	        ;; It is in the set. Set up the alias.

	        (progn
		  (setf missing (remove 'neutral-equivalent= missing))
		  (if (assoc neutral-equivalent aliases)
		      (rplacd (assoc 'neutral aliases)
			      (cons 'neutral
				    (cdr 
				     (assoc 'neutral-equivalent aliases))))
		      (setf aliases
			    (cons 
			     (list neutral-equivalent 'neutral) aliases)))))

	    ;; NEUTRAL is a member of the confidence set.
	    ;; It can be its own NEUTRAL-EQUIVALENT.

	    (format *trace-output* 
		    "NEUTRAL is a member of the confidence set ~S, so NEUTRAL-EQUVALENT ~S is being ignored.~%" 
		    set-name neutral-equivalent)))



    ;; build the giant list of slot descriptors to send to DEFCLASS

    ;;  (if missing
    ;;	(error "The specifications for ~S have not been dealt with.~%" 
    ;;	       missing))

    (setf slot-list
	  (list 
	   `(type :initarg type :initform ',type :type symbol)
	   `(min :initarg min :initform ,min
		 ,@(if (numberp min) (list :type 'number)))
	    `(max :initarg max :initform ,max
		  ,@(if (numberp max) (list :type 'number)))
	    `(values-list :initarg values-list
			  :initform ',values-list
			  :type list)
	    `(ge :initarg ge :type function
		 :initform ',(if (and (listp ge)
				      (eq (car ge) 'lambda))
				 `',ge
			         ge))
	    `(gt :initarg gt :type function
		 :initform ',(if (and (listp gt)
				      (eq (car gt) 'lambda))
				 `',gt gt))
	    `(le :initarg le :type function 
		 :initform ',(if (and (listp le)
				      (eq (car le) 'lambda))
				 `',le le))
	    `(lt :initarg lt :type function 
		 :initform ',(if (and (listp lt)
				      (eq (car lt) 'lambda))
				 `',lt lt))
	    `(eq :initarg eq :type function 
		 :initform ',(if (and (listp eq)
				      (eq (car eq) 'lambda))
				 `',eq eq))
	    `(member-confidence-set :initarg member-confidence-set
				    :type function
				    :initform
				    ',(if (and (listp member-confidence-set)
					       (eq
						(car member-confidence-set)
						'lambda))
					  `',member-confidence-set
					  member-confidence-set))
	    `(aliases :initarg aliases :type list
		      :initform ',aliases)
	    `(neutral-equivalent :initarg neutral-equivalent
				 :initform
				 ',neutral-equivalent)))
    (values slot-list superclass)))


(defun transform-noop-references (tree max min values-list)

  ;; insert current max, min, and values-list values into the
  ;; (v-list x), (max-val x), and (min-val x) forms
  (cond
   ((symbolp tree) tree)
   ((listp tree)
    (case (car tree)
	  (max-val
	   (if max
	       `(max-val ,max)
	       tree))
	  (min-val
	   (if min
	       `(min-val ,min)
	       tree))
	  (v-list
	   (if values-list
	       `(v-list ',values-list)
	       tree))
	  (otherwise
	   (cons
	    (transform-noop-references
	     (car tree) max min values-list)
	    (transform-noop-references
	     (cdr tree) max min values-list)))))
   (t tree)))

(defun generate-standard-fn (which-fn type max min values-list gt lt eq ge le)
  ;; first try the non-composite fns
  ;; try the max/min version, else do the values-list version
  (if (member which-fn '(gt lt eq))
	      (if (and max min)
		  (case which-fn
			(gt `#'>)
			(lt `#'<)
			(eq `#'=))
		  (if values-list
		      (case which-fn
			    (gt
			     `(lambda (x y)
				(member y
					(cdr
					 (member x 
					      (v-list ',values-list))))))
			    (lt
			     `(lambda (x y)
				(member x
					(cdr
					 (member y 
					      (v-list ',values-list))))))
			    (eq `#'eq))
		      (error
	   "Neither max and min nor values-list was specified for function ~S."
	   			which-fn)))

	        (case which-fn
		      (ge
		       (if (and gt eq)
			   `(lambda (x y) 
			      (or (apply ,(if
					      (and (listp gt)
						   (eq (car gt) 'lambda))
					      `',gt gt)
					 (list x y))
				  (apply ,(if
					      (and (listp eq)
						   (eq (car eq) 'lambda))
					      `',eq eq)
					      (list x y))))
			   (if values-list
			       `(lambda (x y)
				  (member y
					  (member x
						  (v-list 
						   ',values-list))))
			       `#'>=)))
		      (le 
		       (if (and lt eq)
			   `(lambda (x y) 
			      (or (apply ,(if (and (listp lt)
						   (eq (car lt) 'lambda))
					      `',lt lt)
					 (list x y))
				  (apply ,(if (and (listp eq)
						   (eq (car eq) 'lambda))
					      `',eq eq)
					 (list x y))))

			   (if values-list
			       `(lambda (x y)
				  (member x 
					  (member y
						  (v-list 
						   ',values-list))))
			       `#'<=)))
		      (member-confidence-set
		       (if (and ge le max min)
			   `(lambda (x)
			      (if (not (typep x ',type))
				  (return nil))
			      (and
			       (apply ,(if (and (listp ge)
						(eq (car ge) 'lambda))
					   `',ge ge)
				      (list x (min-val ,min)))

			       (apply ,(if (and (listp le)
						(eq (car le) 'lambda))
					   `',le le)
				(list x (max-val ,max)))))
			   `(lambda (x) (member x 
						(v-list ',values-list)))))
		      (otherwise
		       (error 
			"This should not happen: unknown fn ~S." which-fn)))))
		 
		


(defun transform-user-fn (fn-spec max min values-list)
  ;;change values-list, max, and min specs into (v-list ,values-list)
  ;; etc.
  (cond
   ((symbolp fn-spec)
    (case fn-spec
	  (max `(max-val ,max))
	  (min `(min-val ,min))
	  (values-list `(v-list ,values-list))
	  (otherwise fn-spec)))
   ((listp fn-spec)
    (cons (transform-user-fn (car fn-spec) max min values-list)
	  (transform-user-fn (cdr fn-spec) max min values-list)))
   (t fn-spec)))

			   
(defun parse-alias-list (alias-list member-confidence-fn)
  ;; alias-list should be of the form:
  ;; ((conf-set-item alias1 alias2 alias3)
  ;;  (conf-set-itema alias1a alias2a alias3a)
  ;;  (conf-set-itemb alias1b))

  (dolist (alias-spec alias-list)
	  (if (not (listp alias-spec))
	      (error "Alias= specification is of the wrong form. ~S should be a list." alias-spec)
	      (if 
		  (not (> (length alias-spec) 1))
		  (error
		"In aliases= specification, no aliases given in specifier ~S."
		   alias-spec)
		  (if (not
		       (eval `(apply ,(if
				    (and 
				     (listp member-confidence-fn)
				     (eq (car member-confidence-fn)
					 'lambda))
				    `',member-confidence-fn
				    member-confidence-fn)
			      (list ',(car alias-spec)))))
		      (error
		       "In aliases= specification, ~S is not a member of the confidence set, so cannot be aliased." (car alias-spec))))))
  
  alias-list)




(defun return-generic-instance (conf-set &optional (no-scream nil))
  (if (not (symbolp conf-set))
      (error "~S is not a symbol." conf-set)
    (if (not (find-class conf-set nil))
	(if no-scream
	    (return-from return-generic-instance nil)
	    (error "~S does not name a CLOS class." conf-set))
        (if (not (subclassp (find-class conf-set)
			    (find-class 'confidence-set)))
	    (if no-scream
		(return-from return-generic-instance nil)
	        (error "~S does not name a subclass of confidence-set."
		       conf-set))
	    ;; it is a class and a subclass of confidence-set, so construct
	    ;; its generic instance's name, and return the instance
	    (return-from return-generic-instance
			 (eval
			  (read-from-string
			   (concatenate 'string
			    (string conf-set) "-GENERIC-INSTANCE"))))))))


;; same function, different name

(defmacro confidence-set-p (set)
  `(if (symbolp ,set)
       (return-generic-instance ,set t)
       (if (typep ,set 'confidence-set)
	   ,set
	   nil)))
	   


(defun conf-set-name (vocab)
  (if (not (typep vocab 'confidence-set))
      (error "~S is not an object of type CONFIDENCE-SET.~%" vocab)
      (class-name (class-of vocab))))


(defun check-alias-compare-form (fn arg1 arg2 vocabulary)
  (let (ok1 ok2)
    (if (member-confidence arg1 vocabulary)
	(setf arg1 (unalias arg1 vocabulary) ok1 t))
    (if (member-confidence arg2 vocabulary)
	(setf arg2 (unalias arg2 vocabulary) ok2 t))
    
    `(apply ,(slot-value vocabulary fn)
	    (list
	     ,(if ok1 `,(if (numberp arg1)
			    arg1 `',arg1)
		`(unalias ,arg1 ',(conf-set-name vocabulary)))
	     ,(if ok2 `,(if (numberp arg2)
			    arg2 `',arg2)
		`(unalias ,arg2 ',(conf-set-name vocabulary)))))))


;; would these be faster as functions?

(defmethod confidence-values ((conf-set symbol))
  (let ((confidence-set (return-generic-instance conf-set)))
    (if (slot-value confidence-set 'values-list)
	(slot-value confidence-set 'values-list)
        (list (slot-value confidence-set 'max) 
	      (slot-value confidence-set 'min)))))

(defmethod confidence-values ((conf-set confidence-set))
  (if (slot-value conf-set 'values-list)
      (slot-value conf-set 'values-list)
      (list (slot-value conf-set 'max) (slot-value conf-set 'min))))


(defmethod member-confidence (conf-value (conf-set symbol))
  (let ((confidence-set (return-generic-instance conf-set)))
    (if (not (slot-empty-p confidence-set 'type))
	(if (not (typep conf-value (slot-value confidence-set 'type)))
	    (return-from member-confidence nil)))
    (if (rassoc conf-value (slot-value confidence-set 'aliases)
		:test #'member)
	t
	 (apply (eval (slot-value confidence-set
			      'member-confidence-set))
		 (list conf-value)))))

(defmethod member-confidence (conf-value (conf-set confidence-set))
    (if (not (slot-empty-p conf-set 'type))
	(if (not (typep conf-value (slot-value conf-set 'type)))
	    (return-from member-confidence nil)))
    (if (rassoc conf-value (slot-value conf-set 'aliases)
		:test #'member)
	t
      (eval `(apply ,(slot-value conf-set 'member-confidence-set)
		    ',(list conf-value)))))


(defmethod confidence-compare (fn arg1 arg2 (vocabulary confidence-set))
  (let ((real-arg1 (unalias arg1 vocabulary))
	(real-arg2 (unalias arg2 vocabulary)))
    (if (not (typep real-arg1 (slot-value vocabulary 'type)))
	(error "~S is not of the correct type for the confidence set." 
	       real-arg1)
        (if (not (typep real-arg2 (slot-value vocabulary 'type)))
	    (error "~S is not of the correct type for the confidence set." 
		   real-arg2)))
    (eval `(apply ,(slot-value vocabulary fn) ',(list real-arg1 real-arg2)))))

(defmethod confidence-compare (fn arg1 arg2 (vocab symbol))
  (let* ((vocabulary (return-generic-instance vocab))
	 (real-arg1 (unalias arg1 vocabulary))
	 (real-arg2 (unalias arg2 vocabulary)))
    (if (not (typep real-arg1 (slot-value vocabulary 'type)))
	(error "~S is not of the correct type for the confidence set." 
	       real-arg1)
      (if (not (typep real-arg2 (slot-value vocabulary 'type)))
	  (error "~S is not of the correct type for the confidence set." 
		 real-arg2)))
    (eval `(apply ,(slot-value vocabulary fn) ',(list real-arg1 real-arg2)))))

(defmethod unalias (value (vocabulary confidence-set))
  (if (rassoc value (slot-value vocabulary 'aliases)
	      :test #'member)
      (car (rassoc value (slot-value vocabulary 'aliases)
		   :test #'member))
      value))

(defmethod unalias (value (vocab symbol))
  (let ((vocabulary (return-generic-instance vocab)))
    (if (rassoc value (slot-value vocabulary 'aliases)
		:test #'member)
	(car (rassoc value (slot-value vocabulary 'aliases)
		     :test #'member))
        value)))

(defmethod maximum-value (value (vocab symbol))
  (let ((vocabulary (return-generic-instance vocab)))
    (eq (unalias value vocabulary)
	(car (confidence-values vocabulary)))))

(defmethod maximum-value (value (vocabulary confidence-set))
  (return-from maximum-value 
	       (eq (unalias value vocabulary)
		   (car (confidence-values vocabulary)))))

(defmethod minimum-value (value (vocab symbol))
  (let ((vocabulary (return-generic-instance vocab)))
    (eq (unalias value vocabulary)
	(car (last (confidence-values vocabulary))))))

(defmethod minimum-value (value (vocabulary confidence-set))
  (return-from minimum-value 
	       (eq (unalias value vocabulary)
		   (car (last (confidence-values vocabulary))))))

;;functions to return values-list, max, min, n-eqiv, etc from either
;; a conf-set name or a conf-set

;; should it be ,x or ', x?

(defmacro min-val (x)
  `,x)

(defmacro max-val (x) `,x)

(defmacro v-list (x) `,x)


