;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)


(defmethod controllers ((idb-instance stub-idb))
  (return-from 
   controllers
    (acons
     'ask
     `(lambda 
       (question &optional (answer-type 'usual-3-val))
       (declare (special *current-dragon* 
			 toolbed::*idb-flushed-list*))

       (if (not (member 
		 (slot-value *current-dragon* 'unique-name) 
		 toolbed::*idb-flushed-list*))

	   ;; if it's not on the flushed list, this is a new case
	   ;; so reset answer-cache using saved cases

	   (progn
	     (setf 
	      (slot-value *current-dragon* 'answer-cache) nil
	      toolbed::*idb-flushed-list* 
	      (cons (slot-value *current-dragon* 'unique-name) 
		    toolbed::*idb-flushed-list*))

	     (when *idb-use-saved-cases*
		   (setf (slot-value *current-dragon* 'answer-cache)
			 (cached-saved-cases *current-dragon*
					     *current-case*)))))

       ;; if idb is in the idb-flushed-list,
       ;; case is same as previous, so leave cache
       ;; alone

       ;; now that cache is set up, get on with things:
       ;; first, error-checking

       (if (not (stringp question))
	   (error "Question ~S to IDB ~S not a string."
		  question ',(slot-value idb-instance 'unique-name)))
       (if (not (or (confidence-set-p answer-type)
		    (member answer-type '(symbol number))))
	   (error "Answer type ~S is not SYMBOL NUMBER or a confidence set."
		  answer-type))

       (let ((answer nil))

	 ;; see if it's in the cache
	 (setf answer (cdr (assoc question (slot-value *current-dragon* 
						  'answer-cache)
			     :test #'string-equal)))

	 ;; it's not in the cache, so go to the user

	 (if (null answer)

	     (progn
	       (when (>= (length (slot-value *current-dragon* 'answer-cache))
			 (slot-value *current-dragon* 'answer-cache-limit))

		     ;; no room to add new question/answer to cache --
		     ;; so fix it by making space, then go on -- only 
		     ;; exit here on user request

		     (increase-answer-cache-space *current-dragon*))

	       ;; we've got space in the cache, let's go on
	       ;; get answer from user, add it to cache

	       (loop
		(format *query-io* "~% ~A " question)
		(setf answer (read *query-io*))

		;;type checking of answers here
		(if (confidence-set-p answer-type)
		    (if (member-confidence answer answer-type)
			(return answer)))
		(if (typep answer answer-type)
		    (return answer))
		(format *query-io*
			"~% Please give an answer of type: ~A ~%" 
			(if (confidence-set-p answer-type) 
			    (confidence-values answer-type) 
			  answer-type)))
	       
	       (remember question answer)
	       (setf (slot-value *current-dragon* 'answer-cache)
		     (acons question answer
			    (slot-value *current-dragon* 'answer-cache)))

	       ;; check for answer-cache-limit warning

	       (if (<= (- (slot-value *current-dragon* 'answer-cache-limit)
			  (length (slot-value *current-dragon* 'answer-cache)))
		       5)
		   (warn
		    "The IDB ~S has ~S entries in its cache, approaching its limit of ~S."
		    (slot-value *current-dragon* 'unique-name)
		    (length (slot-value *current-dragon* 'answer-cache))
		    (slot-value *current-dragon* 'answer-cache-limit)))))

	 answer)) nil)))



(defun cached-saved-cases (idb case)
  ;; idb is *current-dragon*, case is *current-case*

  (let ((saved-cases (cases-saved *current-dragon*))
	(cache-list nil))

    ;; if *current-case* itself is in the saved-case-cache
    (if (member case saved-cases :test #'equal)
	(setf cache-list
	      (append
	       (slot-value idb 'answer-cache)
	       (cdr (assoc case (slot-value idb 'saved-case-cache)
			   :test #'equal)))))
		   
    ;; if *current-case* is a list, put any saved answers
    ;; for the individual members of the list onto the
    ;; answer cache, in the order they are listed in
    ;; *current-case*, and after any answers for the
    ;; case as a whole (from above)

    (if (listp case)
	(dolist (acase case)
		(if (member acase saved-cases :test #'equal)
		    (setf cache-list
			  (append cache-list
				  (cdr (assoc
					acase
					(slot-value 
					 idb
					 'saved-case-cache)
					:test #'equal)))))))

    cache-list))



(defun increase-answer-cache-space (idb)

  (let ((error-in nil))
    (format *query-io*
	    "Warning: storage limit of ~S has been reached for IDB ~S.~%
			Do you wish to increase the cache size (i), delete
			  	some of the cached data (d), or exit (e)? "
	    (slot-value idb 'answer-cache-limit)
	    (slot-value idb 'answer-cache))
    (loop
     (setf error-in (read *query-io*))
     (if (member error-in '(d e i))
	 (return error-in))
     (format *query-io* "~%Please answer i, d, or e: "))
    (case error-in 
	  (i
	   (format *query-io*
		   "Current cache size is ~S. New size: ")
	   (loop
	    (setf error-in (read *query-io*))
	    (if (numberp error-in) 
		(if (> error-in 
		       (slot-value idb 'answer-cache-limit))
		    (return error-in)))
	    (format *query-io* 
		    "~%Please enter a number greater than ~S: "
		    (slot-value idb 'answer-cache-limit)))

	   (setf (slot-value idb 'answer-cache-limit) error-in))

	  (d
	   (format *query-io*
		   "Delete how many entries from the end of the cache? ")
	   (loop
	    (setf error-in (read *query-io*))
	    (if (numberp error-in) 
		(if (> error-in 0)
		    (return error-in)))
	    (format 
	     *query-io* 
	     "~%Please enter a number (greater than 0): "))
	   
	   (setf (slot-value idb 'answer-cache)
		 (nbutlast (slot-value idb 'answer-cache) error-in)))

	  (e
	   (error "~%IDB ~S has run out of answer-cache space."
		  (slot-value idb 'unique-name))))))

