;;
;;      Title :  MorphConcat
;;
;;      Function :  Takes a list of lexical formas and concatenates them
;;                  using the spelling rules.
;;
;;      Author :  Alan W Black     Oct 1986
;;                Dept of A.I. University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;       2.2  This was introduced as another option for use in 
;;            word generation.  
;;
;;      Description :
;;       These functions implement the reverse form of spelling rule
;;       application, instead of segmenting they implement concatenation.
;;       This is a trivial change to the code from segmentation.  This time
;;       the lexical string is fixed but is matched against all possible
;;       surface strings (well almost all within the constraints of the 
;;       feasible pairs set.
;;
;;       This file deals with the actual concatenation the user-level function
;;       is defined in mafuncs.
;;
;;       The actual formalism and interpretation of rules is dealt with
;;       in the file sprulemoveau (via the function D-CheckPairMatch)
;;       Hence these functions are independant of the actual spelling rule
;;       form used (well within reason)
;;
;;      Parameters :
;;          morphs   a list of atoms each made up from characters in the 
;;                   lexical alphabet.
;;
;;      Returns :
;;          list of surfaceforms
;;                   a list of possible surface forms that can validily 
;;                   correspond to the given lexical string
;;
;;      Side Effect :
;;         none                   
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;


(declare
   (special
      D-LEXICON
      D-ENTRYFILEID
      D-INITCONFIG    ;; initial configuration of the spelling rules
      D-OTHERPAIRS
   )
   (localf
      D-GenerateSurface
      D-CheckAndContinue
   )
)

(include "subrout")
;;(include sprulemove)   ;; new spelling rule form
(include "spmoveau")   ;; The spelling rule formalism functions

(defun D-MorphConcat (morphs)  
;;
;;  Concatenates morphemes with respect to the currently
;;  loaded spelling rules
;;
   (setq D-OTHERPAIRS
      (D-RemoveDuplicates
	 (mapcar
	    #'(lambda (pair)
		  (aexplodec pair)
            )
	    D-UNUSUALFEASIBLES
         )
      ))
   (mapcar
      #'(lambda (surfaceform)
	 (concatl (reverse surfaceform))   ;;  build surface form
      )
      (D-GenerateSurface (aexplodec (concatl morphs)) nil D-INITCONFIG)
   )
)

(defun D-GenerateSurface (lexical surface config)
;;
;;  Finds the next possible character that can correspond to the next
;;  lexical character. 
;;
   (cond
      ((null lexical)    ;; end of lexical string
	 (cond           ;; check if a valid final form
	    ((D-Final config) (ncons surface))
	    (t  nil))
      )
      (t
	 (append
	    (D-CheckAndContinue    ;; simple surface eq lexical case
	       (car lexical) (car lexical)
	       (cdr lexical)
	       surface config)
            (mapcan                ;; unusual feasible case
	       #'(lambda (unusualpair)
		  (cond
		    ((eq (cadr unusualpair) (car lexical))
		        (D-CheckAndContinue
		           (car lexical) (caddr unusualpair)
		           (cdr lexical)
		           surface config))
                    (t nil))
               )
	       D-OTHERPAIRS
            )
         )
      )
   )
)

(defun D-CheckAndContinue (lex surf lexrest recognised config)
;;
;;   Checks the given lexical and surface pair if ok continues
;;   search otherwise stops and returns nil
;;
   (let ((newconfig (D-CheckPairMatch lex surf config)))
      (cond
	 ((eq newconfig 'ERROR) nil)   ;; not a valid surface form
	 (t
	    (D-GenerateSurface
	       lexrest
	       (cons surf recognised)
	       newconfig)
         )
      )
   )
)

