
#|-----------------------------------------------------------------------------
Artificial Intelligence, Second Edition
Elaine Rich and Kevin Knight
McGraw Hill, 1991

This code may be freely copied and used for educational or research purposes.
All software written by Kevin Knight.
Comments, bugs, improvements to knight@cs.cmu.edu
----------------------------------------------------------------------------|#

#|----------------------------------------------------------------------------
		      RAILROAD TRAVEL DOMAIN
			  "travel.lisp"
----------------------------------------------------------------------------|#

#|----------------------------------------------------------------------------
This file contains code for a city-to-city railroad travel problem.  The 
task is to travel from San Francisco to Los Angeles, perhaps usign the 
shortest path.  The important functions are (s, s1 and s2 are states):

 (goal-state? s)		t if s is a goal state
 (eq-states s1 s2)		are s1 and s2 equal?
 (expand s)			successor states of s
 (hash-state s)			some has value for s
 (print-state s)		print function
 (destroy-state s)		dispose of state structure
 (heuristic s)			heuristic distance between s and the goal
 (generate-problem)		randomly generated start-state
 (cost-of-move s1 s2)		cost of moving from s1 to s2

The important variables are:

 *sample-initial-state*
 *goal-state*

These functions and variables can all be called from an outside search program.
In fact, these are the functions called by our implementations of depth-first,
breadth-first, hill-climbing, A*, DFID, IDA*, and RTA* search.

The problems in this domain are not particularly difficult to solve, but 
clearly illustrate different search strategies.  To watch different strategies
in action, load the files "dfs.lisp, "bfs.lisp", "dfid.lisp", "hill.lisp",
"a-star.lisp", and "rta-star.lisp" and execute the following commands:

	(dfs 'san-francisco t)
	(dfs-avoid-loops 'san-francisco t)
	(bfs-graph 'san-francisco t)
	(dfid 'san-francisco t)
	(ida-star 'san-francisco t)
	(a-star 'san-francisco t)
	(hill-climbing 'san-francisco t)
	(rta-star 'san-francisco 1 t)

----------------------------------------------------------------------------|#

#|----------------------------------------------------------------------------

			MAP OF CITIES AND RAIL DISTANCES



									  *BOS

					     272   *DET			   229
					*CHI
									 *NY
*SEA					  184	  303
									 91

900					      *IND   371   *PIT  348    *PHI
									 133

									*WAS
*SF    821	*SLC			 518

			    *TPK
470            700		60
				  *KC				761

*LA  450
	 *PHX			      517  		*BIR
	       440
		    *EP	   646     *DAL	             355	
					        *NO		804
			827	  264       363
				      *HOU				

									*MIA
----------------------------------------------------------------------------|#

;; Variable *INFINITY* will be used as the largest possible number.
;; MOST-POSITIVE-FIXNUM is a Lisp symbol that provides it.

(defvar *infinity* most-positive-fixnum)

;;--------------------------------------------------------------------------
;; Variable *CITIES* holds the list of cities that will be used as states
;; in the search.
;;
;; Variable *NUMBER-OF-CITIES* is the total number of cities.
;;
;; Variable *RAIL-DISTANCES* is a list of adjacent cities and their distances
;; by rail.
;;
;; Variable *AIRLINE-DISTANCES* stores, for each pair of cities, the Euclidean
;; distance between them.

(defvar *cities*)
(setq *cities*
  '(san-francisco  los-angeles  salt-lake-city  phoenix  el-paso  houston  
    dallas  kansas-city  topeka  new-orleans  birmingham  miami  washington  
    philadelphia  pittsburgh  indianapolis  chicago  detroit  new-york  
    boston  seattle))

(defvar *number-of-cities*)
(setq *number-of-cities* (length *cities*))

(defvar *rail-distances*)
(setq *rail-distances* 
  '(
    (san-francisco (seattle 900) (salt-lake-city 821) (los-angeles 470))
    (los-angeles (san-francisco 470) (phoenix 450))
    (salt-lake-city (san-francisco 821) (phoenix 450))
    (phoenix (el-paso 440) (salt-lake-city 700) (los-angeles 450))
    (el-paso (phoenix 440) (houston 827) (dallas 646))
    (houston (el-paso 827) (new-orleans 363) (dallas 264))
    (dallas (el-paso 646) (kansas-city 517) (houston 264))
    (kansas-city (topeka 60) (indianapolis 518) (dallas 517))
    (topeka (kansas-city 60))
    (new-orleans (houston 363) (birmingham 355))
    (birmingham (washington 761) (new-orleans 355) (miami 804))
    (miami (birmingham 804))
    (washington (philadelphia 133) (birmingham 761))
    (philadelphia (washington 133) (pittsburgh 348) (new-york 91))
    (pittsburgh (philadelphia 348) (indianapolis 371))
    (indianapolis (pittsburgh 371) (kansas-city 518) (detroit 303) 
		  (chicago 184))
    (chicago (indianapolis 184) (detroit 272))
    (detroit (indianapolis 303) (chicago 272))
    (new-york (philadelphia 91) (boston 229))
    (boston (new-york 229))
    (seattle (san-francisco 900))
   ))

(defvar *airline-distances*)
(setq *airline-distances*
   (make-array (list *number-of-cities* *number-of-cities*)
      :initial-contents
        '(
	  (   0  347  600  653 1202 1645 1483 1506 1456 1926 2385 
	   2594 2442 2523 2264 1949 1858 2091 2571 2699  678)

	  ( 347    0  579  357  800 1374 1240 1356 1306 1673 2078
	   2339 2300 2394 2136 1809 1745 1983 2451 2596  959)

	  ( 600  579    0  504  877 1200  999  925  885 1434 1805 
	   2089 1848 1925 1668 1356 1260 1492 1972 2099  701)

	  ( 653  357  504    0  402 1017  887 1049 1009 1316 1680
	   1982 1983 2083 1828 1499 1453 1690 2145 2300 1114)
	  
	  (1202  800  877  402    0  756  625  936  896 1121 1278 
	   1957 1997 2065 1778 1418 1439 1696 2147 2358 1760)
	  
	  (1645 1374 1200 1017  756    0  225  644  664  318  692 
	    968 1220 1341 1137  865  940 1105 1420 1605 1891)
	  
	  (1483 1240  999  887  625  225    0  451  486  443  653
	   1111 1185 1299 1070  763  803  999 1374 1551 1681)
	  
	  (1506 1356  925 1049  936  644  451    0   50  680  703 
	   1241  945 1038  781  453  414  645 1097 1251 1506)
	  
	  (1456 1306  885 1009  896  664  486   50    0  720  753
	   1291  995 1088  821  473  424  665 1137 1291 1466)
	  
	  (1926 1673 1434 1316 1121  318  443  680  720    0  347 
	    892 1098 1241 1118  839  947 1101 1330 1541 2606)
	  
	  (2385 2078 1805 1680 1278  692  653  703  753  347    0
	    777  751  894  792  492  657  754  983 1194 2612)
	  
	  (2594 2339 2089 1982 1957  968 1111 1241 1291  892  777
	      0 1101 1239 1250 1225 1390 1409 1328 1539 3389)
	  
	  (2442 2300 1848 1983 1997 1220 1185  945  995 1098  751 
	   1101    0  123  192  494  597  396  205  393 2329)
	  
	  (2523 2394 1925 2083 2065 1341 1299 1038 1088 1241  894
	   1239  123    0  259  585  666  443   83  271 2380)
	  
	  (2264 2136 1668 1828 1778 1137 1070  781  821 1118  792
	   1250  192  259    0  330  410  205  317  483 2183)
	  
	  (1949 1809 1356 1499 1418  865  763  453  473  839  492
	   1225  494  585  330    0  165  240  646  807 1872)

	  (1858 1745 1260 1453 1439  940  803  414  424  947  657 
	   1390  597  666  410  165    0  279  840  983 2052)
	  
	  (2091 1983 1492 1690 1696 1105  999  645  665 1101  754
	   1409  396  443  205  240  279    0  671  702 2339)
	  
	  (2571 2451 1972 2145 2147 1420 1374 1097 1137 1330  983
	   1328  205   83  317  646  840  671    0  213 2900)
	  
	  (2699 2596 2099 2300 2358 1605 1551 1251 1291 1541 1194
	   1539  393  271  483  807  983  702  213    0 3043)
	  
	  ( 678  959  701 1114 1760 1891 1681 1506 1466 2606 2612
	   3389 2329 2380 2183 1872 2052 2339 2900 3043    0)
	 )))


;; -------------------------------------------------------------------------


(defvar *sample-initial-state* 'san-francisco)

(defvar *goal-state* 'new-york)

;; -------------------------------------------------------------------------
;; Functions required by search programs in the railroad travel domain.

(defun goal-state? (s)
  (eq-states s *goal-state*))

(defun eq-states (s1 s2)
  (eq s1 s2))

(defun expand (s &optional randomize)
  (let ((succs (mapcar #'car (cdr (assoc s *rail-distances*)))))
     (if randomize
	 (random-permute succs)
	 succs)))


;; Function RANDOM-PERMUTE mixes up the elements of a list.

(defun random-permute (x)
  (do ((y x)
       (res nil))
      ((null y) res)
    (let ((next (nth (random (length y)) y)))
       (setq res (cons next res))
       (setq y (delete next y)))))


(defun hash-state (s)
   (sxhash s))		; built-in Common Lisp function

(defun print-state (s &rest ignore)
   (declare (ignore ignore))
   (format *standard-output* "~d" s))

(defun heuristic (s)
   (aref *airline-distances*
	 (position s *cities*)
	 (position *goal-state* *cities*)))

(defun destroy-state (s)
  (declare (ignore s))
  nil)

(defun generate-problem ()
  (nth (random *number-of-cities*) *cities*))

(defun cost-of-move (s1 s2)
  (cadr (assoc s2
	       (cdr (assoc s1 *rail-distances*)))))

