structure SemiSpaceGC : GC =
struct

  (* Raised to indicate that an allocation or dereference has failed (due to
  lack of resources (or breaking of another invariant) and a collection should
  be invoked. *) 
  exception OutOfMemory

  exception Error of string

  type location = int ref

  datatype sVal =
      Int of int
    | True 
    | False 
    | Nil 
    | Loc of location
    | Unit
  and lVal = 
      Closure of Env * DBMinML.exp
    | Cons of sVal * sVal
    | Pair of sVal * sVal
  and Bind =
      ValBind of sVal 
    | ExpBind of sVal
  
  withtype Env = Bind list
  
  datatype Frame =
      FPrimopN of P.primop * sVal list * DBMinML.exp list
    | If1 of DBMinML.exp * DBMinML.exp
    | App1 of DBMinML.exp 
    | App2 of sVal
    | Cons1 of DBMinML.exp 
    | Cons2 of sVal 
    | FPair1 of DBMinML.exp
    | FPair2 of sVal
    | FFst
    | FSnd
    | Case1 of DBMinML.exp * (DBMinML.bind * DBMinML.bind * DBMinML.exp)
    | Let1 of DBMinML.bind * DBMinML.exp
  
  datatype Cont =
      Frame of Frame
    | Env of Env
  
  type stack = Cont list
  
  (* Types shared between the GC and the abstract machine. *)
  datatype HeapElement = Unused | Fwd of int | Cell of lVal;

  val (heap, upper, lower, next) = (ref (Array.array (0,Unused)), ref 0, ref 0, ref 0)

  val allocs = ref 0
  val reads = ref 0
  val collects = ref 0
  val accesses = ref 0


  (* Initialize the GC heap. *)
  fun init i = 
    let in
      heap := Array.array (i,Unused);
      upper := i div 2;
      lower := 0;
      next := 0
    end

  fun copyStack nil 					= ()
    | copyStack (Env(eta)::k) 				= (copyEnv eta; copyStack k)
    | copyStack (Frame(FPrimopN(po,svl,el))::k) 	= copyStack k
    | copyStack (Frame(If1(e2,e3))::k) 			= copyStack k
    | copyStack (Frame(App1(e2))::k) 			= copyStack k
    | copyStack (Frame(App2(Loc(r)))::k) 		= (copyPtr r; copyStack k)
    | copyStack (Frame(Cons1(e2))::k) 			= copyStack k
    | copyStack (Frame(Cons2(v1))::k) 			= (copysVal v1; copyStack k)
    | copyStack (Frame(Case1(e2,(x,y,e3)))::k) 		= copyStack k
    | copyStack (Frame(Let1(x,e2))::k) 			= copyStack k
    | copyStack (Frame(FPair1 e) :: k) 			= copyStack k
    | copyStack (Frame(FPair2 v) :: k) 			= (copysVal v; copyStack k)
    | copyStack (Frame(FFst) :: k) 			= copyStack k
    | copyStack (Frame(FSnd) :: k) 			= copyStack k
    | copyStack _ 					= raise Error "Copying malformed stack"

  and copyEnv nil 		  = ()
    | copyEnv (ExpBind(v)::eta) = (copysVal v; copyEnv eta)
    | copyEnv (ValBind(v)::eta) = (copysVal v; copyEnv eta)

  and copyPtr r =
      if !r < !lower orelse !r >= !upper
        then (* r points to from-space: not yet copied *)
      (case Array.sub(!heap,!r)
         of Cell(w) => (Array.update(!heap,!r,Fwd(!next));
  		      Array.update(!heap,!next,Cell(w));
  		      r := !next; next := !next+1;  (* effect here *)
  		      copylVal w)
          | Unused => raise Error "Dereferencing Unused cell"
          | Fwd _ => raise Error "Dereferencing Forward")
         else (* r points to to-space: already copied *)
  	 ()
  
  and copysVal (Loc r) = copyPtr r 
    | copysVal _ = ()

  and copylVal (Cons(v1,v2)) = (copysVal v1; copysVal v2)
    | copylVal (Closure(eta,e)) = copyEnv eta
    | copylVal (Pair(v1,v2)) = (copysVal v1; copysVal v2)

  fun flip () =
      if !lower = 0
        then (lower := (Array.length (!heap)) div 2; upper := (Array.length (!heap)); next := !lower)
      else (lower := 0; upper := (Array.length (!heap)) div 2; next := !lower)
  
  fun gc (k,w) = 
      let
         val _ = print "Calling GC\n";
         val _ = flip ()
         val _ = copylVal w
         val _ = copyStack k
         val _ = if !next >= !upper then raise OutOfMemory
  	       else ()
         val _ = print ("Returning from GC (collected: "^Int.toString (!upper - !next)^")\n")
       in
         ()
       end
  
  fun alloc k w =
      let
        val _ = if !next >= !upper then gc (k,w) else () (* could raise *)
        val l = !next
        val _ = next := l+1
        val _ = Array.update(!heap, l, Cell(w))
        val r = ref(l)
      in 
        r 
      end
  
  fun extract(Cell(w)) = w
    | extract _ = raise Error "Extracting contents of non-Cell"
  
  fun read r = 
    let in
      extract (Array.sub(!heap, !r))
    end

  fun numAllocs () = !allocs

  fun numReads () = !reads

  fun numCollects () = !collects

  fun numAccesses () = !accesses


end;
