;  Tile World Experiment  --  Martha Pollack

(defstruct agent
  ;parameters that are invariant through an experiment
  think-strategy            ;top-level strategy for deliberating or planning
  act-time                  ;length of time for an atomic action (in ms)
  ;"mind" (vary during each experiment)
  options                   ;list of option-sets
  intentions                ;list of acts
)

(defstruct option-set
  purpose                   ;intention id or 'SELF'
  alt-acts                  ;list of acts
)

(defstruct act
  id
  type
  parms                     ;list of parm name and binding pairs
  subacts                   ;list of acts
  enabling-conds
  postconds
  time
  atomic                    ;flag to indicate whether a base-level act
)

(defstruct parm
  name
  binding
)

;;;;;;;;;;

;Initialize a world and an agent, and then play a tileworld game.
; *Future:  standardize initialization of agent and world; create interface
;     for initializing both

(defun tw-experiment ()
    (tw-init w0)
    (tw-score (play-tw-game w0
		       (specify-agent-parms)
		       (specify-game-length))))

;Play a tileworld game:  Simulate concurrent acting and thinking by
;first performing one think step, keeping track of how long you take,
;and then acting for the same period of time, allowing the world to
;change then as well.  Because of the concurrency, the act cycle makes 
;use of the initial state of mind:  at the end of both acting and thinking
;the resulting changes to the agent's mental state are combined.  Note
;that the world only changes during the act cycle, so it doesn't need
;to be copied and then updated to simulate concurrency.
;Returns:  --world that results from playing the game

(defun play-tw-game (world mind game-length)
  (let ( think-time
	 (mind1 (tw-copy-agent mind))
	 (mind2 (tw-copy-agent mind)))
    (agent-init mind world)
    (do ( (elapsed 0 (+ elapsed think-time)))
	( (>= elapsed game-length) world)
	(multiple-value-setq (mind1 think-time) (think world mind))
	(setf mind2 (act world mind think-time))
	(merge-minds mind mind1 mind2))
    world))

(defun agent-init (mind world)
  (setf (agent-options mind)
	(create-new-options (tw-hole-list world))))

;Perform one think step.  The strategy for determining what this step
;will be depends upon the agent's "think-strategy", chosen at 
;time.  Keep track of the amount of time (in computer msec's) used.
; Returns:  MULTIPLE VALUES:  --modified "mind" (input arg1 is mutated)
;                             --length of time used

(defun think (world mind)
  (let (start-time stop-time)
    (setf start-time (get-internal-run-time))
    (funcall (agent-think-strategy mind) world mind)
    (setf stop-time (get-internal-run-time))
    (values mind
	    (* 1000 (/ (- stop-time start-time) 
		       internal-time-units-per-second)))))
                                           ;compute time used in msec's

;Act for the same amount of time as you just thought for.  If act
;is successful, remove the satisfied intention from the mental state.
;After attempting an act, note any changes that occurred in the world
;while you were acting.
;Returns:  --modified "mind"  (input arg2 is mutated)
;          (input arg1, representing the world, is also mutated)

(defun act (world mind time)
  (let* ( (cycle-length (agent-act-time mind))
	  (cycles (round time cycle-length))
	  (agent (first (tw-agent-list world)))
	  old-world success next-act)
    (dotimes (i cycles mind)
	(setf old-world (tw-copy-world world))
	(setf next-act (select-next-act mind))
	(when (tw-step world agent (act-type next-act) cycle-length)
	      (setf (agent-intentions mind)
		    (remove-intentions (agent-intentions mind) next-act)))
	(perceive mind old-world world))
    mind))


;Since we're simulating concurrent acting and thinking, after one iteration
;of both processes, we've got to merge the resulting mental states.
;Thinking removes options and adds intentions; acting removes intentions;
;perceiving, which occurs during the act cycle, removes intentions and options
;and also adds options
;Returns:  --"t"(irrelevant):  mutates "mind"  (arg1)

(defun merge-minds (mind think-res act-res)
  (let ( (think-add-ints (remove-intentions (agent-intentions think-res)
					    (agent-intentions mind)))
	 (think-del-opts (remove-options (agent-options mind)
					 (agent-options act-res)))
	 (act-del-ints (remove-intentions (agent-intentions mind)
					  (agent-intentions act-res)))
	 (act-del-opts (remove-options (agent-options mind)
				       (agent-options act-res)))
	 (act-add-ints (remove-intentions (agent-intentions act-res)
					  (agent-intentions mind))))
    (setf (agent-intentions mind)
	  (set-difference
	        (union  (agent-intentions mind) 
		        think-add-ints
			:test #'equalp)
		act-del-ints
		:test #'equalp))
    (setf (agent-options mind)
	  (union
	     (set-difference (agent-options mind)
			     (set-difference think-del-opts
					     act-del-opts
					     :test
					     #'equalp)
			     :test
			     #'equalp)
	     act-add-opts
	     #'equalp)))
  t)
						 

;Routines for removing options and intentions.  Each takes two 
;lists:  a list of options or intentions, and a list of acts, and returns
;a list of options or intentions consisting of those in arg1 with objects
;from arg2 removed.
;Return:  --the resulting list of options (intentions)

(defun remove-options (opts acts)
  (when acts
      (remove-options (remove-act-opt opts (first acts)) (rest acts))))

(defun remove-act-opt (opts act)
  (mapcan #'(lambda (x) (remove-act-opt-top x act)) opts))

(defun remove-act-opt-top (opt act)
  (setf (option-set-alt-acts opt)
	(mapcan #'(lambda (x) (remove-act-opt-tree x act)) 
		(option-set-alt-acts opt)))
  (if (option-set-alt-acts opt) 
      (list opt)
      nil))

(defun remove-act-opt-tree (act1 act2)
  (if (same-act act1 act2)
      nil
      (progn (setf (act-subacts act1)
		   (mapcan #'(lambda (x) (remove-act-opt-tree x act2))
			   (act-subacts act1)))
	     (if (act-subacts act1)
		 (list act1)
	         nil))))

(defun remove-intentions (ints acts)
  (when acts
     (remove-intentions (remove-act-int ints (first acts)) (rest acts))))

(defun remove-act-int (ints act)
  (mapcan #'(lambda (x) (remove-act-int-tree x act)) ints))

(defun remove-act-int-tree (int act)
  (if (same-act int act)
      nil
     (progn (setf (subacts int)
		  (mapcan #'(lambda (x) (remove-act-int-tree x act))
			  (act-subacts int)))
	    (if (act-subacts int)
		(list int)
	        nil))))


(defun same-act (act1 act2)
  (and (eq (act-type act1) (act-type act2))
       (null (set-difference
	           (act-parms act2) 
		   (act-parms act1)
		   :test
		   #'equalp))))
                         ;act1 counts as the same act as act2 if its parms
                         ;subsume those of act2

;Perceive environmental changes--both new options (new holes that have
;appears), and lost options and intentions (holes that have disappeared).
; *Future:  generalize to deal with other kinds of environmental changes,
;  e.g., new tiles
;Returns:  "t" (irrelevant) --mutates arg1


(defun perceive (mind old-world new-world)
  (let ( (new-holes (set-difference (tw-hole-list new-world)
				      (tw-hole-list old-world)
				      :key 'obj-name))
	 (lost-holes (set-difference (tw-hole-list old-world)
				     (tw-hole-list new-world)
				     :key 'obj-name)))
    (when new-holes (setf (agent-options mind)
			  (append (agent-options mind)
				  (create-new-options new-holes))))
    (when lost-holes (let ((lost-opps (create-lost-opps lost-holes)))
		       (setf (agent-options mind)
			     (remove-options (agent-options mind)
					  lost-opps))
		       (setf (agent-intentions mind)
			   (remove-intentions (agent-intentions mind)
					      lost-opps)))))
    t)


;Create a list of top-level (hole-filling) option sets; each set contains 
;two alt-acts:  one to do the fill now, and one to do it later
; * Future:  create other types of new options
;Returns:  such a list

(defun create-new-options (hole-list)
  (mapcar #'create-new-option hole-list))

(defun create-new-option (hole)
  (make-option-set 
           :purpose 'self
	   :alt-acts (list (make-act :id (gensym)
				     :type 'fill
				     :parms (list (make-parm :name 'hole
							     :binding (obj-name hole)))
				     :time 'now)
			   (make-act :id (gensym)
				     :type 'fill
				     :parms (list (make-parm :name 'hole
							      :binding (obj-name hole)))
				     :time 'later))))

;Create a list of actions representing the holes that used to be available
;for filling, but have now disappeared
; *Future:  create other types of lost opportunities

(defun create-lost-opps (hole-list)
  (mapcar #'create-lost-opp hole-list))

(defun create-lost-opp (hole)
  (make-act :type 'fill
	    :parms (list (make-parm :name 'hole
			            :binding hole))))



;Subroutine to determine the binding of arg1 in the parms list that is
;arg2.  Returns the binding.

(defun binding (name parms)
  (cond ( (null parms) nil)
	( (eq (parm-name (first parms)) name) (parm-binding (first parms)))
	( t (binding name (rest parms)))))


;Find the next atomic action to perform.  If there's a currently active
;intention, find the next available atomic subpart.  Otherwise, return
;the null action.
;Returns:  an atomic action to perform

(defun select-next-act (mind)
  (let ( (current (current-int (agent-intentions mind))))
    (cond ( (and current 
		 (find-base-level current)))
	  ( (make-null-act)))))

(defun current-int (intentions)
  (cond ( (null intentions) nil)
	( (eq (act-time (first intentions)) 'now) (first intentions))
	(t (current-int (rest intentions)))))

(defun find-base-level (int)
  (cond ( (act-atomic int) int)
	( (act-subacts int) (find-base-level (first (act-subacts int))))))


;Engage in deliberation if there are any available options.  Otherwise do
;some planning.
;Returns: --  (irrelevant):  mutates mind (arg1)

(defun deliberate-first (world mind)
  (if (agent-options mind)
      (deliberate mind world)
      (expand-plans mind world)))


;Perform a deliberation step:  Choose some option set to consider,
;filter incompatible options excepting those that trigger an override,
;and then evaluate those that remain.
;Returns: -- "t" (irrelevant):  mutates mind (arg2) 

(defun deliberate (mind world)
  (let ( (option (select-option-set (agent-options mind))))
    (setf (agent-options mind) (remove option (agent-options mind)))
                 ;directly remove the options you're deliberating on here
   (evaluate-options (filter (option-set-alt-acts option)
				    mind)
		     (option-set-purpose option)
		     mind
		     world))
  t)

(defun select-option-set (options)
  (first options))               ;be arbitrary for now


(defun filter (options mind)
  (mapcan #'(lambda (x) (or (compatible x mind)
			    (override x mind)))
	  options))

(defun compatible (option mind)
  (case (act-type option) 
	(fill (compat-fill option mind))
	(push (compat-push option mind))
	(otherwise (list option))))     ;anything else is compatible

(defun compat-fill (option mind)
  (let ( (intentions (agent-intentions mind))
	 (time (act-time option))
	 (hole (binding 'hole (act-parms option))))
    (if (eq time 'now)
	(if (conflicts 'hole hole intentions) nil (list option))
        (list option))))

(defun conflicts (parm binding intentions)
  (cond ( (null intentions) nil)
	( (eq (binding parm (act-parms (first intentions))) binding) nil)
	( t (conflicts parm binding (rest intentions)))))

(defun compat-push (option mind)
  (let ( (intentions (agent-intentions mind))
	 (tile (binding 'tile (act-parms option))))
    (if (conflicts 'tile tile intentions) nil (list option))))

(defun override (x y) nil)  ;stub

(defun evaluate-options (options purpose mind world)
  (let ( (new-int (choose-best-option options mind)))
    (if new-int
	(setf (agent-intentions mind)
	      (insert-new-int new-int purpose (agent-intentions mind)))
        (setf (agent-options mind)
	      (append (agent-options mind)
		      (list (make-options :purpose purpose
					  :alt-acts options)))))))


(defun choose-best-option (options mind)
    (when options
	  (cond ( (find-now-option options))
		(t (first options)))))

(defun find-now-option (options)
  (cond ( (null options) nil)
	( (eq (act-time (first options)) 'now) (first options))
	( t (find-now-option (rest options)))))

(defun insert-new-int (int purpose ints)
  (if (eq purpose 'self)
      (cons int ints)
      (attach-new-int int purpose ints)))

(defun attach-new-int (int purpose ints)
  (cond ( (eq (act-id (first ints)) purpose)
	    (setf (act-subacts (first ints)) int))
	( (and (act-subacts (first ints))
	       (attach-new-int int purpose (act-subacts (first ints)))))
	(t (attach-new-int int purpose (rest ints))))
  ints)

(defun expand-plans (mind world)
  (let ( (selected (select-int (agent-intentions mind))))
    (when selected
	(setf (act-subacts selected) (move-sequence)))))
    
(defun select-int (ints)
  (cond ((null ints) nil)
	( (eq (act-type (first ints)) 'fill) (first ints))
	(t (select-int (cdr ints)))))

(defun move-sequence ()
  (list (make-act :id (gensym)
		  :type 'L
		  :atomic 't)
	(make-act :id (gensym)
		  :type 'R
		  :atomic 't)))

(defun make-null-act ()
  (make-act :id (gensym)
	    :type 'N
	    :atomic 't))

(defun tw-copy-agent (agent)
  (copy-agent agent))    ; stub:  rewrite to copy recursively

;(defmacro set-diff-structs (structs1 structs2 key)
;  `(set-difference ,structs1
;		   ,structs2
;		   :test
;		   #'(lambda (x y) (eq (,key x) (,key y)))))

;(defmacro union-structs (structs1 structs2 key)
;  `(union ,structs1
;	  ,structs2
;	  :test
;	  #,(lambda (x y) (eq (,key x) (,key y)))))
    
