;
; path planning functions
;
(defvar c-grid (with-static-area (make-string 4000)))
(defvar occupied (with-static-area (make-string 256 
					:initial-element (int-char 0))))
(setf (aref occupied (char-int #\#)) #\#)
(setf (aref occupied (char-int #\T)) #\#)
(setf (aref occupied (char-int #\t)) #\#)
(setf (aref occupied (char-int #\O)) #\#)
(setf (aref occupied (char-int #\o)) #\#)

; convert plan from string to list
(defun tw-convert-plan (str)
    (let (result)
	(dotimes (i (length str))
	    (setf result 
		  (cons 
		    (case (aref str i)
			(#\u 'u)
			(#\d 'd)
			(#\l 'l)
			(#\r 'r)
			(#\x 'x))
		    result)))
	(reverse result)))

; plan move to a square
(defun tw-plan-move (world agent row col)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (setf (aref c-grid (+ (* row (tw-cols world)) col)) #\D)
    (tw-convert-plan 
	(moveplan (tw-rows world) (tw-cols world) c-grid occupied 
				(obj-row agent) (obj-col agent))))

; plan move to any empty square adjacent to an obj
(defun tw-plan-move-near (world agent obj)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (dolist (cell (obj-cells obj))
	(let ((row (cell-row cell))
	      (col (cell-col cell)))
	    (mapc #'(lambda(i j) 
			(if (eq (aref (tw-grid world) (+ row i) (+ col j)) 
				nothing-char)
			    (setf (aref c-grid (+ (* (+ row i) 
					 (tw-cols world)) col j)) #\D)))
		  '(-1  0  1 -1  1 -1  0  1)
		  '(-1 -1 -1  0  0  1  1  1))))
    (tw-convert-plan (moveplan (tw-rows world) (tw-cols world) c-grid occupied
				(obj-row agent) (obj-col agent))))

(defun move (i j)
    (let ((plan (tw-plan-move w0 (first (tw-agent-list w0)) i j)))
	(if (not (equal plan '(x)))
	    (run-plan plan)
	    'no-path-found)))

(defun moveto (i j)
    (let (plan tile)
	(if (member (aref (tw-grid w0) i j) tile-chars)
	    (setf tile (aref (tw-grid-which w0) i j))
	    (return-from moveto 'not-a-tile))
	(setf plan (tw-plan-move-near w0 (first (tw-agent-list w0)) tile))
	(if (not (equal plan '(x)))
	    (run-plan plan)
	    'no-path-found)))

(defun run-plan (plan)
	(twinitdisp)
	(dolist (move plan)
	    (tw-step w0 nil move 0))
	(twclosedisp))
	
(defun tw-plan-push (world agent tile holecells)
  (let (dir)
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref c-grid (+ (* i (tw-cols world)) j))
		  (aref (tw-grid world) i j))))
    (setf (aref c-grid (+ (* (obj-row agent) (tw-cols world)) 
			  (obj-col agent)))
	  nothing-char)
    (setf (aref c-grid (+ (* (obj-row tile) (tw-cols world)) (obj-col tile))) 
	  nothing-char)
    (dolist (cell holecells)
	(setf (aref c-grid (+ (* (cell-row cell) (tw-cols world))
			      (cell-col cell))) #\D))
    (setf dir (find-agent-dir tile agent))
    (if (>= dir 0)
	(tw-convert-plan
	    (pushplan (tw-rows world) (tw-cols world) c-grid occupied 
		(obj-row tile) (obj-col tile) dir))
	'agent-not-by-the-tile)))

(defun find-agent-dir (tile agent)
    (let* ((drow (- (obj-row agent) (obj-row tile)))
	   (dcol (- (obj-col agent) (obj-col tile)))
	   dir)
	(if (= drow -1)
	    (if (= dcol -1)
		(setf dir 7)
		(if (= dcol 0)
		    (setf dir 0)
		    (if (= dcol 1)
			(setf dir 1)
			(setf dir -1))))
	    (if (= drow 0)
		(if (= dcol -1)
		    (setf dir 6)
		    (if (= dcol 1)
			(setf dir 2)
			(setf dir -1)))
		(if (= drow 1)
		    (if (= dcol -1)
			(setf dir 5)
			(if (= dcol 0)
			    (setf dir 4)
			    (if (= dcol 1)
				(setf dir 3)
				(setf dir -1))))
		    (setf dir -1))))
	dir))

(defun pushto (tilerow tilecol destrow destcol)
    (let (plan tile dir result)
	(if (member (aref (tw-grid w0) tilerow tilecol) tile-chars)
	    (setf tile (aref (tw-grid-which w0) tilerow tilecol ))
	    (return-from pushto 'not-a-tile))
	(setf dir (find-agent-dir tile (first (tw-agent-list w0))))
	(when (< dir 0)
	    (setf result (moveto tilerow tilecol))
	    (if (eq result 'no-path-found)
		(return-from pushto result))
	    (setf dir (find-agent-dir tile (first (tw-agent-list w0)))))
	(setf plan (tw-plan-push w0 (first (tw-agent-list w0)) tile
			(list (make-cell :row destrow :col destcol))))
	(if (equal plan '(x))
	    'unable-to-push-it
	    (run-plan plan))))


;
; Find the closest tile to 'obj' in 'world'.  Returns the tile and the
; parameters for future calls to next-closest-tile, as seven values.
; 
(defun tw-closest-tile (obj world)
    (let ((grid-mark (make-array (list (tw-rows world) (tw-cols world))
				 :initial-element nil))
	  frontier)
	(dolist (c (obj-cells obj))
	    (setf frontier (cons (list (cell-row c) (cell-col c)) frontier))
	    (setf (aref grid-mark (cell-row c) (cell-col c)) t))
	(next-closest-tile frontier nil nil grid-mark (tw-grid world)
			   (tw-grid-which world))))

; 
; Find the next closest tile, in number of moves, assuming tiles and agents
; are not in the way.  'unexpanded' are cells which have been checked but
; not expanded.  'limbo' are cells which have been generated but not checked
; for tiles.  'expanded' are cells which have been checked.
; 
(defun next-closest-tile (unexpanded limbo expanded marked twgrid twgridwhich)
  (let (this)
     (loop
	(when (null limbo)
	    (when (null unexpanded)
		(when (null expanded)
		    (return-from next-closest-tile
			(values nil nil nil nil nil nil nil)))
		(setf unexpanded expanded)
		(setf expanded nil))
	    (setf this (car unexpanded))
	    (setf unexpanded (cdr unexpanded))
	    (mapc #'(lambda(drow dcol)
			(let ((row (+ (car this) drow))
			      (col (+ (cadr this) dcol)))
			    (when (and (not (aref marked row col))
				       (not (member (aref twgrid row col)
						    obstacle-or-hole-chars)))
				(setf limbo (cons (list row col) limbo)))))
		  '( 0  0 -1  1)
		  '(-1  1  0  0)))
	(when limbo
	    (setf this (car limbo))
	    (setf expanded (cons this expanded))
	    (setf limbo (cdr limbo))
	    (setf (aref marked (car this) (cadr this)) t)
	    (when (member (aref twgrid (car this) (cadr this)) tile-chars)
	       (return-from next-closest-tile 
		  (values (aref twgridwhich (car this) (cadr this)) unexpanded
			limbo expanded marked twgrid twgridwhich)))))))


; Simulate a plan.  This should be fast, but isn't really right now
; because it's just lisp code.  That's life.
; If the planner's right, this should always work.  If it isn't, this
; isn't always sufficient to notice, so more checking should be done.
(defun tw-simulate (world agent plan)
  (let ((olddisp display-on))
    (setf display-on nil)
    (dolist (move plan)
	(when (not (tw-step world agent move 0))
	    (setf display-on olddisp)
	    (return-from tw-simulate nil)))
    (setf display-on olddisp)
    t))
		
(defun find-same-obj (obj objlist)
    (dolist (x objlist)
	(when (eq (obj-name obj) (obj-name x))
	    (return-from find-same-obj x)))
    nil)

; 
; Fill the hole at (row,col) using the simplest "push the closest tile into
; the easiest cell of the hole, repeatedly" scheme.
; 
(defun tw-closest-fill (world agent hole)
    (let (next-tile p1 p2 p3 p4 p5 p6
	  (ok-world world)
	  new-world
	  (ok-hole hole)
	  new-hole
	  (ok-agent agent)
	  new-agent
	  plan-to plan-from whole-plan)
	(multiple-value-setq (next-tile p1 p2 p3 p4 p5 p6)
			     (tw-closest-tile hole world))
	(loop
	    (when (not next-tile)
		(return-from tw-closest-fill '(x)))
	    (setf plan-to (tw-plan-move-near ok-world ok-agent next-tile))
	    (when (not (equal plan-to '(x)))
		(setf new-world (tw-copy-world ok-world))
		(setf new-agent (find-same-obj ok-agent 
				    (tw-agent-list new-world)))
		(tw-simulate new-world new-agent plan-to)
  		(setf new-hole
		      (find-same-obj ok-hole (tw-hole-list new-world)))
		(setf new-agent (find-same-obj ok-agent 
				    (tw-agent-list new-world)))
		(setf plan-from (tw-plan-push new-world new-agent next-tile 
				    (obj-cells new-hole)))
		(when (not (equal plan-from '(x)))
		    (tw-simulate new-world new-agent plan-from)
		    (setf whole-plan (append whole-plan plan-to plan-from))
		    (setf new-hole 
			  (find-same-obj ok-hole (tw-hole-list new-world)))
		    (when (not new-hole)
			(return-from tw-closest-fill whole-plan))
		    (setf ok-world new-world)
		    (setf ok-hole new-hole)
		    (setf ok-agent new-agent)))
	    (multiple-value-setq (next-tile p1 p2 p3 p4 p5 p6)
		(next-closest-tile p1 p2 p3 p4 p5 p6)))))



(defun fillhole (i j)
    (let (plan hole)
	(if (member (aref (tw-grid w0) i j) hole-chars)
	    (setf hole (aref (tw-grid-which-hole w0) i j))
	    (return-from fillhole 'not-a-hole))
	(setf plan (tw-closest-fill w0 (first (tw-agent-list w0)) hole))
	(if (equal plan '(x))
	    'unable-to-fill-it
	    (run-plan plan))))
