;
; tw.l - tile world simulator.  Marc Ringuette, 7/89
;
(defstruct tw 
    ; knob settings
    rows
    cols
    slice
    is-automatic
    must-match-shapes
    hole-knobs
    obstacle-knobs
    tile-knobs
    agent-knobs
    ; other properties of world
    score
    ; object-based world state (elements are 'obj's)
    hole-list
    obstacle-list
    tile-list
    agent-list
    ; grid-based world state (arrays [rows*cols])
    grid		    ; elements are chars
    grid-which		    ; elements are ptrs to obj's
    grid-which-hole	    ; elements are ptrs to obj's
)

(defstruct obj
    type
    name
    row
    col
    rel-cells	; (list of 'cell's where top left is (0,0)).
    cells	; (list of 'cell's where top left is (row,col))
    score	; for holes only
    timeout	; nil means no timeout.
)

(defstruct cell
    row
    col
    depth	; for cells of holes only
)

(defstruct knob
    ms-appear	    ; nil means no new ones appear
    ms-disappear    ; nil means they never disappear
    num		    ; a subknob
    cells	    ; a subknob
    timeout	    ; a subknob.  nil means no timeout.
    depth	    ; a subknob.
    score	    ; a subknob
)

(defstruct subknob
    start
    min
    max
    mean	; not implemented yet
    variance	; not implemented yet
    (distribution 'flat)
)

(defstruct move
    name
    delta-row
    delta-col
    agent-char
)

(defconstant r (make-move :name 'r :delta-row 0 :delta-col 1 :agent-char #\>))
(defconstant l (make-move :name 'l :delta-row 0 :delta-col -1 :agent-char #\<))
(defconstant u (make-move :name 'u :delta-row -1 :delta-col 0 :agent-char #\^))
(defconstant d (make-move :name 'd :delta-row 1 :delta-col 0 :agent-char #\v))
(defconstant n (make-move :name 'n :delta-row 0 :delta-col 0 :agent-char #\a))

(defconstant agent-chars '(#\> #\< #\^ #\v #\a #\o))
(defconstant tile-chars '(#\T #\t))
(defconstant obstacle-chars '(#\#))
(defconstant hole-chars '(#\O #\o #\t))
(defconstant obstacle-or-agent-chars (append obstacle-chars agent-chars))
(defconstant bare-tile-char '#\T)
(defconstant bare-hole-char '#\O)
(defconstant obstacle-char '#\#)
(defconstant tile-on-hole-char '#\t)
(defconstant agent-on-hole-char '#\o)
(defconstant nothing-char '#\space)

(defvar display-on t)

; 
; tw-step: update the simulation of 'world' for 'delta-t' milliseconds
; and then make 'move' for 'agent'.
; 
(defun tw-step (world agent move delta-t)
    (let ((new-world (copy-tw world))
	  success moving-objects)
	(setf move (eval move))
	; update world for each time-slice up to delta-t, rounded up.
	(setf delta-t (+ delta-t (- (tw-slice world)
				    (mod delta-t (tw-slice world)))))
	(do ((i 0 (tw-slice world)))
	    ((>= i delta-t))
	    (tw-update-timeouts new-world)
	    (if (tw-is-automatic world)
		(tw-toss-dice new-world)
		(tw-toss-loaded-dice new-world))
	)
	; execute 'move'
        (if (not (eq move n))
	    (setf moving-objects (tw-propagate-move new-world agent move))
	(tw-update-move new-world moving-objects move)
	; drop tiles in holes
	(setf (tw-tile-list new-world)
	      (tw-drop-tiles new-world (tw-tile-list new-world)))
	(setf (tw-agent-list new-world)
	      (tw-drop-tiles new-world (tw-agent-list new-world)))
	; finish up
	(if display-on (tw-display new-world))
	(setf success (or (eq move 'n)
			  moving-objects))
	(values new-world success)))

; 
; decrease timeouts and remove hole if it disappears
; 
(defun tw-update-timeouts (new-world)
	    (setf (tw-hole-list new-world)
		  (tw-update-timeouts-2 (tw-hole-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-tile-list new-world)
		  (tw-update-timeouts-2 (tw-tile-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-agent-list new-world)
		  (tw-update-timeouts-2 (tw-agent-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-obstacle-list new-world)
		  (tw-update-timeouts-2 (tw-obstacle-list new-world)
				        (tw-slice new-world))))


(defun tw-update-timeouts-2 (objlist slice)
	(mapcan #'(lambda(obj)
	    (if (not (null (obj-timeout obj)))
		(progn
		      (if (> (obj-timeout obj) 0)
			  (if (> (obj-timeout obj) (- slice 1))
			      (setf (obj-timeout obj) (- (obj-timeout obj) 
						         slice))
			      (setf (obj-timeout obj) 0))
		      (if (or (> (obj-timeout obj) 0)
			      (= (obj-timeout obj) -1))
		          (list obj))))))
		    objlist))

; 
; agent-object is to go in direction 'move'.  Propagate its move to all
; pushable objects and form a list of all moving objects.  If an obstacle
; is encountered, the returned list is nil.
; 
(defun tw-propagate-move (new-world agent-object move)
    (let ((moving (list agent-object))
	  new-moving
	  (front (tw-front-edge (obj-cells agent-object) move)))
	(loop
	    (if (tw-is-touching-front front (tw-grid new-world) move
		    obstacle-or-agent-chars)
		(return-from tw-propagate-move nil))
	    (if (not (tw-is-touching-front front (tw-grid new-world) move
			 tile-chars))
		(return-from tw-propagate-move moving))
	    (setf new-moving (tw-touching-front front (tw-tile-list new-world)
			         move))
	    (setf front 
		  (tw-front-edge (mapcan #'(lambda(tile)
					      (obj-cells tile))
					 new-moving)
				 move))
	    (setf moving (append new-moving moving)))))

(defun tw-front-edge (cells move)
    (remove-if #'(lambda(cell)
		    (find-if #'(lambda(cell2)
				   (and (= (+ (cell-row cell) 
					      (move-delta-row move))
				           (cell-row cell2))
					(= (+ (cell-col cell)
					      (move-delta-col move))
					   (cell-col cell2))))
			     cells))
	       cells))

(defun tw-is-touching-front (front grid move chars)
    (dolist (cell front)
	(if (member (aref grid
			  (+ (cell-row cell) (move-delta-row move))
			  (+ (cell-col cell) (move-delta-col move)))
		    chars)
	    (return-from tw-is-touching-front t))))


(defun tw-touching-front (front objlist move)
    (mapcan #'(lambda(obj)
		  (dolist (cell (obj-cells obj))
		      (if (find-if #'(lambda(cell2)
				       (and (= (+ (cell-row cell2) 
					          (move-delta-row move))
					       (cell-row cell))
					    (= (+ (cell-col cell2)
						  (move-delta-col move))
					       (cell-col cell))))
				   front)
			  (return (list obj)))))
	    objlist))

; 
; update world to correspond to the moving-objects going in direction 'move'
; 
(defun tw-update-move (new-world moving-objects move)
    (tw-remove-objects new-world moving-objects)
    (mapc #'(lambda (obj)
		    (obj-set-pos obj (+ (obj-row obj) (move-delta-row move))
		                     (+ (obj-col obj) (move-delta-col move))))
	  moving-objects)
    (tw-place-objects new-world moving-objects move))


; Remove objs from world.  Can be any kind of obj; this is used in the
; dice-tossing routines as well as in tw-update-move.
(defun tw-remove-objects (world objs)
    (mapc #'(lambda(obj)
		(mapc #'(lambda(cell)
			    (if (and (member (aref (tw-grid world)
					           (cell-row cell)
					           (cell-col cell))
					     hole-chars)
				     (not (eq (obj-type obj) 'hole)))
				(setf (aref (tw-grid world)
					    (cell-row cell)
					    (cell-col cell))
				      bare-hole-char)
				(setf (aref (tw-grid world)
					    (cell-row cell)
					    (cell-col cell))
				      nothing-char)))
		      (obj-cells obj)))
	  objs))

(defun tw-place-objects (world objs move)
    (mapc #'(lambda(obj)
	(mapc #'(lambda(cell)
		    (let ((ch (aref (tw-grid world)
					    (cell-row cell)
					    (cell-col cell)))
			  newch)
			(cond ((eq (obj-type obj) 'tile)
			       (if (member ch hole-chars)
				   (setf newch tile-on-hole-char)
				   (setf newch bare-tile-char)))
			      ((eq (obj-type obj) 'agent)
			       (if (member ch hole-chars)
				   (setf newch agent-on-hole-char)
				   (setf newch (move-agent-char move))))
			      ((eq (obj-type obj) 'hole)
			       (if (member ch tile-chars)
				   (setf newch tile-on-hole-char)
				   (if (member ch agent-chars)
				       (setf newch agent-on-hole-char)
				       (setf newch bare-hole-char))))
			      ((eq (obj-type obj) 'obstacle)
			       (setf newch obstacle-char)))
			(setf (aref (tw-grid world)
				    (cell-row cell)
				    (cell-col cell))
			      newch)
			(if (eq (obj-type obj) 'hole)
			    (setf (aref (tw-grid-which-hole world)
					(cell-row cell)
					(cell-col cell))
				  obj)
			    (setf (aref (tw-grid-which world)
					(cell-row cell)
					(cell-col cell))
				  obj))))
	      (obj-cells obj)))
	  objs))

(defun obj-set-pos (obj row col)
    (setf (obj-row obj) row)
    (setf (obj-col obj) col)
    (mapc #'(lambda(relcell abscell)
		(setf (cell-row abscell)
		      (+ (cell-row relcell) (obj-row obj)))
		(setf (cell-col abscell)
		      (+ (cell-col relcell) (obj-col obj))))
	  (obj-rel-cells obj)
	  (obj-cells obj)))

; 
; If any tile in 'tiles' is hanging over a hole, drop it in and
; credit points if any holes are full.  'tiles' can be an agent
; list; treat them as tiles and if they fall in, fill in the hole and
; kill them.
; 
(defun tw-drop-tiles (new-world tiles)
    (let (some-fell result)
      (setf result
	(mapcan #'(lambda(tile)
	   (if (mapcan #'(lambda(cell)
		    (if (member (aref (tw-grid new-world)
			              (cell-row cell)
				      (cell-col cell))
			        hole-chars)
		        nil
		        (list t)))
		    (obj-cells tile))
	    ; doesn't drop; leave in the list
		(list tile)
	    ; does drop; fill in all cells, remove from list
		(progn
		    (setf some-fell t)
		    (mapc #'(lambda(cell)
			   (let ((hole (aref (tw-grid-which-hole tile)
					     (cell-row cell)
					     (cell-col cell))))
				(mapc #'(lambda(hole-cell)
					    (if (and (= (cell-row cell) 
							(cell-row hole-cell))
						     (= (cell-col cell)
							(cell-col hole-cell)))
						(setf (cell-depth hole-cell)
						      (- (cell-depth hole-cell)
							 1))))
				      (obj-cells hole))))
		          (obj-cells tile))
		    nil)))
	   tiles))
      (if some-fell (tw-update-holes new-world))
      result))

; 
; remove any full holes and add to the score
; 
(defun tw-update-holes (new-world)
  (setf (tw-hole-list new-world)
    (remove-if #'(lambda(hole)
			  (setf (obj-cells hole)
				(remove-if #'(lambda(cell)
						 (= (cell-depth cell) 0))
					   (obj-cells hole)))
			  (if (null (obj-cells hole))
			      (progn
				  (setf (tw-score new-world)
					(+ (tw-score new-world)
					   (obj-score hole)))
				  t)
			      nil))
	       (tw-hole-list new-world))))


;
; A time tick has occurred.  Toss the dice and add/delete objects as
; needed.
; 
(defun tw-toss-dice (new-world)
    (setf (tw-hole-list new-world)
	  (toss1 (tw-hole-knobs new-world) (tw-hole-list new-world) 
		 'hole new-world))
    (setf (tw-tile-list new-world)
	  (toss1 (tw-tile-knobs new-world) (tw-tile-list new-world)
		 'tile new-world))
    (setf (tw-agent-list new-world)
	  (toss1 (tw-agent-knobs new-world) (tw-agent-list new-world)
		 'agent new-world))
    (setf (tw-obstacle-list new-world)
	  (toss1 (tw-obstacle-knobs new-world) (tw-obstacle-list new-world)
		 'obstacle new-world)))

(defun toss1 (knobs objlist objtype new-world)
  (let (x newobj)
    ; delete an obj if dice say so and not at min #
    (if (and (not (null (knob-ms-disappear knobs)))
	     (< (rrand (knob-ms-disappear knobs)) (tw-slice new-world))
	     (> (length objlist) (subknob-min (knob-num knobs))))
	(setf x (rrand (length objlist)))
	(setf objlist (append (subseq objlist 0 (- x 1))
			      (subseq objlist (+ x 1)))))
    ; add an obj if dice say so and not at max #
    (if (and (not (null (knob-ms-appear knobs)))
	     (< (rrand (knob-ms-appear knobs)) (tw-slice new-world))
	     (< (length objlist) (subknob-max (knob-num knobs))))
	(setf newobj (make-new-obj type))
	(if (tw-generate-random-values new-world newobj knobs)
	    (setf objlist (cons newobj objlist))))
    objlist))

(defun make-new-obj (type)
    (let ((obj (make-obj :type type :name (make-name type))))
	(setf (eval (obj-name obj)) obj)
	obj))

(defun make-name (root)
    (intern (symbol-name (gensym (string parent)))))

(defun tw-init (world)
    (tw-init-grids world)
    (setf (tw-obstacle-list world)
	  (tw-init-list world 'obstacle 'tw-obstacle-knobs))
    (tw-place-objects world (tw-obstacle-list world) n)
    (setf (tw-hole-list world)
	  (tw-init-list world 'hole 'tw-hole-knobs))
    (tw-place-objects world (tw-hole-list world) n)
    (setf (tw-tile-list world)
	  (tw-init-list world 'tile 'tw-tile-knobs))
    (tw-place-objects world (tw-tile-list world) n)
    (setf (tw-agent-list world)
	  (tw-init-list world 'agent 'tw-agent-knobs))
    (tw-place-objects world (tw-agent-list world) n)
    (if display-on (tw-display world)))

(defun tw-init-list (world type knob-name)
   (let* ((lst nil) 
	  (len 0)
	  (iter 0) 
	  (newobj (make-new-obj type))
	  (start-num (subknob-start 
			    (knob-num (funcall knob-name world))))
	  (max-iter (* 10 start-num)))
	(loop
	    (if (tw-generate-random-values world newobj
			(funcall knob-name world))
		(progn (setf lst (cons newobj lst))
		       (setf len (+ len 1))
		       (setf newobj (make-new-obj type))))
	    (setf iter (+ iter 1))
	    (if (or (= len start-num)
		    (> iter max-iter))
		(return lst)))))

(defun tw-init-grids (world)
	(setf (tw-grid world) (make-array (list (tw-rows world)
					        (tw-cols world))
					  :element-type 'standard-char))
	(dotimes (i (tw-rows world))
	    (dotimes (j (tw-cols world))
		(setf (aref (tw-grid world) i j) nothing-char)))
	(dotimes (i (tw-cols world))
	    (setf (aref (tw-grid world) 
			0 
			i)
		  obstacle-char)
	    (setf (aref (tw-grid world) 
			(- (tw-rows world) 1)
			i)
		  obstacle-char))
	(dotimes (i (tw-rows world))
	    (setf (aref (tw-grid world) 
			i
			0)
		  obstacle-char)
	    (setf (aref (tw-grid world) 
			i
			(- (tw-cols world) 1))
		  obstacle-char))
	(setf (tw-grid-which world) (make-array (list (tw-rows world)
						      (tw-cols world))))
	(setf (tw-grid-which-hole world) (make-array (list (tw-rows world)
						           (tw-cols world)))))

;
; returns t if successful, or nil if it landed on top of another obj.
; 
(defun tw-generate-random-values (new-world obj knobs)
    (let (max-row max-col x y)
	(setf (obj-rel-cells obj) (list (make-cell :row 0 :col 0)))
	(dotimes (i (- (tw-random-knob (knob-cells knobs)) 1))
	    (setf (obj-rel-cells obj)
		  (tw-add-random-cell-to (obj-rel-cells obj))))
	(multiple-value-setq (max-row max-col)
			     (tw-normalize-cells (obj-rel-cells obj)))
	(setf (obj-cells obj) nil)
	(dolist (c (reverse (obj-rel-cells obj)))
	    (setf (obj-cells obj) 
		  (cons (copy-cell c) (obj-cells obj))))
	(if (eq (obj-type obj) 'hole)
	    (let (dep)
		(setf dep (tw-random-knob (knob-depth knobs)))
		(dolist (c (obj-rel-cells obj))
		    (setf (cell-depth c) dep))
		(setf (obj-score obj) (tw-random-knob (knob-score knobs)))))
	(if (not (null (knob-timeout knobs)))
	    (setf (obj-timeout obj) (tw-random-knob (knob-timeout knobs))))
	(setf x (+ 1 (rrand (- (tw-rows new-world) 1 max-row))))
	(setf y (+ 1 (rrand (- (tw-cols new-world) 1 max-col))))
	(obj-set-pos obj x y)
	(if (tw-is-on-something (obj-cells obj) (tw-grid new-world))
	    nil
	    t)))

; Set top left corner (not necessarily itself a cell) to 0,0
; and return two values: max row and col of any cell in the new
; coordinates.
(defun tw-normalize-cells (cells)
    (let ((rmin 0) (rmax 0) (cmin 0) (cmax 0))
	(dolist (c cells)
	    (if (< (cell-row c) rmin) (setf rmin (cell-row c)))
	    (if (> (cell-row c) rmax) (setf rmax (cell-row c)))
	    (if (< (cell-col c) cmin) (setf cmin (cell-col c)))
	    (if (> (cell-col c) cmax) (setf cmax (cell-col c))))
	(dolist (c cells)
	    (setf (cell-row c) (- (cell-row c) rmin))
	    (setf (cell-col c) (- (cell-col c) cmin)))
	(values (- rmax rmin) (- cmax cmin))))


(defun tw-is-on-something (cells grid)
    (dolist (cell cells)
	(if (not (eql (aref grid
		           (cell-row cell)
		           (cell-col cell))
		     nothing-char))
	    (return-from tw-is-on-something t))))


(defun tw-random-knob (subknob)
    (if (eq (subknob-distribution subknob)
	    'flat)
	(+ (subknob-min subknob)
	   (rrand (- (subknob-max subknob) (subknob-min subknob))))
	'oops-only-flat-distributions))

(defun tw-add-random-cell-to (cell-list)
    (let (cell move (new-cell (make-cell)))
	(loop
	    (setf cell (nth (rrand (length cell-list)) cell-list))
	    (setf move (eval (nth (rrand 4) '(u d l r))))
	    (setf (cell-row new-cell) 
		  (+ (cell-row cell) (move-delta-row move)))
	    (setf (cell-col new-cell) 
		  (+ (cell-col cell) (move-delta-col move)))
	    (if (not (find-if #'(lambda(x) 
				       (and (eq (cell-row new-cell)
					        (cell-row x))
					    (eq (cell-col new-cell)
					        (cell-col x))))
			      cell-list))
		(return (cons new-cell cell-list))))))

; 
; repeatable random number generator
; 
(defvar rrand-seed 0)
; 
(defun rrand (max)
    (setf rrand-seed (mod (+ (* 43107 rrand-seed) 102713) 900337))
    (if (> max 0)
	(mod rrand-seed max)
        0))
; 
; Put a normal dist. here?!  Bounds crufty also.
(defun rdist (mean variance min max)
    (let ((x (+ mean
		(- (rrand (* 4 variance))
		   (* 2 variance)))))
	 (if (null min)
	     x
	     (if (and (<= min x)
		      (<= x max))
		 x
		 (rdist mean variance min max)))))


; 
; Simple "display"; may be replaced by one using curses.
; 
(defun tw-display (world)
	(princ #\page)
	(dotimes (row (tw-rows world))
	    (dotimes (col (tw-cols world))
		(princ (aref (tw-grid world) row col))
		(princ #\space))
	    (princ #\newline)))
