;
; tw.l - tile world simulator.  Marc Ringuette, 7/89
;
(defstruct tw 
    ; knob settings
    rows
    cols
    slice
    is-automatic
    must-match-shapes
    hole-knobs
    obstacle-knobs
    tile-knobs
    agent-knobs
    ; object-based world state (elements are 'obj's)
    hole-list
    obstacle-list
    tile-list
    agent-list
    ; grid-based world state (arrays [rows*cols])
    grid-what		    ; elements are chars
    grid-which-obj	    ; elements are ptrs to obj's
    grid-which-hole	    ; elements are ptrs to obj's
)

(defstruct obj
    type
    row
    col
    rel-cells	; (list of 'cell's where one of them is (0,0))
    abs-cells	; (list of 'cell's where one of them is (row,col))
    score	; for holes only
    timeout	; for holes only
)

(defstruct cell
    row
    col
    depth	; for cells of holes only
)

(defstruct knob
    ms-appear
    ms-disappear
    num		    ; a subknob
    cells	    ; a subknob
    timeout	    ; a subknob
)

(defstruct subknob
    start
    min
    max
    mean	; not implemented yet
    variance	; not implemented yet
    (distribution 'flat)
)

(defstruct move
    name
    delta-row
    delta-col
    agent-char
)

(setq r (make-move :name 'r :delta-row  0 :delta-col  1 :agent-char #\>))
(setq l (make-move :name 'l :delta-row  0 :delta-col -1 :agent-char #\<))
(setq u (make-move :name 'u :delta-row -1 :delta-col  0 :agent-char #\^))
(setq d (make-move :name 'd :delta-row  1 :delta-col  0 :agent-char #\v))

(defconstant agent-chars '(#\> #\< #\^ #\v #\o))
(defconstant tile-chars '(#\T #\t))
(defconstant obstacle-chars '(#\#))
(defconstant hole-chars '(#\O #\o #\t))
(defconstant obstacle-or-agent-chars (append obstacle-chars agent-chars))


(defconstant bare-tile-char '#\T)
(defconstant bare-hole-char '#\O)
(defconstant obstacle-char '#\#)
(defconstant tile-on-hole-char '#\T)
(defconstant agent-on-hole-char '#\o)
(defconstant nothing-char '#\space)

; 
; tw-step: update the simulation of 'world' for 'delta-t' milliseconds
; and then make 'move' for 'agent'.
; 
(defun tw-step (world agent move delta-t)
    (let ((new-world (copy-tw world)))
	; update world for each time-slice up to delta-t, rounded up.
	(setq delta-t (+ delta-t (- (tw-slice world)
				    (mod delta-t (tw-slice world)))))
	(do ((i 0 (tw-slice world)))
	    ((>= i delta-t))
	    (setf (tw-hole-list new-world)
		  (tw-update-timeouts new-world (tw-hole-list new-world)))
	    (setf (tw-tile-list new-world)
		  (tw-update-timeouts new-world (tw-tile-list new-world)))
	    (setf (tw-agent-list new-world)
		  (tw-update-timeouts new-world (tw-agent-list new-world)))
	    (setf (tw-obstacle-list new-world)
		  (tw-update-timeouts new-world (tw-obstacle-list new-world)))
	    (if (tw-is-automatic world)
		(toss-dice new-world)
		(toss-loaded-dice new-world))
	)
	; execute 'move'
        (setq moving-objects (tw-propagate-move new-world agent move))
	(tw-update-move new-world moving-objects)
	; drop tiles in holes
	(setf (tw-tile-list new-world)
	      (tw-drop-tiles new-world (tw-tile-list new-world)))
	(setf (tw-agent-list new-world)
	      (tw-drop-tiles new-world (tw-agent-list new-world)))))

; 
; decrease timeouts and remove hole if it disappears
; 
(defun tw-update-timeouts (new-world objlist)
	    (setf (tw-hole-list new-world)
		  (tw-update-timeouts-2 (tw-hole-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-tile-list new-world)
		  (tw-update-timeouts-2 (tw-tile-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-agent-list new-world)
		  (tw-update-timeouts-2 (tw-agent-list new-world)
				        (tw-slice new-world)))
	    (setf (tw-obstacle-list new-world)
		  (tw-update-timeouts-2 (tw-obstacle-list new-world)
				        (tw-slice new-world))))


(defun tw-update-timeouts-2 (objlist slice)
		  (mapcan #'(lambda(obj)
		      (if (> (obj-timeout obj) 0)
			  (if (> obj-timeout obj) (- slice 1)
			      (setf (obj-timeout obj) (- (obj-timeout obj) 
						         slice))
			      (setf (obj-timeout obj) 0))
		      (if (or (> (obj-timeout obj) 0)
			      (= (obj-timeout obj) -1))
		          (list obj)
		          nil))
		    objlist)))

; 
; agent-object is to go in direction 'move'.  Propagate its move to all
; pushable objects and form a list of all moving objects.  If an obstacle
; is encountered, the returned list is nil.
; 
(defun tw-propagate-move (new-world agent-object move)
    (let ((moving (list agent-object))
	  (front (tw-front-edge (obj-cells agent-object) move)))
	(loop
	    (if (tw-is-touching-front front (tw-grid new-world) move
		    agent-or-obstacle-chars)
		(return-from tw-propagate-move nil))
	    (if (not (tw-is-touching-front front (tw-grid new-world) move
			 tile-chars))
		(return-from tw-propagate-move moving))
	    (setq new-moving (tw-touching-front front (tw-tile-list new-world)
			         move))
	    (setq front 
		  (tw-front-edge (mapcan #'(lambda(tile)
					      (obj-cells tile))
					 new-moving)
				 move))
	    (setq moving (append new-moving moving)))))

(defun tw-front-edge (cells move)
    (remove-if #'(lambda(cell)
		    (find-if #'(lambda(cell2)
				   (and (= (+ (cell-row cell) 
					      (move-delta-row move))
				           (cell-row cell2))
					(= (+ (cell-col cell)
					      (move-delta-col move))
					   (cell-col cell2))))
			     cells))
	       cells))

(defun tw-is-touching-front (front grid move chars)
    (dolist (cell front)
	(if (member (aref grid
			  (+ (cell-row cell) (move-delta-row move))
			  (+ (cell-col cell) (move-delta-col move)))
		    chars)
	    return-from tw-is-touching-front t)))


(defun tw-touching-front (front objlist move)
    (mapcan #'(lambda(obj)
		  (dolist (cell (obj-cells obj))
		      (if (find-if #'(lambda(cell2)
				       (and (= (+ (cell-row cell2) 
					          (move-delta-row move))
					       (cell-row cell))
					    (= (+ (cell-col cell2)
						  (move-delta-col move))
					       (cell-col cell))))
				   front)
			  return (list obj))))
	    objlist))

; 
; update world to correspond to the moving-objects going in direction 'move'
; 
(defun tw-update-move (new-world moving-objects move)
    ; remove images of moving-objects
    (mapc #'(lambda(obj)
		(mapc #'(lambda(cell)
			    (if (member (aref (tw-grid new-world)
					      (cell-row cell)
					      (cell-col cell))
					bare-hole-chars)
				(setf (aref (tw-grid new-world)
					    (cell-row cell)
					    (cell-col cell))
				      bare-hole-char)))
		      (obj-cells obj)))
	  moving-objects)
    ; edit the object's position info
    (obj-set-pos obj (+ (obj-row obj) (move-delta-row move))
		     (+ (obj-col obj) (move-delta-col move)))
    ; place new images of objects into grids
    (mapc #'(lambda(obj)
	      (if (eq (obj-type obj) 'tile)
		(mapc #'(lambda(cell)
			    (if (member (aref (tw-grid new-world)
					      (cell-row cell)
					      (cell-col cell))
					bare-hole-chars)
				(setf (aref (tw-grid new-world)
					    (cell-row cell)
					    (cell-col cell))
				      tile-on-hole-char)
				(setf (aref (tw-grid new-world)
					    (cell-row cell)
					    (cell-col cell))
				      bare-tile-char))
			    (setf (aref (tw-grid-which new-world)
					(cell-row cell)
					(cell-col cell))
				  obj))
		      (obj-cells obj))
		(mapc #'(lambda(cell)	; guess it was an agent!
			    (if (member (aref (tw-grid new-world)
					      (cell-row cell)
					      (cell-col cell))
					bare-hole-chars)
				(setf (aref (tw-grid new-world)
					    (cell-row cell)
					    (cell-col cell))
				      agent-on-hole-char)
				(setf (aref (tw-grid new-world)
					    (cell-row cell)
					    (cell-col cell))
				      (move-agent-char move)))
			    (setf (aref (tw-grid-which new-world)
					(cell-row cell)
					(cell-col cell))
				  obj))
		      (obj-cells obj))))
	  moving-objects))

(defun obj-set-pos (obj row col)
    (setf (obj-row obj) (+ (obj-row obj)
			   (move-delta-row move)))
    (setf (obj-col obj) (+ (obj-col obj)
			   (move-delta-col move)))
    (mapc #'(lambda(relcell abscell)
		(setf (cell-row abscell)
		      (+ (cell-row relcell) (obj-row obj)))
		(setf (cell-col abscell)
		      (+ (cell-col relcell) (obj-col obj))))
	  (obj-rel-cells obj)
	  (obj-abs-cells obj)))

; 
; If any tile in 'tiles' is hanging over a hole, drop it in and
; credit points if any holes are full.  'tiles' can be an agent
; list; treat them as tiles and if they fall in, fill in the hole and
; kill them.
; 
(defun tw-drop-tiles (new-world tiles)
    (let (some-fell result)
      (setq result
	(mapcan #'(lambda(tile)
	   (if (mapcan #'(lambda(cell)
		    (if (member (aref (tw-grid new-world)
			              (cell-row cell)
				      (cell-col cell))
			        hole-chars)
		        nil
		        (list t)))
		    (obj-abs-cells tile))
	    ; doesn't drop; leave in the list
		(list tile)
	    ; does drop; fill in all cells, remove from list
		(progn
		    (setq some-fell t)
		    (mapc #'(lambda(cell)
			   (let ((hole (aref (tw-grid-which-hole tile)
					     (cell-row cell)
					     (cell-col cell))))
				(mapc #'(lambda(hole-cell)
					    (if (and (= (cell-row cell) 
							(cell-row hole-cell))
						     (= (cell-col cell)
							(cell-col hole-cell)))
						(setf (cell-depth hole-cell)
						      (- (cell-depth hole-cell)
							 1)))))))
		          (obj-cells tile))
		    nil)))
	   tiles))
      (if some-fell (tw-update-holes new-world))
      result))

; 
; remove any full holes and add to the score
; 
(defun tw-update-holes (new-world)
  (setf (tw-hole-list new-world)
    (remove-if #'(lambda(hole)
			  (setf (obj-cells hole)
				(remove-if #'(lambda(cell)
						 (= (cell-depth cell) 0))
					   (obj-cells hole)))
			  (if (null (obj-cells hole))
			      (progn
				  (setf (tw-score new-world)
					(+ (tw-score new-world)
					   (obj-score hole)))
				  t)
			      nil))
	       (tw-hole-list new-world))))


;
; A time tick has occurred.  Toss the dice and add/delete objects as
; needed.
; 
(defun tw-toss-dice (new-world)
    (setf (tw-hole-list new-world)
	  (toss1 (tw-hole-knobs new-world) (tw-hole-list new-world) 
		 'hole new-world))
    (setf (tw-tile-list new-world)
	  (toss1 (tw-tile-knobs new-world) (tw-tile-list new-world)
		 'tile new-world))
    (setf (tw-agent-list new-world)
	  (toss1 (tw-agent-knobs new-world) (tw-agent-list new-world)
		 'agent new-world))
    (setf (tw-obstacle-list new-world)
	  (toss1 (tw-obstacle-knobs new-world) (tw-obstacle-list new-world)
		 'obstacle new-world)))

(defun toss1 (knobs objlist objtype new-world)
  (let (x)
    ; delete an obj if dice say so and not at min #
    (if (and (< (rrand (knob-ms-disappear knobs)) (tw-slice new-world))
	     (> (length objlist) (subknob-min (knob-numparam knobs))))
	(setq x (rrand (length objlist)))
	(setq objlist (append (subseq objlist 0 (- x 1))
			      (subseq objlist (+ x 1)))))
    ; add an obj if dice say so and not at max #
    (if (and (< (rrand (knob-ms-appear knobs)) (tw-slice new-world))
	     (< (length objlist) (subknob-max (knob-numparam knobs))))
	(setq newobj (make-obj :type objtype))
	(if (tw-generate-random-values new-world newobj knobs)
	    (setq objlist (cons newobj objlist))))
    objlist))

(defun tw-init (world)
    (let (i l)
	(dotimes (i (subknob-start (knob-num (tw-hole-knobs world))))
	    (setq newobj (make-obj :type 'hole))
	    (dotimes (j 5)
		(if (tw-generate-random-values world newobj
			(tw-hole-knobs world))
		    (return (cons 

;
; returns t if successful, or nil if it landed on top of another obj.
; 
(defun tw-generate-random-values (new-world obj knobs)
    (let (i c cell-list d)
	(setq cell-list (list (make-cell :row 0 :col 0)))
	(dotimes (i (tw-random-knob (knob-cellparam knobs)))
	    (setq cell-list (tw-add-random-cell-to cell-list)))
	(if (eq (obj-type obj) 'hole)
	    (progn
		(setq d (tw-random-knob (knob-depthparam knobs)))
		(dolist (c cell-list)
		    (setf (cell-depth c) d))))
	(setq x (rrand (tw-rows new-world)))
	(setq y (rrand (tw-cols new-world)))
	(obj-set-pos obj x y)
	(if (tw-is-on-something obj (tw-grid new-world))
	    nil
	    t)))


(defun tw-is-on-something (cells grid)
    (dolist (cell cells)
	(if (not (eq (aref grid
		           (cell-row cell)
		           (cell-col cell))
		     nothing-char))
	    return-from tw-is-on-something t)))
	
	
(defun tw-random-knob (subknob)
    (if (eq (subknob-distribution subknob)
	    'flat)
	(+ (subknob-min subknob)
	   (rrand (- (subknob-max subknob) (subknob-min subknob))))
	'oops-only-flat-distributions))

(defun tw-add-random-cell-to (cell-list)
    (let (cell move (new-cell (make-cell)))
	(loop
	    (setq cell (nth (rrand (length cell-list)) cell-list))
	    (setq move (nth (rrand 4) '(u d l r)))
	    (setf (cell-row new-cell) 
		  (+ (cell-row cell) (move-delta-row move)))
	    (setf (cell-col new-cell) 
		  (+ (cell-col cell) (move-delta-col move)))
	    (if (not (find-if #'(lambda(x) 
				       (and (eq (cell-row new-cell)
					        (cell-row x))
					    (eq (cell-col new-cell)
					        (cell-col x))))
			      cell-list))
		(return (cons new-cell cell-list))))))

; 
; repeatable random number generator
; 
(defvar rrand-seed 0)
; 
(defun rrand (max)
    (setq rrand-seed (mod (+ (* 43107 rrand-seed) 102713) 900337))
    (mod rrand-seed max))
; 
; Put a normal dist. here?!  Bounds crufty also.
(defun rdist (mean variance min max)
    (let ((x (+ mean
		(- (rrand 4*variance)
		   2*variance))))
	 (if (null min)
	     x
	     (if (and (<= min x)
		      (<= x max))
		 x
		 (rdist mean variance min max)))))

