;  Tile World Experiment  --  Martha Pollack

(defstruct agent
  ;parameters that are invariant through an experiment
  think-strategy            ;top-level strategy for deliberating or planning
  act-time                  ;length of time for an atomic action (in ms)
  wait-time                 ;length of time to wait if nothing's happening
  ;"mind" (vary during each experiment)
  options                   ;list of option-sets
  intentions                ;list of acts
  removed-intentions        ;for the 'act' mind: intentions removed
                            ; because they've been done by 'act' or
                            ; seen to disappear by 'perceive'
)

(defstruct option-set
  purpose                   ;intention id or 'SELF'
  alt-acts                  ;list of acts
)

(defstruct act
  id
  type
  parms                     ;list of parm name and binding pairs
  subacts                   ;list of acts
  enabling-conds
  postconds
  time
  atomic                    ;flag to indicate whether a base-level act
  failed
)

(defstruct parm
  name
  binding
)


(defvar *history*)
(defvar game-length 100000)
;;;;;;;;;;

;Initialize a world and an agent, and then play a tileworld game.
; *Future:  standardize initialization of agent and world; create interface
;     for initializing both

(defun tw-experiment ()
    (if display-on (twinitdisp))
    (tw-init w0)
    (setf (agent-options a0) nil)
    (setf (agent-intentions a0) nil)
    (setf (agent-removed-intentions a0) nil)
    (setf w0 (play-tw-game w0 a0 game-length))
    (if display-on (twclosedisp))
    (tw-score w0))

;Play a tileworld game:  Simulate concurrent acting and thinking by
;first performing one think step, keeping track of how long you take,
;and then acting for the same period of time, allowing the world to
;change then as well.  Because of the concurrency, the act cycle makes 
;use of the initial state of mind:  at the end of both acting and thinking
;the resulting changes to the agent's mental state are combined.  Note
;that the world only changes during the act cycle, so it doesn't need
;to be copied and then updated to simulate concurrency.
;Returns:  --world that results from playing the game

(defun play-tw-game (world mind game-length)
  (with-open-file
   (*history* "~mnr/tile/history.data" :direction :output
	      :if-exists :rename)
  (let ( think-time cycles-to-act mind1 mind2)
    (agent-init mind world)
    (when keephistory 
	  (write "HOLES INITIALLY AROUND:"  :stream *history*)
	  (write-char #\Newline *history*)
	  (write-holes (tw-hole-list world) *history*)
	  (write-char #\Newline *history*)
	  (write "INITIAL MENTAL STATE:" :stream *history*)
          (write-char #\Newline *history*)
	  (write-mind mind *history*)
	  (write-char #\Newline  *history*))
    (do ( (elapsed 0 (+ elapsed think-time)))
	( (>= elapsed game-length) world)
	(setf mind1 (agent-copy mind))
	(setf mind2 (agent-copy mind))
	(setf think-time (think world mind1))
	(when keephistory
	      (write "THINK TIME USED:"  :stream *history*)
	      (write-char #\Newline *history*)
	      (write think-time :stream *history*)
	      (write-char #\Newline *history*))
	(setf cycles-to-act (- (floor (/ (+ elapsed think-time)
				         (agent-act-time mind)))
			        (floor (/ elapsed
				          (agent-act-time mind)))))
	(multiple-value-setq (world mind2)
	    		     (act world mind2 cycles-to-act))
	(merge-minds mind mind1 mind2))
    world))
)

(defun agent-init (mind world)
    (setf (agent-options mind)
	  (create-new-options (tw-hole-list world))))

;Perform one think step.  The strategy for determining what this step
;will be depends upon the agent's "think-strategy", chosen at 
;time.  Keep track of the amount of time (in computer msec's) used.
; Returns:  MULTIPLE VALUES:  --modified "mind" (input arg1 is mutated)
;                             --length of time used

(defun think (world mind)
  (let (start-time stop-time work-done think-time)
    (setf start-time (get-internal-run-time))
    (setf work-done (funcall (agent-think-strategy mind) world mind))
    (setf stop-time (get-internal-run-time))
    (setf think-time (if work-done
			 (/ (- stop-time start-time) 1000)  ; milliseconds used
		         (agent-wait-time mind))) ; else pretend it took longer
    think-time))

;Act for the same amount of time as you just thought for.  If act
;is successful, remove the satisfied intention from the mental state.
;After attempting an act, note any changes that occurred in the world
;while you were acting.
;Returns:  --modified "mind"  (input arg2 is mutated)
;          (input arg1, representing the world, is also mutated)

(defun act (world mind cycles)
  (let* ( (cycle-length (agent-act-time mind))
	; (cycles (round time cycle-length))
        ; (agent (first (tw-agent-list world)))
	  old-world success next-act moved-ok)
    (dotimes (i cycles mind)
	(setf old-world (tw-copy-world world))
	(setf next-act (select-next-act mind))
	(when keephistory (write "SELECTING ACT:  " :stream *history*)
	                  (write-char #\Newline *history*)
	                  (write-each-act (list next-act)  *history*)
			  (write-char #\Newline  *history*)
			  (write-char #\Newline *history*))
	(multiple-value-setq (moved-ok world) 
	    	(safely-make-move old-world world 
		       (act-type next-act) cycle-length))
	(when moved-ok 
	      (setf (agent-intentions mind)
		    (remove-intentions (agent-intentions mind) 
				       (list next-act)))
	      (setf (agent-removed-intentions mind)
		    (cons next-act (agent-removed-intentions mind))))
	(if (not moved-ok)
	    (kill-all-plans mind))
	(when keephistory (write "MIND AFTER ACT:  " :stream *history*)
	                  (write-char #\Newline *history*)
	                  (write-mind mind *history*)
			  (write-char #\Newline  *history*))
	(when keephistory (write "HOLES NOW AROUND:  "  :stream *history*)
	                  (write-holes (tw-hole-list world) *history*)
			  (write-char #\Newline *history*))
	(perceive mind old-world world)
	(when keephistory (write "MIND AFTER PERCEIVE:  " :stream *history*)
	                  (write-char #\Newline *history*)
	                  (write-mind mind *history*)
			  (write-char #\Newline  *history*)))
    (values world mind)))


(defun safely-make-move (old-world world move delta-t)
    (let ((stepres (tw-step world nil move delta-t))
	  fellin)
	 (when (null (tw-agent-list world))
	     (when keephistory 
		   (write "FELL INTO A HOLE!" :stream *history*)
		   (write-char #\Newline  *history*))
	     (setf fellin t)
	     (setf world (tw-copy-world old-world)))
	 (values (and stepres (not fellin)) world)))




;Since we're simulating concurrent acting and thinking, after one iteration
;of both processes, we've got to merge the resulting mental states.
;Thinking removes options and adds intentions; acting removes intentions
;which may or may not have succeeded (they may contain the 'failed' slot,
;in which case we shouldn't remove an empty parent intention, since we may
;want to try again);
;perceiving, which occurs during the act cycle, removes intentions and options
;and also adds options
;Returns:  --"t"(irrelevant):  mutates "mind"  (arg1)




(defun merge-minds (mind think-res act-res)
  (let ( (mind-int (acts-copy (agent-intentions mind))) 
	 (think-int (acts-copy (agent-intentions think-res)))
	 (act-int (acts-copy (agent-intentions act-res)))
	 (mind-opt (opts-copy (agent-options mind)))
	 (think-opt (opts-copy (agent-options think-res)))
	 (act-opt (opts-copy (agent-options act-res))))
    (setf (agent-intentions mind)
	  (remove-intentions think-int
		    (agent-removed-intentions act-res)))
    (setf (agent-options mind)
	  (remove-options act-opt
			  (act-list
			     (remove-options mind-opt (act-list think-opt)))))
  (when keephistory (write "MERGE: ACT-INT, ACT-REM'D:  " :stream *history*)
	            (write-char #\Newline *history*)
		    (write "Intentions:" :stream *history*)
		    (write-char #\Newline *history*)
		    (write-intentions (agent-intentions act-res) *history*)
		    (write-char #\Newline *history*)
		    (write "Removed-intentions:" :stream *history*)
		    (write-char #\Newline *history*)
		    (write-intentions (agent-removed-intentions act-res) *history*)
		    (write-char #\Newline *history*))

  (when keephistory (write "MIND AFTER MERGE:  " :stream *history*)
	            (write-char #\Newline *history*)
                    (write-mind mind *history*)
	            (write-char #\Newline  *history*)
  

))
  t)


;Routines for removing options and intentions.  Each takes two 
;lists:  a list of options or intentions, and a list of acts, and returns
;a list of options or intentions consisting of those in arg1 with objects
;from arg2 removed.
;Return:  --the resulting list of options (intentions)

(defun remove-options (opts acts)
  (if acts
      (remove-options (remove-option opts (first acts)) (rest acts))
  opts))
  

(defun remove-option (opts act)
  (mapcan #'(lambda (x) (remove-option-top x act)) opts))

(defun remove-option-top (opt act)
  (setf (option-set-alt-acts opt)
	(mapcan #'(lambda (x) (remove-option-from-tree x act)) 
		(option-set-alt-acts opt)))
  (if (option-set-alt-acts opt) 
      (list opt)
      nil))

(defun remove-option-from-tree (act1 act2)
  (if (same-act act1 act2)
      nil
      (if (act-subacts act1)
	  (progn (setf (act-subacts act1)
		       (mapcan #'(lambda (x) (remove-option-from-tree x act2))
			       (act-subacts act1)))
		 (if (act-subacts act1)
		 (list act1)
	         nil))
	   (list act1))))

(defun remove-intentions (ints acts)
   (if acts
       (remove-intentions (remove-act-int ints (first acts)) 
	   		  (rest acts))
   ints))


(defun remove-act-int (ints act)
  (mapcan #'(lambda (x) (remove-act-int-tree x act)) ints))

(defun remove-act-int-tree (int act)
  (if (same-act int act)
      nil
      (if (act-subacts int)
	 (progn
	    (setf (act-subacts int)
		  (mapcan #'(lambda (x) (remove-act-int-tree x act))
			  (act-subacts int)))
	    (if (or (act-failed act)
		    (act-subacts int))
		(list int)    ; If the 'failed' slot of the act = t, keep
			      ;   the parent even if it becomes empty.
	        nil))
	 (list int))))


(defun same-act (act1 act2)
  (and (eq (act-type act1) (act-type act2))
       (null (set-difference
	           (act-parms act2) 
		   (act-parms act1)
		   :test
		   #'equalp))))
                         ;act1 counts as the same act as act2 if its parms
                         ;subsume those of act2

;Perceive environmental changes--both new options (new holes that have
;appears), and lost options and intentions (holes that have disappeared).
; *Future:  generalize to deal with other kinds of environmental changes,
;  e.g., new tiles
;Returns:  "t" (irrelevant) --mutates arg1
; 
; modification: if 'all-new' is true, return all holes


(defun perceive (mind old-world new-world)
  (let ( (new-holes (set-difference (tw-hole-list new-world)
				      (tw-hole-list old-world)
				      :key 'obj-name))
	 (lost-holes (set-difference (tw-hole-list old-world)
				     (tw-hole-list new-world)
				     :key 'obj-name)))
    (when new-holes (setf (agent-options mind)
			  (append (agent-options mind)
				  (create-new-options new-holes))))
    (when lost-holes (let ((lost-opps (create-lost-opps lost-holes)))
		       (when keephistory
			(write "Lost-opps:" :stream *history*)
			(write-char #\Newline *history*)
			(write-each-act lost-opps *history*)
			(write-char #\Newline *history*))
		       (setf (agent-options mind)
			     (remove-options (agent-options mind)
					  lost-opps))
		       (setf (agent-intentions mind)
			   (remove-intentions (agent-intentions mind)
					      lost-opps))
		       (setf (agent-removed-intentions mind)
			     (append lost-opps
			         (agent-removed-intentions mind)))))
    t))


;Create a list of top-level (hole-filling) option sets; each set contains 
;two alt-acts:  one to do the fill now, and one to do it later
; * Future:  create other types of new options
;Returns:  such a list

(defun create-new-options (hole-list)
  (mapcar #'create-new-option hole-list))

(defun create-new-option (hole)
  (make-option-set 
           :purpose 'self
	   :alt-acts (list (make-act :id (make-name 'act)
				     :type 'fill
				     :parms (list (make-parm :name 'hole
							     :binding (obj-name hole))
						  (make-parm :name 'score
							     :binding (obj-score hole)))
				     :time 'now)
			   (make-act :id (make-name 'act)
				     :type 'fill
				     :parms (list (make-parm :name 'hole
							     :binding (obj-name hole))
						  (make-parm :name 'score
							     :binding (obj-score hole)))
				     :time 'later))))

;Create a list of actions representing the holes that used to be available
;for filling, but have now disappeared
; *Future:  create other types of lost opportunities

(defun create-lost-opps (hole-list)
  (mapcar #'create-lost-opp hole-list))

(defun create-lost-opp (hole)
  (make-act :type 'fill
	    :parms (list (make-parm :name 'hole
			            :binding (obj-name hole)))))



;Subroutine to determine the binding of arg1 in the parms list that is
;arg2.  Returns the binding.

(defun binding (name parms)
  (cond ( (null parms) nil)
	( (eq (parm-name (first parms)) name) (parm-binding (first parms)))
	( t (binding name (rest parms)))))


;Find the next atomic action to perform.  If there's a currently active
;intention, find the next available atomic subpart.  Otherwise, return
;the null action.
;Returns:  an atomic action to perform

(defun select-next-act (mind)
  (let ( (current (current-int (agent-intentions mind))))
    (cond ( (and current 
		 (find-base-level current)))
	  ( (make-null-act)))))

(defun current-int (intentions)
  (cond ( (null intentions) nil)
	( (eq (act-time (first intentions)) 'now) (first intentions))
	(t (current-int (rest intentions)))))

(defun find-base-level (int)
  (cond ( (act-atomic int) int)
	( (act-subacts int) (find-base-level (first (act-subacts int))))))

; Kill all plans (our action failed)

(defun kill-all-plans (mind)
    (let (flat)
	(dolist (i (agent-intentions mind))
	    (when (act-subacts i)
		(dolist (subi (act-subacts i))
		    (dolist (subsubi (act-subacts subi))
		       (setf flat (cons subsubi flat)))
		    (setf flat (append flat (list subi))))
		(setf (act-subacts i) nil)))
	(dolist (i flat)
	    (setf (act-failed i) t))
	(setf (agent-removed-intentions mind) 
	      (append (agent-removed-intentions mind) flat))
	(setf (agent-intentions mind) 
	      (remove-intentions (agent-intentions mind) flat))))


;Engage in deliberation if there are any available options.  Otherwise do
;some planning.
;Returns: --  t if something useful was done; nil if nothing was.
;         --  mutates mind (arg1)

(defun deliberate-first (world mind)
    (cond ( (agent-options mind) (deliberate mind world) t)
	  ( (not (current-int (agent-intentions mind))) (make-current mind))
	  ( (something-to-plan mind world) (expand-plans mind world) t)
	( t nil)))

; something to plan if there's a current intention and it's not fully planned
(defun something-to-plan (mind world)
  (let ((curint (current-int (agent-intentions mind))))
    (if curint
	(or (not (act-subacts curint))
            (eq (act-type (car (last (act-subacts curint)))) 'rest)))))

(defun make-current (mind)
    (let ((best -1) whichbest this)
        (when (agent-intentions mind)
	    (dolist (i (agent-intentions mind))
		 (setf this (binding 'score (act-parms i)))
		 (when (> this best)
		     (setf best this)
		     (setf whichbest i)))
	    (setf (act-time whichbest) 'now))))

;Perform a deliberation step:  Choose some option set to consider,
;filter incompatible options excepting those that trigger an override,
;and then evaluate those that remain.
;Returns: -- "t" (irrelevant):  mutates mind (arg2) 

(defun deliberate (mind world)
  (let ( (option (select-option-set (agent-options mind))))
    (setf (agent-options mind) (remove option (agent-options mind)))
                 ;directly remove the options you're deliberating on here
   (evaluate-options (filter (option-set-alt-acts option)
				    mind)
		     (option-set-purpose option)
		     mind
		     world))
    (when keephistory (write "MIND AFTER DELIBERATE:  " :stream *history*)
	              (write-char #\Newline *history*)
                      (write-mind mind *history*)
	              (write-char #\Newline  *history*))
  t)

; currently: choose the best 'self' option-set first.  If no 'self's, just
; take the first.
(defun select-option-set (options)
    (let ((bestval -1) bestset)
        (dolist (optset options)	;  choose the
	    (when (and (eq (option-set-purpose optset) 'self)
		     (option-set-alt-acts optset))
		(setf thisval (binding 'score (act-parms (first 
				   (option-set-alt-acts optset)))))
		(when (> thisval bestval)
		    (setf bestval thisval)
		    (setf bestset optset))))
	(if bestset bestset (first options))))

(defun filter (options mind)
   (mapcan #'(lambda (x) (if (or (compatible x mind)
			          (override x mind))
			     (list x)))
	  options))

(defun compatible (option mind)
  (case (act-type option) 
	(fill (compat-fill option (agent-intentions mind)))
;	(push (compat-push option (agent-intentions mind)))
	(otherwise (list option))))     ;anything else is compatible

(defun compat-fill (option ints)
  (if (eq (act-time option) 'now)
	(if (current-fill ints)
	    nil
	    t)
    t))

(defun current-fill (ints)
  (cond ( (null ints) nil)
	(  (and (eq (act-time (first ints)) 'now)
		(eq (act-type (first ints)) 'fill))  t)
	(t (current-fill (rest ints)))))

;(defun conflicts (parm binding intentions)
;  (cond ( (null intentions) nil)
;	( (eq (binding parm (act-parms (first intentions))) binding) nil)
;	( t (conflicts parm binding (rest intentions)))))
;
;(defun compat-push (option mind)
;  (let ( (intentions (agent-intentions mind))
;	 (tile (binding 'tile (act-parms option))))
;    (if (conflicts 'tile tile intentions) nil (list option))))

(defun override (x y) nil)  ;stub

(defun evaluate-options (options purpose mind world)
  (when options
	(let ((new-int (choose-best-option options mind)))
	  (if new-int
	      (setf (agent-intentions mind)
		    (insert-new-int new-int purpose (agent-intentions mind)))
	      (setf (agent-options mind)
		    (append (agent-options mind)
		    (list (make-option-set :purpose purpose
					     :alt-acts options))))))))


(defun choose-best-option (options mind)
    (when options
	  (cond ( (find-now-option options))
		(t (first options)))))

(defun find-now-option (options)
  (cond ( (null options) nil)
	( (eq (act-time (first options)) 'now) (first options))
	( t (find-now-option (rest options)))))

(defun insert-new-int (int purpose ints)
  (if (eq purpose 'self)
      (cons int ints)
      (attach-new-int int purpose ints)))

(defun attach-new-int (int purpose ints)
  (cond ( (eq (act-id (first ints)) purpose)
	    (setf (act-subacts (first ints)) int))
	( (and (act-subacts (first ints))
	       (attach-new-int int purpose (act-subacts (first ints)))))
	(t (attach-new-int int purpose (rest ints))))
  ints)

(defun expand-plans (mind world)
  (let ( (selected (select-int (agent-intentions mind))))
    (when selected
	  (if (and (eq (act-type selected) 'fill)
    		   (binding 'hole (act-parms selected))
		   (or (null (act-subacts selected))
		       (eq (act-type (first (last (act-subacts selected)))) 'rest)))
	      (expand-fill selected world)))
    (when keephistory (write "MIND AFTER PLAN:  " :stream *history*)
	              (write-char #\Newline *history*)
                      (write-mind mind *history*)
                      (write-char #\Newline  *history*))))
    
(defun select-int (ints)
  (cond ((null ints) nil)
	( (and (eq (act-type (first ints)) 'fill) 
	       (eq (act-time (first ints)) 'now))
	  (first ints))    ;NOT REALLY RIGHT:  should plan for later
                           ;intentions when there's nothing to do now
	(t (select-int (cdr ints)))))

(defun move-sequence ()
  (list (make-act :id (make-name 'act)
		  :type 'L
		  :atomic 't)
	(make-act :id (make-name 'act)
		  :type 'R
		  :atomic 't)))

(defun make-null-act ()
  (make-act :id (make-name 'act)
	    :type 'N
	    :atomic 't))


;(defmacro set-diff-structs (structs1 structs2 key)
;  `(set-difference ,structs1
;		   ,structs2
;		   :test
;		   #'(lambda (x y) (eq (,key x) (,key y)))))

;(defmacro union-structs (structs1 structs2 key)
;  `(union ,structs1
;	  ,structs2
;	  :test
;	  #,(lambda (x y) (eq (,key x) (,key y)))))


(defun write-mind (mind stream)
  (write "Options:"  :stream stream)
  (write-char #\Newline stream)
  (write-options (agent-options mind) stream)
  (write-char #\Newline stream)
  (write-char #\Newline stream)
  (write "Intentions:" :stream stream)
  (write-char #\Newline stream)
  (write-intentions (agent-intentions mind) stream)
  (write-char #\Newline stream)
  (write-char #\Newline stream))

(defun write-options (options stream)
  (when options
	(write "  Purpose:  " :stream stream)
	(write (option-set-purpose (first options)) :stream stream)
	(write-char #\Newline stream)
	(write-each-act (option-set-alt-acts (first options)) stream)
	(write-char #\Newline stream)
	(write-options (rest options) stream)))


(defun write-each-act (acts stream)
  (when acts
	(write "    Id:  " :stream stream)
	(write (act-id (first acts)) :stream stream)
	(write-char #\Newline stream)
	(write "    Type:  " :stream stream)
	(write (act-type (first acts)) :stream stream)
	(write-char #\Newline stream)
	(write "    Parms:  " :stream stream)
	(write (act-parms (first acts)) :stream stream)
	(write-char #\Newline stream)
	(write "    Time:  " :stream stream)
	(write (act-time (first acts)) :stream stream)
	(write "    Subacts:  "  :stream stream)
	(write (mapcar #'act-id (act-subacts (first acts))) :stream stream)
	(write-char #\Newline stream)
	(write-char #\Newline stream)
	(write-each-act (rest acts) stream)))
	       
  
(defun write-intentions (ints stream)
  (when ints
	(write-each-act ints stream)))


(defun write-holes (hole-list stream)
  (write (mapcar #'obj-name hole-list) :stream stream))

(defun agent-copy (agent)
  (let ( (new-agent (copy-agent agent)))
    (setf (agent-intentions new-agent) (acts-copy (agent-intentions agent)))
    (setf (agent-options new-agent) (opts-copy (agent-options agent)))
    new-agent))

(defun acts-copy (acts)
  (when acts (mapcar #'act-copy acts)))

(defun act-copy (act)
   (let ( (new-act (copy-act act)))
     (setf (act-parms new-act) (copy-tree (act-parms act)))
     (setf (act-subacts new-act) (acts-copy (act-subacts act)))
;  reset enabling conds too
    new-act))

(defun opts-copy (opts)
  (when opts (mapcar #'opt-copy opts)))

(defun opt-copy (opt)
  (let ( (new-opt (copy-option-set opt)))
    (setf (option-set-purpose new-opt) (option-set-purpose opt))
    (setf (option-set-alt-acts new-opt) (acts-copy (option-set-alt-acts opt)))
    new-opt))

(defun act-list (opts)
  (cond ( (null opts) nil)
	(t (append (option-set-alt-acts (first opts))
		   (act-list (rest opts))))))

(defun specify-agent-parms ()
  (make-agent
   :think-strategy 'deliberate-first
   :act-time 200
   :wait-time 1000))

(setf a0 (specify-agent-parms))

(setf keephistory t)
