;  AGENT
;  Routines for having an agent perform in a tileworld

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  DO A TILEWORLD EXPERIMENT                                           ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; Initialize a world and an agent, and then play a tileworld game.
; Returns: -- score obtained.

;(defun tw-experiment ()
;  (if *display-on* (twinitdisp))
;  (tw-init w0)
;  (agent-init a0 w0)
;  (play-tw-game w0 a0 *game-length*)
;  (if *display-on* (twclosedisp))
;  (tw-score w0))

(defun tw-exp (&optional seed)
   (setf (tw-rand-seed w0) seed)
   (if *display-on* (twinitdisp))
   (tw-init w0)
   (agent-init a0 w0)
   (play-tw-game w0 a0 *game-length*)
   (if *display-on* (twclosedisp))
   (tw-score w0))

(defun tw-redo ()
   (setf (tw-rand-seed w0) (tw-rand-seed-was w0))
   (if *display-on* (twinitdisp))
   (tw-init w0)
   (agent-init a0 w0)
   (setf w0 (play-tw-game w0 a0 *game-length*))
   (if *display-on* (twclosedisp))
   (tw-score w0))

(defun agent-init (mind world)
  (setf (agent-options mind) 
	(create-new-options (tw-hole-list world)))
  (setf (agent-intentions mind) nil)
  (setf (agent-removed-intentions mind) nil))
  
;Play a tileworld game:  Simulate concurrent acting and thinking by
;first performing one think step, keeping track of how long you take,
;and then acting for the same period of time, allowing the world to
;change then as well.  Because of the concurrency, the act cycle makes 
;use of the initial state of mind:  at the end of both acting and thinking
;the resulting changes to the agent's mental state are combined.  Note
;that the world only changes during the act cycle, so it doesn't need
;to be copied and then updated to simulate concurrency.
;Mutates:  --arg1 and arg2 (world and mind)
;Returns:  --world that results from playing the game

(defun play-tw-game (world mind *game-length*)
  (with-open-file
   (*history*  *historyfile* :direction :output
	      :if-exists :rename)
  (let ( think-time cycles-to-act mind1 mind2)
    (keephist  "HOLES INITIALLY AROUND:"  'write-holes (tw-hole-list world))
    (keephist "INITIAL MENTAL STATE:" 'write-mind  mind)
    (do ( (elapsed 0 (+ elapsed think-time)))
	( (>= elapsed *game-length*) world)
	(setf mind1 (agent-copy mind))
	(setf mind2 (agent-copy mind))
	(setf think-time (think world mind1))
	(keephist "THINK TIME USED:"  'write-plain think-time)
	(setf cycles-to-act (- (floor (/ (min (+ elapsed think-time)
					      *game-length*)
				         (agent-act-time mind)))
			        (floor (/ elapsed
				          (agent-act-time mind)))))
 	(multiple-value-setq (world mind2)
 	    		     (act world mind2 cycles-to-act))
	(merge-minds mind mind1 mind2))
    world))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; THINK CYCLE                                                          ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Perform one think step.  The strategy for determining what this step
;will be depends upon the agent's "think-strategy", chosen at initialization
;time.  Keep track of the amount of time (in computer msec's) used.
;Mutates:  arg2 (mind)
;Returns:  amount of time used

(defun think (world mind)
  (let (start-time stop-time work-done think-time)
    (setf start-time (get-internal-run-time))
    (setf work-done (funcall (agent-think-strategy mind) world mind))
    (setf stop-time (get-internal-run-time))
    (setf think-time (if work-done
			 (/ (- stop-time start-time) 1000)  ; milliseconds used
		         (agent-wait-time mind))) ; else pretend it took longer
    think-time))


;Engage in deliberation if there are any available options.  Otherwise do
;some planning.
;Returns: --  t if something useful was done; nil if nothing was.
;         --  mutates mind (arg1)

(defun deliberate-first (world mind)
  (cond ( (or (agent-options mind)
	      (and (agent-intentions mind)
	            (not (current-int (agent-intentions mind)))))
         (deliberate world mind)  t)
	( (something-to-plan world mind) (expand-plans mind world) t)
	(t nil)))


; something to plan if there's a current intention and it's not fully planned
(defun something-to-plan (world mind)
  (let ((curint (current-int (agent-intentions mind))))
    (if curint
	(or (not (act-subacts curint))
            (eq (act-type (car (last (act-subacts curint)))) 'rest)))))


; select the best option and make it the current intention
;(defun make-current (mind)
;    (let ((best -1) whichbest this)
;        (when (agent-intentions mind)
;	    (dolist (i (agent-intentions mind))
;		 (setf this (binding 'score (act-parms i)))
;		 (when (> this best)
;		     (setf best this)
;		     (setf whichbest i)))
;	    (setf (act-time whichbest) 'now))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; THINK CYCLE---DELIBERATION                                           ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun deliberate (world mind)
  (let ( (curint (current-int (agent-intentions mind))))
    (setf mval mind)
    (setf wval world)
    (evaluate (filter mind curint) curint mind)
    (setf (agent-options mind) nil)
    (keephist "MIND AFTER DELIBERATE:  " 'write-mind mind)))

(defun filter (mind curint)
  (let ( (opts (make-opt-list (agent-options mind))))
;    (mapcan #'(lambda (x) (when (override x curint) (list x)))
;	    opts)))
   opts))

(defun make-opt-list (options)
  (if (null options)
      nil
      (append (option-set-alt-acts (first options))
	       (make-opt-list (rest options)))))

(defun override (option curint)
  (if curint
    (>= (- (binding 'score (act-parms option))
	   (binding 'score (act-parms curint)))
        *threshhold*)
    t))

(defun evaluate (options curint mind)
  (if curint
      (let* ( (maxopt (findmax options))
	      (restopts (allbut options maxopt))
	     (maxoptval (if maxopt (binding 'score (act-parms maxopt))
			    0))
	     (curintval (if curint (binding 'score (act-parms curint))
			    0)))
	(if (>= curintval (- maxoptval *threshhold*))
	    (make-opts-later-ints options mind)
	    (and (make-current-int maxopt mind)
		 (make-opts-later-ints restopts mind)
		 (demote-to-later-int curint)
		 (kill-all-subplans mind))))
    (let* ( (maxopt (findmax options))
	    (restopts (allbut options maxopt))
	   (maxoptval (if maxopt (binding 'score (act-parms maxopt))
			  0))
	   (maxint (findmax (agent-intentions mind)))
	   (maxintval (if maxint (binding 'score (act-parms maxint))
			  0)))
      (if (> maxoptval maxintval)
	  (and (make-current-int maxopt mind)
	       (make-opts-later-ints restopts mind))
	  (and (promote-to-current-int maxint)
	       (make-opts-later-ints options mind))))))

(defun findmax (acts)
  (let ( (bestval -1) (restacts acts) thisval new best)
    (do () ((null restacts))
	(setf new (first restacts))
	(setf thisval (binding 'score (act-parms new)))
	(when (> thisval bestval) 
	      (setf best new)
	      (setf bestval thisval))
	(setf restacts (rest restacts)))
  best))

(defun promote-to-current-int (int)
  (setf (act-time int) 'now)
  (setf (act-status1 int) 'intention)
  (setf (act-status2 int) 'to-be-planned))

(defun demote-to-later-int (int)
  (setf (act-time int) 'later)
  (setf (act-status1 int) 'intention))

(defun make-current-int (opt mind)
  (setf (act-time opt) 'now)
  (setf (act-status1 opt) 'intention)
  (setf (act-status2 opt) 'to-be-planned)
  (setf (agent-intentions mind) (cons opt (agent-intentions mind))))

(defun make-opts-later-ints (opts mind)
  (mapcar #'(lambda (x) 
	      (setf (act-time x) 'later)
	      (setf (act-status1 x) 'intention))
	  opts)
  (setf (agent-intentions mind) (append (agent-intentions mind) opts)))

(defun allbut (list elt)
  (cond ( (null elt) list)
	( (eq (first list) elt) (rest list))
	(t (cons (first list) (allbut (rest list) elt)))))
	
; Kill all subplans (our action failed or we changed goals).

(defun kill-all-subplans (mind)
    (let (flat)
	(dolist (i (agent-intentions mind))
	    (when (act-subacts i)
		(dolist (subi (act-subacts i))
		    (dolist (subsubi (act-subacts subi))
		       (setf flat (cons subsubi flat)))
		    (setf flat (append flat (list subi))))
		(setf (act-subacts i) nil)))
	(dolist (i flat)
	    (setf (act-failed i) t))
	(setf (agent-removed-intentions mind) 
	      (append (agent-removed-intentions mind) flat))
	(setf (agent-intentions mind) 
	      (remove-intentions (agent-intentions mind) flat))))

;Perform a deliberation step:  Choose some option set to consider,
;filter incompatible options excepting those that trigger an override,
;and then evaluate those that remain.
;Returns: -- "t" (irrelevant):  mutates mind (arg2) 

;(defun deliberate (mind world)
;  (let ( (option (select-option-set (agent-options mind))))
;    (setf (agent-options mind) (remove option (agent-options mind)))
;                 ;directly remove the options you're deliberating on here
;   (evaluate-options (filter (option-set-alt-acts option)
;				    mind)
;		     (option-set-purpose option)
;		     mind
;		     world))
;  (keephist "MIND AFTER DELIBERATE:  " 'write-mind mind)
;  t)


; currently: choose the best 'self' option-set first.  If no 'self's, just
; take the first.
;(defun select-option-set (options)
;    (let ((bestval -1) bestset)
;        (dolist (optset options)	;  choose the
;	    (when (and (eq (option-set-purpose optset) 'self)
;		     (option-set-alt-acts optset))
;		(setf thisval (binding 'score (act-parms (first 
;				   (option-set-alt-acts optset)))))
;		(when (> thisval bestval)
;		    (setf bestval thisval)
;		    (setf bestset optset))))
;	(if bestset bestset (first options))))
;
;(defun filter (options mind)
;   (mapcan #'(lambda (x) (if (or (compatible x mind)
;			          (override x mind))
;			     (list x)))
;	  options))
;
;(defun compatible (option mind)
;  (case (act-type option) 
;	(fill (compat-fill option (agent-intentions mind)))
;;	(push (compat-push option (agent-intentions mind)))
;	(otherwise (list option))))     ;anything else is compatible
;
;(defun compat-fill (option ints)
;  (if (eq (act-time option) 'now)
;	(if (current-fill ints)
;	    nil
;	    t)
;    t))
;
;(defun current-fill (ints)
;  (cond ( (null ints) nil)
;	(  (and (eq (act-time (first ints)) 'now)
;		(eq (act-type (first ints)) 'fill))  t)
;	(t (current-fill (rest ints)))))

;(defun conflicts (parm binding intentions)
;  (cond ( (null intentions) nil)
;	( (eq (binding parm (act-parms (first intentions))) binding) nil)
;	( t (conflicts parm binding (rest intentions)))))
;
;(defun compat-push (option mind)
;  (let ( (intentions (agent-intentions mind))
;	 (tile (binding 'tile (act-parms option))))
;    (if (conflicts 'tile tile intentions) nil (list option))))
;
;(defun override (options mind)
;  (>= (- (binding 'score (act-parms options))
;	(binding 'score (act-parms (current-int (agent-intentions mind)))))
;     *threshhold*))

;(defun evaluate-options (options purpose mind world)
;  (when options
;	(let ((new-int (choose-best-option options mind)))
;	  (if new-int
;	      (setf (agent-intentions mind)
;		    (insert-new-int new-int purpose (agent-intentions mind)))
;	      (setf (agent-options mind)
;		    (append (agent-options mind)
;		    (list (make-option-set :purpose purpose
;					     :alt-acts options))))))))

;(defun choose-best-option (options mind)
;  (fix) (break)
;    (when options
;;	  (cond ( (find-now-option options))
;		(t (first options)))))

;(defun find-now-option (options)
;  (cond ( (null options) nil)
;	( (eq (act-time (first options)) 'now) (first options))
;	( t (find-now-option (rest options)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; DELIBERATION -- PLAN                                                 ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun expand-plans (mind world)
  (let ( (selected (select-int (agent-intentions mind))))
    (when selected
	  (if (and (eq (act-type selected) 'fill)
    		   (binding 'hole (act-parms selected))
		   (or (null (act-subacts selected))
		       (eq (act-type (first (last (act-subacts selected)))) 'rest)))
	      (expand-fill selected world)))
    (keephist "MIND AFTER PLAN:  " 'write-mind mind)))

    
(defun select-int (ints)
  (cond ((null ints) nil)
	( (and (eq (act-type (first ints)) 'fill) 
	       (eq (act-time (first ints)) 'now))
	  (first ints))    ;NOT REALLY RIGHT:  should plan for later
                           ;intentions when there's nothing to do now
	(t (select-int (cdr ints)))))

(defun move-sequence ()
  (list (make-act :id (make-name 'act)
		  :type 'L
		  :atomic 't)
	(make-act :id (make-name 'act)
		  :type 'R
		  :atomic 't)))

(defun make-null-act ()
  (make-act :id (make-name 'act)
	    :type 'N
	    :atomic 't))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ACT/PERCEIVE CYCLE                                                   ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Act for the same amount of time as you just thought for.  If act
;is successful, remove the satisfied intention from the mental state.
;After attempting an act, note any changes that occurred in the world
;while you were acting.
;Mutates:  arg1 and arg2 (world and mind)

(defun act (world mind cycles)
  (let* ( (cycle-length (agent-act-time mind))
	  old-world success next-act moved-ok)
    (dotimes (i cycles mind)
	(setf old-world (tw-copy-world world))
	(setf next-act (select-next-atomic-act mind))
	(keephist "SELECTING ACT:  " 'write-each-act (list next-act))
 	(multiple-value-setq (moved-ok world) 
 	    	(safely-make-move old-world world 
 		       (act-type next-act) cycle-length))
 	(when moved-ok 
	      (setf (agent-intentions mind)
		    (remove-intentions (agent-intentions mind) 
				       (list next-act)))
	      (setf (agent-removed-intentions mind)
		    (cons next-act (agent-removed-intentions mind))))
	(if (not moved-ok)
	    (kill-all-subplans mind))
	(keephist "MIND AFTER ACT:  " 'write-mind mind)
	(keephist "HOLES NOW AROUND:  "  'write-holes (tw-hole-list world))
	(perceive mind old-world world)
	(keephist "MIND AFTER PERCEIVE:  " 'write-mind mind)))
    (values world mind))

(defun safely-make-move (old-world world move delta-t)
     (let ((stepres (tw-step world nil move delta-t))
 	  fellin)
	 (when (null (tw-agent-list world))
	     (when *keephistory* 
		   (write "FELL INTO A HOLE!" :stream *history*)
		   (write-char #\Newline  *history*))
	     (setf fellin t)
	     (setf world (tw-copy-world old-world)))
	 (values (and stepres (not fellin)) world)))



;Perceive environmental changes--both new options (new holes that have
;appears), and lost options and intentions (holes that have disappeared).
; *Future:  generalize to deal with other kinds of environmental changes,
;  e.g., new tiles
; Mutates arg1 (mind)

(defun perceive (mind old-world new-world)
  (let ( (new-holes (set-difference (tw-hole-list new-world)
				      (tw-hole-list old-world)
				      :key 'obj-name))
	 (lost-holes (set-difference (tw-hole-list old-world)
				     (tw-hole-list new-world)
				     :key 'obj-name)))
    (when new-holes (setf (agent-options mind)
			  (append (agent-options mind)
				  (create-new-options new-holes))))
    (when lost-holes (let ((lost-opps (create-lost-opps lost-holes)))
		       (keephist "LOST-OPPS:  "  'write-each-act lost-opps)
		       (setf (agent-options mind)
			     (remove-options (agent-options mind)
					  lost-opps))
		       (setf (agent-intentions mind)
			   (remove-intentions (agent-intentions mind)
					      lost-opps))
		       (setf (agent-removed-intentions mind)
			     (append lost-opps
			         (agent-removed-intentions mind)))))))


;Find the next atomic action to perform.  If there's a currently active
;intention, find the next available atomic subpart.  Otherwise, return
;the null action.
;Returns:  an atomic action to perform

(defun select-next-atomic-act (mind)
  (let ( (current (current-int (agent-intentions mind))))
    (cond ( (and current 
		 (find-base-level current)))
	  ( (make-null-act)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; MERGE RESULTS OF THINK AND ACT CYCLES                                ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;Since we're simulating concurrent acting and thinking, after one iteration
;of both processes, we've got to merge the resulting mental states.
;Thinking removes options and adds intentions; acting removes intentions
;which may or may not have succeeded (they may contain the 'failed' slot,
;in which case we shouldn't remove an empty parent intention, since we may
;want to try again);
;perceiving, which occurs during the act cycle, removes intentions and options
;and also adds options
;Mutates: --arg1 (mind)

(defun merge-minds (mind think-res act-res)
  (let ( (new-ints-think (acts-copy (agent-intentions think-res)))
	 (old-opts (opts-copy (agent-options mind)))
	 (think-opts (opts-copy (agent-options think-res)))
	 (act-opts (opts-copy (agent-options act-res))))
    (setf (agent-intentions mind)
	  (remove-intentions new-ints-think
		    (agent-removed-intentions act-res)))
    (setf (agent-options mind)
	  (remove-options act-opts
			  (act-list
			    (remove-options old-opts (act-list think-opts)))))
    (keephist "MIND AFTER MERGE:  "  'write-mind mind)))

