;  AGENT-ACCESS
;  Routines for creating, copying, removing, and examining the components
;   of an agent structure.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  CREATE A NEW OPTION SET                                             ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;Create a list of top-level (hole-filling) option sets; each set contains 
;two alt-acts:  one to do the fill now, and one to do it later
; * Future:  create other types of new options
;Returns:  such a list

(defun create-new-options (hole-list)
  (mapcar #'create-new-option hole-list))

(defun create-new-option (hole)
  (make-option-set 
           :purpose 'self
	   :alt-acts (list (make-act :id (make-name 'act)
				     :type 'fill
				     :status1 'option
				     :status2 'to-be-deliberated
				     :parms (list (make-parm :name 'hole
							     :binding (obj-name hole))
						  (make-parm :name 'score
							     :binding (obj-score hole)))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  CREATE LOST OPPORTUNITY LIST                                        ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Create a list of actions representing the holes that used to be available
;for filling, but have now disappeared
; *Future:  create other types of lost opportunities

(defun create-lost-opps (hole-list)
  (mapcar #'create-lost-opp hole-list))

(defun create-lost-opp (hole)
  (make-act :type 'fill
      	    :status1 'intention
	    :status2 'to-be-deleted
	    :parms (list (make-parm :name 'hole
			            :binding (obj-name hole)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  COPY AN AGENT STRUCTURE (OR PARTS OF ONE)                           ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun agent-copy (agent)
  (let ( (new-agent (copy-agent agent)))
    (setf (agent-intentions new-agent) (acts-copy (agent-intentions agent)))
    (setf (agent-options new-agent) (opts-copy (agent-options agent)))
    new-agent))

(defun acts-copy (acts)
  (when acts (mapcar #'act-copy acts)))

(defun act-copy (act)
   (let ( (new-act (copy-act act)))
     (setf (act-parms new-act) (copy-tree (act-parms act)))
     (setf (act-subacts new-act) (acts-copy (act-subacts act)))
;  reset enabling conds too
    new-act))

(defun opts-copy (opts)
  (when opts (mapcar #'opt-copy opts)))

(defun opt-copy (opt)
  (let ( (new-opt (copy-option-set opt)))
    (setf (option-set-purpose new-opt) (option-set-purpose opt))
    (setf (option-set-alt-acts new-opt) (acts-copy (option-set-alt-acts opt)))
    new-opt))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  REMOVE OPTIONS OR INTENTIONS FROM AN AGENT                          ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Routines for removing options and intentions.  Each takes two 
;lists:  a list of options or intentions, and a list of acts, and returns
;a list of options or intentions consisting of those in arg1 with objects
;from arg2 removed.
;Return:  --the resulting list of options (intentions)

(defun remove-options (opts acts)
  (if acts
      (remove-options (remove-option opts (first acts)) (rest acts))
  opts))
  

(defun remove-option (opts act)
  (mapcan #'(lambda (x) (remove-option-top x act)) opts))

(defun remove-option-top (opt act)
  (setf (option-set-alt-acts opt)
	(mapcan #'(lambda (x) (remove-option-from-tree x act)) 
		(option-set-alt-acts opt)))
  (if (option-set-alt-acts opt) 
      (list opt)
      nil))

(defun remove-option-from-tree (act1 act2)
  (if (same-act act1 act2)
      nil
      (if (act-subacts act1)
	  (progn (setf (act-subacts act1)
		       (mapcan #'(lambda (x) (remove-option-from-tree x act2))
			       (act-subacts act1)))
		 (if (act-subacts act1)
		 (list act1)
	         nil))
	   (list act1))))

(defun remove-intentions (ints acts)
   (if acts
       (remove-intentions (remove-act-int ints (first acts)) (rest acts))
   ints))


(defun remove-act-int (ints act)
  (mapcan #'(lambda (x) (remove-act-int-tree x act)) ints))

(defun remove-act-int-tree (int act)
  (if (same-act int act)
      nil
      (if (act-subacts int)
	 (progn
	    (setf (act-subacts int)
		  (mapcan #'(lambda (x) (remove-act-int-tree x act))
			  (act-subacts int)))
	    (if (or (act-failed act)
		    (act-subacts int))
		(list int)    ; If the 'failed' slot of the act = t, keep
			      ;   the parent even if it becomes empty.
	        nil))
	(list int))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; DETERMINE A PARM BINDING                                             ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;Subroutine to determine the binding of arg1 in the parms list that is
;arg2.  Returns the binding.

(defun binding (name parms)
  (cond ( (null parms) nil)
	( (eq (parm-name (first parms)) name) (parm-binding (first parms)))
	( t (binding name (rest parms)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; MISCELLANEOUS                                                        ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; Extract the current intention from an intention list


(defun current-int (intentions)
  (cond ( (null intentions) nil)
	( (eq (act-time (first intentions)) 'now) (first intentions))
	(t (current-int (rest intentions)))))

(defun find-base-level (int)
  (cond ( (act-atomic int) int)
	( (act-subacts int) (find-base-level (first (act-subacts int))))))


; Determine whether two acts are the same: act1 counts as the same act
;  as act2 if its parms subsume those of act2

(defun same-act (act1 act2)
  (and (eq (act-type act1) (act-type act2))
       (null (set-difference
	           (act-parms act2) 
		   (act-parms act1)
		   :test
		   #'equalp))))

; Extract the list of alternative acts from an option set

(defun act-list (opts)
  (cond ( (null opts) nil)
	(t (append (option-set-alt-acts (first opts))
		   (act-list (rest opts))))))


; Insert a new intention into an intention list: if it's a top-level 
; intention, just cons it onto the list.  Otherwise, attach it below
; the appropriate intention

(defun insert-new-int (int purpose ints)
  (if (eq purpose 'self)
      (cons int ints)
      (attach-new-int int purpose ints)))

(defun attach-new-int (int purpose ints)
  (cond ( (eq (act-id (first ints)) purpose)
	    (setf (act-subacts (first ints)) int))
	( (and (act-subacts (first ints))
	       (attach-new-int int purpose (act-subacts (first ints)))))
	(t (attach-new-int int purpose (rest ints))))
  ints)
