;
; tw.l - tile world simulator.  Marc Ringuette, 7/89
;
(defstruct 
  (tw 
   (:print-function (lambda (w s d) (tw-print-briefly w s d))))
    name
    ; knob settings
    rows
    cols
    slice
    is-automatic
    must-match-shapes
    hole-knobs
    obstacle-knobs
    tile-knobs
    agent-knobs
    ; other properties of world
    score
    elapsed-time
    rand-seed
    rand-seed-was
    rand-stepseed
    rand-subseed
    ; object-based world state (elements are 'obj's)
    hole-list
    obstacle-list
    tile-list
    agent-list
    ; grid-based world state (arrays [rows*cols])
    grid		    ; elements are chars
    grid-which		    ; elements are ptrs to obj's
    grid-which-hole	    ; elements are ptrs to obj's
)

(defun tw-print-briefly (the-world the-stream depth)
  (format the-stream 
		  "#s<TW ~a: ~d/~d>" 
		  (tw-name the-world)
		  (tw-rows the-world)
		  (tw-cols the-world)))

(defstruct obj
    type
    name
    row
    col
    rel-cells	; (list of 'cell's where top left is (0,0)).
    cells	; (list of 'cell's where top left is (row,col))
    score	; for holes only
    score-decay	; ms per point, nil means no decay
    timeout	; nil means no timeout.
    creation-time
)

(defstruct cell
    row
    col
    depth	; for cells of holes only
)

(defstruct knob
    ms-appear	    ; nil means no new ones appear
    ms-disappear    ; nil means they never disappear
    num		    ; a subknob
    cells	    ; a subknob
    timeout	    ; a subknob.  nil means no timeout.
    depth	    ; a subknob.
    score	    ; a subknob
    score-decay     ; a subknob.  If score-decay-is-relative, it's ms until
		    ;   the entire score decays.  Otherwise, it's ms until
    		    ;   the score decays by one point.  nil --> no decay.
    score-decay-is-relative		; flag, t or nil
)

(defstruct subknob
    start
    min
    max
    mean	; not implemented yet
    variance	; not implemented yet
    (distribution 'flat)
)

(defstruct move
    name
    delta-row
    delta-col
    agent-char
)

(defvar r (make-move :name 'r :delta-row 0 :delta-col 1 :agent-char #\>))
(defvar l (make-move :name 'l :delta-row 0 :delta-col -1 :agent-char #\<))
(defvar u (make-move :name 'u :delta-row -1 :delta-col 0 :agent-char #\^))
(defvar d (make-move :name 'd :delta-row 1 :delta-col 0 :agent-char #\v))
(defvar n (make-move :name 'n :delta-row 0 :delta-col 0 :agent-char #\a))

(defconstant agent-chars '(#\> #\< #\^ #\v #\a #\o))
(defconstant tile-chars '(#\T #\t))
(defconstant obstacle-chars '(#\#))
(defconstant hole-chars '(#\O #\o #\t))
(defconstant obstacle-or-hole-chars (append obstacle-chars hole-chars))
(defconstant obstacle-or-agent-chars (append obstacle-chars agent-chars))
(defconstant tile-or-agent-chars (append tile-chars agent-chars))
(defconstant bare-tile-char '#\T)
(defconstant bare-hole-char '#\O)
(defconstant obstacle-char '#\#)
(defconstant tile-on-hole-char '#\t)
(defconstant agent-on-hole-char '#\o)
(defconstant nothing-char '#\space)

(defvar *display-on* t)

; 
; tw-step: update the simulation of 'world' for 'delta-t' milliseconds
; and then make 'move' for 'agent'.
; 
(defun tw-step (world agent move delta-t)
    (let (moving-objects)
	(setf move (eval move))
	(when (null agent) 
	    (setf agent (first (tw-agent-list world)))
	    (when (null agent)
		(setf move n)))
	; update world for each time-slice crossed.
	(dotimes (i (- (floor (/ (+ (tw-elapsed-time world) delta-t)
			         (tw-slice world)))
		       (floor (/ (tw-elapsed-time world)
			         (tw-slice world)))))
	    (tw-update-timeouts world)
	    (if (tw-is-automatic world)
		(tw-toss-dice world)
		(tw-toss-loaded-dice world)))
        (setf (tw-elapsed-time world) (+ (tw-elapsed-time world) delta-t))
	; execute 'move'
        (when (not (or (eq move n)
		       (null agent)))
            (setf moving-objects (tw-propagate-move world agent move))
	    (tw-update-move world moving-objects move))
	(if (and agent
		 (or (eq move n)
		     (eq moving-objects nil)))
	    (tw-update-move world (list agent) n))
	; finish up
	(if *display-on* (tw-display world))
	(return-from tw-step (or (and (eq move n)
				      (tw-agent-list world))
				 moving-objects))))

; 
; decrease timeouts and remove hole if it disappears
; 
(defun tw-update-timeouts (world)
	    (setf (tw-hole-list world)
		  (tw-update-timeouts-2 (tw-hole-list world)
				        world))
	    (setf (tw-tile-list world)
		  (tw-update-timeouts-2 (tw-tile-list world)
				        world))
	    (setf (tw-agent-list world)
		  (tw-update-timeouts-2 (tw-agent-list world)
				        world))
	    (setf (tw-obstacle-list world)
		  (tw-update-timeouts-2 (tw-obstacle-list world)
				        world)))


(defun tw-update-timeouts-2 (objlist world)
	(mapcan #'(lambda(obj)
	  (let (newobj nobj)
	    (if (null (obj-timeout obj))
		(setf newobj (list obj))
		(progn
		          (if (> (obj-timeout obj) (- (tw-slice world) 1))
			      (setf (obj-timeout obj) (- (obj-timeout obj) 
						         (tw-slice world)))
			      (setf (obj-timeout obj) 0))
			  (if (> (obj-timeout obj) 0)
			      (setf newobj (list obj))
			      (progn (tw-remove-objects world (list obj))
				     (setf newobj nil)))))
	    (when newobj
	      (setf nobj (first newobj))
	      (if (null (obj-score-decay nobj))
		(setf newobj (list nobj))
		(let ((points (- (floor (/ (- (tw-elapsed-time world)
					      (obj-creation-time obj))
					   (obj-score-decay nobj)))
				 (floor (/ (- (tw-elapsed-time world) 
					      (obj-creation-time obj)
					      (tw-slice world))
					   (obj-score-decay nobj))))))
		     (if (< points (obj-score nobj))
			 (setf (obj-score nobj) 
			       (- (obj-score nobj) points))
			 (setf (obj-score nobj) 0))
		     (if (> (obj-score nobj) 0)
			      (setf newobj (list nobj))
			      (progn (tw-remove-objects world (list nobj))
				     (setf newobj nil))))))
	      newobj))		; return-value within mapcan
	    objlist))

; 
; agent-object is to go in direction 'move'.  Propagate its move to all
; pushable objects and form a list of all moving objects.  If an obstacle
; is encountered, the returned list is nil.
; 
(defun tw-propagate-move (world agent-object move)
    (let ((grid-mark (make-array (list (tw-rows world) (tw-cols world))
				 :initial-element nil))
	  moving-objects)
	(setf moving-objects
	    (tw-prop-2 (tw-grid world) (tw-grid-which world) grid-mark
	        (cell-row (first (obj-cells agent-object)))
	        (cell-col (first (obj-cells agent-object)))
	        (move-delta-row move) (move-delta-col move) agent-object))
	(if (eq moving-objects t)
	    nil
	    (remove-duplicates moving-objects))))

;
; Mark the cell at (row,col), propagate further, then add this object
; to the set of moving objects, possibly duplicating it, and return it.
;
(defun tw-prop-2 (grid grid-which grid-mark row col drow dcol this-agent)
    (let (partial more)
	(setf (aref grid-mark row col) t)
	; if an obstacle or other agent is in the way, return t
	(if (or (member (aref grid (+ row drow) (+ col dcol))
		         obstacle-chars)
		(and (member (aref grid (+ row drow) (+ col dcol))
		              agent-chars)
		     (not (eql (aref grid-which (+ row drow) (+ col dcol))
			       this-agent))))
	    (return-from tw-prop-2 t))
	; if what's in front of this cell is a tile or this-agent, do next
	(when (and (or (member (aref grid (+ row drow) (+ col dcol))
			       tile-chars)
		       (and (member (aref grid (+ row drow) (+ col dcol))
				    agent-chars)
			    (eql (aref grid-which (+ row drow) (+ col dcol))
				 this-agent)))
		   (not (aref grid-mark (+ row drow) (+ col dcol))))
	    (setf partial (tw-prop-2 grid grid-which grid-mark (+ row drow)
			      (+ col dcol) drow dcol this-agent))
	    (if (eq partial t) (return-from tw-prop-2 t)))
	; go through the other 3 directions & see if this object extends there
	(mapcar #'(lambda (i j)
		(when (not (and (= i drow) (= j dcol)))
		    (when (and (member (aref grid (+ row i) (+ col j))
				       tile-or-agent-chars)
			       (eql (aref grid-which (+ row i) (+ col j))
			            (aref grid-which row col))
			       (not (aref grid-mark (+ row i) (+ col j))))
			(setf more (tw-prop-2 grid grid-which grid-mark
			            (+ row i) (+ col j) drow dcol this-agent))
			(if (eq more t) 
			    (return-from tw-prop-2 t)
			    (setf partial (append partial more))))))
		'(-1 0 0 1)
		'(0 -1 1 0))
	; add the object at this cell to the partial set and return it
	(cons (aref grid-which row col) partial)))


; 
; update world to correspond to the moving-objects going in direction 'move'
; 
(defun tw-update-move (world moving-objects move)
    (tw-remove-objects world moving-objects)
    (dolist (obj moving-objects)
	    (obj-set-pos obj (+ (obj-row obj) (move-delta-row move))
	                     (+ (obj-col obj) (move-delta-col move))))
    (setf moving-objects (tw-drop-tiles world moving-objects))
    (tw-place-objects world moving-objects move))


; Remove objs from world.  Can be any kind of obj; this is used in the
; dice-tossing routines as well as in tw-update-move.
(defun tw-remove-objects (world objs)
    (mapc #'(lambda(obj)
		(mapc #'(lambda(cell)
		    (let ((ch (aref (tw-grid world) (cell-row cell) 
				    (cell-col cell))))
			(if (eq (obj-type obj) 'hole)
			    (if (member ch agent-chars)
				(setf ch (move-agent-char n))
			        (if (member ch tile-chars)
				    (setf ch bare-tile-char)
				    (setf ch nothing-char)))
			    (if (member ch hole-chars)
				(setf ch bare-hole-char)
				(setf ch nothing-char)))
			(setf (aref (tw-grid world) (cell-row cell)
				    (cell-col cell))
			      ch)))
		      (obj-cells obj)))
	  objs))

; place any kind of obj.
(defun tw-place-objects (world objs move)
    (mapc #'(lambda(obj)
	(mapc #'(lambda(cell)
		    (let ((ch (aref (tw-grid world)
					    (cell-row cell)
					    (cell-col cell)))
			  newch)
			(cond ((eq (obj-type obj) 'tile)
			       (if (member ch hole-chars)
				   (setf newch tile-on-hole-char)
				   (setf newch bare-tile-char)))
			      ((eq (obj-type obj) 'agent)
			       (if (member ch hole-chars)
				   (setf newch agent-on-hole-char)
				   (setf newch (move-agent-char move))))
			      ((eq (obj-type obj) 'hole)
			       (if (member ch tile-chars)
				   (setf newch tile-on-hole-char)
				   (if (member ch agent-chars)
				       (setf newch agent-on-hole-char)
				       (setf newch bare-hole-char))))
			      ((eq (obj-type obj) 'obstacle)
			       (setf newch obstacle-char)))
			(setf (aref (tw-grid world)
				    (cell-row cell)
				    (cell-col cell))
			      newch)
			(if (eq (obj-type obj) 'hole)
			    (setf (aref (tw-grid-which-hole world)
					(cell-row cell)
					(cell-col cell))
				  obj)
			    (setf (aref (tw-grid-which world)
					(cell-row cell)
					(cell-col cell))
				  obj))))
	      (obj-cells obj)))
	  objs))

(defun obj-set-pos (obj row col)
    (setf (obj-row obj) row)
    (setf (obj-col obj) col)
    (mapc #'(lambda(relcell abscell)
		(setf (cell-row abscell)
		      (+ (cell-row relcell) (obj-row obj)))
		(setf (cell-col abscell)
		      (+ (cell-col relcell) (obj-col obj))))
	  (obj-rel-cells obj)
	  (obj-cells obj)))

; 
; If any tile in 'tiles' is hanging over a hole, drop it in and
; credit points if any holes are full.  'tiles' contains the agent;
; treat it as a tile and if it falls in, fill in the hole and kill it.
; 
(defun tw-drop-tiles (world tiles)
	(mapcan #'(lambda(tile)
	   (if (find-if-not #'(lambda(cell)
				    (member (aref (tw-grid world)
						  (cell-row cell)
						  (cell-col cell))
					    hole-chars))
			    (obj-cells tile))
	    ; doesn't drop; leave in the list
		(list tile)
	    ; does drop; fill in all cells, remove from moving-tiles and list
		(progn
		  (dolist (cell (obj-cells tile))
		    (tw-update-hole world (aref (tw-grid-which-hole world)
			             (cell-row cell) (cell-col cell))
				    (cell-row cell) (cell-col cell)))
		  (when (eq (obj-type tile) 'tile)
		      (setf (tw-tile-list world)
			    (remove tile (tw-tile-list world))))
		  (when (eq (obj-type tile) 'agent)
		      (setf (tw-agent-list world)
			    (remove tile (tw-agent-list world))))
		  nil)))
	    tiles))

; 
; A tile has fallen into hole at (row,col).  Update the hole, hole list,
; and score.
; 
(defun tw-update-hole (world hole row col)
    (let (cell)
	(setf cell (find-if #'(lambda(cell) (and (= (cell-row cell) row)
						 (= (cell-col cell) col)))
		       (obj-cells hole)))
	(setf (cell-depth cell) (- (cell-depth cell) 1))
	(if (= (cell-depth cell) 0)
	  (progn
	    (setf (aref (tw-grid world) row col) nothing-char)
	    (setf (obj-cells hole)
		  (remove cell (obj-cells hole)))
	    (when (null (obj-cells hole))
		(setf (tw-score world)
		      (+ (tw-score world)
			 (obj-score hole)))
		(setf (tw-hole-list world)
		      (remove hole (tw-hole-list world)))))
	  (setf (aref (tw-grid world) row col) bare-hole-char))))

;
; A time tick has occurred.  Toss the dice and add/delete objects as
; needed.  Random numbers are generated from a sub-seed which is
; re-initialized by rrand-step in order to smooth out differences
; between random number sequences beginning with the same seed.
; 
(defun tw-toss-dice (world)
    (rrand-step world)
    (setf (tw-hole-list world)
	  (toss1 (tw-hole-knobs world) (tw-hole-list world) 
		 'hole world))
    (rrand-step world)
    (setf (tw-tile-list world)
	  (toss1 (tw-tile-knobs world) (tw-tile-list world)
		     'tile world))
    (rrand-step world)
    (setf (tw-agent-list world)
	  (toss1 (tw-agent-knobs world) (tw-agent-list world)
		 'agent world))
    (rrand-step world)
    (setf (tw-obstacle-list world)
	  (toss1 (tw-obstacle-knobs world) (tw-obstacle-list world)
		 'obstacle world)))

(defun toss1 (knobs objlist objtype world)
  (let (which newobj)
    ; delete an obj if dice say so and not at min #
    (if (and (not (null (knob-ms-disappear knobs)))
	     (< (rrand world (knob-ms-disappear knobs)) (tw-slice world))
	     (> (length objlist) (subknob-min (knob-num knobs))))
	(progn
	      (setf which (rrand world (length objlist)))
	      (tw-remove-objects world (list (nth which objlist)))
	      (setf objlist (remove (nth which objlist) objlist))))
    ; add an obj if dice say so and not at max #
    (when (and (not (null (knob-ms-appear knobs)))
	       (< (rrand world (knob-ms-appear knobs)) (tw-slice world))
	       (< (length objlist) (subknob-max (knob-num knobs))))
	(setf newobj (make-new-obj objtype))
	(tw-generate-random-values world newobj knobs)
	(dotimes (i 20)
	    (when (tw-random-pos world newobj)
		(setf objlist (cons newobj objlist))
		(tw-place-objects world (list newobj) n)
		(when (and (eq objtype 'hole)
			   (tw-must-match-shapes world))
		    (let ((newtile (tw-convert-hole-list world 
				       (list newobj))))
			(when newtile
			    (tw-place-objects world newtile n)
			    (setf (tw-tile-list world)
				  (cons (car newtile)
					(tw-tile-list world))))))
		(return))))
    objlist))

(defun make-new-obj (type)
    (let ((obj (make-obj :type type :name (make-name type))))
;	(eval `(setf ,(obj-name obj) ',obj))	; removed this line to GC
	obj))

(defun make-name (type)
    (intern (symbol-name (gensym (string type)))))

(defun tw-init (world)
    (setf (tw-score world) 0)
    (setf (tw-elapsed-time world) 0)
    (setf (tw-rand-stepseed world)
	  (setf (tw-rand-seed-was world)
		(if (null (tw-rand-seed world))
		    (get-internal-real-time)
		    (tw-rand-seed world))))
    (rrand-step world)
    (tw-init-grids world)
    (setf (tw-obstacle-list world)
	  (tw-init-list world 'obstacle 'tw-obstacle-knobs))
    (setf (tw-hole-list world)
	  (tw-init-list world 'hole 'tw-hole-knobs))
    (setf (tw-tile-list world)
	  (tw-init-list world 'tile 'tw-tile-knobs))
    (if (tw-must-match-shapes world)
	(setf (tw-tile-list world)
	      (append (tw-convert-hole-list world (tw-hole-list world))
		      (tw-tile-list world))))
    (setf (tw-agent-list world)
	  (tw-init-list world 'agent 'tw-agent-knobs))
    (when *display-on*
	(twclear)
	(tw-display world)))

(defun tw-init-list (world type knob-name)
   (let* ((lst nil) 
	  (len 0)
	  (iter 0) 
	  (newobj (make-new-obj type))
	  (start-num (subknob-start 
			    (knob-num (funcall knob-name world))))
	  (max-iter (* 10 start-num)))
	(loop
	    (tw-generate-random-values world newobj
			(funcall knob-name world))
	    (loop
		(setf iter (+ iter 1))
		(if (or (= len start-num)
			(> iter max-iter))
		    (return-from tw-init-list lst))
		(when (tw-random-pos world newobj)
		    (setf lst (cons newobj lst))
		    (tw-place-objects world (list newobj) n)
		    (setf len (+ len 1))
		    (setf newobj (make-new-obj type))
		    (return))))))

(defun tw-init-grids (world)
	(setf (tw-grid world) (make-array (list (tw-rows world)
					        (tw-cols world))
					  :element-type 'standard-char))
	(dotimes (i (tw-rows world))
	    (dotimes (j (tw-cols world))
		(setf (aref (tw-grid world) i j) nothing-char)))
	(dotimes (i (tw-cols world))
	    (setf (aref (tw-grid world) 
			0 
			i)
		  obstacle-char)
	    (setf (aref (tw-grid world) 
			(- (tw-rows world) 1)
			i)
		  obstacle-char))
	(dotimes (i (tw-rows world))
	    (setf (aref (tw-grid world) 
			i
			0)
		  obstacle-char)
	    (setf (aref (tw-grid world) 
			i
			(- (tw-cols world) 1))
		  obstacle-char))
	(setf (tw-grid-which world) (make-array (list (tw-rows world)
						      (tw-cols world))))
	(setf (tw-grid-which-hole world) (make-array (list (tw-rows world)
						           (tw-cols world)))))
; 
; If must-match-shapes is true, copy the tile list shapes from the hole list.
; 
(defun tw-convert-hole-list (world hole-list)
  (let (iter lst1 lst2)
    (setf lst1 (tw-copy-objlist hole-list))
    (dolist (obj (reverse lst1))
	(setf iter 0)
	(loop
		(when (tw-random-pos world obj)
		    (setf (obj-type obj) 'tile)
		    (setf (obj-name obj) (make-name 'tile))
		    (setf lst2 (cons obj lst2))
		    (return))
		(setf iter (+ iter 1))
		(if (> iter 50) (return))))
    lst2))
;
; generates object qualities *except for absolute position*
; 
(defun tw-generate-random-values (world obj knobs)
	(setf (obj-rel-cells obj) (list (make-cell :row 0 :col 0)))
	(dotimes (i (- (tw-random-knob world (knob-cells knobs)) 1))
	    (setf (obj-rel-cells obj)
		  (tw-add-random-cell-to world (obj-rel-cells obj))))
	(tw-normalize-cells (obj-rel-cells obj))
	(if (eq (obj-type obj) 'hole)
	    (let (dep)
		(setf dep (tw-random-knob world (knob-depth knobs)))
		(dolist (c (obj-rel-cells obj))
		    (setf (cell-depth c) dep))
		(setf (obj-score obj) 
		      (tw-random-knob world (knob-score knobs)))
		(when (not (null (knob-score-decay knobs)))
		    (setf (obj-score-decay obj)
		          (tw-random-knob world (knob-score-decay knobs)))
		    (if (knob-score-decay-is-relative knobs)
			(if (> (obj-score obj) 0)
			  (progn
			    (setf (obj-score-decay obj)
			          (floor (/ (obj-score-decay obj)
					    (obj-score obj))))
			    (if (= (obj-score-decay obj) 0)
				(setf (obj-score-decay obj) 1)))
			  (setf (obj-score-decay obj) nil))))))
	(if (not (null (knob-timeout knobs)))
	    (setf (obj-timeout obj) 
		  (tw-random-knob world (knob-timeout knobs))))
	(setf (obj-creation-time obj) (tw-elapsed-time world))
	(setf (obj-cells obj) nil)
	(dolist (c (reverse (obj-rel-cells obj)))
	    (setf (obj-cells obj) 
		  (cons (copy-cell c) (obj-cells obj)))))


(defun tw-random-pos (world obj)
    (let (x y max-row max-col)
	(multiple-value-setq (max-row max-col) 
			     (tw-find-size (obj-rel-cells obj)))
	(setf x (+ 1 (rrand world (- (tw-rows world) 1 max-row))))
	(setf y (+ 1 (rrand world (- (tw-cols world) 1 max-col))))
	(obj-set-pos obj x y)
	(if (tw-is-on-something (obj-cells obj) (tw-grid world))
	    nil
	    t)))


; Set top left corner (not necessarily itself a cell) to 0,0
(defun tw-normalize-cells (cells)
    (let ((rmin 0) (cmin 0))
	(dolist (c cells)
	    (if (< (cell-row c) rmin) (setf rmin (cell-row c)))
	    (if (< (cell-col c) cmin) (setf cmin (cell-col c))))
	(dolist (c cells)
	    (setf (cell-row c) (- (cell-row c) rmin))
	    (setf (cell-col c) (- (cell-col c) cmin)))))

; Find two values: max row and col of any cell where top left is 0,0
(defun tw-find-size (cells)
    (let ((rmax 0) (cmax 0))
	(dolist (c cells)
	    (if (> (cell-row c) rmax) (setf rmax (cell-row c)))
	    (if (> (cell-col c) cmax) (setf cmax (cell-col c))))
	(values rmax cmax)))


(defun tw-is-on-something (cells grid)
    (dolist (cell cells)
	(if (not (eql (aref grid
		           (cell-row cell)
		           (cell-col cell))
		     nothing-char))
	    (return-from tw-is-on-something t))))


(defun tw-random-knob (world subknob)
    (if (eq (subknob-distribution subknob)
	    'flat)
	(+ (subknob-min subknob)
	   (rrand world (- (subknob-max subknob) (subknob-min subknob) -1)))
	'oops-only-flat-distributions))

(defun tw-add-random-cell-to (world cell-list)
    (let (cell move (new-cell (make-cell)))
	(loop
	    (setf cell (nth (rrand world (length cell-list)) cell-list))
	    (setf move (eval (nth (rrand world 4) '(u d l r))))
	    (setf (cell-row new-cell) 
		  (+ (cell-row cell) (move-delta-row move)))
	    (setf (cell-col new-cell) 
		  (+ (cell-col cell) (move-delta-col move)))
	    (if (not (find-if #'(lambda(x) 
				       (and (eq (cell-row new-cell)
					        (cell-row x))
					    (eq (cell-col new-cell)
					        (cell-col x))))
			      cell-list))
		(return (cons new-cell cell-list))))))

; 
; repeatable random number generator.  Uses the sub-seed initialized
; periodically by rrand-step.
; 
(defun rrand (world max)
    (setf (tw-rand-subseed world) 
	  (mod (+ (* 43573 (tw-rand-subseed world)) 172021) 200003))
    (if (> max 0)
	(floor (/ (* (tw-rand-subseed world) max) 200003))
        0))
; 
; 2nd generator - called 4 times per tick.  Makes sure that games with 
; the same random number seed don't diverge too much.
; 
(defun rrand-step (world)
    (setf (tw-rand-stepseed world) 
	  (mod (+ (* 61121 (tw-rand-stepseed world)) 83571) 192221))
    (setf (tw-rand-subseed world) (tw-rand-stepseed world)))

; 
; Copy-world utility - copies everything recursively.  Nontrivial because
; pointers in 'grid-which' and 'grid-which-hole' change.
; 
(defun tw-copy-world (world)
    (let ((new-world (copy-tw world)))
	 (setf (tw-obstacle-knobs new-world)
	       (tw-copy-knob (tw-obstacle-knobs world)))
	 (setf (tw-hole-knobs new-world)
	       (tw-copy-knob (tw-hole-knobs world)))
	 (setf (tw-tile-knobs new-world)
	       (tw-copy-knob (tw-tile-knobs world)))
	 (setf (tw-agent-knobs new-world)
	       (tw-copy-knob (tw-agent-knobs world)))

	 (setf (tw-obstacle-list new-world)
	       (tw-copy-objlist (tw-obstacle-list world)))
	 (setf (tw-hole-list new-world)
	       (tw-copy-objlist (tw-hole-list world)))
	 (setf (tw-tile-list new-world)
	       (tw-copy-objlist (tw-tile-list world)))
	 (setf (tw-agent-list new-world)
	       (tw-copy-objlist (tw-agent-list world)))

	 (tw-init-grids new-world)

	 (tw-place-objects new-world (tw-obstacle-list new-world) n)
	 (tw-place-objects new-world (tw-hole-list new-world) n)
	 (tw-place-objects new-world (tw-tile-list new-world) n)
	 (tw-place-objects new-world (tw-agent-list new-world) n)

	 new-world))


(defun tw-copy-knob (knob)
    (let ((new-knob (copy-knob knob)))
	 (if (knob-num knob) 
	     (setf (knob-num new-knob)
	           (copy-subknob (knob-num knob))))
	 (if (knob-cells knob) 
	     (setf (knob-cells new-knob)
	           (copy-subknob (knob-cells knob))))
	 (if (knob-timeout knob) 
	     (setf (knob-timeout new-knob)
	           (copy-subknob (knob-timeout knob))))
	 (if (knob-depth knob) 
	     (setf (knob-depth new-knob)
	           (copy-subknob (knob-depth knob))))
	 (if (knob-score knob) 
	     (setf (knob-score new-knob)
	           (copy-subknob (knob-score knob))))
	 new-knob))

(defun tw-copy-objlist (objlist)
    (mapcar #'(lambda(obj)
		 (let ((newobj (copy-obj obj)))
		     (setf (obj-rel-cells newobj)
			   (tw-copy-cells (obj-rel-cells obj)))
		     (setf (obj-cells newobj)
			   (tw-copy-cells (obj-cells obj)))
		     newobj))
	    objlist))

(defun tw-copy-cells (cells)
    (mapcar #'(lambda(cell)
		 (copy-cell cell))
	cells))

(defvar prev-holes 0)

(defun tw-display (world)
  (let ((row 0) (col (+ (* 2 (tw-cols world)) 4))
	(gr (make-array (list (tw-rows world) (tw-cols world))
		:element-type 'standard-char)))
    (dotimes (i (tw-rows world))
	(dotimes (j (tw-cols world))
	    (setf (aref gr i j)
		  (if (eql (aref (tw-grid world) i j) bare-hole-char)
		      (if (< (obj-score (aref (tw-grid-which-hole world) i j))
			     100)
			  (int-char (+ 48 (floor (/ (obj-score 
				(aref (tw-grid-which-hole world) i j)) 10))))
			  bare-hole-char)
		      (aref (tw-grid world) i j)))))
;    (tw-display-array 1 1 gr)
        (tw-display-array (tw-rows world) (tw-cols world) gr)
    (tw-print row col (format nil "Time:  ~6,0F" (* 1.0 *elapsed*)))
    (tw-print (+ row 1) col (format nil "Score: ~5A" (tw-score world)))
    (setf row (+ row 3))
    (dolist (hole (tw-hole-list world))
	(tw-show-obj row col hole)
	(setf row (+ row 5)))
    (dotimes (i (- prev-holes (length (tw-hole-list world))))
	(dotimes (j 5)
	    (tw-print row col (format nil "~15A" ""))
	    (setf row (+ row 1))))
    (setf prev-holes (length (tw-hole-list world)))
    (tw-print (tw-rows world) 0 "]")
    (twrefresh)
))

(defun tw-show-obj (row col obj)
    (let (j)
	 (tw-print row col (format nil "~15A" (obj-name obj)))
	 (tw-print (+ row 1) col
	     (format nil "Row ~2A Col ~4A" 
		     (cell-row (first (obj-cells obj)))
		     (cell-col (first (obj-cells obj)))))
	 (setf j 2)
	 (when (not (null (obj-timeout obj)))
	     (tw-print (+ row j) col
	         (format nil "Out ~11A" (obj-timeout obj)))
	     (setf j (+ j 1)))
	 (when (eq (obj-type obj) 'hole)
	     (tw-print (+ row j) col
	         (format nil "Sco ~11A" (obj-score obj)))
	     (setf j (+ j 1)))
	 (tw-print (+ row j) col (format nil "~15A" ""))
	 (+ j 1)))
