/**************************************************************************/
/* main.c                                                       /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <signal.h>

#define GLOBAL_EXTERN

#include "global_communication.h"
#include "command.h"

#include "communication.h"
#include "main.h"
#include "matrix.h"
#include "vector.h"




void main(int argc, char *argv[])
{
  int i;
  extern int robots, simulators;

  if (argc != 2)
  {
	fprintf(stderr,"Usage:  %s <r|s>\n",argv[0]);
	exit(1);
  }

  /*
   * Check how many robots and simulators must be connected.
   */
  for (i=0; i<strlen(argv[1]); i++)
	if (argv[1][i] == 'r')
		robots++;
	else if (argv[1][i] == 's')
		simulators++;
	else
	{
		fprintf(stderr, "illegal argument %c\n", argv[1][i]);
		exit(1);
	}

  if (robots + simulators == 0)
  {
	fprintf(stderr, "No robots connected---ready\n");
	exit(0);
  }



  (void) signal(SIGQUIT, (void (*)()) ok_exit);
  (void) signal(SIGINT, (void (*)()) ok_exit);

  init_kinematics("DHfile");
  on_robot();
  on_camera();
  loop();
  ok_exit();
}




void loop(void)
{
  vector cam_1_eef, cam_1_obj, cam_2_eef, cam_2_obj, hand_cam, area;
  vector end_effector, dummy_vector, object_pos;
  char *s;
  joint_values joint_pos;

  /*
   * A nasty situation results when the robot simulator
   * has no clue as to how large the object supposedly
   * is.  So, before we generate any robot action, tell it.
   */
  E = 100.0; focus = 1.0;
  object_pos[0] = object_pos[1] = object_pos[2] = 0.0; object_pos[3] = 1.0;
  send_object_position(object_pos);

  /*
   * Check if command.h contains inconsistencies.
   */
  check_command_integrity();

  reset_robot();
  
  while (1)
    {
      long code;
      
      do printf("> "); while (get_next_word(&s) == '\0');
      
      switch (code = parse_command(&s))
	{
	case 0:
	  break;

	case C_HELP1: case C_HELP2:
	  help();
	  break;
	  
	case C_DESCRIBE:
	  describe();
	  break;
	  
	case C_VERSION:
	  get_version_data();
	  break;
	  
	case C_JOINTS:
	  {
	    joint_values q, dq, ddq;
	    int status;
	    
	    if (NR_LINKS != 6)
	      {
		fprintf(stderr, "Only for 6DoF robot\n");
		return;
	      }
	    status = get_joint_values(0, 0, q, dq, ddq);
	    if (status > 0)
	      printf("still moving;\n");
	    printf("  q = [%.2f %.2f %.2f %.2f %.2f %.2f]\n",
		   q[1], q[2], q[3], q[4], q[5], q[6]);
	    printf(" dq = [%.2f %.2f %.2f %.2f %.2f %.2f]\n",
		   dq[1], dq[2], dq[3], dq[4], dq[5], dq[6]);
	    printf("ddq = [%.2f %.2f %.2f %.2f %.2f %.2f]\n",
		   ddq[1], ddq[2], ddq[3], ddq[4], ddq[5], ddq[6]);
	  }
	  break;
	  
	case C_CAMERA:
	  if (get_camera(0, CAMERA_1, cam_1_eef, cam_1_obj))
	    printf("error camera 1\n");
	  else
	    printf("cam1: [%.2f %.2f %.2f] [%.2f %.2f %2f]\n",
		   cam_1_eef[0], cam_1_eef[1], cam_1_eef[2],
		   cam_1_obj[0], cam_1_obj[1], cam_1_obj[2]);
	  
	  if (get_camera(0, CAMERA_2, cam_2_eef, cam_2_obj))
	    printf("error camera 2\n");
	  else
	    printf("cam2: [%.2f %.2f %.2f] [%.2f %.2f %2f]\n",
		   cam_2_eef[0], cam_2_eef[1], cam_2_eef[2],
		   cam_2_obj[0], cam_2_obj[1], cam_2_obj[2]);
	  
	  if (get_camera(0, HAND_CAMERA, hand_cam, area))
	    printf("error hand camera\n");
	  else
	    printf("handcam: [%.2f %.2f %.2f] [%.2f %.2f %2f]\n",
		   hand_cam[0], hand_cam[1], hand_cam[2],
		   area[0], area[1], area[2]);
	  
	  if (get_camera(0, END_EFFECTOR, end_effector, dummy_vector))
	    printf("error end effector position\n");
	  else
	    printf("end_effector: [%.2f %.2f %.2f]\n",
		   end_effector[0], end_effector[1],
		   end_effector[2]);
	  break;
	  
	case C_RESET:
	  reset_robot();
	  break;
	  
	case C_MOVE_RELATIVE: case C_MOVE_ABSOLUTE:
	  user_move_robot(code);
	  break;
	  
	case C_FIX_TARGET:
	  fix_target();
	  break;
	  
	case C_INVERSE_KINEMATICS:
	  inverse_kinematics();
	  break;

        case C_JUST_GET_IT:
          camera_based_inverse_kinematics(joint_pos);
          move_robot(NULL, joint_pos, NULL, NULL, 1.0, 1.0);
          break;

	case C_QUIT:
	  return;
	  
	default:
	  printf("unknown command %o---adapt main.c\n", code);
	  break;
	}
    }
}
