/**************************************************************************/
/* inverse.c                                                    /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. van der Smagt and A. Jansen                    _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         ajansen@fwi.uva.nl                               | | \/\/ | |  */
/*                                                          | \______/ |  */
/*                                                           \________/   */
/*                                                                        */
/*                                                              /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <math.h>

#include "global.h"
#include "inverse.h"
#include "matrices.h"
#include "io.h"
#include "moving.h"
#include "ieee.h"
#include "command.h"



void init_kinematics(char *robotdefinition)
{
  static already_read = 0;

  if (already_read == 0)
  {
	read_definition(robotdefinition);
	already_read = 1;
  }
}



void inverse_kinematics(void)
{
  joint_values cur_q, dq, ddq;
  double d_x, d_y, d_z;
  REAL x, y, z;
  char *s;

  /*
   * Wait till robot is ready with its move and
   * then read the current position.
   */
  get_joint_values(0, 1, cur_q, dq, ddq);

  printf("give target position: ");
  while (get_next_word(&s) == '\0'); sscanf(s, "%lf", &d_x); x = (REAL) d_x;
  while (get_next_word(&s) == '\0'); sscanf(s, "%lf", &d_y); y = (REAL) d_y;
  while (get_next_word(&s) == '\0'); sscanf(s, "%lf", &d_z); z = (REAL) d_z;
  do_inverse(x, y, z);
}



/*
 * Perform an action opposite to that in couple_joints, after
 * having received a set of joint values from the neural
 * network.
 */
void decouple_joints(joint_values u_q, joint_values u_dq, joint_values u_ddq,
                joint_values q, joint_values dq, joint_values ddq)
{
  unsigned short i;
  for (i=1; i<=NR_LINKS; i++)
  {
        u_q[i] = q[i];
        u_dq[i] = dq[i];
        u_ddq[i] = ddq[i];
  }
  u_q[3] -= q[2];
  u_dq[3] -= dq[2];
  u_ddq[3] -= ddq[2];
}




void full_forward_kinematics(joint_values q2, homo joint_homo[])
     /*
      * This is the full DH-forward kinematics
      * and is copied straight out of simmel/kinematics.c,
      * except for the addition of four words.
      */
{
  homo m1, m2, m3;
  REAL *p1, *p2, *p3, *res;
  REAL theta, d, a, alfa;
  int i;
  joint_values target_position, dq1, dq2, ddq1, ddq2;

  init_kinematics("DHfile");

  decouple_joints(target_position-1, dq1, ddq1, q2, dq2, ddq2);

  p1 = m1; p2 = m2; p3 = m3;
  init_matrix(p1, HOMO_DEG, HOMO_DEG);

  /*
   * In the neural network code and camera/robot simulator, joint
   * values are target_position[1]...target_position[6].  This
   * simulator expects target_position[0]...target_position[5].
   */

   /*
    * Fill in the values...
    */
   substitute(target_position);

   /* loop through the links */
   for (i=1; i<=NR_LINKS ; i++)
   {
        theta = get_param(i, DH_THETA);
        d = get_param(i, DH_D);
        a = get_param(i, DH_A);
        alfa = get_param(i, DH_ALFA);

        dh_transformation(p2, theta, d, a, alfa);
        mul_matrix(p1, HOMO_DEG, HOMO_DEG, p2, HOMO_DEG, HOMO_DEG,
                        p3, HOMO_DEG, HOMO_DEG);

        /*                 0
         * joint_homo[i] =  A
         *                   i
         */
        copy_homo(joint_homo[i], p3);


        res = p1;
        p1 = p3;
        p3 = res;
  }
}



void do_inverse(REAL x, REAL y, REAL z)
{
  vector approach_vector, target_position;
  joint_values cur_q, dq, ddq, new_q, rad_move;

  approach_vector[0] = 0.0;
  approach_vector[1] = 0.0;
  approach_vector[2] = -1.0;
  approach_vector[3] = 1.0;
  target_position[0] = x;
  target_position[1] = y;
  target_position[2] = z;

  if (inverse_kinematics_OSCAR(approach_vector, target_position, rad_move))
	return;

  new_q[1] = rad2deg(rad_move[1]);
  new_q[2] = rad2deg(rad_move[2]);
  new_q[3] = rad2deg(rad_move[3]);
  new_q[4] = rad2deg(rad_move[4]);
  new_q[5] = rad2deg(rad_move[5]);
  new_q[6] = rad2deg(rad_move[6]);
  if (move_robot(NULL, new_q, NULL, NULL, FAST_DQ, FAST_DDQ))
	fprintf(stderr, "Illegal target position\n");

  /*
   * Wait for completion of move.
   */
  get_joint_values(0, 1, cur_q, dq, ddq);
}


REAL my_atan2(REAL y, REAL x)
{
  if (y >= 0.0 && x >= 0.0)
  {
	if (iszero((double) x) == 1 && iszero((double) y) == 1)
		return (REAL) 0.0;

	if (iszero((double) x) == 1)
		return (REAL) (0.5 * M_PI);

	return (REAL) atan((double) (y/x));
  }

  if (y >= 0.0 && x < 0.0)
	return (REAL) (M_PI - atan((double) (y/-x)));

  if (y < 0.0 && x >= 0.0)
  {
	if (iszero((REAL) x) == 1)
		return (REAL) (- 0.5 * M_PI);

	return (REAL) atan((double) (y/x));
  }

  return (REAL) (- M_PI + atan((double) (y/x)));
}



void calc_p_vector(REAL *p_6, REAL *a, REAL *p)
/*
 * calculate the vector to the intersection of the last three joints.
 * This is p = p_6 - d_6 * a.
 */
{
  REAL d;
  short i;

  d = get_param(NR_LINKS, DH_D);

  for (i = 0; i < HOMO_DEG - 1; i++)
	p[i] = p_6[i] - d * a[i];

  p[HOMO_DEG - 1] = 1.0;
}
 



REAL joint_1_CRAIG_solution(REAL *p)
{
  return my_atan2(p[1], p[0]);
}





REAL joint_3_CRAIG_solution(REAL theta_1, REAL *p, REAL *p_2,
			      short *possible)
/*
 * Calculate theta_3 joint value. This must be done before calculating the
 * value of theta_2.  If possible is 0 the target position is unreachable.
 * *p_2 is a pointer to a vector p expressed in the (x_1,y_1,z_1) coordinate
 * system. It must be allocated before passing to this function.
 * The joint value is returned.
 */
{
  REAL c_1, s_1, c_a, s_a;
  REAL alfa_1, d_1, a_2, d_4;
  REAL **A_0_1, **A_1_0;
  REAL Q;

  A_0_1 = alloc_matrix();
  A_1_0 = alloc_matrix();

  c_1 = (REAL) cos((double) theta_1);
  s_1 = (REAL) sin((double) theta_1);

  alfa_1 = (get_param(1, DH_ALFA) / 180.0) * M_PI;

  c_a = (REAL) cos((double) alfa_1);
  s_a = (REAL) sin((double) alfa_1);

  d_1 = get_param(1, DH_D);
  d_4 = (REAL) fabs((double) get_param(4, DH_D));
  a_2 = (REAL) fabs((double) get_param(2, DH_A));

  /* 
   * make matrix 0_A_1.
   */

  A_0_1[0][0] = c_1; A_0_1[0][1] = -c_a * s_1;
	A_0_1[0][2] = s_a * s_1; A_0_1[0][3] = 0.0;
  A_0_1[1][0] = s_1; A_0_1[1][1] = c_a * s_1;
	A_0_1[1][2] = -s_a * c_1; A_0_1[1][3] = 0.0;
  A_0_1[2][0] = 0.0; A_0_1[2][1] = s_a;
	A_0_1[2][2] = c_a; A_0_1[2][3] = d_1;
  A_0_1[3][0] = 0.0; A_0_1[3][1] = 0.0;
	A_0_1[3][2] = 0.0; A_0_1[3][3] = 1.0;

  /*
   * Express p in coordinate system (x_1,y_1,z_1).
   */

  invert_h_matrix(A_0_1,A_1_0);

  mv_prod(A_1_0,p,p_2);

  delete_matrix(A_0_1);
  delete_matrix(A_1_0);

  Q = (REAL) sqrt(Msqr(p_2[0]) + Msqr(-p_2[1]));

  /*
   * check to see of the target position can be reached
   */

  if (Q > a_2 + d_4) {
	*possible = 0;
	return 0.0;
  }

  *possible = 1; /* target can be reached */

  /*
   * Return always the above solution.
   * The below solution can be given by -theta_3.
   */
  return (REAL) acos(((-Msqr(p_2[0]) - Msqr(p_2[1]) + Msqr(a_2) + Msqr(d_4)) /
		  (2.0 * a_2 * d_4)));
}






REAL joint_2_CRAIG_solution(REAL theta_3, REAL *p_2)
/*
 * To calculate theta_2 first theta_3 must be known.  In this function
 * is the vector p_2 calculated too, so it must not be calculated twice.
 */
{
  REAL alpha, beta;
  REAL a_2, d_4;

  d_4 = (REAL) fabs(get_param(4, DH_D));
  a_2 = (REAL) fabs(get_param(2, DH_A));

  alpha = my_atan2(-p_2[1], p_2[0]);

  if (fabs(fabs(alpha) - M_PI) < 0.00001)
	alpha = 0.0;

  beta = (REAL) acos((Msqr(p_2[0]) + Msqr(-p_2[1]) + Msqr(a_2) - Msqr(d_4)) /
	      (2.0 * a_2 * sqrt(Msqr(p_2[0]) + Msqr(-p_2[1]))));

  /*
   * dependent on theta_3 the solution of theta_2 is alpha +/- beta.
   */

  if (theta_3 >= 0.0)
	return alpha + beta;

  return alpha - beta;
}




REAL joint_5_CRAIG_solution(REAL theta_2, REAL theta_3)
/*
 * With the constraint that the gripper is always pointing
 * downwards theta_5 can be easily calculated according to
 * theta_5 = 1.5 * pi - theta_2 - theta_3.
 */
{
  return 1.5 * M_PI - theta_2 - theta_3;
}





int inverse_kinematics_OSCAR(vector a, vector p_6, joint_values thetas)
{
  REAL theta_1, theta_2, theta_3, theta_5;
  REAL *p, *p_2;
  short possible;

  p = alloc_vector();
  p_2 = alloc_vector();

  calc_p_vector(p_6,a,p);

  theta_1 = joint_1_CRAIG_solution(p);
  theta_3 = joint_3_CRAIG_solution(theta_1,p,p_2,&possible);

  if (!possible) {
	fprintf(stderr, "not a valid solution possible\n");
	delete_vector(p);
	delete_vector(p_2);
	return 1;
  }

  theta_2 = joint_2_CRAIG_solution(theta_3,p_2);
  theta_5 = joint_5_CRAIG_solution(theta_2,theta_3);

  /*
   * fill in the thetas corresponding to the OSCAR definition
   * in an array from 0 .. nr_links - 1.
   */

  /*
   * For the OSCAR an upwards rotation of link2 expects a negative
   * theta_2 value.
   */
  thetas[1] = theta_1;
  thetas[2] = -theta_2;

  /*
   * For the OSCAR the above solution for theta_3 = 1.5 * pi - theta_3 and
   * the below solution is theta_3 = theta_3 - 0.5 * pi.
   */

  if (theta_3 >= 0.0)
	thetas[3] = 1.5 * M_PI - theta_3 + thetas[2];
  else
	thetas[3] = theta_3 - 0.5 * M_PI + thetas[2];

  /*
   * For the OSCAR the solution for theta_5 is theta_5 = pi - theta_5.
   */

  thetas[5] = M_PI - theta_5;

  /*
   * theta_4 and theta_6 are not calculated yet.
   */

  thetas[4] = 0.0;
  thetas[6] = 0.0;

  delete_vector(p);
  delete_vector(p_2);

  return 0;
}
