/**************************************************************************/
/* communication.c                                              /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <signal.h>


#include <errno.h>

#include "global_communication.h"
#include "data_code.h"
#include "ieee.h"
#include "version.h"


/* for sockets */
#include <sys/types.h>
#include <sys/errno.h>

#define COMMUNICATION_EXTERN
#include "communication.h"




void ok_exit(void)
{
  close_down(STOP, STOP);
}

void panic_exit(void)
{
  close_down(PANIC, PANIC);
}


/*
 * Sends the new joint values to the controller.  Returns:
 * 0 if everything ok, otherwise the number of the
 * failing joint as returned by the other side of the socket.
 */
unsigned short send_joint_values(int nr, joint_values q, joint_values dq,
			joint_values ddq, int check_reach)
{
  char buffer[BUF_SIZE];
  unsigned short length;
  char *return_buffer, type;
  char *p_buffer;
  unsigned short i, chunk;

  /*
   * The 0th character indicates whether the robot should check
   * on validness of the target position.
   */
  buffer[0] = check_reach;
  /*
   * For user-friendly encoding, the sprintf-like function encode() 
   * is devised.  Call it as follows:
   */
  p_buffer = buffer+1;
  length = 1;
  for (i=1; i<=NR_LINKS; i++)
  {
        chunk = encode(p_buffer, "%f%f%f", (float) q[i], (float) dq[i],
                                                         (float) ddq[i]);
        length += chunk;
	if (length > BUF_SIZE)
	{
		fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
		panic_exit();
	}
        p_buffer += chunk;
  }

  send_data(d_robot_socket[nr], NEWPOS, length, buffer);
  length = receive_data(d_robot_socket[nr], &return_buffer, &type);

  if (type == STOP)
	ok_exit();
  if (type == PANIC)
	panic_exit();
  if (type == NEWPOS_BAD)
  {
	unsigned short joint;
	float j_min, j_val, j_max;

	/*
	 * First stop all other robots, too.
	 */
	stop_robot();

	decode(return_buffer, length, "%d%f%f%f",
				&joint, &j_val,&j_max,&j_min);
	if (joint == 0)
	{
		/*
		 * This is an illegal end-effector error.
		 */
		printf("illegal end-effector position, robot %d\n", nr);
		joint = NR_LINKS+1;
	}
	else
		printf("joint %d robot %d: not %f < %f < %f\n",
				joint, nr, j_min, j_val, j_max);

	free(return_buffer);
	return joint;
  }

  if (type != NEWPOS_OK)
  {
	fprintf(stderr, "send_joint_values() got bad message `%c'\n", type);
	panic_exit();
  }

  if (length != 0)
  {
	fprintf(stderr, "send_joint_values() got far too many bytes\n");
	panic_exit();
  }

  return 0;
}



int get_joint_values(int nr, int wait,
			joint_values q, joint_values dq, joint_values ddq)
/*
 * Read the current joint values.
 * If wait != 0, then it waits until the robot has
 * finished moving.  Also, it returns a flag indicating
 * whether the robot moves or not.
 *
 * Data is read from the connected robot as given by nr; usually, nr
 * is set to 0.
 *
 * Any one of the pointers q, dq, and ddq may be NULL pointers, in
 * which case they are not assigned.  This facilitates calls which
 * are only used to wait for the robot to stop: 
 *	get_joint_values(0, 1, NULL, NULL, NULL).
 */
{
  float j[NR_LINKS+1], dj[NR_LINKS+1], ddj[NR_LINKS+1];
  char *p_buffer;
  unsigned short i, chunk;
  int moving;

  unsigned short length;
  char *buffer, type;

  do
  {
	moving = 0;

	send_data(d_robot_socket[nr], SEND_JOINTS, 0, (char *) NULL);
	length = receive_data(d_robot_socket[nr], &buffer, &type);

	if (type == STOP)
		ok_exit();
	if (type == PANIC)
		panic_exit();
	if (type != SENT_JOINTS)
	{
		fprintf(stderr, "Expected SENT_JOINTS from robot (got `%c')\n",
								type);
		panic_exit();
	}

	chunk = length / NR_LINKS;
	p_buffer = buffer;
	for (i=1; i<=NR_LINKS; i++, p_buffer += chunk)
	{
        	if (decode(p_buffer,chunk,"%f%f%f",&j[i],&dj[i],&ddj[i]) != 3)
        	{
                	fprintf(stderr,
				"get_joint_values failed for joint %d\n", i);
                	panic_exit();
        	}

		if (q != NULL) q[i] = (REAL) j[i];
		if (dq != NULL) dq[i] = (REAL) dj[i];
		if (ddq != NULL) ddq[i] = (REAL) ddj[i];
		if (!ciszero(dj[i]) || !ciszero(ddj[i]))
			moving++;
	}

	if (buffer != NULL)
		free(buffer);
  } while (wait && moving);

  return moving;
}



void send_robot(char msg)
{
  int i;

  for (i=0; i<robots+simulators; i++)
  	send_data(d_robot_socket[i], msg, 0, (char *) NULL);
}




/*
 * Camera info is ONLY read from the first connected robot.
 */
short get_camera(int robot_nr, char cam_type, vector view_1, vector view_2)
{
  char *buffer, type;
  unsigned short length;
  int fields;
  float a1, a2, a3, a4, a5, a6;
  short error;

  send_data(d_camera_socket[robot_nr], cam_type, 0, (char *) NULL);
  length = receive_data(d_camera_socket[robot_nr], &buffer, &type);

  if (type == STOP)
	ok_exit();
  if (type == PANIC)
	panic_exit();
  if (type == NO_OBJECT)
	return -2;
  if (type != cam_type)
  {
	fprintf(stderr, "get_camera() received bad message\n");
	panic_exit();
  }

  fields = decode(buffer, length, "%d%f%f%f%f%f%f",
				&error, &a1, &a2, &a3, &a4, &a5, &a6);
  if (fields != 7)
  {
	fprintf(stderr, "get_camera() got %d fields, expected 7\n",
	fields);
	panic_exit();
  }

  /*
   * Check to see if some image processing error occurred.
   */
  if (error)
	return error;

  view_1[0] = (REAL) a1;
  view_1[1] = (REAL) a2;
  view_1[2] = (REAL) a3;
  view_1[3] = 1.0;
  view_2[0] = (REAL) a4;
  view_2[1] = (REAL) a5;
  view_2[2] = (REAL) a6;
  view_2[3] = 1.0;

  if (buffer != NULL)
	free(buffer);

  return 0;
}



/* 
 * Here is the difference between the real and the simulated
 * worlds: the target position is indicated by typing in
 * coordinates and sending them to the simulator.  In fact,
 * they should be generated by the bemmel's.  Not yet, not yet.
 */
void send_object_position(vector target)
{
  char buffer[100];		/* surely not too small. */
  char type;
  unsigned short length;
  int i;

  for (i=0; i<robots+simulators; i++)
  {
	/*
	 * ONLY send this to connected simulators.
	 */
	if (type_of_connected[i] == TYPE_ROBOT)
		continue;

 
	length = encode(buffer, "%f%f%f%f%f",
		(float) target[0], (float) target[1], (float) target[2],
		(float) E, (float) focus);

	send_data(d_camera_socket[i], OBJECT_POSITION, length, buffer);

	length = receive_data(d_camera_socket[i], (char **) NULL, &type);
	if (type == STOP)
		ok_exit();
	if (type == PANIC)
		panic_exit();
	if (type != OBJECT_RECEIVED)
	{
		fprintf(stderr,"send_object_position() got strange message\n");
		panic_exit();
	}
	if (length != 0)
	{
		fprintf(stderr, "send_object_position() in big trouble\n");
		panic_exit();
	}
  }
}



int camera_based_inverse_kinematics(joint_values target_position)
/*
 * Read the camera data, and apply
 * inverse kinematics to reach the
 * calculated point.
 */
{
  vector camera, area, world_position, hand_position, approach_vector;
  homo joint_homos[NR_LINKS+1], inverse_eef;
  joint_values cur_q, dq, ddq, rad_move;
  int i;
  double rad2deg();

  get_camera(0, HAND_CAMERA, camera, area);
  printf("rel obj %f %f %f %f\n", camera[0], camera[1], camera[2], camera[3]);

  /*
   * The "camera[]" now contains the position of the
   * blob relative to the hand of the robot.  So
   * to get the blob position in world coordinates,
   * which is required for the inverse kinematics
   * module, first the forward kinematics must be
   * applied to the current position of the robot,
   * rendering the position of the end-effector in
   * world coordinates.
   */
  get_joint_values(0, 0, cur_q, dq, ddq);
  full_forward_kinematics(cur_q, joint_homos);

  printf("eef %f %f %f\n",
         joint_homos[NR_LINKS][3],
         joint_homos[NR_LINKS][7],
         joint_homos[NR_LINKS][11]);

  invert_homo(inverse_eef, joint_homos[NR_LINKS]);
  homo_vec_mul(world_position, joint_homos[NR_LINKS], camera);
  printf("world %f %f %f\n",
         world_position[0], world_position[1], world_position[2]);

  approach_vector[0] = 0.0;
  approach_vector[1] = 0.0;
  approach_vector[2] = -1.0;
  approach_vector[3] = 1.0;

  if (inverse_kinematics_OSCAR(approach_vector, world_position, rad_move))
    {
      fprintf(stderr, "Illegal target point\n");
      return 1;
    }

  for (i=1; i<=NR_LINKS; i++)
    target_position[i] = rad2deg(rad_move[i]);

  return 0;
}



void close_down(char m_robot, char m_camera)
{
  int i;

  /*
   * Allow no interruptions whil closing the communication channels.
   */
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT, SIG_IGN);

  for (i=0; i<robots+simulators; i++)
  {
	if (robot_connected != NULL && robot_connected[i])
	{
		robot_connected[i] = 0;
		send_data(d_robot_socket[i], m_robot, 0, (char *) NULL);
		close(d_robot_socket[i]);
		close(c_robot_socket[i]);
	}

	if (camera_connected != NULL && camera_connected[i])
	{
		camera_connected[i] = 0;
		send_data(d_camera_socket[i], m_camera, 0, (char *) NULL);
		close(d_camera_socket[i]);
		close(c_camera_socket[i]);
	}
  }
  exit(0);
}


void stop_robot(void)
{
  char type;
  int i;

  for (i=0; i<robots+simulators; i++)
	send_data(d_robot_socket[i], HALT, 0, (char *) NULL);
  for (i=0; i<robots+simulators; i++)
  {
	receive_data(d_robot_socket[i], (char **) NULL, &type);
	if (type != HALT_OK)
	{
		fprintf(stderr, "robot %d won't stop moving!!\n", i+1);
		panic_exit();
	}
  }
}



void on_robot(void)
{
  int i;

  c_robot_socket = (int *) calloc(robots+simulators, sizeof(int));
  d_robot_socket = (int *) calloc(robots+simulators, sizeof(int));
  robot_connected = (int *) calloc(robots+simulators, sizeof(int));
  if (type_of_connected == NULL)
	type_of_connected = (char *) calloc(robots+simulators, sizeof(char));

  if (c_robot_socket==NULL || d_robot_socket==NULL ||
      robot_connected==NULL || type_of_connected==NULL)
  {
	perror("on_robot()");
	panic_exit();
  }

  for (i=0; i<robots+simulators; i++)
  {
	char socket_file_name[BUF_SIZE];

	sprintf(socket_file_name, "../socketnr/robot%d", i+1);
	setup_communication(socket_file_name, &c_robot_socket[i],
							&d_robot_socket[i]);
	robot_connected[i] = 1;
	type_of_connected[i] = (i < robots ? TYPE_ROBOT : TYPE_SIMULATOR);
  }
}


void on_camera(void)
{
  int i;

  c_camera_socket = (int *) calloc(robots+simulators, sizeof(int));
  d_camera_socket = (int *) calloc(robots+simulators, sizeof(int));
  camera_connected = (int *) calloc(robots+simulators, sizeof(int));
  if (type_of_connected == NULL)
	type_of_connected = (char *) calloc(robots+simulators, sizeof(char));

  if (c_camera_socket==NULL || d_camera_socket==NULL ||
      camera_connected==NULL || type_of_connected==NULL)
  {
	perror("on_camera()");
	panic_exit();
  }

  for (i=0; i<robots+simulators; i++)
  {
	char socket_file_name[BUF_SIZE];

	sprintf(socket_file_name, "../socketnr/camera%d", i+1);
	setup_communication(socket_file_name, &c_camera_socket[i],
							&d_camera_socket[i]);
	camera_connected[i] = 1;
	type_of_connected[i] = (i < robots ? TYPE_ROBOT : TYPE_SIMULATOR);
  }
}




void get_version_data(void)
{
  int i;
  printf("This is %s\n"
	 "written by Patrick van der Smagt\n"
	 "Department of Computer Systems\n"
	 "University of Amsterdam\n"
	 "Amsterdam, The Netherlands\n"
	 "email <smagt@fwi.uva.nl>\n\n", local_version);
  for (i=0; i<robots+simulators; i++)
    {
      send_data(d_robot_socket[i], VERSION, 0, (char *) NULL);
      send_data(d_camera_socket[i], VERSION, 0, (char *) NULL);
    }
}





/************************************************************************/
/*									*/
/************************************************************************/
void setup_communication(char *socketfile,
			int *control_socket, int *data_socket)
{
  char type;
  int socknr;
  FILE *sockfile;

  if ((sockfile = fopen(socketfile,"r")) == NULL)
  {
	fprintf(stderr, "can't open %s\n", socketfile);
	exit(1);
  }
  fscanf(sockfile, "%d", &socknr);
  fclose(sockfile);

  printf("waiting at socket %d\n", socknr);

  /* Setup socket.  The number of the connection is
     socknr (see common.h).  The socket line is
     in control_socket, whereas communication goes over the
     file descriptor data_socket.  The last parameter
     is the maximum number of clients that is allowed
     to connect.
   */
  if (setup_server_socket(socknr, control_socket, 1) == -1)
	exit(1);

  /* Now for each client accept the connection.  In this case,
     we only have one client and have to accept only once.
   */
  accept_client(*control_socket, data_socket);

  printf("client accepted\n");

  /* set up connection with client wait for alive char */
  send_data(*data_socket, START, 0, (char *) NULL);
  do
  {
	receive_data(*data_socket, (char **) NULL, &type);
	if (type == START_ERROR)
	{
		fprintf(stderr, "client says it can't start\n");
		panic_exit();
	}
	if (type != START_OK)
		fprintf(stderr, "received wrong message from client\n");
  } while (type != START_OK);

  printf("chris: client accepts start\n");
}




/************************************************************************/
/*			Socket routines					*/
/************************************************************************/

void send_data(int sock, char code, unsigned short length, char *data)
{
  char *buf;
  unsigned short i;
  int status;

  if ((buf = (char *) malloc(length+3)) == NULL)
  {
	fprintf(stderr, "send_data(): out of buffer space\n");
	panic_exit();
  }

  buf[0] = code;
  us_encode(length, buf+1);

  if (data != NULL)
	for (i=0; i<length; i++) 
		buf[i+3] = data[i];


  do
	status = send(sock, buf, (int) length+3, 0);
  while (status == -1 && errno == EWOULDBLOCK);

  if (status == -1)
  {
	perror("send_data()");
	panic_exit();
  }
  else if (status != length+3)
  {
	fprintf(stderr, "couldn't send all bytes\n");
	perror("send_data()");
	panic_exit();
  }

  free(buf);
}



unsigned short receive_data(int sock, char **buffer, char *type)
{
  extern int errno;
  unsigned short message_len;
  char buf[3];

  /* Read the header of the message. */
  while (1)
  {
	int status;

	status = recv(sock, buf, 3, 0);
	if (status == 3) break;
	if (status == -1 && errno == EWOULDBLOCK) continue;
	if (status != -1)
	{
		fprintf(stderr, "receive_data() received too few bytes");
		fprintf(stderr, "; received %d, expected 3\n", status);
	} else
		perror("receive_data()");
	panic_exit();
  }

  *type = buf[0];
  if (*type == PANIC)
  {
	fprintf(stderr, "Robot request an exit\n");
	panic_exit();
  }

  message_len = us_decode(buf+1);

  /*
   * Read the rest of the message.
   */
  if (buffer != (char **) NULL)
	*buffer = NULL;
  if (message_len > 0)
  {
	if (buffer == (char **) NULL)
	{
		fprintf(stderr, "receive_data(): message too long\n");
		panic_exit();
	}

	*buffer = (char *) malloc(message_len+1);
	if (*buffer == NULL)
	{
		fprintf(stderr,"receive_data(): out of buffer space\n");
		panic_exit();
	}

	while (1)
  	{
		int status;

		status = recv(sock, *buffer, message_len, 0);
		if (status == message_len) break;
		if (status == -1 && errno == EWOULDBLOCK) continue;
		if (status != -1)
		{
			fprintf(stderr,
				"receive_data() received too few bytes");
			fprintf(stderr, "; received %d, expected %d\n",
				status, message_len);
		} else
			perror("receive_data()");
		panic_exit();
  	}
  }

  return message_len;
}



/*
 * Check if the number x is near enough to 0.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int ciszero(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return iszero(rounded_x);

}


/*
 * Check if the number x is near enough to 0 or positive.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cispos(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return (iszero(rounded_x) || !signbit(rounded_x));
}



/*
 * Check if the number x is near enough to 0 or negative.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cisneg(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return (iszero(rounded_x) || signbit(rounded_x));
}


