/**************************************************************************/
/* io.c --- almost identical to a version by F. J. Jungen       /\/\      */
/* Version 2.2.1 --  January 1992                               \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#include "config.h"

#include <stdio.h>
#include <ctype.h>
#include <string.h>


#include "io.h"

#define	START		0
#define COMMENT		1
#define	READING		2
#define	READY		3

#define	COMMENT_CHAR	'#'
#define	WORD_SIZE	512



char line[WORD_SIZE];
int lineno = 1;
int no_links = 0;


struct link
{
	int variable;
	REAL par[NR_DH_PARAMS];
} robot[MAX_NR_LINKS];




/*
 * parsing the robot definition file
*/
static void exception(char *s)
{
  fprintf(stderr, "On line %d of the DHfile: ", lineno);
  fprintf(stderr, s);
  exit(1);
}



static void skip_spaces(FILE *fp)
{
  int c;

  do
  {
	c = fgetc(fp);
	if (c == '\n') lineno ++;
  } while (isspace(c));
  ungetc(c, fp);
}



static void skip_rest(FILE *fp)
{
  int c;

  while ((c = fgetc(fp)) != '\n' && c != EOF);
  lineno++;
}



/*
 * read_word reads a word from file pointed to by fp.
 * Comments etc. are skipped.
 */
static int read_word(FILE *fp, char *line)
{
  int state, i;
  int c;

  state = START;
  i = 0;
  while (state != READY)
	switch(state){
		case START:
			skip_spaces(fp);
			if ((c = fgetc(fp)) == COMMENT_CHAR)
				state = COMMENT;
			else if (c == EOF)
			{
				line[0] = '\0';
				state = READY;
			}
			else
				state = READING;
			ungetc(c, fp);
				break;
		case COMMENT:
			skip_rest(fp);
			state = START;
			break;
		case READING:
			while (! isspace((c = fgetc(fp))) &&
				      c != COMMENT_CHAR && i < WORD_SIZE - 1)
				line[i ++] = c;
			if (i == WORD_SIZE - 1 && c != COMMENT_CHAR &&
				   ! isspace(c))
				exception("word too long.\n");
			ungetc(c, fp);
			while (isspace(line[--i]));
			line[++i] = '\0';
			state = READY;
			break;
	}
  if (line[0] == '\0') return 0;

  return 1;
}



/*
 * operations on the robot datastructure
*/



/*
 * returns an int which specifies which parameter of the link is variable.
 */
int get_variable(int link)
{
  if (link < 1 || link > no_links) return 0;
  return(robot[link - 1].variable);
}



/*
 * get_param returns the current value of parameter par.
 */
REAL get_param(int link, int par)
{
  if (link < 1 || link > no_links) return 0;
  if (par < DH_THETA || par > DH_ALFA) return 0;
  return(robot[link - 1].par[par]);
}



/*
 * variable_check checks if only THETA or D are variable, if not
 * a error message is produced and we exit.
 */
static void variable_check(void)
{
  int i;

  for (i = 1 ; i <= no_links ; i ++)
  {
	if (get_variable(i) == DH_A)
	{
		fprintf(stderr, "At link %d: A is not allowed", i);
		fprintf(stderr, " to be variable.\n");
		exit(1);
	}
	if (get_variable(i) == DH_ALFA)
	{
		fprintf(stderr, "At link %d: ALFA is not ", i);
		fprintf(stderr, "allowed to be variable.\n");
		exit(1);
	}
  }
}



/*
 * read_definition reads a robotdefinition from a file specified by filename.
 */
void read_definition(char filename[])
{
  FILE *fp;
  int i;
  struct link *p;
  double dummy;

  if ((fp = fopen(filename, "r")) == NULL)
  {
	fprintf(stderr, "Could not open %s.\n", filename);
	exit(1);
  }

  i = 0;
  p = & robot[0];
  p -> variable = NO_VAR;
  while (read_word(fp, line))
  {
	if (no_links >= MAX_NR_LINKS)
		exception("To many links specified.\n");
	if (! strcmp(line, VARIABLE))
		if (p -> variable == NO_VAR)
		{
			p -> variable = i;
			i ++;
		}
		else
			exception("More than one variable in link\n");
	else
		if (sscanf(line, "%lf", &dummy) != 1)
			exception("dh_parameter expected.\n");
		else 
		{
			p -> par[i] = (REAL) dummy;
			i ++;
		}
	if (i == NR_DH_PARAMS)
	{
		if (p -> variable == NO_VAR)
			exception("No variable at link.\n");
		i = 0;
		p ++;
		no_links ++;
		p -> variable = NO_VAR;
	}
  }
  if (no_links == 0) exception("No links specified.\n");
  if (i != 0) exception("EOF encountered during read.\n");
  fclose(fp);
  variable_check();
}



/*
 * substitute fills the joint variables specified in buffer
 */
void substitute(REAL buffer[])
{
  int i;

  for (i = 0 ; i < no_links ; i ++)
	robot[i].par[robot[i].variable] = buffer[i];
}



void print_robot(void)
{
  int i, j;

  for (i = 1 ; i <=  no_links ; i ++)
  {
	for (j = DH_THETA ; j <= DH_ALFA ; j ++)
		if (j == get_variable(i)) printf("variable ");
		else printf("%9.4lf ", (double) get_param(i, j));
	printf("\n");
  }
}
