/**************************************************************************/
/* datastruct.c --- after a version by F. J. Jungen             /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/


#include "config.h"

#define DATASTRUCT_EXTERN

#include <stdio.h>
#include <stdlib.h>

#include "datastruct.h"
#include "matrix.h"
#include "main.h"



/*
 * add_link adds a link to the robot: if the maximum number of links is exceeded
 * then an error message is produced and the program halts.
 */
void add_link(void)
{
  nr_links++;

  if (link_end == NULL)
  {
	/*
	 * The first point.
	 */
	link_start=(struct link_struct *) calloc(1, sizeof(struct link_struct));
	link_start -> next = NULL;
	link_end = link_start;
	return;
  }

  link_end -> next=(struct link_struct *) calloc(1, sizeof(struct link_struct));
  link_end = link_end -> next;
  link_end -> next = NULL;
  link_end -> nr_points = link_end -> nr_lines = 0;
}



/*
 * add_point: adds a point to the points. If the maximum  number of points is
 * exceeded then an error message is produced and the program halts. all the
 * points are stored in a 4 X n matrix where each point is a column where the
 * last coordinate is 1: (x,y,z,1).
 */
void add_point(REAL x, REAL y, REAL z)
{
  if (link_end -> point_start == NULL)
  {
	/*
	 * Create the first node.
	 */
	link_end -> point_start =
		(struct point_struct *) calloc(1, sizeof(struct point_struct));
	link_end -> point_end = link_end -> point_start;
	link_end -> point_end -> next = NULL;
  }
  else
  {
	link_end -> point_end -> next =
		(struct point_struct *) calloc(1, sizeof(struct point_struct));
	link_end -> point_end = link_end -> point_end -> next;
	link_end -> point_end -> next = NULL;
  }
  link_end -> point_end -> points[0] = x;
  link_end -> point_end -> points[1] = y;
  link_end -> point_end -> points[2] = z;
  link_end -> point_end -> points[3] = 1.0;
  link_end -> nr_points++;
}



/*
 * add_line adds a line to the current link. If the maximum number of lines is
 * exceeded an error message is produced and the program exits.
 */
void add_line(int p1, int p2)
{
  if (z_projection)
	no_segments += 2;
  else
	no_segments ++;

  if (link_end -> line_start == NULL)
  {
	link_end -> line_start =
		(struct line_struct *) calloc(1, sizeof(struct line_struct));
	link_end -> line_end = link_end -> line_start;
	link_end -> line_end -> next = NULL;
  }
  else
  {
	link_end -> line_end -> next =
		(struct line_struct *) calloc(1, sizeof(struct line_struct));
	link_end -> line_end = link_end -> line_end -> next;
	link_end -> line_end -> next = NULL;
  }
  link_end -> line_end -> start = p1 - 1;
  link_end -> line_end -> finish = p2 - 1;

  link_end -> nr_lines++;
}





/*
 * convert converts the datastructure in one that uses less space, by making 
 * the points matrices exactly the correct size, and creates the result matrices
 */
void convert(void)
{
  int r,c;
  struct link *p;
  struct link_struct *link_ptr;

  /*
   * First allocate the links.
   */
  if ((links = (struct link *) calloc(nr_links, sizeof(struct link))) == NULL)
  {
	perror("convert(): links");
	exit(1);
  }
  link_ptr = link_start;

  for (p = &links[0]; p < &links[nr_links]; p++)
  {
	struct point_struct *point_ptr;
	struct line_struct *line_ptr;

	if (link_ptr == NULL)
	{
		fprintf(stderr, "PANIC: 1 bad data structure in convert()\n");
		exit(1);
	}

	/*
	 * In each link, allocate the points.
	 */
	if ((p -> points = (matrix)
		calloc(4 * link_ptr -> nr_points, sizeof(REAL))) == NULL)
	{
		perror("p -> points");
		panic_exit();
	}
	if ((p -> result = (matrix)
		calloc(4 * link_ptr -> nr_points, sizeof(REAL))) == NULL)
	{
		perror("p -> result");
		panic_exit();
	}
	p -> nr_points = link_ptr -> nr_points;

	point_ptr = link_ptr -> point_start;
	for (c = 0; c < link_ptr -> nr_points; c++)
	{
		if (point_ptr == NULL)
		{
			fprintf(stderr, 
			"PANIC: 2 bad data structure in convert()\n");
			exit(1);
		}

		for (r = 0; r < 4; r++)
			p->points[r*p->nr_points + c] = point_ptr -> points[r];

		/*
		 * Throw away the point we just stored.
		 */
		point_ptr = point_ptr -> next;
		free(link_ptr -> point_start);
		link_ptr -> point_start = point_ptr;
	}

	/*
	 * Next, allocate the lines.
	 */
	if ((p -> lines = (struct line *) calloc(link_ptr -> nr_lines,
					sizeof(struct line))) == NULL)
	{
		perror("convert(): lines");
		exit(1);
	}
	p -> nr_lines = link_ptr -> nr_lines;

	line_ptr = link_ptr -> line_start;
	for (c = 0; c < link_ptr -> nr_lines; c++)
	{
		if (line_ptr == NULL)
		{
			fprintf(stderr, 
			"PANIC: 3 bad data structure in convert()\n");
			exit(1);
		}
		 
		p -> lines[c].start = line_ptr->start;
		p -> lines[c].finish = line_ptr->finish;

		/*
		 * Throw away the structure just copied.
		 */
		line_ptr = line_ptr->next;
		free(link_ptr -> line_start);
		link_ptr -> line_start = line_ptr;
	}

	/*
	 * Finally, throw away and increment the main structure.
	 */
  	link_ptr=link_ptr -> next;
	free(link_start);
	link_start = link_ptr;
  }


  segs = (struct segment *) calloc(no_segments, sizeof(struct segment));
  if (segs == NULL)
  {
	perror("convert(): segs");
	exit(1);
  }
}
