/**************************************************************************/
/* main.c                                                       /\/\      */
/* Version 2.2.1 --  August  1991                               \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

/*
 * Warning: this file contains some OSCAR-specific code.
 */

#define DEF_EXTERN
#define GLOBAL_EXTERN
#define MAIN_EXTERN

#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#include "global_simmel.h"
#include "main.h"
#include "communication.h"
#include "global_communication.h"

#include "setup.h"


int connect_to_neural_network;


/*
 * The name of the executable should consist of the string
 * defined here.  Insert a `%d' to indicate where the
 * numeral indicating the sequence number is defined.
 */
#define EXECUTABLE_NAME		"simmel%d"
#define ROBOT_SOCKET_NAME     	"../socketnr/robot"
#define CAMERA_SOCKET_NAME	"../socketnr/camera"
#define BEMMEL_SOCKET_NAME	"../socketnr/bemmel"


void main(int argc, char *argv[])
{
  int moving, wait;
  joint_values new_q, new_dq;
  homo cam_1_homo, cam_2_homo;
  /*
   * The sequence number (i.e., numeral in name string) of this executable.
   */
  int sequence;
  char socket_name[100];

  /*
   * This one is only to initialise the robot drawings.
   */

  if (argc > 2)
  {
	fprintf(stderr, "Usage:  %s [n][ss...]\n", argv[0]);
	fprintf(stderr, "n means connect to neural network\n");
	fprintf(stderr, "each s means connect to one simulator\n");
	exit(1);
  }

  if (argc == 2)
  {
	int i;

	for (i=0; i<strlen(argv[1]); i++)
		if (argv[1][i] == 'n')
			connect_to_neural_network++;
		else if (argv[1][i] == 's')
			bemmels_connected++;
		else
		{
			fprintf(stderr, "illegal argument %c\n", argv[1][i]);
			exit(1);
		}
  }

  signal(SIGQUIT, leave);
  signal(SIGINT, leave);
  
  /*
   * Place the object at a `random' position.
   */
  object_position[0] = 20.0;
  object_position[1] = 20.0;
  object_position[2] = 20.0;
  object_position[3] = 1.0;

  if (sscanf(argv[0], EXECUTABLE_NAME, &sequence) != 1)
  {
	fprintf(stderr, "Program has an unexpected name, change main()\n");
	exit(0);
  }

  /*
   * Position of the fixed cameras.
   */
  
  /*
   * Initialize the DH Matrix for fixed camera 1.
   * We actually never need this matrix, but rather its
   * inverse, which we calculate immediately.
   */
  dh_transformation(cam_1_homo,  90.0, 100.0, 200.0, 0.0);
  invert_homo(cam_1_inv_homo, cam_1_homo);
 
  
  /*
   * DH Matrix for fixed camera 2.
   * We actually never need this matrix, but rather its
   * inverse, which we calculate immediately.
   */
  dh_transformation(cam_2_homo, 180.0, 100.0, 200.0, 0.0);
  invert_homo(cam_2_inv_homo, cam_2_homo);
  
  
  if (sscanf(argv[0], EXECUTABLE_NAME, &sequence) != 1)
  {
	fprintf(stderr, "Program has an unexpected name, change main()\n");
	exit(0);
  }

 
  
  /*
   * Initialise forward kinematics.
   */
  init_forwardkinematics(DHfile);

  /*
   * The order of first starting the robot and next
   * the camera must correspond with the order in
   * the neural network simulator.
   */
  if (connect_to_neural_network)
  {
    	sprintf(socket_name, "%s%d", ROBOT_SOCKET_NAME, sequence);
	open_robot_communication(socket_name);
	sprintf(socket_name, "%s%d", CAMERA_SOCKET_NAME, sequence);
  	open_camera_communication(socket_name);
  }

  open_bemmel_communication(BEMMEL_SOCKET_NAME, bemmels_connected);

  /*
   * Only for OSCAR.
   */
  if (NR_LINKS == 6)
  {
	joint_values tmp_q;

	q[1] =   0.0;
	q[2] = -90.0;
	q[3] =  90.0;
	q[4] =   0.0;
	q[5] =  90.0;
	q[6] =   0.0;
	decouple_joints(tmp_q, dq, ddq, q, dq, ddq);

	forward_kinematics(tmp_q);
	display_robot();
  }
  else
	fprintf(stderr, "Warning: joint values not initialised\n");

  if (connect_to_neural_network == 0)
  {
	printf("Starting in test mode...\n");
	test_simulator();
  }

  /*
   * Initially, there is no move to make.  So, set target time to 0
   * to prevent motion.
   */
  moving = wait = 0;

  while (1)
  {
	int command, i;
	unsigned short length;
	char *buffer, type;
	joint_values tmp_q, tmp_dq, tmp_ddq;
	vector dummy;


	/*
	 * This is rather essential: move the robot.  If this
	 * is not done here, it is possible that the simulator
	 * is continuously interrupted, such that the trajectory
	 * is never followed.
	 *
	 * Meanwhile, communication channels are checked for
	 * new incoming data.  There are two calls, since the
	 * second one is allowed to be blocking, but the first
	 * one isn't.  Thus we assure that, when this program
	 * may be dormant, it'll sleep in the select_receive_
	 * _next_command() function; but when the robot is still
	 * in motion, it will only check the incoming channels for
	 * data and read if there is data available.
	 */
	if (moving && !wait)
	{
		moving = move(new_q, new_dq, ddq);
		command = select_receive_next_command(&type,&buffer,&length,0);
	}
	else
		command = select_receive_next_command(&type,&buffer,&length,1);

	switch(command)
	{
	case NO_DATA:
		/*
		 * Check for the most probable one first.
		 */
		break;

	case ROBOT:
		/*
		 * Got a message for the robot.
		 */

		switch(type)
		{
		case NEWPOS:
			if (decode_joints(buffer, length, new_q, new_dq, ddq))
				break;
			moving = 1;
			break;

		case SEND_JOINTS:
			send_joints(q, dq, ddq);
			break;

		case HALT:
			for (i=1; i<=NR_LINKS; i++)
				new_dq[i] = dq[i] = ddq[i] = 0.0;
			accept_halt();
			break;

		case STOP:
		case STOP_OK:
			close_down(STOP_OK, STOP, STOP);
			/*
			 * Never reached.
			 */
			break;

		case PANIC:
			close_down(PANIC_OK, PANIC, PANIC);
			/*
			 * Never reached.
			 */
			break;

		case START:
			/*
			 * Might have received another START when
			 * the other party didn't receive our acceptation
			 * in time.  Ignore it.
			 */
			break;

		case WAIT:
			/*
			 * Make sure the robot doesn't move for
			 * a while.  Should always be followed
			 * by a CONTINUE.
			 */
			wait = 1;
			break;

		case CONTINUE:
			/*
			 * Undo a WAIT.
			 */
			wait = 0;
			break;

		default:
			fprintf(stderr,
			"Robot received illegal message `%c' (%d)\n",
							type, type);
			close_down(PANIC, PANIC, PANIC);
			break;
		}
		break;


	case CAMERA:
		/*
		 * Got a message for the camera.
		 */

		switch(type)
		{
			/*
			 * There are four sets of values sent from
			 * the camera:
			 *	1. the position of the end-effector
			 *	   and object as seen by fixed camera 1;
			 *	2. the position of the end-effector
			 *	   and object as seen by camera 2;
			 *	3. the position of the object
			 *	   RELATIVE to the robot's fingers;
			 *	4. the area of the object
			 *	   RELATIVE to the camera's CCD.
			 *	   This CCD is mounted above the
			 *	   fingers, at a height CCD_FINGERS_DISTANCE.
			 */
		case CAMERA_1:
			if (!wait) moving = move(new_q, new_dq, ddq);
			send_camera(CAMERA_1, cam_1_eef, cam_1_obj);
			break;
		case CAMERA_2:
			if (!wait) moving = move(new_q, new_dq, ddq);
			send_camera(CAMERA_2, cam_2_eef, cam_2_obj);
			break;
		case HAND_CAMERA:
			if (!wait) moving = move(new_q, new_dq, ddq);
			send_camera(HAND_CAMERA, hand_camera, area);
			break;

		case END_EFFECTOR:
			send_end_effector(effector);
			break;
			
		case OBJECT_POSITION:
			/*
			 * Tell the simulator where the object must be.
			 */
			decode_camera(buffer, length, object_position);
			display_robot();
			break;

		case STOP:
		case STOP_OK:
			close_down(STOP, STOP_OK, STOP);
			/*
			 * Never reached.
			 */
			break;

		case PANIC:
			close_down(PANIC, PANIC_OK, PANIC);
			/*
			 * Never reached.
			 */
			break;

		case START:
			/*
			 * Might have received another START when
			 * the other party didn't receive our acceptation
			 * in time.  Ignore it.
			 */
			break;

		default:
			fprintf(stderr, "Camera got illegal message `%c'\n",
								type);
			close_down(PANIC, PANIC, PANIC);
			break;
		}
		break;


	default:
		fprintf(stderr,"receive_next_command() gives illegal return\n");
		close_down(PANIC, PANIC, PANIC);
		/*
		 * Never reached.
		 */
		break;
	}

	/*
	 * This is a good moment to see if the graphics
	 * simulators have called it a day.  They might also
	 * have sent a HALT command, in which case the
	 * robot must be halted.  Note that the accept_halt()
	 * is already performed in check_bemmel_channels_for_data().
	 */
	if (check_bemmel_channels_for_data() == HALT)
		for (i=1; i<=NR_LINKS; i++)
			new_dq[i] = dq[i] = ddq[i] = 0.0;
  }

  /*
   * Never reached, too.
   */
  close_down(STOP, STOP, STOP);
}



/*
 * Run a test of the kinematic simulator.
 */
void test_simulator(void)
{
  E = 100.0; focus = 4.0;

  if (NR_LINKS != 6)
  {
	fprintf(stderr, "test_simulator() only implemented for 6DoF robot\n");
	return;
  }

  while (1)
  {
	joint_values new_q, new_dq;
	int i;

	printf("current position: %.0f %.0f %.0f %.0f %.0f %.0f\n",
				q[1], q[2], q[3], q[4], q[5], q[6]);
	printf("q1");
	for (i=1; i<=NR_LINKS; i++)
	{
		double p;
		scanf("%lf", &p);
		new_q[i] = (REAL) p;
		getchar();
	}
	for (i=1; i<=NR_LINKS; i++) printf(".   "); printf("\n");
	printf("q1");
	for (i=2; i<=NR_LINKS; i++) printf(", q%d", i); printf(": ");
	for (i=1; i<=NR_LINKS; i++)
	{
		double p;
		scanf("%lf", &p);
		new_dq[i] = (REAL) p;
		getchar();
	}
	for (i=1; i<=NR_LINKS; i++) printf("..  "); printf("\n");
	printf("q1");
	for (i=2; i<=NR_LINKS; i++) printf(", q%d", i); printf(": ");
	for (i=1; i<=NR_LINKS; i++)
	{
		double p;
		scanf("%lf", &p);
		ddq[i] = (REAL) p;
		getchar();
	}

	while (move(new_q, new_dq, ddq))
		printf("Hand v=(%.2f %.2f %.2f) a=(%.2f %.2f %.2f), e=[%.2f %.2f %.2f)\n",
			v[6][0], v[6][1], v[6][2], a[6][0], a[6][1], a[6][2],
			area[0], area[1], area[2]);
  }
}
