/* 
 * Routines for reading/writing packets
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"


write_sint_to_client(int i)	/* write a short integer as two bytes, low-order first */
{
  write_byte_to_client(0xff & i);
  write_byte_to_client(0xff & (i>>8));
}

write_string_to_client(char *s)	/* write string s, with length as first byte */
{
  write_byte_to_client(strlen(s));
  while(*s)
    write_byte_to_client(*s++);
}

int
read_sint_from_client()		/* reads a short integer from input stream */
{
  int n;
  n = read_byte_from_client();
  n = n | (read_byte_from_client() << 8);
  if (n > 0x7fff) n -= 0x10000;
  return(n);
}

int 
read_word_from_client()		/* reads a long integer (4 bytes) */
{
  int n;
  n = (read_byte_from_client() << 24);
  n = n | (read_byte_from_client() << 16);
  n = n | (read_byte_from_client() << 8);
  n = n | read_byte_from_client();
  return(n);
}


int
sync_client()			/* syncs to a new client by echoing packets */
{
  int loop = 1;
  int c;
  int count = 0;
  char buf[20];
  while (client_data_waiting_timeout(5000))
    {
      if (process_input() >= 0)
	{
	  c = read_byte_from_client();
	  sprintf(buf,"SYNC>>> %d",c);
	  connect_message(buf);
	  process_events();
	  echo_packet();
	  if (c != count++) count = 0;
	  if (count > 2) return(0);
	}
    }
  return(-1);
}


/* Variables used in sending out info packets */

int motor_packet_count = 0;	/* packet sending variables */
int sonar_packet_count = 0;
int sonar_polling_flag = 0;

int sonar_num;			/* sonar parameters */
int current_sonar_num = 0;

float head_control = -1.0;	/* heading control variable */
int say_control = 0;		/* acknowledgment number for speech */

extern int single_step_mode;
extern int step_wait;
extern int nohalt;

/* Send out robot info packets to client */

send_client_packets()		/* send out info */
{
  if (motor_packet_count-- > 0)	/* motor packets */
    { send_motor_packet(); }
  if (sonar_polling_flag && sonar_packet_count-- > 0)
    { send_sonar_packet(); }	/* sonar packets */
}


write_packet_header(int n)	/* writes a packet header of type n */
{
  write_byte_to_client(n);
}

float 
dist_conv_factor = 1.0;		/* different robots have different values */

int
norm_pos(float f)		/* normalize from 0 to 2^15 - 1 */
{
  int d;
  d = (int)(f/dist_conv_factor) % 0x8000;
  if (d < 0) d += 0x8000;
  return(d);
}


float 
angle_conv_factor = RAW_TO_RAD;	/* robot angular units */
float
vel_conv_factor = 1.0;		/* robot velocity units, initially mm/sec */
float
diff_conv_factor = 1.0/ANG_VEL_TO_WHEEL_DIFF; /* robot wheel differential */

send_motor_packet()
{
  int d;
  int x, y;
  
  write_packet_header(motor_header());
  write_sint_to_client(norm_pos(erratic.dx));
  write_sint_to_client(norm_pos(erratic.dy));
  write_sint_to_client((int)(ang_to_rad(erratic.dth)/angle_conv_factor));
  d = (int)(erratic.rv/diff_conv_factor);
  write_sint_to_client((int)((erratic.tv - d)*0.5/vel_conv_factor));
				/* left wheel velocity */
  write_sint_to_client((int)((erratic.tv + d)*0.5/vel_conv_factor));
				/* right wheel velocity */
  write_byte_to_client(25);	/* battery voltage */
  write_sint_to_client(0);	/* bumpers */
  write_sint_to_client((head_control < 0) ? -1 : 
		       (int)(head_control/angle_conv_factor));
  write_byte_to_client(say_control); /* ack always */
  output_packet();
}

int
motor_header()			/* motor header flag */
{
  return(MOTORpac | ((erratic.tv == 0.0 && erratic.rv == 0.0) ? 2 : 3));
}
 

int 
sonar_header()			/* sonar header flag */
{
  if (++current_sonar_num >= sonar_num)
    current_sonar_num = 0;
  return(SONARpac | current_sonar_num);
}


send_sonar_packet()		/* send out two sonar packets */
{
  write_packet_header(sonar_header());
  write_sint_to_client(norm_pos(erratic.dx));
  write_sint_to_client(norm_pos(erratic.dy));
  write_sint_to_client((int)(ang_to_rad(erratic.dth)/angle_conv_factor));
  write_sint_to_client(current_sonar_value());
  output_packet();

  write_packet_header(sonar_header());
  write_sint_to_client(norm_pos(erratic.dx));
  write_sint_to_client(norm_pos(erratic.dy));
  write_sint_to_client((int)(ang_to_rad(erratic.dth)/angle_conv_factor));
  write_sint_to_client(current_sonar_value());
  output_packet();
}

set_sonar_polling(int arg)	/* sets up sonar polling */
{
}


/* Process all waiting incoming packets */


int
client_packet_waiting()		/* returns 1 if packets are waiting in the input queue */
{
  if (process_input() >= 0)
    return(1);
  else 
    return(0);
}

int
wait_client_packet()		/* waits until a client packet appears */
{
  while (process_input() < 0)
    { 
      ms_pause(100);
    }
}


void
process_waiting_packets()	/* processes any input packets */
{
  while(client_packet_waiting())
    process_one_client_packet();
}


process_one_client_packet()	/* process one packet */
{
  int pac_type, arg, count;
  static int stepcount;
  char *str;
  char * form_arg_string();
				/* Read the Packet */
  arg = 0; count = 0;
  pac_type = read_byte_from_client();
  switch(arg = read_byte_from_client()) /* read args */
    {
    case ARGchar:
      arg = read_sint_from_client();
      break;
    case NARGchar:
      arg =  - read_sint_from_client();
      break;
    case SARGchar:
      str = form_arg_string();
      break;
    case ENDchar:
      arg = 0;
      break;
    default:
      printf("\n*** Warning: bad argument type %d in packet.",arg);
      break;
    }

  if (debug) printf("\nPac type: %d   Arg: %d",pac_type,arg);

  switch(pac_type)
      {
      case STEP_COM:		/* stepping mode */
	if (arg == 0)		/* turn it off */
	  { message("Returning to continuous simulation");
	    single_step_mode = 0;
	  }
	else if (arg > 0)
	  { message("Starting single step mode");
	    single_step_mode = 1;
	    step_wait = 1;
	    stepcount = 1;
	  }
	else
	  { 
	    smessage("step...%d",stepcount++);
	    step_wait = 0;
	  }
	break;

      case PULSE_COM:		/* pulse motor or sonar */
	count = (arg & 0xff00) >> 8;
	if (count <= 0) count = 1;
	if (arg & MOTOR_ARG)
	  motor_packet_count = count;
	if (arg & SONAR_ARG)
	  sonar_packet_count = count;
	break;

      case OPEN_COM:
	message("Client open request");
	break;

      case CLOSE_COM:		/* close client connection */
	message("Client close request");
	if (arg > 0)
	  {
	    shutdown_client();
	    nohalt = 0;
	    step_wait = 0;
	  }
	break;

      case POLLING_COM:		/* start or stop sonar polling */
	if (arg)
	  {
	    sonar_polling_flag = 1;
	    set_sonar_polling(arg);
	  }
	else sonar_polling_flag = 0;
	break;

      case SETV_COM:		/* set max velocity, in mm/sec */
	erratic.mtv = (float)arg;
	break;
	
      case SETRV_COM:		/* set rot velocity increment, in deg/sec */
	arg = deg_to_ang((float)arg);
	erratic.mrv = arg;
	break;

      case SETA_COM:		/* set acceleration, in mm/sec/sec */
	erratic.mta = (float)arg;
	erratic.mra = (float)arg * TRANS_TO_ROT_ACC; 
	break;
	 
      case SETO_COM:		/* set origin */
	robot_set_origin();
	break;

      case MOVE_COM:		/* move forward, in mm */
	head_control = -1.0;
	c_move((float)arg);
	break;

      case ROTATE_COM: /* rotate to a heading, in degrees */
	head_control = -1.0;
	c_rotate_deg((float)arg);
	break;

      case DROTATE_COM:		/* rotate arg degrees */
	head_control = -1.0;
	c_rotate_deg((float)arg);
	break;

      case VEL_COM:		/* achieve a velocity and hold it, in mm/sec */
	c_trvel((float)arg);
	break;

      case RVEL_COM:		/* achieve a rot velocity and hold it, in ang/sec */
	head_control = -1.0;
	c_rotvel((float)arg);
	break;

      case HEAD_COM:		/* achieve a heading, in deg */
	head_control = norm_angle(deg_to_ang((float)arg));
	c_rotate(sub2_ang(head_control,erratic.dth));
	break;

      case DHEAD_COM:		/* change heading, in deg */
	if (arg)
	  {
	    if (head_control < 0) head_control = erratic.dth;
	    head_control = add_ang(head_control,deg_to_ang((float)arg));
	    c_rotate(sub2_ang(head_control,erratic.dth));
	  }
	else
	  {
	    head_control = -1;
	    c_rotvel(0.0);
	  }
	break;

      case DCHEAD_COM:		/* change heading from current, in deg */
	head_control = erratic.dth;
	head_control = add_ang(head_control,deg_to_ang((float)arg));
	c_rotate(sub2_ang(head_control,erratic.dth));
	break;

      case SAY_COM:
	smessage("SAYING: %s",str);
	say_control++;
	break;
      default:
	printf("\n*** Warning: bad packet type %d.",pac_type);
	break;

      }
}


