/* 
 * Routines for reading/writing packets
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sgtty.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

write_sint_to_client(int i)	/* write a short integer as two bytes, low-order first */
{
  write_byte_to_client(0xff & i);
  write_byte_to_client(0xff & (i>>8));
}

write_string_to_client(char *s)	/* write string s, with length as first byte */
{
  write_byte_to_client(strlen(s));
  while(*s)
    write_byte_to_client(*s++);
}

int
read_sint_from_client()		/* reads a short integer from input stream */
{
  int n;
  n = read_byte_from_client();
  n = n | (read_byte_from_client() << 8);
  if (n > 0x7fff) n -= 0x10000;
  return(n);
}

int 
read_word_from_client()		/* reads a long integer (4 bytes) */
{
  int n;
  n = (read_byte_from_client() << 24);
  n = n | (read_byte_from_client() << 16);
  n = n | (read_byte_from_client() << 8);
  n = n | read_byte_from_client();
  return(n);
}


int
sync_client()			/* syncs to a new client by echoing end-chars */
{
  int i, c;
  for (i = 0; i < 3; i++)
    {
      message("Synching...");
      clear_client_input();
      write_byte_to_client(i);
      output_packet();
      if (client_data_waiting_timeout(1000))
	{
	  ms_pause(100);
	  process_input();
	  c = read_byte_from_client();
	  printf("\nSYNC>>> %d",c); fflush(stdout);
	  if (c == 2) return(0);
	}
    }
  return(-1);
}


/* sending commands to flakey */

write_command(c)
{
  write_byte_to_client(c);
}


int trace_commands = 0;

command_to_flakey(com)
{
  if (trace_commands)
    printf("\nCom: %d",com);
  write_command(com);
  output_packet();
}

command_to_flakey_i(com,arg)	/* integer argument */
{
  if (trace_commands)
    printf("\nCom: %d %d",com,arg);
  write_command(com);
  if (arg >= 0) write_byte_to_client(ARGchar);
  else write_byte_to_client(NARGchar);
  write_sint_to_client(ABS(arg));
  output_packet();
}

command_to_flakey_s(int com,char *str) /* string argument */
{
  if (trace_commands)
    printf("\nCom: %d %s",com,str);
  write_command(com);
  write_byte_to_client(SARGchar);
  write_string_to_client(str);
  output_packet();
}
 


open_motor_controller(high_power, speech) /* try opening motor controller */
{
  int i, arg;
/*  command_to_flakey(CLOSE_COM); */ 
  for (i=0; i<4; i++)
    {
      clear_client_input();
      if (!speech) arg = -1;
      else if (high_power) arg = HIGH_POWER_ARG;
      else arg = NORMAL_POWER_ARG;
      command_to_flakey_i(OPEN_COM,arg);
      command_to_flakey_i(PULSE_COM,MOTOR_ARG); /* send a packet requesting info */
      if (wait_client_packet(1000))
	{
	  process_one_client_packet(1);
	  return(0);
	}
    }
  return(-1);
}
      


/* Process all waiting incoming packets */


int
client_packet_waiting()		/* returns 1 if packets are waiting in the input queue */
{
  while(client_data_waiting())
    {
      if (process_input() >= 0)
	return(1);
      else 
	return(0);
    }
  return(0);

}

int
wait_client_packet(int wait)	/* waits until a client packet appears */
{
  int count = 0;
  while (process_input() < 0)
    { 
      ms_pause(100);
      count += 100;
      if (count >= wait) return(0);
    }
  return(1);
}


process_waiting_packets()	/* processes any input packets */
{
  while(client_packet_waiting())
    process_one_client_packet(0);
}


int trace_packets = 0;

sdata sbucket[MAX_SONARS];	/* holds sonar data */

process_sonar_packet(float x,float y,float th,int n,int s)
{
  sdata *r;
  if (n >= 0 && n < MAX_SONARS)	/* check bounds */
    {
      r = &sbucket[n];
      r->fx = x; r->fy = y; r->fth = th; 
      r->range = s; r->new = True;
      sonar_coords(n);		/* calculate RW coordinates */
    }
  else
    {
      printf("\n*** Illegal sonar number: %d\n",n); 
      fflush(stdout);
    }
}


process_one_client_packet(int init) /* process one packet */
{
  int pac_type, arg, range;
  int status;
  char *s;
  int x,y,th, tx, ty, tth;	/* odometry readings */
  float dx, dy, dth;		/* incremental movement */
  static int last_x, last_y, last_th;

  /* Read the Packet */
  pac_type = read_byte_from_client();
  status = pac_type & 0x0f;	/* extract status */
  pac_type  = pac_type & 0xf0;	/* and pac type nibbles */
  if (trace_packets)
    { 
      switch(pac_type)
	{
	case MOTORpac:
	  s = "mot";
	  break;
	case DEPTHpac:
	  s = "dep";
	  break;
	case SONARpac:
	  s = "son";
	  break;
	default: 
	  s = "oth";
	  break;
	}
      printf(" %s",s);
      fflush(stdout);
    }
  x = read_sint_from_client() & 0x7fff;	/* odometer absolute values in mm */
  y = read_sint_from_client() & 0x7fff;
  if (init)
    { last_x = x; last_y = y; }
  tx = x - last_x;
  ty = y - last_y;
  if (tx > 0x4000) tx -= 0x8000;	/* overflow check */
  if (ty > 0x4000) ty -= 0x8000;	/* overflow check */
  dx = (float)tx; dy = (float)ty;
  prot(&dx,&dy,sub_ang((RAW_TO_RAD * last_th),flakey.th)); /* rotate to current direction */
  
  th = read_sint_from_client();
  if (init)
    { last_th = th; }
  tth = th - last_th;		/* this is the incremental change in angle */
  dth = norm2_angle(RAW_TO_RAD * tth);

  switch(pac_type)		/* do differential packet processing */
    {
    case MOTORpac:
      flakey.motor_packet_count++;
      flakey.status = status;
      last_th = th;		/* update previous readings */
      last_x  = x;
      last_y  = y;
      flakey.x += dx;
      flakey.y += dy;
      flakey.th = add_ang(dth,flakey.th);
      flakey.leftv = read_sint_from_client();
      flakey.rightv = read_sint_from_client();
      flakey.tv = (flakey.leftv + flakey.rightv) * 0.5;
      flakey.rv =  (flakey.leftv - flakey.rightv) / ANG_VEL_TO_WHEEL_DIFF;
      flakey.battery = read_byte_from_client();
      flakey.bumpers = read_sint_from_client();
      th = (float)read_sint_from_client(); /* heading control */
      if (th < 0) flakey.control = 1000.0;
      else flakey.control = norm2_angle(RAW_TO_RAD*(th - last_th));
      flakey.say = read_byte_from_client();
      break;
      
    case SONARpac:
      flakey.sonar_packet_count++;
      range = read_sint_from_client();
      process_sonar_packet(dx+flakey.x,dy+flakey.y,add_ang(dth,flakey.th),status,range);
      break;
    }
}




