/*#########################################
 * win.c
 *
 * Window display functions for the Erratic simulator.
 *
 *#########################################
 */

#include <malloc.h>
#include <stdio.h>
#include "structures.h"
#include "constants.h"
#include "parameters.h"

int draw_wake = 0;		/* 1 if drawing the wake, 0 if not */
int wake_step = 10;		/* leave one bean every 10 ticks */
int wake_cur_step = 0;		/* counter */

int win_flag = 0;		/* 1 if window present */
int win_w, win_h;		/* window width and height */

int refresh_flag = 0;		/* window refreshing flag */

int win_scroll_x = 0;		/* window scrolling vars */
int win_scroll_y = 0;

#define check_win if (win_flag == 0) return

/* scaling values */

float pix_to_mm = 15.0;		/* 15 mm per pixel, initially */
int err_pix_rad = 15.0 / 290.0;	/* erratic radius in pixels */

#define MAX_WIN_H 700		/* maximum screen sizes */
#define MAX_WIN_W 800
#define START_WIN_H 400		/* startup screen sizes */
#define START_WIN_W 500

int heading_length = 0;		/* in pixels, for heading vector display */

window_scale(ptomm)		/* changes pix_to_mm */
     float ptomm;
{
  pix_to_mm = ptomm;
  heading_length = (int)(200.0 / pix_to_mm);
  err_pix_rad = (int)(erratic.rad / ptomm);
}


setup_display_window(argc,argv) /* do a display with this width and height */
     int argc; char **argv;
{
  window_scale(pix_to_mm);
  win_w = START_WIN_W;
  win_h = START_WIN_H;
  win_scroll_x = win_scroll_y = 0;
  if (win_flag) kill_window();
  init_window(win_w,win_h,argc,argv);
  win_flag = 1;
}

kill_display_window()
{
  if (win_flag) kill_window();
  win_flag = 0;
}


int old_err_x = 0;		/* old position to erase */
int old_err_y = 0;
int old_err_dx = 0;
int old_err_dy = 0;


draw_erratic()			/* draw the robot, erasing old one if necessary */
{
  check_win;
  set_graphics(XOR);		/* use XOR mode */
  if (refresh_flag) refresh_flag = 0;
  else
    {
      draw_oval(SCRX(old_err_x) - err_pix_rad, /* erase old robot */
		SCRY(old_err_y) - err_pix_rad,
		err_pix_rad * 2, err_pix_rad * 2);
      draw_dline(SCRX(old_err_x),SCRY(old_err_y),old_err_dx,old_err_dy);
    }
				/* now draw the new one */
  old_err_x = (int)(erratic.x / pix_to_mm);
  old_err_y = (int)(erratic.y / pix_to_mm);
  old_err_dx = (int)((err_pix_rad + 3) * cos(erratic.th));
  old_err_dy = (int)((err_pix_rad + 3) * sin(erratic.th));
  if (draw_wake)		/* drawing the wake... */
    {
      if (wake_cur_step == 0)
	{
	  draw_dline(SCRX(old_err_x),SCRY(old_err_y),0,0);
	  wake_cur_step = wake_step;
	}
      else wake_cur_step--;
    }
				/* draw erratic */
  old_err_dx = heading_length * cos(erratic.th);
  old_err_dy = - (heading_length * sin(erratic.th));
  draw_oval(SCRX(old_err_x) - err_pix_rad,
	    SCRY(old_err_y) - err_pix_rad,
	    err_pix_rad * 2, err_pix_rad * 2);
  draw_dline(SCRX(old_err_x),SCRY(old_err_y),old_err_dx,old_err_dy);
  set_graphics(DRAW);
}

/* segment operations */

void
draw_seg(s)			/* draws a single segment */
     seg *s;
{
  check_win;
  draw_line(SCRX(SCALE(s->x1)), SCRY(SCALE(s->y1)),
	    SCRX(SCALE(s->x2)), SCRY(SCALE(s->y2)));
}

void
draw_world()			/* redraw all segments */
{
  int i; seg **p;
  p = (seg **)world.ss.arr;
  for (i=0; i<world.ss.n; i++)
    {
      draw_seg(*p);
      p++;
    }
}




window_refresh()		/* clear and redraw */
{
  check_win;
  refresh_flag = 1;
  clear_window();
  draw_erratic();
  draw_world();
  display_buffer();
}


void
scroll_to_position(x,y)		/* scrolls center window to this absolute scr position */
     int x,y;
{
  win_scroll_x = x - win_w / 2;
  win_scroll_y = y + win_h / 2;
  window_refresh();
}

void
scroll_to_screen_position(x,y)	/* scrolls center window to this relative scr position */
     int x,y;
{
  win_scroll_x += x - win_w / 2;
  win_scroll_y -= y - win_h / 2;
  window_refresh();
}


float too_close = 1500.0;	/* robot won't get closer than this (mm) to window sides */
void
scroll_to_erratic_position()	/* scrolls window if necessary */
{
  int h,v,c;
  check_win;
  h = old_err_x - win_scroll_x;	/* offset window coords */
  v = win_scroll_y - old_err_y;
  c = too_close / pix_to_mm;
  if (h < c || v < c || h > win_w - c || v > win_h - c)
    {
      scroll_to_position(old_err_x,old_err_y);
    }
}


#define CHAR_H 14
#define CHAR_W  8

void 
display_pos_info()		/* write out x,y,th absolute position */
{
  draw_num((int)(erratic.x * 0.01), "%4d ", CHAR_W * 4, CHAR_H);
  draw_num((int)(erratic.y * 0.01), "%4d ", CHAR_W * 4, CHAR_H * 2);
  draw_num((int)(RAD_TO_DEG * erratic.th), "%4d ", CHAR_W * 4, CHAR_H * 3);
}


void
update_window()			/* update all info in the window */
{
  check_win;
  draw_erratic();
  scroll_to_erratic_position();
  display_pos_info();
  display_buffer();
}


