/* #########################################
 * sonar.c
 *
 * sonar sensor routines
 * for the erratic simulator
 *#########################################
 */

/*
 * 12 sonars are distributed  almost uniformly around the body of
 *  the robot
 *
 * This should be parameterizable
 */

#include "structures.h"
#include "constants.h"
#include "parameters.h"

#define NUM_SONARS 12

int sonar_num = NUM_SONARS;	/* number of sonars */
float sonar_long_range = SONAR_LONG_RANGE; /* long distance reflection */
float sonar_short_range = SONAR_SHORT_RANGE;	/* short distance reflection */
float sonar_out_of_range = SONAR_OUT_OF_RANGE; /* out of range */
float sonar_out_of_range2 = (SONAR_OUT_OF_RANGE * SONAR_OUT_OF_RANGE);

float sonar_good_reflection_angle = deg_to_ang(SONAR_GOOD_ANGLE);
float sonar_bad_reflection_angle = deg_to_ang(SONAR_BAD_ANGLE);
float sonar_jitter = 30;
float sonar_close_reflection_angle = deg_to_ang(SONAR_CLOSE_ANGLE);
float sonar_max_reflection_angle = deg_to_ang(30.0);
float sonar_close_dist = SONAR_SHORT_RANGE; /* where we start the close reflection, in mm */
float sonar_specular_offset = 500.0; /* distance to add for specular reflections, in mm */

float beam_half_ang = deg_to_ang(8.0); /* half-angle of sonar beam */
int beam_fan_num = 4;		/* number of fans to use */


/* sonar_buffer is an array holding the readings */

float sonar_buffer[NUM_SONARS];

#define CP4 0.7071067811865
#define SP4 0.7071067811865
#define MSP4 -0.7071067811865

float sonar_spec_array[NUM_SONARS][5] =	/* sonar specifications: x,y coord and direction */
{
  {+290,+80,1,0,0},
  {+290,-80,1,0,0},
  {+240,-200,0,-1,(- ANG90)},
  {+200,-240,1,0,0},
  {-200,-240,-1,0,ANG180},
  {-240,-200,0,-1,(- ANG90)},
  {+220,+220,CP4,SP4,ANG45},
  {+220,-220,CP4,(- SP4),(- ANG45)},
  {-240,+200,0,+1,ANG90},
  {-200,+240,-1,0,ANG180},
  {+200,+240,+1,0,0},
  {+240,+200,0,+1,ANG90}
};

float sonar_x(int i,float r)	/* returns x coord of sonar reading I range R */
{
  return((float)(sonar_spec_x(i) + r * sonar_spec_dx(i)));
}

float sonar_y(int i,float r)	/* returns y coord of sonar reading I range R */
{
  return((float)(sonar_spec_y(i) + r * sonar_spec_dy(i)));
}

/* calculates a reading for the Nth sonar element, given a segment S */

float
sonar_value(int n,seg *s)
{
  float x = erratic.x;
  float y = erratic.y;
  float th = erratic.th;
  float xp, yp;
  float x1 = s->x1, x2 = s->x2, y1 = s->y1, y2 = s->y2;
  float xi = 0.0, yi = 0.0, cs, sn;

  cs = cos(- th); sn = sin(- th); /* inverse transform */
  x +=  cs * sonar_spec_x(n) +  sn * sonar_spec_y(n);
  y +=  cs * sonar_spec_y(n) -  sn * sonar_spec_x(n);
  xp = x + cs * sonar_spec_dx(n) * sonar_long_range
         + sn * sonar_spec_dy(n) * sonar_long_range;
  yp = y + cs * sonar_spec_dy(n) * sonar_long_range
         + sn * sonar_spec_dx(n) * sonar_long_range;
  intpps(x,y,xp,yp,s,&xi,&yi);	/* xi,yi contains the intersection point */
  if (xi == 0.0 && yi == 0.0) return(sonar_long_range);
  else if (between(x,y,xp,yp,xi,yi) &&
	   between(x1,y1,x2,y2,xi,yi))
    return(sqdist(xi,yi,x,y));
  else return(sonar_long_range);
}
    

float
add_sonar_jitter(float range, float echo)
{
  float x;

  if (echo > 0.0)
    echo = randy(echo,1.0);
  
  x = (range + 1000.0) / sonar_jitter;
  if (x == 0.0) return(range+echo);
  else return(range+echo+(x*.2)+randy(x,1.0));
}


float check_seg();
float return_seg_ang();

float
sonar_values(int n,buclist *segs)   /* loops thru segs, returns nearest value */
{
  float x = erratic.x, y = erratic.y, th = erratic.th,
        res = sonar_out_of_range2,
        good, rth;
  int i; seg **p;
  
  setup_sonar(n,sonar_long_range,x,y,th,beam_half_ang,beam_fan_num);

  if (cur_seg_list != NULL)
    {
      p = (seg **)&cur_seg_list->segs;
      for (i=0; i<cur_seg_list->n; i++)
	{
	  res = check_seg(*p);	/* find intersection with current segments */
	  p++;
	}
    }

  if (res >= sonar_out_of_range2)
    return(sonar_out_of_range);
  else if (sonar_jitter == 0.0) return((float)sqrt(res));
  else
    {
      res = (float)sqrt(res);
      rth = return_seg_ang();
      rth = ABS(rth);
      good = res > sonar_close_dist ? sonar_good_reflection_angle :
       sonar_good_reflection_angle - sonar_close_reflection_angle * (res / sonar_close_dist);
      if (rth > good) return(add_sonar_jitter(res,0.0));
      else
	{
	  res += sonar_specular_offset + 
	    ((good - rth) * (res + sonar_specular_offset)) / sonar_max_reflection_angle;
	  res = randy(res,1.0);
	  return(add_sonar_jitter(res,0.0));
	}
    }
}


extern int current_sonar_num;

int
current_sonar_value()		/* get the next sonar firing */
{
  return((int)sonar_values(current_sonar_num,cur_seg_list));
}

