;;; -*- Package: Toolset; Syntax: Common-Lisp; Mode: Lisp; Base: 10 -*-

(in-package 'toolset)

(DEFCLASS HIERARCHY (TOOLBED::DATA-STRUCTURES) ((NODE-TYPE :TYPE SYMBOL 
                                                       :INITARG 
                                                       NODE-TYPE)
                                                (SUB-LIST :TYPE LIST 
                                                       :INITARG 
                                                       SUB-LIST 
                                                       :INITFORM NIL)
                                                (SUPER-LIST :TYPE LIST 
                                                       :INITARG 
                                                       SUPER-LIST 
                                                       :INITFORM NIL)))


(defmacro make-hierarchy (&optional (node-type T))
  `(pcl::*make-instance 'hierarchy :node-type ,node-type))


(DEFmacro GET-NODES (HIER) 
  `(MAPCAR #'CAR 
	  (SLOT-VALUE (if (symbolp ,hier)
			  (slot-value (eval ,hier) 'hierarchy)
			  (if (typep ,hier 'classifier)
			      (slot-value ,hier 'hierarchy)
			      ,hier)) 
		      'super-list)))


(DEFMACRO HIERARCHY-CONTENTS (HIER)
  `(GET-SUBS-PRESERVING-STRUCTURE (GET-TOP-NODE ,HIER)
                 ,HIER))


(DEFMACRO IS-NODE (NODE-NAME HIER) 
       
       ;; assumes that if node is in hier, it will be on both
       ;; sub-list and super-list
 `(FIND ,NODE-NAME (GET-NODES ,HIER)))



(DEFMACRO IS-SUB (CHILD PARENT HIER) 
  `(FIND ,CHILD (GET-SUBS
		 ,PARENT
		 ,HIER)))


(DEFMACRO IS-SUPER (PARENT CHILD HIER)
   `(FIND ,PARENT (GET-SUPERS ,CHILD ,HIER)))

(DEFmethod GET-SUBS (OBJ (HIER hierarchy)) 
  (IF (NOT (IS-NODE OBJ HIER))
      (ERROR 
       "~% The node named ~S doesn't exist." 
       OBJ)
      (CDR (ASSOC OBJ (SLOT-VALUE HIER
				  'SUB-LIST)))))

(DEFmethod GET-SUBS (OBJ (class classifier))
  (let ((hierarchy (slot-value class 'hierarchy)))
    (IF (NOT (IS-NODE OBJ HIERarchy))
	(ERROR 
	 "~% The node named ~S doesn't exist." 
	 OBJ)
	(CDR (ASSOC OBJ (SLOT-VALUE HIERarchy
				    'SUB-LIST))))))

(DEFmethod GET-SUBS (OBJ (classifier-name symbol))
  (let ((hierarchy (slot-value (eval classifier-name) 'hierarchy)))
    (IF (NOT (IS-NODE OBJ HIERarchy))
	(ERROR 
	 "~% The node named ~S doesn't exist." 
	 OBJ)
	(CDR (ASSOC OBJ (SLOT-VALUE HIERarchy
				    'SUB-LIST))))))



(DEFUN GET-SUBS-PRESERVING-STRUCTURE (NODE HIER)
   (IF (NOT (IS-NODE NODE HIER))
       (ERROR "~S is not a node." NODE))
   (RETURN-FROM GET-SUBS-PRESERVING-STRUCTURE
          (IF (NULL (GET-SUBS NODE HIER))
	      NODE
              (CONS NODE (MAPCAR #'(LAMBDA (X)
                                         (GET-SUBS-PRESERVING-STRUCTURE
                                                   X HIER))
                                (GET-SUBS NODE HIER))))))


(DEFmethod GET-SUPERS (OBJ (HIER hierarchy)) 
  (IF (NOT (IS-NODE OBJ HIER))
      (ERROR  
	     "~% The node named ~S doesn't exist." 
	     OBJ)
      (CDR (ASSOC OBJ (SLOT-VALUE
		       HIER
		       'SUPER-LIST)))))

(DEFmethod GET-SUPERS (OBJ (class classifier)) 
  (let ((hier (slot-value class 'hierarchy)))
    (IF (NOT (IS-NODE OBJ HIER))
	(ERROR  
	 "~% The node named ~S doesn't exist." 
	 OBJ)
	(CDR (ASSOC OBJ (SLOT-VALUE
			 HIER
			 'SUPER-LIST))))))

(DEFmethod GET-SUPERS (OBJ (classifier-name symbol)) 
  (let ((hier (slot-value (eval classifier-name) 'hierarchy)))
    (IF (NOT (IS-NODE OBJ HIER))
	(ERROR  
	 "~% The node named ~S doesn't exist." 
	 OBJ)
	(CDR (ASSOC OBJ (SLOT-VALUE
			 HIER
			 'SUPER-LIST))))))


(DEFSETF GET-SUBS (OBJ HIER)
   (NEW-SUB-LIST)
   `(PROGN 
     (RPLACD 
      (ASSOC ,OBJ 
	     (SLOT-VALUE 
	      (if (typep ,HIER 'hierarchy)
		   hier
		   (if (typep ,hier 'classifier)
		       (slot-value ,hier 'hierarchy)
		       (if (symbolp ,hier)
			   (slot-value (eval ,hier)
				       'hierarchy))))
	      'SUB-LIST))
      ,NEW-SUB-LIST)
     ,NEW-SUB-LIST))


(DEFSETF GET-SUPERS (OBJ HIER)
   (NEW-SUPERS-LIST)
   `(PROGN 
     (RPLACD 
      (ASSOC ,OBJ 
	     (SLOT-VALUE
	      (if (typep ,HIER 'hierarchy)
		   ,hier 
		   (if (typep ,hier 'classifier)
		       (slot-value ,hier 'hierarchy)
		       (if (symbolp ,hier)
			   (slot-value (eval ,hier)
				       'hierarchy))))
	      'SUPER-LIST))
      ,NEW-SUPERS-LIST)
     ,NEW-SUPERS-LIST))

(DEFUN GET-ALL-SUBS (NODE-NAME HIER)
   (IF (NOT (IS-NODE NODE-NAME HIER))
       (ERROR "~% The node named ~S doesn't exist." NODE-NAME)
       (LET ((SUBS-LIST (GET-SUBS NODE-NAME HIER)))
            (DOLIST (CHILD SUBS-LIST)
                   (SETF SUBS-LIST (APPEND (GET-ALL-SUBS CHILD HIER)
                                          SUBS-LIST)))
            (REMOVE-DUPLICATES SUBS-LIST))))


(DEFUN GET-ALL-SUPERS (NODE-NAME HIER)
   (IF (NOT (IS-NODE NODE-NAME HIER))
       (ERROR  "~% The node named ~S doesn't exist." NODE-NAME)
       (LET ((SUPERS-LIST (GET-SUPERS NODE-NAME HIER)))
            (DOLIST (PARENT SUPERS-LIST)
                   (SETF SUPERS-LIST (APPEND (GET-ALL-SUPERS PARENT 
                                                    HIER)
                                            SUPERS-LIST)))
            (REMOVE-DUPLICATES SUPERS-LIST))))


(DEFMACRO TIP-NODEP (NODE HIER)
   `(IF (NULL (GET-SUBS ,NODE ,HIER))
        T NIL))


(DEFmacro GET-TIP-NODES (HIER) 
  `(LET ((TIP-LIST NIL))
       (DOLIST (NODE (GET-NODES ,HIER))
	       (IF (NULL (GET-SUBS NODE ,HIER))
		   (SETF TIP-LIST
			 (CONS NODE TIP-LIST))))
       TIP-LIST))



(DEFMACRO TOP-NODEP (NODE HIER)
   `(IF (NULL (GET-SUPERS ,NODE ,HIER))
        T NIL))


(DEFMACRO GET-TOP-NODE (HIER)
   `(DOLIST (NODE (GET-NODES ,HIER))
           (IF (NULL (GET-SUPERS NODE ,HIER))
               (RETURN NODE))))



(DEFMACRO TREEP (HIER)
   `(DOLIST (CURRENT-NODE (GET-NODES ,HIER) T)
           (IF (> (LENGTH (GET-SUPERS CURRENT-NODE ,HIER))
                  1)
               (RETURN NIL))))


(DEFUN BREAK-LINK (NODE1 NODE2 HIER &KEY (DELETE NIL))
   (COND
      ((NOT (IS-NODE NODE1 HIER))
       (ERROR  " The node named ~S doesn't exist." NODE1))
      ((NOT (IS-NODE NODE2 HIER))
       (ERROR  " The node named ~S doesn't exist." NODE2))
      (T (IF (NOT (OR (IS-SUB NODE2 NODE1 HIER)
                      (IS-SUPER NODE2 NODE1 HIER)))
             (ERROR  "~% There is not a link between ~S and ~S." NODE1 
                    NODE2)
             (LET (CHILD PARENT)
                  (IF (IS-SUB NODE2 NODE1 hier) ; node2 is child of node1

                      (SETF CHILD NODE2 PARENT NODE1)
                      (SETF CHILD NODE1 PARENT NODE2))
                  (IF (OR (NOT DELETE)
                          (NOT (> (LENGTH (GET-SUPERS CHILD HIER))
                                  1)))
                      (ERROR 
                     "Breaking link between ~S and ~S would orphan ~S." 
                             PARENT CHILD CHILD)
		      (delete-node child hier :recursive t))
                  (SETF (GET-SUBS PARENT HIER)
                        (REMOVE CHILD (GET-SUBS PARENT HIER)))
                  (SETF (GET-SUPERS CHILD HIER)
                        (REMOVE PARENT (GET-SUPERS CHILD HIER)))
                  HIER)))))

(defun add-link (node1 node2 hier)
  ;; add link from node1 to node2
   (COND
      ((NOT (IS-NODE NODE1 HIER))
       (ERROR  " The node named ~S doesn't exist." NODE1))
      ((NOT (IS-NODE NODE2 HIER))
       (ERROR  " The node named ~S doesn't exist." NODE2))
      ((is-super node1 node2 hier)
       ;; link is already there, just return
       (return-from add-link hier))
      (t
       (setf (get-subs node1 hier)
	     (cons node2 (get-subs node1 hier))
	     (get-supers node2 hier)
	     (cons node1 (get-supers node2 hier)))
       hier)))

(DEFUN CHANGE-NODE (OLD-OBJ NEW-OBJ HIER) "Function to replace an object in a hierarchy with another object occupying the same position."
   (COND
      ((NOT (IS-NODE OLD-OBJ HIER))
       (ERROR "~% The node ~S doesn't exist." OLD-OBJ))
      ((IS-NODE NEW-OBJ HIER)
       (ERROR  "~% A node ~S already exists." NEW-OBJ))
      (T (DOLIST (CHILD (GET-SUBS OLD-OBJ HIER))
                (SETF (GET-SUPERS CHILD HIER)
                      (CONS NEW-OBJ (GET-SUPERS CHILD HIER))))
         (DOLIST (PARENT (GET-SUPERS OLD-OBJ HIER))
                (SETF (GET-SUPERS PARENT HIER)
                      (CONS NEW-OBJ (GET-SUPERS PARENT HIER))))
         (SETF HIER (CAR (MULTIPLE-VALUE-LIST
                          (ADD-NODE NEW-OBJ (get-supers old-obj
							hier)
                                 (get-supers old-obj hier)
                                 HIER))))
         (SETF HIER (DELETE-NODE OLD-OBJ HIER))
         HIER)))



(DEFUN DESTROY-HIER (HIER) 
  (DELETE-NODE (get-TOP-NODE HIER) HIER :RECURSVE T)
  (SETF (slot-value hier 'NODE-TYPE) NIL)
  (SETF HIER NIL))


(DEFUN ADD-NODE (NODE PARENT-LIST CHILD-LIST HIER 
		      &optional &key (redefine nil))
  (if (symbolp hier)
      (setf hier (slot-value (eval hier) 'hierarchy))
      (if (typep hier 'classifier)
	  (setf hier (slot-value hier 'hierarchy))
	  (if (not (typep hier 'hierarchy))
	      (error "~S is not of type hierarchy, classifier, or symbol."
		     hier))))
  (IF (IS-NODE NODE HIER)
      (if redefine
	  (progn
	    (change-node node node hier)
	    (return-from add-node (values hier nil)))
	  (ERROR "Attempt to redefine node ~S.~%" NODE)))
  
  (IF (NOT (TYPEP NODE (SLOT-VALUE HIER 'NODE-TYPE)))
      (ERROR "Node not of type ~S.~%" (SLOT-VALUE HIER 'NODE-TYPE)))
  (LET ((UNRESOLVED-LIST NIL))
       (IF (NOT (NULL PARENT-LIST))
	   (DOLIST (PARENT PARENT-LIST)

            ;; if parent hasn't been defined yet		   

             (IF (NOT (IS-NODE PARENT HIER))
		   
	      ;; if parent already on too-be-checked list, just add
	      ;; this relationship to its entry

                       (IF (ASSOC PARENT UNRESOLVED-LIST)
                           (RPLACD (ASSOC PARENT UNRESOLVED-LIST)
                                  (CONS (CONS PARENT NODE)
                                        (CDR (ASSOC PARENT 
                                                    UNRESOLVED-LIST))))
       
       ;; else add parent and relationship to unresolved list

                           (SETF UNRESOLVED-LIST
                                 (ACONS PARENT (LIST (CONS PARENT NODE)
                                                     )
                                        UNRESOLVED-LIST)))
       
       ;; else parent already defined, check relationship

                       (IF (NOT (IS-SUB NODE PARENT HIER))
                           (ERROR 
    "~S claims to be a child of ~S, but was not defined as such by ~S." 
                                  NODE PARENT PARENT)))))
        (IF (NOT (NULL CHILD-LIST))
            (DOLIST (CHILD CHILD-LIST)
       
       ;; if child hasn't been defined yet

                   (IF (NOT (IS-NODE CHILD HIER))
       
       ;; if child already on too-be-checked list, just add
       
       ;; this relationship to its entry

                       (IF (ASSOC CHILD UNRESOLVED-LIST)
                           (RPLACD (ASSOC CHILD UNRESOLVED-LIST)
                                  (CONS (CONS CHILD NODE)
                                        (CDR (ASSOC CHILD 
                                                    UNRESOLVED-LIST))))
       
       ;; else add child and relationship to unresolved list

                           (SETF UNRESOLVED-LIST
                                 (ACONS CHILD (LIST (CONS NODE CHILD))
                                        UNRESOLVED-LIST)))
       
       ;; else child already defined, check relationship

                       (IF (NOT (IS-SUPER NODE CHILD HIER))
                           (ERROR 
   "~S claims to be a parent of ~S, but was not defined as such by ~S." 
                                  NODE CHILD CHILD)))))
       
       ;; all the relationships we can check are okay, add node
       
       ;; to hierarchy -- implementation-dependent section of
       ;; add-node
       
       ;; double-check if this adds two top nodes to hier 

        (IF (NULL PARENT-LIST)
            (IF (NOT (NULL (GET-TOP-NODE HIER)))
                (ERROR "Adding ~S would give the hierarchy two top nodes - ~S is already the top node." 
                       NODE (GET-TOP-NODE HIER))))
        (SETF (SLOT-VALUE HIER 'SUPER-LIST) (ACONS NODE PARENT-LIST
                                                  (SLOT-VALUE
                                                   HIER
                                                   'SUPER-LIST)))
        (SETF (SLOT-VALUE HIER 'SUB-LIST) (ACONS NODE CHILD-LIST
                                                (SLOT-VALUE
                                                 HIER
                                                 'SUB-LIST)))
        (VALUES HIER UNRESOLVED-LIST)))


(DEFUN DELETE-NODE (NODE HIER &KEY (RECURSIVE NIL))
  (if (symbolp hier)
      (setf hier (slot-value (eval hier) 'hierarchy))
      (if (typep hier 'classifier)
	  (setf hier (slot-value hier 'hierarchy))
	  (if (not (typep hier 'hierarchy))
	      (error "~S is not of type hierarchy, classifier, or symbol."
		     hier))))

   (COND
      ((NOT (IS-NODE NODE HIER))
       (ERROR  "~% The node named ~S doesn't exist." NODE))
      ((not (or recursive
		(dolist (CHILD (GET-SUBS NODE HIER) t)
		       (IF (NOT (> (LENGTH (GET-SUPERS CHILD HIER)) 1))
			   (ERROR "Deleting node ~S would orphan ~S." 
				  node child))))))
      (T (DOLIST (PARENT (GET-SUPERS NODE HIER))
                (SETF (GET-SUBS PARENT HIER)
                      (REMOVE NODE (GET-SUBS PARENT HIER))))
         (DOLIST (CHILD (GET-SUBS NODE HIER))
                (SETF (GET-SUPERS CHILD HIER)
                      (REMOVE NODE (GET-SUPERS CHILD HIER)))
                (IF (NULL (GET-SUPERS CHILD HIER))
                    (IF recursive (DELETE-NODE CHILD HIER :recursive t)
                        (ERROR 
                  "This should not happen: deleting ~S will orphan ~S." 
                               NODE CHILD))))
         (SETF (SLOT-VALUE HIER 'SUB-LIST)
               (REMOVE (ASSOC NODE (SLOT-VALUE HIER 'SUB-LIST))
                      (SLOT-VALUE HIER 'SUB-LIST)))
         (SETF (SLOT-VALUE HIER 'SUPER-LIST)
               (REMOVE (ASSOC NODE (SLOT-VALUE HIER 'SUPER-LIST))
                      (SLOT-VALUE HIER 'SUPER-LIST)))
         HIER)))


(DEFUN INSERT-NODE (NEW-NODE NODE1 NODE2 HIER)
   (COND
      ((IS-NODE NEW-NODE HIER)
       (ERROR  "~% A node ~S already exists." NEW-NODE))
      ((NOT (IS-NODE NODE1 HIER))
       (ERROR  "~% The node ~S doesn't exist." NODE1))
      ((NOT (IS-NODE NODE2 HIER))
       (ERROR  "~% The node ~S doesn't exist." NODE2))
      (T (IF (NOT (OR (IS-SUB NODE2 NODE1 HIER)
                      (IS-SUPER NODE2 NODE1 HIER)))
             (ERROR  " There is not a link between ~S and ~S." NODE1 
                    NODE2)
             (LET (PARENT CHILD)
                  (IF (IS-SUB NODE2 NODE1 HIER)
                      (SETF PARENT NODE1 CHILD NODE2)
                      (SETF PARENT NODE2 CHILD NODE1))
                  (SETF (GET-SUPERS CHILD HIER)
                        (CONS NEW-NODE (GET-SUPERS CHILD HIER)))
                  (SETF (GET-SUBS PARENT HIER)
                        (CONS NEW-NODE (GET-SUBS PARENT HIER)))
       
       ;; got to be able to add to more than one parent/child,
       ;; do we want "add-sub" and "add-super" routines,
       ;; instead of cons (macros)

                  (SETF HIER (CAR (MULTIPLE-VALUE-LIST
                                   (ADD-NODE NEW-NODE (LIST PARENT
                                                                 )
                                          (LIST CHILD)
                                          HIER))))
                  (SETF HIER (BREAK-LINK PARENT CHILD HIER))
                  HIER)))))


(DEFUN MOVE-NODE (NODE NEW-PARENT-LIST NEW-CHILD-LIST HIER 
		       &optional &KEY (DELETE NIL))
   (COND
      ((NOT (IS-NODE NODE HIER))
       (ERROR  "~% The node named ~S doesn't exist." NODE))
       
       ;; check to see if all specified parents and children
       ;; exist

      ((NOT (DOLIST (PARENT NEW-PARENT-LIST T)
                   (IF (NOT (IS-NODE PARENT HIER))
		       (progn
			 (ERROR  " The parent node named ~S doesn't exist." 
				PARENT)
			 (RETURN NIL))))))
      ((NOT (DOLIST (CHILD NEW-CHILD-LIST T)
                   (IF (NOT (IS-NODE CHILD HIER))
		       (progn
			 (ERROR " The child node ~S doesn't exist." 
				CHILD)
			 (RETURN NIL))))))
       
       ;; make sure that either :DELETE is specified, or all
       ;; nodes no longer children of this one have at least
       ;; one other parent

      ((NOT
        (OR DELETE
            (DOLIST (OLD-CHILD (GET-SUBS NODE HIER)
                           T)
                   (IF (NOT (MEMBER OLD-CHILD NEW-CHILD-LIST))
                       (IF (NOT (> (LENGTH (GET-SUPERS OLD-CHILD HIER))
                                   1))
			   (ERROR 
			    "Orphaned child ~S would result from moving ~S." 
			    OLD-CHILD NODE)))))))
      (T 
       
       ;; fix up the parent connections

         (DOLIST (OLD-PARENT (GET-SUPERS NODE HIER))
                (IF (NOT (MEMBER OLD-PARENT NEW-PARENT-LIST))
                    (SETF (GET-SUPERS NODE HIER)
                          (REMOVE OLD-PARENT (GET-SUPERS NODE HIER))
                          (GET-SUBS OLD-PARENT HIER)
                          (REMOVE NODE (GET-SUBS OLD-PARENT HIER)))))
         (SETF (GET-SUPERS NODE HIER)
               NEW-PARENT-LIST)
       
       ;; fix up the child connections, delete if orphaned and
       ;; :DELETE is T (no other orphans should occur, due to
       ;; earlier check)

         (DOLIST (OLD-CHILD (GET-SUBS NODE HIER))
                (IF (NOT (MEMBER OLD-CHILD NEW-CHILD-LIST))
                    (PROGN (SETF (GET-SUBS NODE HIER)
                                 (REMOVE OLD-CHILD (GET-SUBS NODE HIER))
                                 (GET-SUPERS OLD-CHILD hier)
                                 (REMOVE NODE (GET-SUPERS OLD-CHILD 
                                                     HIER)))
                           (IF (NULL (GET-SUPERS OLD-CHILD HIER))
                               (IF DELETE 
				   (DELETE-NODE OLD-CHILD HIER :recursive t)
                                   (ERROR 
                  "This should not happen -- ~S orphaned by moving ~S." 
                                          OLD-CHILD NODE))))))
         (SETF (GET-SUBS NODE HIER)
               NEW-CHILD-LIST)
       
       ;; return HIER

         HIER)))



