;; -*- Mode: Lisp -*-

;;;; JSAINT: A rational reconstruction of Slagel's SAINT program

;;; Copyright (c) 1991, Kenneth D. Forbus, Northwestern University,
;;; and Johan de Kleer, Xerox Corporation
;;; All rights reserved.
;;; Version 1, Last edited 5/3/91

(in-package 'user)

(defstruct (Jsaint :conc-name
		   (:print-function (lambda (a st ignore)
				      (format st "<Agenda ~A>"
					      (jsaint-title a)))))
  (title "")           ;; Name for printing
  (jtre nil)           ;; Associated JTRE
  (agenda nil)         ;; List of queued subproblems
  (problem nil)        ;; When solved, we are done.
  (solution nil)       ;; Cached answer.
  (n-subproblems 0)    ;; Statistic
  (max-tasks 20)     ;; resource bound
  (debugging nil))     ;; Debugging flag

;; Start with the usual encapsulation

(proclaim '(special *jsaint*))

(defvar *jsaint* nil)

(defun create-jsaint (title problem &key (debugging nil) (max-tasks nil))
  (let ((ag (make-jsaint
	      :TITLE title
	      :problem problem
	      :JTRE (create-jtre 
		      (concatenate 'string "JTRE of " title))
	      :DEBUGGING debugging
	      :MAX-TASKS (if (integerp max-tasks) max-tasks 2000))))
    (change-jtms (jtre-jtms (jsaint-jtre ag))
		 :contradiction-handler
		 #'(lambda (jtms contras)
		     (error "JSAINT ~A shouldn't have contradictions: ~A"
			    ag contras)))
    ag))

(defmacro debugging-jsaint (js msg &rest args)
  `(when (jsaint-debugging ,js) (format t ,msg ,@ args)))

(defun change-jsaint (js &key (debugging :nada) (problem :nada)
		       (max-tasks :nada))
  (unless (eq debugging :nada) (setf (jsaint-debugging js) debugging))
  (unless (eq problem :nada) (setf (jsaint-problem js) problem))
  (unless (eq max-tasks :nada) (setf (jsaint-max-tasks js) max-tasks)))

(defun use-jsaint (js) (setq *jsaint* js))

(defmacro with-jsaint (js &rest forms) `(let ((*ag* ,js)) ,@ forms))

;;;; User entry point

(defvar *jsaint-rules* "/u/bps/code/jtms/jsrules.lisp") ;; Fundamentals
(defvar *jsaint-operators* "/u/bps/code/jtms/jsops.lisp") ;; Operators

(defun solve-integral (integral
		       &key (title (symbol-name (gensym)))
		       (debugging nil)
		       (max-tasks 2000))
  ;; Remove redudancies and canonicalize input. 
  (setq integral (eval (quotize (simplifying-form-of integral))))
  (use-jsaint (create-jsaint title integral
			     :debugging debugging
			     :max-tasks max-tasks))
  (queue-problem (jsaint-problem *jsaint*) nil)
  (with-JTRE (jsaint-jtre *jsaint*) 
	     (load *jsaint-rules*)
	     (load *jsaint-operators*))
  (run-jsaint *jsaint*))

(defun explain-result (&optional (*jsaint* *jsaint*))
  (cond ((null (jsaint-solution *jsaint*))
	 (format t "~% Problem not solved yet."))
	((eq (jsaint-solution *jsaint*) ':FAILED-PROBLEM)
	 (wander-dnet (get-tms-node `(failed ,(jsaint-problem *jsaint*))
				    (jsaint-jtre *jsaint*)))
	 (format t "~% Failed to find a solution."))
	((eq (jsaint-solution *jsaint*) ':FAILED-EMPTY)
	 (format t "~% Ran out of things to do.")
	 (wander-dnet (get-tms-node `(failed ,(jsaint-problem *jsaint*))
				    (jsaint-jtre *jsaint*))))
	(t (format t "~% Solved the problem:")
	   (wander-dnet (get-tms-node
			 `(solution-of ,(jsaint-problem *jsaint*)
				       ,(jsaint-solution *jsaint*))
			 (jsaint-jtre *jsaint*))))))

(defun wander-dnet (node)
  (unless (in-node? node)
	  (format t "~% Sorry, ~A not believed." (view-node node))
	  (return-from wander-dnet node))
  (do ((stack nil)
       (current node)
       (options nil)
       (olen 0)
       (done? nil))
      (done? current)
      (why-node current)
      (setq options (if (typep (tms-node-support current) 'just)
			(just-antecedents (tms-node-support current))
		      nil))
      (setq olen (length options))
      (do ((good? nil)
	   (choice 0))
	  (good? (case good?
		       (Q (return-from wander-dnet current))
		       (0 (if stack
			      (setq current (pop stack))
			    (return-from wander-dnet current)))
		       (t (push current stack)
			  (setq current (nth (1- good?) options)))))
	  (format t "~%>>>")
	  (setq choice (read))
	  (cond ((or (eq choice 'q)
		     (and (integerp choice)
			  (not (> choice olen))
			  (not (< choice 0))))
		 (setq good? choice))
		(t (format t "~% Must be q or an integer from 0 to ~D." olen))))))
			  
(defun read-from-console () (read)) ;; for tracing

;;;; Basic algorithm 

(defun run-jsaint (*jsaint*)
  (when (jsaint-solution *jsaint*)
    (return-from run-jsaint ;; Don't re-solve
      (values (jsaint-solution *jsaint*) *jsaint*)))
  (when (> (jsaint-n-subproblems *jsaint*)
	   (jsaint-max-tasks *jsaint*))
    (return-from run-jsaint ;; Respect resource limits
      (values :time-out *jsaint*)))
  (do ((done? nil)
       (solution (fetch-solution (jsaint-problem *jsaint*) *jsaint*)
		 (fetch-solution (jsaint-problem *jsaint*) *jsaint*))
       (failure-signal `(Failed (Integrate ,(jsaint-problem *jsaint*)))))
      (done? (values (jsaint-solution *jsaint*) *jsaint*))
    (cond (solution
	   (setf (jsaint-solution *jsaint*) solution)
	   (debugging-jsaint
	    *jsaint* "~% ~A: Solved original problem."
	    (jsaint-title *jsaint*))
	   (setq done? t))
	  ((in? failure-signal (jsaint-jtre *jsaint*))
	   (debugging-jsaint
	    *jsaint* "~% ~A: Failed on original problem."
	    (jsaint-title *jsaint*)) 
	   (setf (jsaint-solution *jsaint*) :FAILED-PROBLEM)
	   (setq done? t))
	  ((null (jsaint-agenda *jsaint*))
	   (debugging-jsaint *jsaint* "~% ~A: Agenda empty."
			     (jsaint-title *jsaint*))
	   (setf (jsaint-solution *jsaint*) :FAILED-EMPTY)
	   (setq done? t))
	  (t (process-subproblem
	      (cdr (pop (jsaint-agenda *jsaint*))))))))

;;;; Working out subproblems

(defun process-subproblem (item &aux (jtre (jsaint-jtre *jsaint*))
			   (suggestions nil))
  (debugging-jsaint *jsaint* "~%  Trying to solve ~A." item)
  (when (in? `(expanded ,item) jtre)
	(debugging-jsaint *jsaint* "~%   ..already expanded.")
	(return-from process-subproblem nil))
  (when (out? `(relevant ,item) jtre)
	(debugging-jsaint *jsaint* "~%  ..not relevant.")
	(return-from process-subproblem nil))
  (open-subproblem item)
  (when (fetch-solution item)
	(debugging-jsaint *jsaint* "~%  ..solved directly.")
	(return-from process-subproblem T))
  (dolist (suggestion (fetch `(SUGGEST-FOR ,item ?operator) jtre))
    (when (in? suggestion jtre)
      (queue-problem `(trying ,(third suggestion)) item)
      (push (third suggestion) suggestions)))
  ;; Presume extra subgoals don't come along.
  (assert! `(OR-SUBGOALS ,item ,(mapcar #'(lambda (sg) `(trying ,sg)) suggestions))
	   ':OR-SUBGOALS jtre)
  (run-rules jtre))

(defun open-subproblem (item &aux (jtre (jsaint-jtre *jsaint*)))
  (assert! `(expanded ,item) ':EXPAND-AGENDA-ITEM jtre)
  (assume! `(open ,item) ':EXPAND-AGENDA-ITEM jtre)
  ;; Look for quick win, extra consequences.
  (run-rules jtre))

(defun fetch-solution (problem &optional (*jsaint* *jsaint*)
		       &aux (jtre (jsaint-jtre *jsaint*)))
  (dolist (solution (fetch `(SOLUTION-OF ,problem ?answer) jtre))
    (when (in? solution jtre)
      (return-from fetch-solution (third solution)))))

;;;; Queuing problems
;; Queue entries take the form (<difficulty> . <subproblem>)
;; Difficulty estimates are based on the form of the subproblem
;; alone, since there could be multiple parents for a subproblem.

(defun queue-problem (problem parent &aux entry)
  (setq entry (cons (estimate-difficulty problem) problem))
  (debugging-jsaint *jsaint* "~%   Queueing ~A, difficulty = ~D"
		    problem (car entry))
  (setf (jsaint-agenda *jsaint*)
	(merge 'list (list entry)
	       (jsaint-agenda *jsaint*)
	       #'(lambda (a b) (< (car a) (car b))))))

(defun estimate-difficulty (problem)
  (+ (max-depth problem) (count-symbols problem)))

(defun count-symbols (pr)
  (cond ((null pr) 0)
	((listp pr)
	 (reduce #'+ (mapcar #'count-symbols pr)
		 :initial-value 0))
	(t 1)))

(defun max-depth (pr)
  (cond ((not (listp pr)) 1)
	(t (1+ (reduce #'max (mapcar #'max-depth pr)
		       :initial-value 0)))))

;;;; Defining operators

(defmacro defIntegration (name trigger &rest keyed-items
			  &aux subproblems result test)
  (setq subproblems (cadr (member ':subproblems keyed-items)))
  (setq result (cadr (member ':result keyed-items)))
  (setq test (cadr (member ':test keyed-items)))
  (unless result 
    (error "Integration operator must have result form"))
  `(rule ((:in (expanded (Integrate ,trigger)) :var ?starter
	  ,@ (if test `(:TEST ,test) nil)))
	 (rlet ((?integral ,trigger)
		(?problem (Integrate ,trigger)))
	       (rlet ((?op-instance (,name ?integral)))
	     (rassert! (Operator-Instance ?op-instance) :OP-Instance-Definition)
	     ;; If no subproblems, just create solution
	     ,@ (cond ((null subproblems)
		       `((rlet ((?solution (:eval (simplify ,(quotize result)))))
			       (rassert! (solution-of ?problem ?solution)
					 (,(keywordize name) (Operator-Instance ?op-instance))))))
		      (t ;; Usual case
		       (let ((subs (calculate-subproblem-list subproblems))) 
	 `((rassert! (suggest-for ?problem ?op-instance)
		     (:IntOpExpander ?starter))
     (rule ((:in (expanded (trying ?op-instance)) :var ?trying))
	   (rlet ,subs
		 ,@ (mapcar #'(lambda (sub)
				`(queue-problem ,(car sub) ?problem))
			    subs)
		 (rassert! (AND-SUBGOALS (trying ?op-instance) ;; ?problem
					 ,(mapcar #'car subs))
			   (,(keywordize (format nil "~A-DEF" name))
			    ?trying))
		 ;; Solution detector
		 ,(multiple-value-bind (triggers antes)
		      (calculate-solution-rule-parts subs subproblems)
		    `(rule (,@ triggers)
			   (rlet ((?solution
				    (:eval (simplify ,(quotize result)))))
				 (rassert! (solution-of ?problem ?solution)
					   (,(keywordize name) (Operator-Instance ?op-instance)
					    ,@ antes)))))))))))))))

(defvar *test-operator*
	'(defIntegration Integral-of-Sum
	   (Integral (+ ?t1 ?t2) ?var)
	   :subproblems ((?int1 (Integrate (Integral ?t1 ?var)))
			 (?int2 (Integrate (Integral ?t2 ?var))))
	   :result (+ ?int1 ?int2)))

;;;; Helpers for operator definition

(defun calculate-subproblem-list (subproblems &aux (counter -1))
  ;; Takes list of entries whose form is (?result-var ?form)
  ;; and returns a list of (?goal-var ?form)
  (mapcar #'(lambda (pair)
	      (incf counter)
	      (list (intern (format nil "?GOAL~D" counter)) 
		    (simplifying-form-of (cadr pair))))
	  subproblems))

(defun simplifying-form-of (alg-goal)
  ;; Run simplifier on subgoals, just in case.
  (cond ((null alg-goal) nil)
	((not (listp alg-goal)) alg-goal)
	((eq (car alg-goal) 'INTEGRAL) ;; Simplify as needed
	 `(INTEGRAL (:EVAL (SIMPLIFY ,(quotize (cadr alg-goal)))) ,(caddr alg-goal)))
	(t (cons (simplifying-form-of (car alg-goal))
		 (simplifying-form-of (cdr alg-goal))))))

(defun calculate-solution-rule-parts (sub-pairs res-pairs
				      &aux (counter -1)
				      (antes nil)
				      (triggers nil))
  (setq triggers
	(mapcar #'(lambda (subpair respair)
		    (incf counter)
		    (let ((rvar (intern (format nil "?RESULT~D" counter))))
		      (push rvar antes)
		      `(:in (solution-of ,(car subpair) ,(car respair))
			:var ,rvar)))
		sub-pairs res-pairs))
  (values triggers (nreverse antes)))

(defun keywordize (stuff)
  (cond ((null stuff) (error "Can't keywordize nothing."))
	((listp stuff) (keywordize (car stuff)))
	(t (intern (format nil "~A" stuff) 'keyword))))

;;;; Interrogatives

;;; SHOW-PROBLEM highlights the assertions relevant to
;;; the given problem.

(defun statistics (&optional (*jsaint* *jsaint*))
  (dolist (problem (list-jsaint-problems *jsaint*))
	  (show-problem problem)))

(defun list-jsaint-problems (&optional (*jsaint* *jsaint*))
  (do ((queue (list (jsaint-problem *jsaint*))
	      (nconc (cdr queue) new-problems))
       (new-problems nil nil)
       (problems (list (jsaint-problem *jsaint*))))
      ((null queue) (nreverse problems))
      (dolist (sg-statement
	        (nconc (fetch `(and-subgoals ,(car queue) ?subgoals)
			      (jsaint-jtre *jsaint*))
		       (fetch `(or-subgoals ,(car queue) ?subgoals)
			      (jsaint-jtre *jsaint*))))
	      (when (in? sg-statement (jsaint-jtre *jsaint*))
		    (dolist (subproblem (caddr sg-statement))
			    (unless (member subproblem problems
					    :test #'equal)
				    (push subproblem problems)
				    (push subproblem new-problems)))))))

(defun show-problem (pr &optional (*jsaint* *jsaint*)
			&aux jtre stuff ands ors)
  (format t "~%~A::" pr)
  (setq jtre (jsaint-jtre *jsaint*))
  (setq stuff (fetch `(parent-of ,pr ?x ?type) jtre))
  (if stuff (dolist (p stuff)
		    (format t "~% ~A parent: ~A."
			    (fourth p) (third p)))
    (format t "~% No parents found."))
  (if (fetch `(expanded ,pr) jtre) (format t "~% Expanded,")
    (format t "~% Not expanded,"))
  (if (fetch `(open ,pr) jtre) (format t " opened,")
    (format t " not opened,"))
  (if (in? `(relevant ,pr) jtre) (format t " relevant.")
    (format t " not relevant."))
  (cond ((setq stuff (fetch-solution pr))
	 (format t "~% Solved, solution = ~A" stuff))
	((and (setq stuff (fetch `(failed ,pr) jtre))
	      (in? stuff jtre)) (format t "~%  Failed."))
	(t (format t "~% Neither solved nor failed.")))
  (setq ands (fetch `(and-subgoals ,pr ?ands) jtre))
  (when ands (format t "~% And subgoals:")
	(dolist (subg (third (car ands)))
		(format t "~%   ~A" subg))
	(format t "."))
  (setq ors (fetch `(or-subgoals ,pr ?ors) jtre))
  (when (and ors 
	     (not (and (null (cdr ors))
		       (null (third (car ors))))))
	(format t "~% Or subgoals:")
	(dolist (subg (third (car ors)))
		(format t "~%   ~A" subg))
	(format t ".")))


;;;; Debugging

(defun try-jsaint (problem)
  (solve-integral problem :debugging t))

(defun jfetch (pattern) (fetch pattern (jsaint-jtre *jsaint*)))

(defvar problem1 '(Integrate (Integral 1 x)))
(defvar problem2 '(Integrate (integral (+ x 5) x)))
(defvar problem3 '(Integrate (integral (* 46 (log x %e)) x)))
(defvar problem4 '(Integrate (integral (+ 0.63
					  (* 3.2 (sin (* 1.7 x)))
					  (* 4 (expt %e (* 2 x)))) x)))
(defvar problem5 '(Integrate (integral (+ (expt x 6) (* 8 (expt %e x))) x)))
(defvar problem6 '(Integrate (integral (+ (sqr (sin x)) (sqr (cos x))) x)))
