;; -*- Lisp -*-

;;;; ATMS-based planner using ATRE + ATMS

(in-package 'user)

;; Copyright (c) 1986, 1987, 1988, 1989, 1990 Kenneth D. Forbus, 
;;   Northwestern University, and Johan de Kleer, Xerox Corporation.  
;; All rights reserved.

;; Design: Each state/situation/world consists of an environment
;; and its associated consequences.  Changes are modeled via
;; STRIPS-like operators.  The predicates mentioned in add-lists
;; and delete-lists must always be assumptions.  Ergo, to apply
;; an operator to a world one simply removes the assumptions from
;; the world corresponding to the delete list, and add the
;; assumptions corresponding to the add list.

;; This version has two modes.
;; Mode 1: Classical forward search.  
;;           Each path consists of a sequence of operator
;;           instances and envs, the latter 
;;           corresponding to a particular world.
;; Mode 2: Envisioning mode
;;           First generate all possible states.  Then for
;;           each state find all possible transitions
;;           between them.

(defvar *debug-plan-a* nil)

;;;; Defining operators
;;
;; To find out what operators are applicable in some world,
;;  we generate a pattern-directed rule to make suggestions,
;;  just as in PROVER.
;; To apply an operator to a world requires surgery on the
;;  assumptions comprising the environment, carried out by
;;  the code below.

(defvar *operators* nil) ;; Alist of (<name> , <op structure>)
(defvar *plan* nil)

(defstruct (operator
	    (:predicate Operator?)
	    (:print-function (lambda (n st ignore) (declare (ignore ignore))
			       (format st "<Operator ~A>"
				       (operator-form n)))))
  (form nil)
  (preconditions nil)
  (add-list nil)
  (delete-list nil))

(defmacro DefOperator (form &rest keywords-and-values)
  (let ((test (cadr (member ':TEST keywords-and-values)))
	(preconditions
	  (cadr (member ':PRECONDITIONS keywords-and-values)))
	(add-list
	  (cadr (member ':ADD-LIST keywords-and-values)))
	(delete-list
	  (cadr (member ':DELETE-LIST keywords-and-values))))
    `(progn (let ((entry (assoc ',(car form) *operators*))
		  (op (make-operator
			:form ',form
			:preconditions ',preconditions
			:add-list ',add-list
			:delete-list ',delete-list)))
	      (cond (entry (setf (cdr entry) op))
		    (t (push (cons ',(car form) op)
			     *operators*))))
    ;; make rule that determines when it is applicable
     (rule :INTERN ,preconditions
	   ,(cond (test 
		   `(when ,test
		      (rassert! (applicable ,form)
				(:OP-PCs-Satisfied ,@ preconditions))))
		  (t `(rassert! (applicable ,form)
				(:OP-PCs-satisfied ,@ preconditions))))))))

(defun fetch-operator (op-name) (cdr (assoc op-name *operators*)))

;;;; Defining a problem
;;
;; A problem consists of a distinguished start state and goal,
;; each specified by a set of formulae.

;; In envisioning mode it makes sense to have *start* = () 
;; since the purpose is to build up the set of all states
;; and transitions that can be used to solve several
;; planning problems involving the same scenario. 

(defvar *choice-sets* nil) ;; Choice sets comprising a domain
(defvar *states* nil) ;; List of all states
(defvar *choice-set-finder* #'(lambda () nil)) ;; Domain-specific code
(defvar *start* nil) ;; Conjunctive list of formulae from initial state.
(defvar *goal* nil) ;; Conjunctive list of goal formulae
(defvar *start-state* nil) ;; the starting environment
(defvar *goal-state* nil) ;; a goal-satisfying environment

(defun gather-choice-sets (&optional (*atre* *atre*))
  (funcall *choice-set-finder*) ;assume someone set this up
  (assume-choice-sets *atre*))

(defun assume-choice-sets (&optional (*atre* *atre*))
  (dolist (choice-set *choice-sets*)
    (dolist (choice choice-set)
      (assume! choice :Choice))))

(defun goal-state? (state) (check-goals state *goal*))

(defun check-goals (state conditions)
  (let ((answer (catch 'got-one
		  (check-goal1 state conditions nil))))
    (case (car answer)
      (:WINNER (values T (cdr answer)))
      (t (values nil nil)))))

(defun check-goal1 (state goals bindings)
  (cond ((null goals)
	 (throw 'got-one (cons :WINNER bindings))) ;take any one
	(t (let ((candidates (fetch-implied-by (car goals) state)))
	     (cond ((null candidates) NIL)
		   (t (dolist (candidate candidates)
			(let ((new-bindings 
			       (unify1 (car goals) candidate
				       bindings)))
			(unless (eq new-bindings 'fail)
			  (check-goal1 state (cdr goals)
				       new-bindings))))))))))

(defun fetch-state (conditions)  ;; Useful for finding input states
  (remove-if-not #'(lambda (state)
		     (check-goals state conditions)) *states*))

;;;; Finding operators

(defun find-applicable-operators (state)
  (mapcar #'cadr (fetch-implied-by `(applicable ?x) state)))

(defun fetch-implied-by (form env &aux result)
  (dolist (candidate (fetch form))
    (if (in? candidate env) (push candidate result)))
  result)

(defun apply-operator (state op-inst)
;; This does surgery on the env (i.e., the state)
  (let ((operator (fetch-operator (car op-inst)))
	(vals (cdr op-inst))
	(assumptions (env-assumptions state))
	(bindings nil)
	(add-list nil)
	(delete-list nil)
	(atms (atre-atms *atre*))) 
    ;;; First substitute the values for the variables and create
    ;;; the appropriate add list and delete list
    (setq bindings (mapcar #'(lambda (var val) (cons var val))
			   (cdr (operator-form operator)) vals))
    (setq add-list (sublis bindings
			   (operator-add-list operator)))
    (setq delete-list (sublis bindings
			      (operator-delete-list operator)))
    (when *debug-plan-a*
      (format t "~%   Applying ~A to ~A." op-inst state)
      (format t "~%      Add list: ~A" add-list)
      (format t "~%      Delete list: ~A" delete-list))
    ;;; Remove delete-list assumptions.
    (setq assumptions
	  (remove-if #'(lambda (a) 
			 (member (datum-lisp-form
				   (tms-node-datum a))
				 delete-list
				 :test #'equal)) assumptions))
    (dolist (new add-list)
      (setq assumptions
	    (ordered-insert (get-tms-node new) assumptions
			    #'assumption-order)))
    (find-or-make-env assumptions atms)))

;;;; Mode 1 -- Forward search

;; This algorithm assumes that every choice in each choice set
;; has become an assumption, but that states have not yet been
;; generated.  

;; Notice how the ability to use environments as explicit
;; objects lets us revert back to CPS-like code!

(defun Plan-a (start goal *atre*)
  (setq *goal* goal)
  ;; Here start is a specific environment.
  ;; The goal is a list of conjunctions
  (do ((queue (list (list start))
	      (nconc (cdr queue) new-sprouts))
       (new-sprouts nil nil)
       (found? nil))
      ((or found? (null queue))
       (setq *plan* found?))
    (cond ((goal-state? (caar queue))
	   (setq found? (car queue)))
	  (t (dolist (op-inst (find-applicable-operators
				(caar queue)))
	       (let ((result (apply-operator (caar queue) op-inst)))
		 (unless (member result (car queue))
		   (push
		     (cons result
			   (cons op-inst
				 (car queue)))
		     new-sprouts))))))))

(defun show-plan (plan)
  (do ((steps (reverse plan) (cddr steps)))
      ((null steps))
    (print-env (car steps))
    (when (cadr steps) (format t "~%  then, by ~A, "
			       (cadr steps)))))

;;;; Mode 2 -- Envisioning

;; First compute all possible states, then all transitions
;;  between them. Then for any question, just do lookup

 ;; Alist of (<state> (<op instance> . <next state>) ...  )
(defvar *transitions* nil)

(defun envision (&optional (*atre* *atre*))
  (gather-choice-sets *atre*) ;; Find the "basis set" for the state space
  (run-rules *atre*)          ;; Install the constraints between them.
  (setq *states* (solutions *atre* *choice-sets*)) ;generate all states.
  ;; Find transitions by apply operators to each state
  (setq *transitions* nil)  
  (dolist (state *states*)  
    (apply-all-operators state)))

(defun apply-all-operators (state)
  (dolist (op-inst (find-applicable-operators state))
    (let ((result (apply-operator state op-inst))
	  (entry nil))
      (when (not (member result *states*))
	(error "~% ~A result of ~A on ~A, but not a state!"
		result op-inst state))
      (setq entry (assoc state *transitions*))
      (unless entry
	(setq entry (cons state nil))
	(push entry *transitions*))
      (setf (cdr entry) (cons (cons op-inst result)
			      (cdr entry))))))

;;;; Finding plans given an envisionment

(defun find-plan (start goals)
  (let ((goal-states (fetch-state goals))
	(start-states (fetch-state start)))
    (if *debug-plan-a*
	(format t "~%Goal states are ~A." goal-states))
    (do ((queue (mapcar #'(lambda (state)
			    (list state)) start-states)
		(nconc (cdr queue) new-sprouts))
	 (new-sprouts nil nil)
	 (found? nil))
	((or found? (null queue))
	 (setq *plan* found?))
      (cond ((member (caar queue) goal-states) ;got it
	     (setq found? (car queue)))
	    (t (dolist (transition
			 (cdr (assoc (caar queue)
				     *transitions*)))
		 (unless (member (cdr transition)
				 (cdar queue)) ;avoid loops
		   (if *debug-plan-a*
		       (format t "~% Can reach ~A via ~A from ~A."
			   (cdr transition) (car transition)
			   (caar queue)))
		   (push (cons (cdr transition)
			       (cons (car transition)
				     (car queue)))
			 new-sprouts))))))))

;;;; Debugging stuff

(defun show-states (&optional (stream *standard-input*))
  (cond ((null *states*) (format stream "~%The state space is empty."))
	(t (format stream "~% ~D states have been generated:" (length *states*))
	   (dolist (state *states*)
	     (print-env state stream)))))
