;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Ben Tso

(in-package 'protos)



;;;=============================================================================
;;;
;;;               V E R B S    A N D    Q U A N T I F I E R S
;;;
;;;         O F    T H E    E X P L A N A T I O N    L A N G U A G E
;;;  --------------------------------------------------------------------------
;;;			
;;;  Overview:    This file defines the verbs and quantifiers of the Protos
;;;               explanation language.  (The "verb" and "quantifier"
;;;               structures are defined in "defs.lisp").
;;;  
;;;  Notes:       Each distinct verb and quantifier is represented as an 
;;;               instance of the verb or quantifier structure, respectively.
;;;               Each instance is saved as the value of a distinct global
;;;               parameter.
;;;=============================================================================




;;;-----------------------------------------------------------------------------
;;;  Contents:  quantifiers
;;;
;;;  Notes:     -- There is one quantifier structure for each quantifier
;;;                giving the "pretty name" of the quantifier and its strength.
;;;
;;;             -- The creation of a structure for quanitfiers might seem like
;;;                overkill right now; the assumption is that future work on
;;;                the explanation language might require more information to
;;;                be associated with each quantifier.
;;;
;;;             -- Note that the quantifiers are of four general types:
;;;                context, strength, temporal, and belief.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;;;
;;;  (defstruct quantifier
;;;    name           ; pretty quantifier name, such as "usually".
;;;    symbol         ; symbol that is assigned this structure's contents.
;;;                   ;   (this is used in saving/loading a knowledge base).
;;;    strength       ; multiplicative strength in range 0 to 1.
;;;  )
;;;-----------------------------------------------------------------------------


;;; CONTEXT QUANTIFIERS ----------------------------------------------------

(defparameter *quant-always*       (make-quantifier :name     "always"
						    :abbrev   "al"
						    :strength 1.0))

(defparameter *quant-usually*      (make-quantifier :name     "usually"
						    :abbrev   "us"
						    :strength 0.8))

(defparameter *quant-sometimes*    (make-quantifier :name     "sometimes"
						    :abbrev   "st"
						    :strength 0.6))

(defparameter *quant-occasionally* (make-quantifier :name     "occasionally"
						    :abbrev   "occ"
						    :strength 0.4))



;;; STRENGTH QUANTIFIERS --------------------------------------------------

(defparameter *quant-strongly*     (make-quantifier :name     "strongly"
						    :abbrev   "str"
						    :strength 1.0))

(defparameter *quant-moderately*   (make-quantifier :name     "moderately"
						    :abbrev   "mod"
						    :strength 0.7))

(defparameter *quant-weakly*       (make-quantifier :name     "weakly"
						    :abbrev   "we"
						    :strength 0.4))



;;; TEMPORAL QUANTIFIERS -------------------------------------------------

(defparameter *quant-instantly*    (make-quantifier :name     "instantly"
						    :abbrev   "ins"
						    :strength 1.0))

(defparameter *quant-quickly*      (make-quantifier :name     "quickly"
						    :abbrev   "qu"
						    :strength 0.8))

(defparameter *quant-gradually*    (make-quantifier :name     "gradually"
						    :abbrev   "grad"
						    :strength 0.6))



;;; BELIEF QUANTIFIERS ---------------------------------------------------

(defparameter *quant-certainly*    (make-quantifier :name     "certainly"
						    :abbrev   "cert"
						    :strength 1.0))

(defparameter *quant-probably*     (make-quantifier :name     "probably"
						    :abbrev   "prob"
						    :strength 0.7))

(defparameter *quant-possibly*     (make-quantifier :name     "possibly"
						    :abbrev   "poss"
						    :strength 0.4))

;;;-----------------------------------------------------------------------------
;;;  Contents:   verbs
;;;
;;;  Note:       The verb structure is shown below (from defs.lisp).  The verbs
;;;              are divided into four classes -- active, passive,
;;;              bidirectional, and non-directional -- depending on how many
;;;              antecedents and how many consequents have to be proved.
;;;              For example, "causes" is an active verb.  When you say
;;;              "(A and B) causes C", "A" and "B" must both be proved true
;;;              for the relation to be true.  Also, when you say
;;;              "A causes (B and C)", only one of "B" or "C" need be proved
;;;              true for the relation to be true.
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;;;
;;;  (defstruct verb
;;;    name            ; pretty verb name, such as "is caused by".
;;;    symbol          ; symbol that is assigned this structure's contents.
;;;                    ;   (this is used in saving/loading a knowledge base).
;;;    strength        ; strength (0 to 1) of this verb (without quantifiers).
;;;    from-types      ; a list of legal types of antecedent terms.
;;;    to-types        ; a list of legal types of consequent terms.
;;;    from-proof      ; if non-nil, prove 1 antecedent, else prove all.
;;;    to-proof        ; if non-nil, prove 1 consequent, else prove all.
;;;    inverse         ; pointer to inverse verb's structure.
;;;    iquantifier     ; if nil, then no quantifier applied to inverse verb.
;;;                    ; if 'same, apply this verb's quantifier to inverse.
;;;                    ; otherwise, apply this quantifier name to inverse.
;;;  )
;;;
;;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;;;
;;;         The :to-proof and :from-proof slots can have two possible non-nil
;;;         values: t or 'only1.  The value 'only1 is used by the parser to
;;;         check that this verb has only one antecedent or only one
;;;         consequent.  For example, it is a syntax error to say something
;;;         like "A and B has typical generalization C".
;;;-----------------------------------------------------------------------------


;;;----------------------------------------------------------------------
;;;
;;;  ACTIVE AND PASSIVE VERBS
;;;  The following verbs consist of an active verb followed by its 
;;;  passive counterpart.  
;;;   
;;;  ACTIVE VERBS:
;;;      1) If more than one FROM node    ==> must prove them all
;;;      2) If more than one TO node      ==> must prove one of them
;;;  PASSIVE VERBS:
;;;      1) If more than one FROM node    ==> must prove one of them
;;;      2) If more than one TO node      ==> must prove them all
;;;
;;;----------------------------------------------------------------------
                                                                       
(defparameter  *verb-implies*
	       (make-verb  :name          "implies"
			   :abbrev        "imp"
                           :strength      0.8
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))


(defparameter  *verb-isImpliedBy*
	       (make-verb  :name          "is implied by"
			   :abbrev        "ib"
			   :strength      0.8
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
			   :inverse       *verb-implies*))

(setf (verb-inverse *verb-implies*) *verb-isImpliedBy*)

(defparameter  *verb-causes*
	       (make-verb  :name          "causes"
			   :abbrev        "cs"
			   :strength      1.0
			   :to-proof      t
			   :from-proof    nil
                           :iquantifier   *quant-sometimes*))
				 
(defparameter  *verb-causedBy*
               (make-verb  :name          "is caused by"
			   :abbrev        "cb"
			   :strength      1.0
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
                           :inverse       *verb-causes*))
			   
(setf (verb-inverse *verb-causes*) *verb-causedBy*)


(defparameter  *verb-enables*
	       (make-verb  :name          "enables"
			   :abbrev        "en"
			   :strength      0.9
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))

(defparameter  *verb-isEnabledBy*
	       (make-verb  :name          "is enabled by"
			   :abbrev        "eb"
			   :strength      0.9
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
			   :inverse       *verb-enables*))

(setf (verb-inverse *verb-enables*) *verb-isEnabledBy*)


(defparameter  *verb-hasFunction*
	       (make-verb  :name          "has function"
			   :abbrev        "hf"
			   :strength      1.0
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))

(defparameter *verb-isFunctionOf*
	      (make-verb   :name          "is function of"
			   :abbrev        "fo"
			   :strength      1.0
			   :to-proof      nil
			   :from-proof    t
                           :iquantifier   *quant-sometimes*
			   :inverse       *verb-hasFunction*))

(setf (verb-inverse *verb-hasFunction*) *verb-isFunctionOf*)


(defparameter *verb-hasPart*
	      (make-verb   :name          "has part"
			   :abbrev        "hp"
			   :strength      0.7 
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   'same))

(defparameter *verb-partOf*
	      (make-verb   :name          "is part of"
			   :abbrev        "po"
			   :strength      0.7
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   'same
			   :inverse       *verb-hasPart*))

(setf (verb-inverse *verb-hasPart*) *verb-partOf*)


(defparameter *verb-suggests*
	      (make-verb   :name          "suggests"
			   :abbrev        "sg"
			   :strength      0.8
			   :to-proof      t
			   :from-proof    nil
                           :iquantifier   *quant-sometimes*))
			   
(defparameter *verb-isInferredFrom*
              (make-verb   :name          "is inferred from"
			   :abbrev        "ifr"
			   :strength      0.8
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
                           :inverse       *verb-suggests*))

(setf (verb-inverse *verb-suggests*) *verb-isInferredFrom*)


(defparameter *verb-isRequiredBy*
	      (make-verb   :name          "is required by"
			   :abbrev        "rb"
			   :strength      0.9
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))

(defparameter *verb-requires*
	      (make-verb   :name          "requires"
			   :abbrev        "req"
			   :strength      0.9
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
			   :inverse       *verb-isRequiredBy*))

(setf (verb-inverse *verb-isRequiredBy*) *verb-requires*)


(defparameter *verb-exhibits*
	      (make-verb   :name          "exhibits"
			   :abbrev        "ex"
			   :strength      0.8
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))

(defparameter *verb-isConsistentWith*
	      (make-verb   :name          "is consistent with"
			   :abbrev        "cw"
			   :strength      0.8
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   *quant-sometimes*
			   :inverse       *verb-exhibits*))

(setf (verb-inverse *verb-exhibits*) *verb-isConsistentWith*)


(defparameter *verb-actsOn*
	      (make-verb   :name          "acts on"
			   :abbrev        "ao"
			   :strength      1.0
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-occasionally*))

(defparameter *verb-isActedOnBy*
	      (make-verb   :name          "is acted on by"
			   :abbrev        "ia"
			   :strength      1.0
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   nil
			   :inverse       *verb-actsOn*))

(setf (verb-inverse *verb-actsOn*) *verb-isActedOnBy*)


(defparameter *verb-affects*
	      (make-verb   :name          "affects"
			   :abbrev        "af"
			   :strength      1.0
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   *quant-occasionally*))

(defparameter *verb-isAffectedBy*
	      (make-verb   :name          "is affected by"
			   :abbrev        "ab"
			   :strength      1.0
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   nil
			   :inverse       *verb-affects*))

(setf (verb-inverse *verb-affects*) *verb-isAffectedBy*)



;;;-----------------------------------------------------------------------------
;;;             Q U A L I T A T I V E    M A T H    V E R B S
;;;
;;;  The following verbs fm+, rm+, fm-, and rm- are part of an extension
;;;  to the original Protos to handle qualitative math variables.  The
;;;  "m+" and "m-" part of the names refers to monotonically increasing
;;;  monotonically decreasing constraints between two variables.  The "f" and
;;;  "r" refer to "forward" and "reverse" directions of influence.  For
;;;  example, in a situation where we have Tank A draining into Tank B,
;;;  we want to say that the level in Tank A has a positive effect on the
;;;  level in Tank B, but not vice versa.  Thus, we say:
;;;
;;;                    (level-of-A  fm+  level-of-B)
;;;
;;;  These verbs get special treatment inside kbpm (see function
;;;  "add-variable-nodes") and in heuristics (see "FtoF-heuristic18").
;;;-----------------------------------------------------------------------------

(defparameter *verb-m+*
	      (make-verb   :name          "m+"
			   :abbrev        "m+"
			   :strength      0.99
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   'same))

(setf (verb-inverse *verb-m+*) *verb-m+*)

(defparameter *verb-m-*
	      (make-verb   :name          "m-"
			   :abbrev        "m-"
			   :strength      0.99
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   'same))

(setf (verb-inverse *verb-m-*) *verb-m-*)


(defparameter *verb-fm+*
	      (make-verb   :name          "fm+"
			   :abbrev        "fm+"
			   :strength      0.99
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   'same))

(defparameter *verb-rm+*
	      (make-verb   :name          "rm+"
			   :abbrev        "rm+"
			   :strength      0.99
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   'same
			   :inverse       *verb-fm+*))

(setf (verb-inverse *verb-fm+*) *verb-rm+*)


(defparameter *verb-fm-*
	      (make-verb   :name          "fm-"
			   :abbrev        "fm-"
			   :strength      0.99
			   :to-proof      t
			   :from-proof    nil
			   :iquantifier   'same))

(defparameter *verb-rm-*
	      (make-verb   :name          "rm-"
			   :abbrev        "rm-"
			   :strength      0.99
			   :to-proof      nil
			   :from-proof    t
			   :iquantifier   'same
			   :inverse       *verb-fm-*))

(setf (verb-inverse *verb-fm-*) *verb-rm-*)


;;;------------------------------------------------------------------------
;;;  
;;; BIDIRECTIONAL VERBS
;;; The following verbs are their own inverse.
;;; Both TO  and  FROM nodes require all nodes to be proved.
;;;
;;;------------------------------------------------------------------------

(defparameter *verb-iff*
	      (make-verb   :name          "if and only if"
			   :abbrev        "iff"
			   :strength      0.9
			   :to-proof      nil
			   :from-proof    nil
			   :iquantifier   'same))

(setf (verb-inverse *verb-iff*) *verb-iff*)


(defparameter *verb-equivalent*
	      (make-verb   :name          "equivalent"
			   :abbrev        "eq"
			   :strength      1.0
			   :to-proof      nil
			   :from-proof    nil
			   :iquantifier   'same))

(setf (verb-inverse *verb-equivalent*) *verb-equivalent*)


(defparameter *verb-defimplies*
	      (make-verb   :name          "definition implies"
			   :abbrev        "di"
			   :strength      0.9
			   :to-proof      nil
			   :from-proof    nil
			   :iquantifier   *quant-sometimes*))

(setf (verb-inverse *verb-defimplies*) *verb-defimplies*)


(defparameter *verb-cooccurs*
	      (make-verb   :name          "cooccurs with"
			   :abbrev        "co"
			   :strength      0.8
			   :to-proof      nil
			   :from-proof    nil
                           :iquantifier   'same))
			    
(setf (verb-inverse *verb-cooccurs*) *verb-cooccurs*)
	      

(defparameter *verb-MEx*
	      (make-verb   :name         "mutually exclusive with"
			   :abbrev       "me"
			   :strength     1.0
			   :to-proof     nil
			   :from-proof   nil
			   :iquantifier  'same))

(setf (verb-inverse *verb-MEx*) *verb-MEx*)


;;;---------------------------------------------------------------------------------
;;;										   
;;; NON-DIRECTIONAL VERBS 
;;; The following verbs need one FROM and one TO node to be proved.
;;;
;;;---------------------------------------------------------------------------------

(defparameter *verb-hasTypicalGen*
	      (make-verb   :name         "has typical generalization"
			   :abbrev       "tg"
			   :strength     0.9
			   :to-proof     'only1
			   :from-proof   'only1
			   :iquantifier  nil))

(defparameter *verb-hasTypicalSpec*
	      (make-verb   :name         "has typical specialization"
			   :abbrev       "ts"
			   :strength     0.9
			   :to-proof     'only1
			   :from-proof   'only1
			   :iquantifier  nil
			   :inverse      *verb-hasTypicalGen*))

(setf (verb-inverse *verb-hasTypicalGen*) *verb-hasTypicalSpec*)

(defparameter *verb-spurious*
	      (make-verb   :name         "spurious"
			   :abbrev        "spur"
			   :strength     0.9
			   :to-proof     t
			   :from-proof   t
			   :iquantifier  'same))

(setf (verb-inverse *verb-spurious*) *verb-spurious*)


;(defparameter  *verb-isFeatureOf*
;	       (make-verb  :name          "is a feature of"
;			   :abbrev        "fto"
;			   :strength      1.0
;			   :to-proof      'only1
;			   :from-proof    'only1
;                           :iquantifier   nil))
;				 
;(defparameter  *verb-hasFeature*
;               (make-verb  :name          "has feature"
;			   :abbrev        "hft"
;			   :strength      1.0
;			   :to-proof      'only1
;			   :from-proof    'only1
;			   :iquantifier   nil
;                           :inverse       *verb-isFeatureOf*))
;			   
;(setf (verb-inverse *verb-isFeatureOf*) *verb-hasFeature*)


(defparameter  *verb-isExemplarOf*
	       (make-verb  :name          "is an exemplar of"
			   :abbrev        "exo"
			   :strength      1.0
			   :to-proof      'only1
			   :from-proof    'only1
                           :iquantifier   nil))
				 
(defparameter  *verb-hasExemplar*
               (make-verb  :name          "has exemplar"
			   :abbrev        "hex"
			   :strength      1.0
			   :to-proof      'only1
			   :from-proof    'only1
			   :iquantifier   nil
                           :inverse       *verb-isExemplarOf*))
			   
(setf (verb-inverse *verb-isExemplarOf*) *verb-hasExemplar*)



;;;---------------------------------------------------------------------------
;;;  Note:  The following parameters *verb-noInverse* and *relation-noInverse*
;;;         are used by the function invert-explanation whenever it encounters
;;;         a relation which does not have a corresponding inverse stored in
;;;         the category network.  Normally this is an error and, due to the
;;;         two parameters below, it will appear when the inverted explanation
;;;         is printed, e.g., "Mammals {NO INVERSE FOUND} Humans".
;;;---------------------------------------------------------------------------

(defparameter *verb-noInverse*
	      (make-verb   :name         "{NO INVERSE FOUND}"
			   :abbrev       "{NO INVERSE FOUND}"
			   :strength     0.9
			   :to-proof     t
			   :from-proof   t
			   :iquantifier  'same))

(setf (verb-inverse *verb-noInverse*) *verb-noInverse*)

(defparameter *relation-noInverse*
	      (make-relation  :verb *verb-noInverse*
			      :from-nodes  nil
			      :to-nodes    nil
			      :strength    1.0
			      :creator     "gremlin"))

(setf (relation-inverse *relation-noInverse*) *relation-noInverse*)



;;;-----------------------------------------------------------------------------
;;;  Function:   (correlational verb)
;;;
;;;  Returns:    T if the given verb is a correlational relation, else NIL.
;;;
;;;  Called by:  heuristics
;;;-----------------------------------------------------------------------------

(defun correlational (verb)
  (member verb (list *verb-cooccurs* *verb-isConsistentWith*
		     *verb-exhibits* *verb-isRequiredBy* *verb-requires*
		     *verb-implies*  *verb-isImpliedBy*)))



;;;-----------------------------------------------------------------------------
;;;  Function:   (unidirectional verb)
;;;
;;;  Returns:    T if the given verb is a unidirectional relation, else NIL.
;;;
;;;  Note:       "Unidirectional" means that an explanation may not contain a
;;;              mixture of "forward" and "reverse" verbs, i.e., all the
;;;              unidirectional verbs in a given explanation must all "forward"
;;;              or "reverse".
;;;
;;;  Called by:  heuristics
;;;-----------------------------------------------------------------------------

(defun unidirectional (verb)
  (member verb (list  *verb-fm+*  *verb-rm+*  *verb-fm-*  *verb-rm-*)))



