;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;===========================================================================
;;;
;;;             P R I N T  /  D I S P L A Y    F U N C T I O N S
;;;   -----------------------------------------------------------------------
;;;
;;;  Contents:  This file contains functions for producing various displays
;;;             of information to the user.  Many of these functions display
;;;             selected portions of the knowledge base, and depend upon the
;;;             functions defined in defs.lisp for formatted displays of the
;;;             various structures.
;;;
;;;  Functions: print-it
;;;             print-e-categories
;;;             print-whole-kb
;;;             print-all-transformations
;;;             print-hierarchy
;;;		print-names
;;;             print-relationship
;;;             print-reminding
;;;             check-term-name
;;;             check-proposition-name
;;;             check-predicate-name
;;;             check-args
;;;             define-transformation
;;;===========================================================================



;;;-----------------------------------------------------------------------------
;;;  Function:  (print-it)
;;;
;;;  Purpose:   This function prompts the user for the name of any object and
;;;             then prints it out.
;;;-----------------------------------------------------------------------------

(defun print-it ()
  (let* ((input  (prompt "~%Please enter the name of any object ---> "
			 nil 'termname nil nil))
	 object)

    (if (and (symbolp input) (boundp input) (predicate-p (eval input)))
	(print-predicate (eval input) t 1)
	(progn
	  (setq object (check-term-name input 'fail))
	  (if object
	      (print-term object t 1)
	      (format t "~%Sorry, ~A is unknown." input))))))



;;;---------------------------------------------------------------------------
;;;  Function:  print-e-categories
;;;
;;;  Purpose:   This function prints the names of all exemplar-containing
;;;             categories along with the names of their exemplars.
;;;---------------------------------------------------------------------------

(defun print-e-categories ()
  (dolist (term *history*)
    (let (exemplars)
      (if (and (category-p term)
	       (setq exemplars (category-exemplars term)))
	  (progn
	    (format t "~%~A:   " (getname term))
	    (print-node-names exemplars t ", "))))))


;;;---------------------------------------------------------------------------
;;;  Function:  print-whole-kb
;;;
;;;  Purpose:   This function prints the entire knowledge base.  First it
;;;             prints details about each term and then it prints details
;;;             about each exemplar.
;;;---------------------------------------------------------------------------

(defun print-whole-kb ()
  (dolist (term *history*)
    (terpri)
    (typecase term
      (term      (print-term term t 1))
      (predicate (print-predicate term t 1))
      (case      nil)
      (string    nil)
      (otherwise (format t "~%Unknown term type ~A on history." (type-of term))))))


;;;----------------------------------------------------------------------------
;;;  Function:  print-all-transformations
;;;
;;;  Purpose:   This function displays a list of all transformations in the KB,
;;;             and asks whether the user wants to see all information for a 
;;;             particular transformation.
;;;----------------------------------------------------------------------------

(defun print-all-transformations ()
  (cond ((null *transformations*)
	 (format t "~%No transformations defined."))
	(t (let* ((lst nil)
		  (num 1)
		  (xform-sequence-list (progn
					 (dolist (xf *transformations* lst)
					   (setq lst (cons (cons num xf) lst))
					   (incf num))
					 (reverse lst))))
	     (terpri) (prog ()
			 list
			    (format t "The following transformations are defined: ~%")
			    (dolist (num-xf xform-sequence-list)
			      (format t "~%~A. ~A~
                         ~%     Type:           ~A~
                         ~%     Input-features: ~A"
				      (car num-xf)
				      (transformation-name (cdr num-xf))
				      (transformation-type (cdr num-xf))
				      (transformation-in-predicates (cdr num-xf))))
			    (format t "~2%Choose a number if you want a more detailed description of~
                      ~%some transformation (else type q):~
                      ~%----> ")
			    (setq num (read *query-io* nil nil))
			 choose
			    (cond ((and (numberp num) (<= num (caar (last xform-sequence-list))))
				   (go description))
				  ((equal num 'l) (go list))
				  ((equal num 'q) (return (values)))
				  (t (format t "~%Wrong input! Try again:~
                           ~2%>")
				     (setq num (read *query-io* nil nil))
				     (go choose)))
			    
			 description
			    (let ((old-print-level *print-level*))
			      (setq *print-level* 99) 
			      (print-transformation (cdr (assoc num xform-sequence-list)) t 99)
			      (format t "~2%Enter one of:  <number>  -  description of another transformation~
                                          ~%                  L      -  the list of transformations~
                                          ~%                  Q      -  quit displaying transformations~
                                          ~%>")
			      (setq *print-level* old-print-level)
			      (setq num (read *query-io* nil nil))
			      (go choose)))))))

;;;---------------------------------------------------------------------------
;;;  Function:  (print-hierarchy  verb)
;;;
;;;  Given:     verb, a verb pointing towards the root of a hierarchy
;;;
;;;  Does:      This function prints the hierarchy of terms as related by
;;;             the given verb and its inverse.  The hierarchy is printed
;;;             starting with a root term, with successively deeper levels
;;;             indented from the left margin.  For example:
;;;
;;;             vehicles
;;;                  land-vehicles
;;;                       cars
;;;                            sports-cars
;;;                            sedans
;;;                       trucks
;;;                  space-vehicles
;;;                       rockets
;;;                       satellites
;;;---------------------------------------------------------------------------

(defun print-hierarchy (verbsymbol)
  (let ((closed nil)
	(verb   (eval verbsymbol)))
    (dolist (term *history*)
      (if (and (term-p term)
	       (not (member term closed)))
	  (let ((root (find-root term verb)))
	    (setq closed (print-hierarchy2 (verb-inverse verb) root (cons root closed) 0)))))
    (if (null closed)
	(format t "~%none."))))

(defun print-hierarchy2 (verb node closed level)
  (let ((column (* level 5))			; indent 5 spaces for each level
	(printed nil))                          ; whether this node has been printed
    (if (> level 0)
	(progn
	  (format t "~%~V@T~(~A~)"  column  (getname node))
	  (setq printed t)))
	  
    (dolist (rel (node-relations node))
      (if (eq verb (relation-verb rel))
	  (let* ((to-nodes (relation-to-nodes rel))
		 (to-node  (car to-nodes)))
	    (if (not printed)
		(progn
		  (format t "~%~V@T~(~A~)"  column  (getname node))
		  (setq printed t)))
	    (pushnew to-node closed)
	    (if (cdr to-nodes)
		(progn
		  (format t "  Warning: multiple to-nodes: ")
		  (print-node-names to-nodes t ", ")))
	    (setq closed (print-hierarchy2 verb to-node closed (1+ level))))))
    closed))

(defun find-root (node verb)
  (dolist (rel (node-relations node))
    (if (eq verb (relation-verb rel))
	(return-from find-root (find-root (car (relation-to-nodes rel)) verb))))
  node)

(defun print-names (listname)
  (let ((nodes (eval listname)))
  ;; Check for null nodes since 'spurious has no to-nodes.
  (if (null nodes)
      ;; then print "none"
      (format t "none")
      ;; else if more than one node ...
      (let* ((column 0)
	     (names  nil))
	(setq nodes (delete-duplicates (remove-if-not #'node-p nodes)))
	(dolist (node nodes)
	  (push (princ-to-string (node-name node)) names)
	  (if (node-abbrev node)
	      (push (princ-to-string (node-abbrev node)) names))
	  (if (node-synonyms node)
	      (dolist (syn (node-synonyms node))
		(push (princ-to-string syn) names))))
	(setq names  (sort names #'string-lessp))

	(dolist (name names)
	  (if (= column 0)
	      (terpri))
	  (setq column (mod (1+ column) 3))
	  (format t " ~25A" name))))))


(defparameter *no-exemplar* (make-term :name 'null-exemplar
				       :features nil
				       :category nil
				       :typicality 0))


;;;----------------------------------------------------------------------------
;;;  Function:  (print-relationship)
;;;
;;;  Purpose:   This function prompts the user for the names of two terms and
;;;             then attempts to find a relationship between the two in the
;;;             category network.  If found, it is printed.
;;;----------------------------------------------------------------------------

(defun print-relationship ()
  (let ((faultvars  *fault-variables*)
	term1 term2 pred result)
    (declare (special *fault-variables*))

    ;; Temporarily set *fault-variables* to NIL.
    (setq *fault-variables* nil)

    (if (setq term1 (check-term-name
		      (prompt "~&Enter name of first term:  " nil 'termname nil nil)
		      'ask))
	(if (setq term2 (check-term-name
			  (prompt "~&Enter name of second term: " nil 'termname nil nil)
			  'ask))
	    (progn
	      ;; If one of the terms is a category containing fault variables,
	      ;; then temporarily assert those fault variables.
	      (setq *fault-variables* (or (category-faultvars term1)
					  (category-faultvars term2)))
	      (if *fault-variables*
		  ;; If fault variables found, inform user.
		  (progn
		    (format t "~%Asserting the following fault variables: ")
		    (print-node-names *fault-variables* t ", "))

		  ;; If no fault variables were associated with either term,
		  ;; then if either term is a "variable" term, prompt the user
		  ;; for a fault variable.  If this looks like just an ordinary
		  ;; pair of terms, then don't prompt the user for a fault
		  ;; variable since "faults" is an unadvertised extension.

		  (if (or (and (setq pred (term-predicate term1))
			       (predicate-relations pred))
			  (and (setq pred (term-predicate term2))
			       (predicate-relations pred)))
		      (let* ((predname (prompt "~&Enter fault variable (or nil): "
					       nil 'symbol nil nil))
			     pred)
			(if predname
			    (setq pred (check-pred-name predname 'fail)))
			(setq *fault-variables* (if pred
						    (list pred)
						    nil)))))
						  
	      (setq result (kbpm 'FtoF term2 *importance-big* *no-exemplar* (list term1)))
	      (format t "~%   ")
	      (case (result-type result)
		(identical  (format t "Terms are identical."))
		(spurious   (format t "~A is spurious." (getname term1)))
		(unmatched  (format t "No match found."))
		((explained excluded)
		 (print-explanation (result-explanation result))
		 (format t ", strength = ~4,2F" (result-quality result))))
	      ;; Restore original value of *fault-variables*
	      (setq *fault-variables* faultvars))))))



;;;----------------------------------------------------------------------------
;;;  Function:  (print-reminding  feature  target  strength  update)
;;;
;;;  Purpose:   This function prints a new (or updated) reminding from feature
;;;             to target.  If update is non-nil, then the reminding is taken
;;;             to be an update of a previous reminding.
;;;
;;;  Called by: set-reminding
;;;----------------------------------------------------------------------------

(defun print-reminding (feature target strength update)
  (format t "~%~:[Creating~;Updating~] ~:[reminding~;censor~] from ~A to ~A, strength ~A"
	  update
	  (minusp strength)
	  (getname feature)
	  (getname target)
	  (qualitative-value strength 'reminding)))


(defun print-importance (feature target strength update)
  (format t "~%~:[Creating~;Updating~] importance of ~A to ~A, strength ~A"
	  update
	  (minusp strength)
	  (getname feature)
	  (getname target)
	  (qualitative-value strength 'importance)))


(defparameter *name-menu* (make-menu
   :label nil
   :items '((#\C . ("It's correct; Create it as a new term."  return  create))
	    (#\R . ("Re-spell it."                            return  respell))
	    (#\S . ("this is a Synonym for an existing name." return  synonym))
	    (#\F . ("Forget this name."                       return  skip))
	    (#\D . ("Display existing names."                 return  show)))))

(defparameter *name-menu2* (make-menu
   :label nil
   :items '((#\R . ("Re-spell it."                            return  respell))
	    (#\S . ("this is a Synonym for an existing name." return  synonym))
	    (#\F . ("Forget this name."                       return  skip))
	    (#\D . ("Display existing names."                 return  show)))))

(defparameter *term-types-menu* (make-menu
   :label "~%Please specify the type of term:"
   :twocol t
   :items '((nil . ("OBSERVED EVIDENCE:"))
	    (nil . ("DIAGNOSES:"))
	    (#\H . ("   History data"                 return historydata))
	    (#\B . ("   Basic diagnosis"              return basic-diagnosis))
	    (#\S . ("   Symptom"                      return symptom))
	    (#\G . ("   General diagnosis"            return general-diagnosis))
	    (#\O . ("   Observation"                  return observation))
	    (nil . ("INFERRED EVIDENCE"))
	    (#\T . ("   Test result"                  return testresult))
	    (#\I . ("   Inferred evidence"            return inferred))
	    (nil . (""))
	    (#\N . ("None of the above; prompt for type"  return other)))))


;;;----------------------------------------------------------------------------
;;;  Function:  (check-term-name  name  if-doesnt-exist)
;;;
;;;  Given:     -- the name of an object, which may be a feature or category
;;;                or exemplar (and if it is a feature or category it may be 
;;;                expressed as a simple proposition [e.g., "headache"] or as
;;;                a predicate with arguments [e.g., "(fever mild)"]);
;;;             -- a flag that indicates what to do if the named object does
;;;                not already exist (can be 'ask, 'create, or 'fail).
;;; 
;;;  Returns:   -- the appropriate structure for the object, or
;;;                nil if the name is a number, or
;;;                nil if the teacher decided that the term was a mistake, or
;;;                nil if it didn't exist and if-doesnt-exist was 'fail.
;;;
;;;  Called by:  every function that accepts terms from the user or a file.
;;;----------------------------------------------------------------------------

(defun check-term-name (name if-doesnt-exist)
  ;; If this is a number ...
  (if (numberp name)
      ;; then nothing to do, so return nil.
      nil
      ;; else if this is a list ...
      (if (listp name)
	  ;; and if it has at least two elements ...
	  (if (cdr name)
	      ;; then it looks like a predicate with arguments.
	      (check-predicate-name   name if-doesnt-exist)
	      ;; else it's just a proposition wrapped in parentheses.
	      (check-proposition-name (car name) if-doesnt-exist))
	  ;; not a list, so it must be a proposition.
	  (check-proposition-name name if-doesnt-exist))))


(defun check-pred-name (name if-doesnt-exist)
  (declare (ignore if-doesnt-exist))
  (cond ((null name)
	 (format t "~%check-pred-name: null name.")
	 nil)
	((not (symbolp name))
	 (format t "~%check-pred-name: ~A not a symbol." name)
	 nil)
	((boundp name)
	 (let ((node (eval name)))
	   (if (predicate-p node)
	       node
	       (progn
		 (format t "~%check-pred-name: ~A is a ~A, not a predicate."
			 name (type-of node))
		 nil))))
	(t
	 (format t "~%check-pred-name: ~A not bound." name)
	 nil)))
	

;;;----------------------------------------------------------------------------
;;;  Function:  (check-proposition-name  name  if-doesnt exist)
;;;
;;;  Given:     a name of a feature, category, or exemplar (whether existing
;;;             or brand new) in the form of a proposition, e.g. "headache".
;;;
;;;  Returns:   the term structure for this proposition, or NIL if
;;;             the user has decided that this name was a mistake.
;;;
;;;  Design:    If a structure already exists for the given name,
;;;             then this function simply returns that structure.
;;;             Otherwise, the teacher must select from four choices:
;;;             (1) create the name as a term,
;;;             (2) respell the name [in case of a typing error],
;;;             (3) forget this term [useful if the name is just some garbage
;;;                 that was typed accidentally], or 
;;;             (4) Display the existing names and ask if it was  supposed
;;;                 to be one of these.
;;;----------------------------------------------------------------------------


(defun check-proposition-name (name if-doesnt-exist)
  (prog (menu)
	;; Make sure "name" is a symbol, not a list.
	(if (not (symbolp name))
	    (progn
	      (format t "~%Error: ~A not a proposition!" name)
	      (return-from check-proposition-name nil)))
	
	;; If this name has already been created ...
	(if (boundp name)

	    ;; then if this is supposed to be a brand new term, warn user.
	    (if (eql 'new if-doesnt-exist)
		(progn
		  (format t "~%Overwriting old value of ~A" name)
		  (go CREATE))

		;; else if it is a term (and not something else) ...
		(let ((node (eval name)))
		  (if (term-p node)
		  
		      ;; then return pointer to the node
		      (return-from check-proposition-name node)
		  
		      ;; else report error to user and ask for a different name.
		      (progn
			(format t "~%Sorry, ~A is already in use as a ~A:"
				name (type-of node))
			(setq menu *name-menu2*)
			(go ASK)))))
	    
	    ;; else check with user before creating a new symbol
	    (progn
	      (setq menu *name-menu*)
	      (case if-doesnt-exist
		(ask    (format t "~%~%\"~A\" does not currently exist:"  name)
			(go ASK))
		(create ;(format t "~%~%Creating new term: ~A" name)
			(go CREATE))
		(new    (go CREATE))
		(fail   (return-from check-proposition-name nil)))))
	
	;; The 'go' forms in this case statement were used because I needed
	;; to branch to the same code from two different places, and didn't
	;; want to make functions out of the target code.
	
     ASK
	(case (menu-select menu)
	  (create  (go CREATE))
	  (respell (go RESPELL))
	  (synonym (go SYNONYM))
	  (show    (format t "~%Terms:")
		   (print-names '*history*)
		   (terpri)
		   (go ASK))
	  (skip    (return-from check-proposition-name nil)))
	
	
     CREATE
	(let ((term (make-term :name name)))
	  (set name term)
	  (push term *history*)
;	  (if (eql 'ask if-doesnt-exist)
;	      (let ((abbrev (prompt "~%Enter abbreviation, if any: " nil 'termname nil nil)))
;		(if abbrev
;		    (if (boundp abbrev)
;			(let ((node (eval abbrev)))
;			  (format t "~%Sorry, ~A is already in use as a ~A"
;				  abbrev (type-of node))
;			  (if (node-p node)
;			      (format t " (~A)" (node-name node))))
;			(set abbrev term)))))
	  (if (and *diagnostic-model* (eql 'ask if-doesnt-exist))
	      (let ((termtype (menu-select *term-types-menu*)))
		(case termtype
		  ((testresult
		     observation
		     symptom
		     historydata
		     inferred
		     general-diagnosis
		     basic-diagnosis)    (setf (term-type term) termtype))
		  (other          (format t "~%Please describe the type of term you want to specify,~
                                     ~%in English (terminate with blank line):")
				  (format *logstream* "~%NEW TERM TYPE (for ~A)" name)
				  (get-log-text))
		  (t              (format t "~%Unknown type ~A" termtype)))))
	  
	  (return-from check-proposition-name term))
	
     RESPELL
	(setq name (prompt "~%Enter new name ---> " nil 'termname nil nil))
	(return-from check-proposition-name (check-proposition-name name if-doesnt-exist))

     SYNONYM
	(let ((name2 (prompt "~%Enter existing term name ---> " nil 'termname nil nil))
	      term)
	  (if name2
	      (if (setq term (check-term-name name2 'fail))
		  (progn
		    (set name term)
		    (push name (node-synonyms term))
		    (return-from check-proposition-name term))
		  (progn
		    (format t "~%Sorry, but ~A does not exist.~
                             ~%What do you want to do about ~A ?"
			    name2 name)
		    (go ASK)))
	      (go ASK)))
	))

;;;----------------------------------------------------------------------------
;;;  Function:  (check-predicate-name  name  if-doesnt-exist)
;;;
;;;  Given:     -- a name of a feature (whether existing or brand new) in the
;;;                form of a predicate + arguments, e.g. "(fever mild)"; and
;;;             -- a flag indicating what to do if the name doesn't already
;;;                exist (ask user, or just create the name, or fail).
;;;
;;;  Returns:   the appropriate term structure, or nil if the user has decided
;;;             that this name was a mistake.
;;;
;;;  Design:    If a term structure already exists for the given name,
;;;             then this function simply returns that structure.
;;;             Otherwise, the teacher must select from four choices:
;;;             (1) create the name as a term,
;;;             (2) respell the name [in case of a typing error],
;;;             (3) forget this term [useful if the name is just some garbage
;;;                 that was typed accidentally], or 
;;;             (4) Display the existing names and ask if it was  supposed
;;;                 to be one of these.
;;;----------------------------------------------------------------------------

(defun check-predicate-name (name if-doesnt-exist)
  (prog ((predname (first name))
	 menu
	 predicate
	 predtype)

	;; Predicates such as "no" and "history-of" are treated specially 
	;; since they are variations of their argument.  For example,
	;; if we see "(no hypertension)" then we create a node for
	;; that and relate it to hypertension via a mutual-exclusion.
	;; Similarly, if we see "(history-of hypertension)" then we create
	;; a node for that and relate it to "hypertension" via the relation
	;; "sometimes exhibits".

	(setq predtype (cdr (assoc predname '((no  . no)
					      (not . no)
					      (h-o . h-o)
					      (h/o . h-o)
					      (history-of . h-o)))))
	(if predtype
	    (if (> (length name) 2)
		(progn
		  (format t "~%Error: ~A must have a single argument in ~A!"
			  predname name)
		  (return-from check-predicate-name nil))
		(let (term1 term2)
		  (setq term1 (check-term-name (second name) if-doesnt-exist))
		  (if (null term1) (return-from check-predicate-name nil))
		  (setq term2 (cdr (assoc predtype (term-variations term1))))
		  (if term2
		      (return-from check-predicate-name term2)
		      (if (eql 'fail if-doesnt-exist)
			  (return-from check-predicate-name nil)
			  (progn
			    (setq term2 (make-term :name name))
			    (push (cons predtype term2) (term-variations term1))
			    (ecase predtype
			      (no   (install-relation (list term2) (list term1)
						      nil *verb-MEx* nil))
			      (h-o  (install-relation (list term2) (list term1)
						      (list *quant-sometimes*) *verb-exhibits*
						      nil)))
			    (return-from check-predicate-name term2)))))))

	
	;; If this name has already been created ...
	(if (boundp predname)
	    
	    ;; then if it really is the name of a predicate ...
	    (let ((predicate (eval predname)))
	      (if (predicate-p predicate)
		  (progn
		    ;; if this is a "variable", i.e., a predicate of the form
		    ;; "(variable-name  qualitative-magnitude  qualitative-direction)"
		    ;; but doesn't have the qdir specified, then assign it the
		    ;; the default value "xxx".
		    ;; For the moment, we can tell if this is a variable by whether
		    ;; or not it has attached relations.
		    (if (and (predicate-relations predicate)
			     (null (third name)))
			(setq name (append name '(xxx))))
		  
		    ;; then go find/create the node for this name
		    (multiple-value-bind (node alist)
			(check-args name predicate 1 (predicate-args predicate) if-doesnt-exist)
		      (setf (predicate-args predicate) alist)  ;; <======?????????????????????
		      (return-from check-predicate-name node)))
		  
		  ;; else report that this name is NOT a predicate!
		  (progn
		    (format t "~%Error: ~A already exists as a ~A."
			    predname  (type-of predicate))
		    (setq menu *name-menu2*)
		    (go ASK))))
	    
	    ;; else check with user before creating a new symbol
	    (progn
	      (setq menu *name-menu*)
	      (case if-doesnt-exist
		(ask    (format t "~%~%\"~A\", as in ~A, does not currently exist:"  predname name)
			(go ASK))
		(create ;(format t "~%~%Creating new predicate: ~A" name)
			(go CREATE))
		(fail   (return-from check-predicate-name nil)))))
	
     ASK
	(case (menu-select menu)
	  (create  (go CREATE))
	  (respell (go RESPELL))
	  (synonym (go SYNONYM))
	  (show    (format t "~%Terms:")
		   (print-names '*history*)
		   (terpri)
		   (go ASK))
	  (skip    (return-from check-predicate-name nil)))
	
	
     CREATE
	(setq predicate (make-predicate :name predname))
	(set predname predicate)
	(push predicate *history*)
	(multiple-value-bind (node alist)
	    (check-args name predicate 1 (predicate-args predicate) if-doesnt-exist)
	  (setf (predicate-args predicate) alist)	; <=================???????
	  (if (and *diagnostic-model* (eql 'ask if-doesnt-exist))
	      (let ((termtype (menu-select *term-types-menu*)))
		(case termtype
		  ((testresult
		     observation
		     symptom
		     historydata
		     inferred
		     general-diagnosis
		     basic-diagnosis)    (setf (term-type node) termtype))
		  (other      (format t "~%Please describe the type of term you want to specify,~
                                               ~%in English (terminate with blank line):")
			      (format *logstream* "~%NEW TERM TYPE (for ~A)" name)
			      (get-log-text))
		  (t          (format t "~%Unknown type ~A" termtype)))))
	  
	  (return-from check-predicate-name node))
	
     RESPELL
	(setq predname (prompt (format nil "~%Respell ~A ---> " predname) nil
			       'termname nil nil))
	(setf (first name) predname)
	(return-from check-predicate-name (check-predicate-name name if-doesnt-exist))

     SYNONYM
	(let ((predname2 (prompt "~%Enter existing predicate name ---> " nil 'termname nil nil))
	      pred)
	  (if predname2
	      (if (boundp predname2)
		  (if (predicate-p (setq pred (eval predname2)))
		      (progn
			(push predname (node-synonyms pred))
			(set predname pred)
			(return-from check-predicate-name
			  (check-predicate-name name if-doesnt-exist)))
		      (progn
			(format t "~%Sorry, but ~A is a ~A, not a predicate.~
                                   ~%What do you want to do about ~A ?"
				predname2 (type-of pred) predname)
			(go ASK)))
		  (progn
		    (format t "~%Sorry, but ~A does not exist.~
                               ~%What do you want to do about ~A ?"
			    predname2 predname)
		    (go ASK)))
	      (go ASK)))
	))
      

;;;----------------------------------------------------------------------------
;;;  Function:  (check-args  term predicate argnumber alist if-doesnt-exist)
;;;
;;;  Given:     -- term, the list-form of a term which is a predicate with
;;;                  arguments, such as "(fever mild)";
;;;             -- predicate, the instance of the predicate structure for
;;;                  this term;
;;;             -- argnumber, a number from 1 to N indicating which argument
;;;                  is being checked on this call;
;;;             -- alist, the alist to be searched;
;;;             -- if-doesnt-exist, what to do if the argument doesn't already
;;;                exist (see function check-term-name).
;;;
;;;  Returns:   (values termnode alist), where
;;;             -- termnode is the term node that was either found or created
;;;                during this search, and
;;;             -- alist is the possibly modified version of the alist
;;;                argument (to be stored back into the predicate structure).
;;;
;;;  Notes:     This function searches the alist stored in the 'args' slot of
;;;             the predicate structure and ultimately returns the instance of
;;;             the term node (which is created if not already in existence).
;;;             See the comments preceding the 'predicate' structure.
;;;
;;;             This function correctly handles predicates inside of predicates
;;;             as in (weather sunny (humidity low) warm).
;;;----------------------------------------------------------------------------

(defun check-args (termname predicate argnumber alist if-doesnt-exist)
  
  (if (not (listp alist))
      (progn
	(format t "~%Error3: check-args: ~A, arg ~A." termname argnumber)
	(values nil alist)))
  
  (let* ((argname  (nth argnumber termname))
	 pair)

    (if (not (numberp argname))
	(let ((argnode  (check-term-name argname if-doesnt-exist)))
	  ;; If no node exists/created for this argument ...
	  (if (null argnode)
	      ;; then return a failure ...
	      (return-from check-args (values nil alist))
	      ;; else get the name from the node in case the user respelled
	      ;; it or used a synonym or abbreviation.
	      (setq argname (node-name argnode)))))

    (if (listp argname)
	(setq pair (find argname alist :test #'equal :key #'car))
	(setq pair (assoc argname alist)))

    ;; If this argument value has ocurred before ...
    (if pair
	;; then if this is the end of the arguments ...
	(if (= (1+ argnumber) (length termname))

	    ;; then if this points to a term node ...
	    (if (term-p (cdr pair))
		;; then return it
		(values (cdr pair) alist)
		;; else report error
		(progn
		  (format t "~%Error1: check-args: ~A, arg ~A."
			  termname argnumber)
		  (values nil alist)))

	    ;; else recurse down to next argument
	    (if (listp (cdr pair))
		(multiple-value-bind (termnode newalist)
		    (check-args termname predicate (1+ argnumber) (cdr pair) if-doesnt-exist)
		  (rplacd pair newalist)
		  ;;(push newalist alist)
		  ;;(setq alist newalist)
		  (values termnode alist))
		;; else report error.
		(progn
		  (format t "~%Error2: check-args: ~A, arg ~A."
			  termname argnumber)
		  (values nil alist))))

	;; else create entry for this argument.  If this is the end
	;; of the arguments ...
	(progn 
	  (if (= (1+ argnumber) (length termname))

	      ;; then create new term node and return
	      (let ((termnode (make-term :name termname :predicate predicate)))
		(push termnode *history*)
		(push (cons argname termnode) alist)
		;;(setq alist (nconc alist (list (cons argname termnode))))
		(values termnode alist))

	      ;; else iterate through remaining arguments.
	      (let ((npair  (cons argname nil))
		    nnpair)
		(push npair alist)
		(dolist (argname (nthcdr (1+ argnumber) termname))
		  (if (not (numberp argname))
		      (let ((argnode (check-term-name argname if-doesnt-exist)))
			(if (null argnode)
			    (return-from check-args (values nil alist))
			    ;; Do this in case user changed spelling of argument.
			    (setq argname (node-name argnode)))))
		  (setq nnpair (cons argname nil))
		  (push nnpair (cdr npair))
		  (setq npair nnpair))

		(let ((termnode (make-term :name termname :predicate predicate)))
		  (push termnode *history*)
		  (rplacd npair termnode)
		  (values termnode alist))))))))


