;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Rita T. Duran

(in-package 'protos)

;;; ----------------------------------------------------------------------------
;;; HEURISTICS
;;;
;;; This file contains the code necessary to implement the heuristics
;;; described in Appendix A of Ray's dissertation.
;;;
;;; There are two types of heuristics:
;;;   FtoC-heuristics are meant to prune bad paths while doing forward chaining
;;;         from the feature to the category or other target.
;;;   FtoF-heuristics are meant to prune bad paths while doing backward chaining
;;;         from the exemplar to a feature of the newcase.
;;;
;;; These heuristics are domain-independent, in the sense that they only look
;;; at the links between nodes in the category network, not at the nodes
;;; themselves.
;;;
;;; ----------------------------------------------------------------------------


;;; ----------------------------------------------------------------------------
;;; Function:  (evaluate-strength  parent children relation down-strength)
;;;
;;; Given:     parent,         a graphnode
;;;            children,       to-nodes of relation from parent
;;;            relation,       relation linking parent to children
;;;            down-strength,  strength of path from start to parent
;;;
;;; Returns:   A list of the strengths from the start node to each child in
;;;            children.
;;;
;;; Note:      Some heuristics may modify the strength of only one child
;;;            without affecting the strengths of the others.
;;;
;;; ----------------------------------------------------------------------------

(defun evaluate-strength (parent children relation dstrength)
  (let ((num-children (length children)))
    (if (zerop num-children)
	(setq num-children 1))
    
    (if *heuristic-mode*
	;;use heuristics
	(if (eql *kbpm-mode* 'FtoF)
	    (mapcar #'(lambda (path-strength)
			(/ path-strength num-children))
		    (FtoF-heuristics parent children relation dstrength))
	    (mapcar #'(lambda (path-strength)
			(/ path-strength num-children))
		    (FtoC-heuristics parent children relation dstrength)))
      
	;; else don't use heuristics
	(if (eql *kbpm-mode* 'FtoF)
	    (make-list num-children :initial-element
		       (* dstrength (relation-strength (relation-inverse relation))))
	    (make-list num-children :initial-element
		       (* dstrength (relation-strength relation))))
	)))

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristics parent children relation dstrength)
;;;
;;; Given:     parent,   a graphnode
;;;            children, to-nodes of the relation from parent
;;;            relation, relation linking parent to children
;;;            dstrength, strength of the path from start to parent
;;; 
;;; Returns:   If no heuristics apply, returns a list of length 1 containing 
;;;            the default strength of the inverse of relation times
;;;            the strength of the path to parent.
;;;            Otherwise, return a vector of strengths. Each entry in the vector
;;;            is the strength of the link between the start node and the
;;;            corresponding child node.
;;;
;;; Design:   Calling function for the Feature-to-Feature heuristics.
;;;           Some heuristics are called for all links; others are only 
;;;           called depending on the type of relation.
;;;
;;;   FtoF search is meant to produce an explanation from a newcase feature to
;;;   a feature of the exemplar.  However, Protos begins its search at a feature
;;;   of the exemplar, producing an explanation from the exemplar feature to a
;;;   newcase feature. Thus, this explanation must be reversed (inverted) after
;;;   the search is complete.  Inverting an explanation requires inverting every
;;;   relation within it.  Note, however, that since the user has the capability
;;;   of setting the strength of any relation, the strength of a relation is not
;;;   necessarily equal to the strength of its inverse.  Thus, in order to find
;;;   the best path from the newcase to the exemplar, protos must use the
;;;   strengths of the inverse relations as it does its search.  The FtoF
;;;   heuristics must also look for and remove invalid sequences of the inverse
;;;   relations as the search progresses. 
;;;
;;;   FtoF search
;;;   Direction of Search:  ----------->
;;;
;;;   Exemplar                                                      New Case
;;;                        next-to      
;;;                        last-link     last-link     next-link             
;;;       O <---------- O <---------- O <---------- O <---------- O  . . .
;;;
;;;       inverse relations: <-----------
;;;       relations:          ----------->
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristics (parent children relation dstrength)
  (let* ((inv-relation (relation-inverse relation))
	 (next-link (relation-verb inv-relation))
	 (link-strength (relation-strength inv-relation))
	 (grandparent (graphnode-parent parent))
	 (gstrength 0.0)
	 ;;path-strength = default strength from start to children
	 (path-strength (* dstrength link-strength))
	 (strength-vector nil))
    
    (if *trace-heuristics*
      (progn
         (format t "~%")
	 (if grandparent
	    (format t "~% Grandparent: ~A"
	      (getname (graphnode-name grandparent))))
         (format t "~% Parent: ~A"  (getname (graphnode-name parent)))
	 (format t "~% Children: ")
	 (dolist (c children)
	   (format t "~@(~A~) " (getname c)))
	 (format t "~% Inv-Relation: ~A " (getname next-link))))
    
    (if grandparent
      (progn
        (setq gstrength (graphnode-downstrength grandparent))
        (if (/= (graphnode-certainty grandparent) 1.0)
	   (setq gstrength (graphnode-certainty grandparent)))))
    
    (setq strength-vector
      (cond
	;; no heuristics apply to the spurious relation
        ((null children) 
	 (return-from FtoF-heuristics (list path-strength)))

	((FtoF-heuristic1and2  children path-strength))

	;; if very first link in graph, none of the following heuristics
	;; can possibly apply
	((null grandparent) nil)
	
	((FtoF-heuristic6  parent children))
	
	((correlational next-link)
	 (FtoF-heuristic4  parent children path-strength))

	((unidirectional next-link)
	 (FtoF-heuristic18  parent children next-link))

	((eq next-link *verb-hasTypicalSpec*)
	 (FtoF-heuristic17  parent children next-link link-strength gstrength))

	((eq next-link *verb-partOf*)
	 (cond
	   ((FtoF-heuristic7and8  parent children))
	   ((FtoF-heuristic12     parent children))))

	((eq next-link *verb-causes*)
	 (cond
	   ((FtoF-heuristic10and11  parent children path-strength))
	   ((FtoF-heuristic14       parent children link-strength gstrength))
	   ((FtoF-heuristic15       parent children next-link
				    link-strength gstrength))))

	((eq next-link *verb-hasFunction*)
	 (cond
	   ((FtoF-heuristic10and11  parent children path-strength))
	   ((FtoF-heuristic13       parent children next-link
				    link-strength gstrength))
	   ((FtoF-heuristic16       parent children next-link
				    link-strength gstrength))))	 

	((eq next-link *verb-enables*)
	 (cond
	   ((FtoF-heuristic13  parent children next-link link-strength gstrength))
	   ((FtoF-heuristic16  parent children next-link link-strength gstrength))
	   ))

	((eq next-link *verb-causedBy*)
	   (FtoF-heuristic15  parent children next-link
			      link-strength gstrength))

	((eq next-link *verb-isFunctionOf*)
	  (FtoF-heuristic16  parent children next-link
			     link-strength gstrength))

	((eq next-link *verb-isEnabledBy*)
	  (FtoF-heuristic16  parent children next-link
			     link-strength gstrength))

	((eq next-link *verb-hasTypicalGen*)
	 (cond
	   ((FtoF-heuristic5   parent children))
  	   ((FtoF-heuristic17  parent children next-link
			       link-strength gstrength))))
	))
    
    (when (null strength-vector)
      (setq strength-vector (FtoF-heuristic9  children path-strength)))
    
    (if (null strength-vector)
      (return-from FtoF-heuristics
	(make-list (length children) :initial-element path-strength)) 
      (return-from FtoF-heuristics strength-vector))
    ))

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic1and2 children path-strength)
;;;
;;; Heuristic1:  Two features are not related merely by virtue of their common
;;;              relationship with the category of the exemplar being matched.
;;;
;;; Example:     "Carnivorous is consistent with dog which usually exhibits
;;;              barking" does not imply the equivalence of carnivorous and
;;;              barking.
;;;
;;; Heuristic2:  Two features are not related merely by virtue of their common
;;;              relationship with a sibling category of exemplar's category.
;;;
;;; Example:     If the hypothesized category is dog, then "carnivorous is 
;;;              consistent with coyote which usually exhibits barking" does
;;;              not imply the equivalence of carnivorous and barking.
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic1and2 (children path-strength)
  (do ((siblings  (cons *category* *category-siblings*))
       (strengths nil)
       (success   nil)
       (c         children (cdr c)))
    ((endp c)
     (if success
        (progn (trace-print *trace-heuristics*
			    "~% FtoF-heuristic1and2 result: ~A"
		              (reverse strengths))
		  (nreverse strengths))))
    
    (if (member (car c) siblings)
      (progn (setq success t)
             (push 0 strengths))
      (push path-strength strengths))
    ))




;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic3)
;;;
;;; Heuristic:  Two nodes are not related because they are both in the to-nodes
;;;             or from-nodes of a relation. 
;;;
;;; Note:       This case is already taken care of by the function generate-
;;;             successors.
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic3 ()
  nil)



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic4 parent children strength)
;;;
;;; Heuristic:  A multi-step correlational explanation is weak, especially if
;;;             it involves more than two inference steps.
;;;
;;; Example:    "odor is sometimes consistent with dog sometimes co-occurs with
;;;             boy usually co-occurs with toys."
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link            last-link        next-link             
;;;  . . .   <---------------- O <--------------- O <------------- . . .
;;;            correlational      correlational     correlational
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic4 (gnode children path-strength)
  (prog (parent last-link next-to-lastlink)
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    (setq last-link (relation-verb (relation-inverse
			     (graphlink-relation (graphnode-link gnode)))))
    (if (or (not (correlational last-link))
	    (null (graphnode-link parent)))
      (return nil))
    (setq next-to-lastlink (relation-verb (relation-inverse
			      (graphlink-relation (graphnode-link parent)))))
    (if (not (correlational next-to-lastlink))
      (return nil)
      (progn (trace-print *trace-heuristics*
			  "~% FtoF-4 applied ~A " (* .5 path-strength))
             (return (make-list (length children)
			 :initial-element (* .5 path-strength))))
      )))





;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic5 gnode children)
;;;
;;; Heuristic:  A path through a common generalization is acceptable, but not
;;;             continuable.
;;;
;;; Example:    Chair legs have typical generalization seat support has typical
;;;             specialization pedestal usually has part wheels.
;;;             A --- hasTypicalGen ---> B --- hasTypicalSpec ---> C
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link        last-link         next-link             
;;;  . . .   <----------- O <--------------- O <--------------- . . .
;;;                           hasTypicalSpec      hasTypicalGen
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic5 (gnode children)
  (prog (parent last-link)
    
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    
    (setq last-link (relation-verb (relation-inverse
		  (graphlink-relation (graphnode-link gnode)))))
    
    (if (and (eq *verb-hasTypicalSpec* last-link)
	     (graphnode-link parent))
      (progn (trace-print *trace-heuristics* "~% FtoF-5 applied ")
             (return (make-list (length children) :initial-element 0)))
      (return nil))
    ))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic6 gnode children)
;;;
;;; Heuristic:  A path from a generalization to a specialization is not
;;;             continuable to another generalization.
;;;
;;; Example:    "Hunts-in-packs is consistent with wolf has typical
;;;             generalization canine has typical specialization dog has
;;;             typical generalization pet is sometimes consistent with
;;;             lives-indoors."
;;;
;;;             A --- hasTypicalGen ---> B --- hasTypicalSpec ---> C ------> D
;;;
;;;  Exemplar                                                      New Case
;;;            next-to- 
;;;            last-link             last-link       next-link            
;;;  . . . O <---------------- O <-------------- O < ---------- . . .
;;;            hasTypicalGen      hasTypicalSpec
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic6 (gnode children)
  (prog (parent last-link next-to-last-link)
    
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    
    (setq last-link (relation-verb (relation-inverse
		  (graphlink-relation (graphnode-link gnode)))))
    
    (if (or (not (eq *verb-hasTypicalSpec* last-link))
	    (null (graphnode-link parent)))
      (return nil))
    
    (setq next-to-last-link
      (relation-verb (relation-inverse (graphlink-relation (graphnode-link parent)))))
    
    (if (eq *verb-hasTypicalGen* next-to-last-link)
      (progn (trace-print *trace-heuristics* "~% FtoF-heuristic6 applied")
             (return (make-list (length children) :initial-element 0))))
    ))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic7and8 parent children )
;;;
;;; Heuristic7:  The function of an assembly cannot be ascribed to one of its
;;;              parts.
;;; 
;;; Example:     Volume control is part of tv set has function communication.
;;;              A --- isPartOf ---> B --- hasFunction ---> C
;;;
;;;
;;; Heuristic8:  If a composite system (something which has parts) is the cause
;;;              of something, the causality cannot be ascribed to one of its
;;;              parts.
;;;
;;; Example:    Water is part of acid rain causes ecological damage.
;;;               A --- isPartOf ---> B --- causes ---> C
;;;
;;;  Exemplar                                                      New Case
;;;                       last-link        next-link             
;;;            . . . O <------------- O <------------ . . .
;;;                        causes          partOf
;;;                      hasFunction
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic7and8 (parent children)
  (let ((last-link (relation-verb (relation-inverse 
		      (graphlink-relation (graphnode-link parent))))))
    
    (if (or (eq last-link *verb-causes*)
	    (eq last-link *verb-hasFunction*))
      (progn
       (trace-print *trace-heuristics* "~% FtoF-7and8 applied")
       (make-list (length children) :initial-element 0)))))





;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic9 children path-strength)
;;;
;;; Heuristic:  When selecting an inference step, prefer one which leads to the
;;;             goal feature over one which is not known to lead to the goal.
;;;
;;; Example:    prefer "nose is sometimes equivalent to probosis" over 
;;;             "nose is part of face ... "
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic9 (children path-strength)
; (format t "~%FtoF-heuristic9 path-strength = ~4,2F" path-strength)
  (do ((c children (cdr c))
       (strengths nil)
       (success nil))
      ((endp c) (if success (progn (trace-print *trace-heuristics* 
						"~%FtoF-heuristic9 applied ~A"
						(reverse strengths))
				   (nreverse strengths))))

    (if (member (car c) *case-features*)  ; this global is in kbpm.lisp
	(progn
          (push (max .7 path-strength) strengths)
	  (setq success t))
	(push path-strength strengths))))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic10and11 parent children strength)
;;;
;;; Heuristic10:  Mixing causal and correlational links within a chain of 
;;;               inference is suspicious, but not prohibited.
;;;
;;; Example:      A --- causes ---> B --- correlationally related to ---> C
;;;
;;; Heuristic11:  Mixing functional and correlational links within a chain of 
;;;               inference is suspicious, but not prohibited.
;;;
;;; Example:      A --- hasFunction ---> B --- correlationally related to ---> C
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link          last-link         next-link             
;;;  . . .   <------------- O <--------------- O <-------------- . . .
;;;                             correlational       hasFunction
;;;                                                   causes
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic10and11 (parent children strength)
  (let ((last-link (relation-verb (relation-inverse
		      (graphlink-relation (graphnode-link parent))))))
    
    (if (correlational last-link)
      (progn
       (trace-print *trace-heuristics* "~% FtoF10and11 applied: ~A " (* .75 strength))
       (make-list (length children) :initial-element (* .75 strength)))
      )))




;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic12 parent children)
;;;
;;; Heuristic:  Two features are not strongly related by virtue of being parts
;;;             of the same assembly.
;;;
;;; Example:    Wheels are part of car which has part engine.
;;;             A --- partOf ---> B --- hasPart ---> C
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link        last-link        next-link             
;;;  . . .   <------------- O <------------ O <------------- . . .
;;;                              hasPart          partOf
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic12 (parent children)
  (let ((last-link (relation-verb (relation-inverse 
			      (graphlink-relation (graphnode-link parent))))))
    
    (if (eq last-link *verb-hasPart*)
      (progn
       (trace-print *trace-heuristics* "~% FtoF-12 applied")
       (make-list (length children) :initial-element 0)))
    ))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic13 parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  Two features which share a common function are very similar.
;;; 
;;; Example:    Airplane wings enable lift which is enabled by helicopter rotor.
;;;             A ----- enables ----> B --- isEnabledBy ----> C
;;;             A --- hasFunction --> B --- isFunctionOf ---> C
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link        last-link        next-link             
;;;  . . .   <------------- O <------------ O <------------- . . .
;;;                             isEnabledBy      enables
;;;                            isFunctionOf     hasFunction
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic13 (parent children next-link link-strength gstrength)
  (let* ((last-rel (relation-inverse (graphlink-relation (graphnode-link parent))))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))
    
    (if (or (and (eq last-link *verb-isEnabledBy*) (eq next-link *verb-enables*))
            (and (eq last-link *verb-isFunctionOf*) (eq next-link *verb-hasFunction*)))
    
      (progn
       (setq new-strength (min (max .9 (* last-rel-strength link-strength))
			       (* 2 last-rel-strength link-strength)))

       (trace-print *trace-heuristics* "~% FtoF-13 applied ~A" new-strength)
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))
    ))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic14 parent children gstrength)
;;;
;;; Heuristic:  Two categories which cause the same thing are somewhat similar.
;;;
;;; Example:    Bacterial infection sometimes causes fever which is sometimes
;;;             caused by viral infection.
;;;             A --- causes --> B --- causedBy --> C
;;;
;;;  Exemplar                                                      New Case
;;;              
;;;                             last-link        next-link             
;;;  . . .   <------------- O <------------ O <------------- . . .
;;;                             causedBy        causes
;;;
;;;       inverse relations: <-----------
;;; 
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic14 (parent children link-strength gstrength)
  (let* ((last-rel (relation-inverse (graphlink-relation (graphnode-link parent))))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))
    
    (if (eq last-link *verb-causedby*)
      (progn 
       (setq new-strength (min (max .8 (* last-rel-strength link-strength))
			       (* 2 last-rel-strength link-strength)))
       
       (trace-print *trace-heuristics* "~%FtoF-14 applied ~A  " new-strength)
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))
    ))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic15 
;;;                            parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  The length of a sequence of causal relationships should not 
;;;             diminish its goodness.  
;;; 
;;; Example:    
;;;             A ----- causes ---> B ---- causes ----> C
;;;             A --- causedBy ---> B --- causedBy ---> C
;;;
;;; Note:  Quantifiers on the causal relationships may weaken the strength of
;;;        the path.
;;; ----------------------------------------------------------------------------



(defun FtoF-heuristic15 (parent children next-link link-strength gstrength)
  (let* ((last-rel (relation-inverse (graphlink-relation (graphnode-link parent))))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	(new-strength 0))

    (if (or (and (eq next-link *verb-causes*) (eq last-link *verb-causes*))
	    (and (eq next-link *verb-causedBy*) (eq last-link *verb-causedBy*)))
    
      (progn
        (setq new-strength (min last-rel-strength link-strength))
        (trace-print *trace-heuristics* "~% FtoF-15 applied ~A" new-strength)
        (make-list (length children) :initial-element (* gstrength new-strength))
       ))
    ))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic16
;;;                           parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  The length of a sequence of functional relationships should not 
;;;             diminish its goodness.
;;; 
;;; Example:    "wings enable lift enable flight"
;;;
;;;             A ---- hasFunction ---> B ---- hasFunction ----> C
;;;             A --- isFunctionOf ---> B --- isFunctionOf ----> C
;;;             A ------ enables -----> B ----- enables -------> C
;;;             A ----is EnabledBy ---> B ----isEnabledBy -----> C
;;;
;;;                   next-link               last-link
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic16 (parent children next-link link-strength gstrength)
  (let* ((last-rel (relation-inverse (graphlink-relation (graphnode-link parent))))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))

   (if (or
	(and (eq next-link *verb-hasFunction*) (eq last-link *verb-hasFunction*))
	(and (eq next-link *verb-isFunctionOf*) (eq last-link *verb-isFunctionOf*))
	(and (eq next-link *verb-enables*) (eq last-link *verb-enables*))
	(and (eq next-link *verb-isEnabledBy*) (eq last-link *verb-isEnabledBy*)))

     (progn
       (setq new-strength (min last-rel-strength link-strength))
       (trace-print *trace-heuristics* "~%FtoF-16 applied ~A "new-strength)
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))
   ))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic17 
;;;                           parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  The length of a sequence of generalization relationships should
;;;             not diminish its goodness.
;;; 
;;; Example:    "body covering(hair) is consistent with mammal which has typical
;;;             specialization dog which has typical specialization collie"
;;;               A ---- hasTypicalSpec ----> B ---- hasTypicalSpec ----> C
;;;               A ---- hasTypicalGen  ----> B ---- hasTypicalGen  ----> C
;;;               
;;;                    next-link                     last-link
;;; ----------------------------------------------------------------------------


(defun FtoF-heuristic17 (parent children next-link link-strength gstrength)
  (let* ((last-rel (relation-inverse (graphlink-relation (graphnode-link parent))))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))

    (if (or
	 (and (eq next-link *verb-hasTypicalGen*) (eq last-link *verb-hasTypicalGen*))
	 (and (eq next-link *verb-hasTypicalSpec*)(eq last-link *verb-hasTypicalSpec*)))

      (progn
       (setq new-strength (min last-rel-strength link-strength))
       (trace-print *trace-heuristics* "~% FtoF-17 applied ~A "new-strength)
       (make-list (length children) :initial-element (* gstrength new-strength))
    ))
    ))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoF-heuristic18  parent children next-link)
;;;
;;; Heuristic:  An inference step of a unidirectional verb is continuable only
;;;             if it does not conflict in direction with a previous
;;;             unidirectional verb.
;;;
;;; Example:    "stroke-volume fm+ cardiac-output rm+ pulse" is not continu-
;;;             able.
;;;
;;;  Exemplar                                                      New Case
;;;            next-to-  
;;;            last-link            last-link        next-link             
;;;  . . .   <---------------- O <--------------- O <------------- . . .
;;;                                                  unidirectional
;;;
;;;       inverse relations: <-----------
;;;
;;; ----------------------------------------------------------------------------

(defun FtoF-heuristic18 (gnode1 children next-link)
  (let (last-link next-direction)
    (setq next-direction (direction next-link))
    
    (do* ((gnode  gnode1  parent)
	  (parent (graphnode-parent gnode) (graphnode-parent gnode)))
	 ((null parent) nil)
      
      (setq last-link (relation-verb
			(relation-inverse
			  (graphlink-relation (graphnode-link gnode)))))
      (if (and (unidirectional last-link)
	       (not (eql next-direction (direction last-link))))
	  (return-from FtoF-heuristic18
	    (make-list (length children) :initial-element 0))))))




;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristics parent children relation)
;;;
;;; Calling function for the Feature-to-Category heuristics
;;;
;;; FtoC search
;;;   Direction of Search:  ----------->
;;;
;;;   Feature                                                        Target
;;;                       next-to      
;;;                       last-link      last-link     next-link             
;;;       O ----------> O ----------> O ----------> O ----------> O  . . .
;;;
;;;       relations:          ----------->
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristics (parent children relation dstrength)
  (let* ((next-link (relation-verb relation))
	 (link-strength (relation-strength relation))
	 (grandparent (graphnode-parent parent))
	 (gstrength 0.0)
	 ;;path-strength = default strength from start to children
	 (path-strength (* dstrength link-strength))
	 (strength-vector nil))
    
    (if *trace-heuristics*
      (progn
         (format t "~%")
	 (if grandparent
	    (format t "~% Grandparent: ~A" (getname (graphnode-name grandparent))))
         (format t "~% Parent: ~A"  (getname (graphnode-name parent)))
	 (format t "~% Children: ")
	 (dolist (c children)
	   (format t "~@(~A~) " (getname c)))
	 (format t "~% Relation: ~A " (getname next-link))))
    
    (if grandparent
      (progn
        (setq gstrength (graphnode-downstrength grandparent))
        (if (/= (graphnode-certainty grandparent) 1.0)
	   (setq gstrength (graphnode-certainty grandparent))))
      )

    (setq strength-vector
      (cond
	;; no heuristics apply to the spurious relation
	((null children) 
	 (return-from FtoC-heuristics (list path-strength)))

	((FtoC-heuristic1 children path-strength))

	;; if first link in search graph, none of the following heuristics apply
	((null grandparent) nil)

	((FtoC-heuristic3 parent children))

	((eq next-link *verb-hasTypicalGen*)
	 (cond
	   ((FtoC-heuristic4 parent children))
	   ((FtoC-heuristic17 parent children link-strength gstrength))))

	((correlational next-link)
	 (cond
	   ((FtoC-heuristic6 parent children path-strength))
	   ((FtoC-heuristic8and9 parent children path-strength))
	   ))

	((eq next-link *verb-hasTypicalSpec*)
	 (FtoC-heuristic7 children path-strength))  ;; always succeeds

	((eq next-link *verb-hasPart*)
	 (FtoC-heuristic10and11and12 parent children))
	
	((eq next-link *verb-hasFunction*)
	 (cond
	  ((FtoC-heuristic10and11and12 parent children))
	  ((FtoC-heuristic16 parent children next-link 
			     link-strength gstrength))))

	((eq next-link *verb-isFunctionOf*)
	 (cond
	   ((FtoC-heuristic13 parent children next-link link-strength gstrength))
	   ((FtoC-heuristic16 parent children next-link link-strength gstrength))
	   ))

	((eq next-link *verb-isEnabledBy*)
	 (cond
	   ((FtoC-heuristic13 parent children next-link link-strength gstrength))
	   ((FtoC-heuristic16 parent children next-link link-strength gstrength))))

	((eq next-link *verb-causedBy*)
	 (cond 
	   ((FtoC-heuristic14 parent children link-strength gstrength))
	   ((FtoC-heuristic15 parent children next-link
			      link-strength gstrength))))

	((eq next-link *verb-causes*)
	 (cond
	   ((FtoC-heuristic10and11and12 parent children))
	   ((FtoC-heuristic15 parent children next-link link-strength gstrength))))

	((eq next-link *verb-enables*)
	 (FtoC-heuristic16 parent children next-link link-strength gstrength))

	))

    (when (null strength-vector)
      (setq strength-vector (FtoC-heuristic5 children next-link path-strength)))
    
    (if (null strength-vector)
      (return-from FtoC-heuristics
	(make-list (length children) :initial-element path-strength))
      (return-from FtoC-heuristics strength-vector))

    ))

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic1  children path-strength)
;;;
;;; Heuristic:  An explanation cannot involve a sibling of *category*.
;;; 
;;; Example:    "carnivorous is consistent with coyote which usually exhibits
;;;             barking which is consistent with dog" does not imply a relation-
;;;             ship between carnivorous and dog.
;;;
;;; Note:       Corresponds to FtoF-heuristic2
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic1 (children path-strength)
  (do ((siblings  *category-siblings*)
       (strengths nil)
       (success   nil)
       (c         children (cdr c)))
    ((endp c)
     (if success
        (progn (trace-print *trace-heuristics*
			    "~% FtoC-heuristic1 result: ~A"
			    (reverse strengths))
	       (nreverse strengths))))
    
    (if (member (car c) siblings)
      (progn (setq success t)
             (push 0 strengths))
      (push path-strength strengths))
    ))

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic2 )
;;;
;;; Heuristic:  Two features are not related just because they were involved in 
;;;             a previous conjunctive explanation.
;;;
;;; Note:       This case already eliminated by the fnnction generate-successors.
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic2 ()
  nil)

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic3 gnode children)
;;;
;;; Heuristic:  A path through a common generalization is acceptable, but not
;;;             continuable.
;;;      
;;; Example:    Chair legs have typical generalization seat support has typical
;;;             specialization pedestal usually has part wheels.
;;;             A --- hasTypicalGen ---> B --- hasTypicalSpec ---> C
;;;
;;;   Direction of Search:  ----------->
;;;
;;;   Feature                                                         Category
;;;                      next-to      
;;;                      last-link           last-link       next-link             
;;;       O --------> O --------------> O ---------------> O ----------> O  . . .
;;;                      hasTypicalGen      hasTypicalSpec
;;;
;;;       relations:          ----------->
;;;
;;; ----------------------------------------------------------------------------
        
(defun FtoC-heuristic3 (gnode children)
  (prog (parent last-link next-to-last-link)
    
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    
    (setq last-link (relation-verb (graphlink-relation (graphnode-link gnode))))
    
    (if (or (not (eq *verb-hasTypicalSpec* last-link))
	    (null (graphnode-link parent)))
      (return nil))
    
    (setq next-to-last-link
      (relation-verb (graphlink-relation (graphnode-link parent))))
    
    (if (eq *verb-hasTypicalGen* next-to-last-link)
      (progn (trace-print *trace-heuristics* "~% FtoC-3 applied ")
             (return (make-list (length children) :initial-element 0)))
      (return nil))
    ))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic4 parent children)
;;;
;;; Heuristic:  A path from a generalization to a specialization is not
;;;             continuable to another generalization.
;;;
;;; Example:    "Hunts-in-packs is consistent with wolf has typical
;;;             generalization canine has typical specialization dog has
;;;             typical generalization pet is sometimes consistent with
;;;             lives-indoors."
;;;
;;;             A --- hasTypicalGen ---> B --- hasTypicalSpec ---> C ------> D
;;;
;;;  Feature                                                      Category
;;;             next-to-
;;;             last-link         lastt-link        next-link              
;;;  . . . O ----------------> O --------------> O ---------------> . . .
;;;                              hasTypicalSpec     hasTypicalGen
;;;
;;;        relations: ----------->
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic4 (gnode children)
  (prog (parent last-link)
    
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    
    (setq last-link (relation-verb (graphlink-relation (graphnode-link gnode))))
    
    (if (or (not (eq *verb-hasTypicalSpec* last-link))
	    (null (graphnode-link parent)))
      (return nil)
      (progn (trace-print *trace-heuristics* "~% FtoC-heuristic4 applied")
             (return (make-list (length children) :initial-element 0))))
    ))




;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic5  children next-link path-strength)
;;;
;;; Heuristic:  When selecting an inference step, one leading to the target
;;;             category is better than one which is not known to do so.
;;; 
;;; Note:      Corresponds to FtoF-heuristic9
;;;		      
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic5 (children next-link path-strength)
  (do ((c children (cdr c))
       (strengths nil)
       (success nil))
      ((endp c)
     (if success (progn (trace-print *trace-heuristics* 
				     "~% FtoC-heuristic5 applied ~A"
				     (reverse strengths))
                                 (nreverse strengths))))
    (if  (eq (car c) *category*)
      (progn
         (if (eq next-link *verb-hasTypicalSpec*)
	   (push 1.0 strengths)
	   (push (max path-strength .7) strengths))
	 (setq success t))
      
      (push path-strength strengths))))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic6  parent children path-strength)
;;;
;;; Heuristic:  A multistep correlational explanation is weak, especially if it
;;;             involves more than two steps.
;;;
;;; Example:    "odor is sometimes consistent with dog sometimes co-occurs with
;;;             boy usually co-occurs with toys."
;;;
;;;  Feature                                                       Category
;;;            next-to-  
;;;            last-link            last-link          next-link             
;;;  . . .   ----------------> O  -------------->- O -------------> . . .
;;;            correlational       correlational     correlational
;;;
;;;       relations: ----------->
;;;
;;; Notes:     Corresponds to FtoF-heuristic4
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic6 (gnode children path-strength)
  (prog (parent last-link next-to-lastlink)
    (setq parent (graphnode-parent gnode))
    (if (null parent)
      (return nil))
    (setq last-link (relation-verb (graphlink-relation (graphnode-link gnode))))
    (if (or (not (correlational last-link))
	    (null (graphnode-link parent)))
      (return nil))
    (setq next-to-lastlink
      (relation-verb (graphlink-relation (graphnode-link parent))))
    (if (not (correlational next-to-lastlink))
      (return nil)
      (progn (trace-print *trace-heuristics*
			  "~% FtoC-6 applied ~A " (* .5 path-strength))
             (return (make-list (length children)
			 :initial-element (* .5 path-strength))))
      )))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic7 children path-strength)
;;;
;;; Heuristic:  When relating a feature to a category, traversing a spec link 
;;;             from a general category is a reasonable step in the explanation.
;;;
;;; Example:   "body covering(hair) is consistent with mammal has typical
;;;            specialization dog"
;;;               A --- relatedTo ---> B --- hasTypicalSpec ---> C
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic7 (children path-strength)
  (trace-print *trace-heuristics*
			  "~% FtoC-7 applied ~A "
			  (min (max .7 path-strength) (* path-strength 2.0)))
  (make-list (length children) :initial-element
	     (min (max .7 path-strength) (* path-strength 2.0))))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic8and9 parent children next-link path-strength)
;;;
;;; Heuristic8: Mixing causal and correlational links within a chain of inference
;;;             is suspicious, but not prohibited.
;;;
;;; Example:    A --- causes ---> B --- correlationally related to ---> C
;;;
;;; Heuristic9: Mixing functional and correlational links within a chain of
;;;             inference is suspicious, but not prohibited.
;;;
;;; Example:    A --- hasFunction ---> B --- correlationally related to ---> C
;;;
;;;  Feature                                                      Category
;;;     
;;;            next-to-  
;;;            last-link          last-link        next-link             
;;;  . . .   ------------- O ---------------> O --------------> . . .
;;;                             hasFunction      correlational
;;;                              causes
;;;
;;;       relations: ----------->
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic8and9 (parent children strength)
  (let ((last-link (relation-verb (graphlink-relation (graphnode-link parent)))))
    
    (if (or (eq last-link *verb-causes*)
	      (eq last-link *verb-hasFunction*))
      (progn
       (trace-print *trace-heuristics* "~% FtoC8and9 applied: ~A " (* .75 strength))
       (make-list (length children) :initial-element (* .75 strength)))
      )))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic10and11and12  parent children)
;;;
;;; Heuristic10:  The function of an assembly cannot be ascribed to one of its
;;;               parts.
;;;
;;; Example:      A --- partOf ---> B --- hasFunction ---> C   
;;;
;;; Note:         Corresponds to FtoF-heuristic7
;;;
;;; Heuristic11:  If a composite system is the cause of something, the causality
;;;               cannot be ascribed to one of it parts.
;;;
;;; Example:      A --- partOf ---> B --- causes ---> C
;;;
;;; Note:         Corresponds to FtoF-heuristic8
;;;
;;; Heuristic12:  Two terms are not strongly related by virtue of being parts of
;;;               the same assembly.
;;;
;;; Example:      A --- partOf ---> B --- hasPart ---> C
;;;
;;; Note:         Corresponds to FtoF-heuristic12
;;;
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic10and11and12 (parent children)
  (let ((last-link (relation-verb (graphlink-relation (graphnode-link parent)))))
    (if (eq last-link *verb-partOf*)
      (progn
       (trace-print *trace-heuristics* "~% FtoC10and11and12 applied: ")
       (make-list (length children) :initial-element 0)))))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic13  parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  Two features which share a common function are very similar.
;;; 
;;; Example:    Airplane wings enable lift which is enabled by helicopter rotor.
;;;             A ----- enables ----> B --- isEnabledBy ----> C
;;;             A --- hasFunction --> B --- isFunctionOf ---> C
;;;
;;;  Feature                                                      Category
;;;
;;;                            last-link        next-link             
;;;  . . .   -------------> O ------------> O -------------> . . .
;;;                              enables       isEnabledby
;;;                            hasFunction     isFunctionOf
;;;                          
;;;
;;;        relations: ----------->
;;;
;;; Note:  Corresponds to FtoF-Heuristic13
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic13 (parent children next-link link-strength gstrength)
  (let* ((last-rel (graphlink-relation (graphnode-link parent)))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))
    
    (if (or (and (eq last-link *verb-enables*)(eq next-link *verb-isEnabledBy*))
            (and (eq last-link *verb-hasFunction*)(eq next-link *verb-isFunctionOf*)))
    
      (progn
       (setq new-strength (min (max .9 (* last-rel-strength link-strength))
			       (* 2 last-rel-strength link-strength)))

       (trace-print *trace-heuristics* "~% FtoC-13 applied ~A" new-strength)
       (make-list (length children) :initial-element 
		  (* gstrength new-strength)))
      )))



;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic14 parent children link-strength gstrength)
;;;
;;; Heuristic14:  Two categories which cause the same thing are somewhat similar.
;;;
;;; Example:    Bacterial infection sometimes causes fever which is sometimes
;;;             caused by viral infection.
;;;             A --- causes --> B --- causedBy --> C
;;;
;;; Note:  Corresponds to FtoF-Heuristic14
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic14 (parent children link-strength gstrength)
  (let* ((last-rel (graphlink-relation (graphnode-link parent)))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))
    
    (if (eq last-link *verb-causes*)
      (progn
       (setq new-strength (min (max .8 (* last-rel-strength link-strength))
			       (* 2 last-rel-strength link-strength)))
       (trace-print *trace-heuristics* "~% FtoC-14 applied ~A" new-strength)
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic15  parent children next-link link-strength gstrength)
;;;
;;; Heuristic:  The length of a sequence of causal relationships should not 
;;;             diminish its goodness.
;;; 
;; Example:    
;;;             A ----- causes ---> B ---- causes ----> C
;;;             A --- causedBy ---> B --- causedBy ---> C
;;;                 last-link           next-link
;;;
;;; ----------------------------------------------------------------------------



(defun FtoC-heuristic15 (parent children next-link link-strength gstrength)
  (let* ((last-rel (graphlink-relation (graphnode-link parent)))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0.0))

    (if (or (and (eq last-link *verb-causes*)(eq next-link *verb-causes*))
	    (and (eq last-link *verb-causedBy*)(eq next-link *verb-causedBy*)))
    
      (progn
       (setq new-strength (min last-rel-strength link-strength))
       (trace-print *trace-heuristics* "~%FtoC-15 applied ~A " new-strength) 
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))))


;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic16 parent children next-link link-strength dstrength)
;;;
;;; Heuristic:  The length of a sequence of functional relationships should not 
;;;             diminish its goodness.
;;; 
;;; Example:    "wings enable lift enable flight"
;;;
;;;             A ---- hasFunction ---> B ---- hasFunction ----> C
;;;             A --- isFunctionOf ---> B --- isFunctionOf ----> C
;;;             A ------ enables -----> B ----- enables -------> C
;;;             A ----is EnabledBy ---> B ----isEnabledBy -----> C
;;;
;;;                   last-link               next-link
;;; ----------------------------------------------------------------------------

(defun FtoC-heuristic16 (parent children next-link link-strength gstrength)
  (let* ((last-rel (graphlink-relation (graphnode-link parent)))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))

   (if (or
	(and (eq last-link *verb-hasFunction*) (eq next-link *verb-hasFunction*))
	(and (eq last-link *verb-isFunctionOf*)(eq next-link *verb-isFunctionOf*))
	(and (eq last-link *verb-enables*)     (eq next-link *verb-enables*))
	(and (eq last-link *verb-isEnabledBy*) (eq next-link *verb-isEnabledBy*)))

     (progn
       (setq new-strength (min last-rel-strength link-strength))
       (trace-print *trace-heuristics* "~%FtoC-15 applied ~A " new-strength) 
       (make-list (length children) :initial-element (* gstrength new-strength))
       ))))

;;; ----------------------------------------------------------------------------
;;; Function:  (FtoC-heuristic17 parent children link-strength gstrength)
;;;
;;; Heuristic:  The length of a sequence of generalization relationships should
;;;             not diminish its goodness.
;;; 
;;; Example:    "collie has typical generalization dog which has typical 
;;;              generalization mammal is consistent with body covering(hair)."
;;;               A ---- hasTypicalGen  ----> B ---- hasTypicalGen  ----> C
;;;               
;;;                   last-link                     next-link
;;; ----------------------------------------------------------------------------


(defun FtoC-heuristic17 (parent children link-strength gstrength)
  (let* ((last-rel (graphlink-relation (graphnode-link parent)))
	 (last-rel-strength (relation-strength last-rel))
	 (last-link (relation-verb last-rel))
	 (new-strength 0))

    (if (eq last-link *verb-hasTypicalGen*)
      (progn
       (setq new-strength (min last-rel-strength link-strength))
       (trace-print *trace-heuristics* "~%FtoC-15 applied ~A " new-strength) 
       (make-list (length children) :initial-element (* gstrength new-strength))
      ))
    ))
      




;;; Testing of heuristics
;;; ----------------------------------------------------------------------------
;(setq *trace-kbpm* t)
;(load "protos-cl/test-heuristics")
;(kbpm 'FtoC carnivorous 1.0 dog)
;(kbpm 'FtoC A 1.0 E)
;(kbpm 'FtoC E 1.0 A)
;(kbpm 'FtoC F 1.0 I)
;(kbpm 'FtoC chair-legs 1.0 wheels)
;(kbpm 'FtoC wheels 1.0 engine)
;(kbpm 'FtoC wings 1.0 propeller)
;(kbpm 'FtoC G 1.0 Q)
;(kbpm 'FtoF A 1.0 exemplar2)
;(setf (feature-of-newcase E) nil)
;(kbpm 'FtoF E 1.0 exemplar2)
;(kbpm 'FtoF carnivorous 1.0 dog-exemplar)
;(setf (feature-of-newcase F) t)
;(setf (feature-of-newcase G) nil)
;(kbpm 'FtoF I 1.0 exemplar2)
;(setf (feature-of-newcase F) nil)
;(setf (feature-of-newcase O) nil)
;(setf (feature-of-newcase Q) t)
;(kbpm 'FtoF I 1.0 exemplar2)
;(kbpm 'FtoF wheels 1.0 exemplar2)
;(kbpm 'FtoF transportation 1.0 exemplar2)
;(setf (feature-of-newcase wings) t)
;(kbpm 'FtoF propeller 1.0 exemplar2)


