;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;              C L A S S I F I C A T I O N    F U N C T I O N S 
;;;   ------------------------------------------------------------------------
;;;
;;;  Overview:   This file contains the main functions for CL-Protos'
;;;		 classification algorithm (except for kbpm, the knowledge-
;;;              based pattern matcher, which is in a separate file).
;;;
;;;  Functions:  build-hypotheses
;;;		 mark-hypotheses
;;;		 classify
;;;              test-hypotheses
;;;              test-hypotheses2
;;;		 compare-new-case
;;;		 compare-feature
;;;=============================================================================



(defvar *hypotheses*)     ; temporary storage for the list of hypotheses.



;;;----------------------------------------------------------------------------
;;;  Function:  (classify  newcase)
;;;
;;;  Given:     newcase, the case to be classified;
;;;
;;;  Returns:   -- NIL, if no classification could be determined; or
;;;             -- an exemplar, if the new case became a new exemplar of
;;;                some category; or
;;;             -- a list of matches of "approved" exemplar-to-newcase matches.
;;;
;;;  Design:    1.  Collect the raw remindings and censors evoked by the
;;;                 features of the new case.
;;;             2.  Heuristically combine the remindings/censors so that the
;;;                 most specific categories and exemplars will have the
;;;                 strongest remindings.
;;;             3.  Form an ordered list of hypotheses of exemplars to be
;;;                 matched to the new case.
;;;             4.  Compare the new case to each of the hypotheses in turn,
;;;                 returning an approved match (or set of matches).
;;;             5.  If either the exemplar or category of any of the approved
;;;                 matches evokes a reminding, then add that to the raw
;;;                 remindings and return to step 2.  This is done in case
;;;                 the match involves a featural exemplar, which means that
;;;                 there may be a higher-level concept yet to be matched.
;;;
;;;  Called by: enter-case-and-classify
;;;
;;;  Note:      This function is the top-level program interface to the Protos
;;;             classification algorithm.  If, after building a suitable
;;;             knowledge base, you wanted to use Protos as a callable,
;;;             non-interactive classification tool, then do:
;;;                 (setq *learning-mode* nil)
;;;                 (classify newcase)
;;;
;;;             The human-interface function "display-classification" takes
;;;             the value returned by this function and displays the
;;;             classification results.
;;;----------------------------------------------------------------------------

(defun classify (newcase)
  (prog ((preclass   (case-category newcase))
	 (approved-matches nil)
	 raw-remindings
	 remindings
	 hypotheses)

     RETRY
	;; Get raw remindings evoked by features of the new case.
	(setq raw-remindings (get-raw-remindings (case-features newcase)))

	;; If preclassified, add strong reminding to that classification.
	(if preclass
	    (push (cons preclass *reminding-strong*) raw-remindings))

     RECOMBINE
	;; Sum duplicate remindings and combine heuristically.
	(setq remindings (combine-remindings raw-remindings))
	
	;; Build list of hypotheses from the reminded categories and exemplars.
	(setq hypotheses (build-hypotheses remindings approved-matches))
	
	;; Evaluate similarity of new case to the hypotheses.
	(multiple-value-bind (action matches)
	    (test-hypotheses newcase hypotheses approved-matches)

	  ;; If a single exemplar returned (which only happens if the
	  ;; teacher has decided to create a new exemplar from the case),
	  ;; then return it to caller.
	  (cond ((null matches)  nil)
		((exemplar-p matches)  (return-from classify matches))
		((listp matches)       (setq approved-matches (nconc approved-matches matches)))
		(t                     (setq approved-matches (nconc approved-matches (list matches)))))

	  (case action
	    (retry    (go RETRY))
	    (done     (let ((new-remindings (get-new-remindings matches)))
			;; If any remindings associated with the new matches,
			;; add them in and add the matched exemplars as features.
			;; This is done in case there are "featural exemplars".
			(if new-remindings
			    (progn
			      (if *learning-mode*
				  (progn
				    (format t "~%The match to:~{ ~A~}"
					    (mapcar #'(lambda (x) (node-name (match-exemplar x))) matches))
				    (format t "~%has evoked remindings to: ")
				    (print-alist new-remindings t 1)
				    (if (not (prompt "~%Should Protos use these to look for higher-level concepts? "
						     nil 'y-or-n nil nil))
					     (go DONE))))

			      (setq raw-remindings (nconc new-remindings raw-remindings))
			      (dolist (match matches)
				(let* ((exemplar (match-exemplar match))
				       (category (exemplar-category exemplar)))
				  (push category (case-features newcase))
				  (push exemplar (case-features newcase))))
			      (go RECOMBINE)))))))

     DONE
	;; Returned list of matches.
        (return approved-matches)))



;;;----------------------------------------------------------------------------
;;;  Function:  (build-hypotheses  remindings  approved-matches)
;;;
;;;  Given:     -- remindings: an alist of remindings of the form
;;;                (node . strength), sorted by strength;
;;;             -- approved-matches, a list of matches whose exemplars and
;;;                categories should be excluded from the hypotheses to be
;;;                generated;
;;;
;;;  Returns:	a list of hypotheses sorted in decreasing order of strength.
;;;		Each hypothesis is of the form (exemplar strength lastp) where:
;;;		-- 'exemplar' is a prototypical exemplar of a reminded category
;;;		   or is a directly-reminded exemplar;
;;;		-- 'strength' is the strength of the combined reminding; and
;;;		-- 'lastp' is non-nil only if this is the last hypotheses
;;;		   having this exemplar's category on the list.
;;;                (This is used later to decide when to reassess remindings
;;;                to incorrect categories).
;;;
;;;  Called by: process-new-case
;;; 
;;;
;;;  Algorithm
;;;  ========================================================================
;;;    For each of the combined remindings
;;;      do
;;;        if reminding is to a category
;;;	      then add the N best exemplars to hypothesis list, with
;;;		   strength = (strength of reminding) * (prototypicality of
;;;				exemplar) / (prototypicality of prototype)
;;;	      else add the reminded exemplar to hypothesis list with its
;;;			strength
;;;	 done
;;;    Sort hypothesis list in increasing order of strength.
;;;    Call 'mark-hypotheses' to mark "last" exemplars and put the list into
;;;      decreasing order of strength.
;;;
;;;
;;;  NOTE:   What's the meaning of "N best exemplars" in the above 
;;;          algorithm?  In the original Protos it was a fixed number, but
;;;          here it has been improved so that the strength of the combined
;;;          reminding to the category controls how many exemplars are chosen.
;;;          For the category having the strongest combined reminding (of all
;;;          the remindings), the number of exemplars chosen is equal to
;;;          *n-best-exemplars*.  For a category having a combined reminding
;;;          that is, say, 40% as strong as the strongest reminding, it can
;;;          select only 40% as many exemplars.  For very weak remindings,
;;;          a minimum of 1 exemplar is chosen.
;;;----------------------------------------------------------------------------

(defun build-hypotheses (remindings approved-matches)
  (declare (special  *hypotheses*  *n-best-exemplars*))
  
  ;; Clear the hypotheses list.
  (setq *hypotheses* nil)
  
  (if *trace-hypotheses*
      (format t "~%Tracing build-hypotheses ..."))
  
  ;; If no combined remindings, then return to caller.
  (if (null remindings)
      (progn
	(if *trace-hypotheses*
	    (format t "~%   No combined remindings, therefore no hypotheses."))
	(return-from build-hypotheses nil)))
  
  ;; Save the strength value of the strongest reminding.  (This code assumes
  ;; that 'remindings' is already sorted in decreasing order of strength).
  (let* ((high-rem-strength   (cdar remindings))
	 (approved-exemplars  (mapcar #'match-exemplar approved-matches))
	 (approved-categories (mapcar #'exemplar-category approved-exemplars)))
    
    ;; For each of the combined remindings...
    (dolist (rem remindings)
      (let ((node     (car rem))		; pointer to reminded category or exemplar.
	    (strength (cdr rem)))		; strength of the reminding.
	
	;;;;;(format t "~%build-hypotheses: rem = ~A ~A, type = ~A"
	;;;;;              (getname node) strength (type-of node))
	
	;; If this node is an exemplar ...
	(if (my-exemplar-p node)
	    ;; then add it to the hypotheses
	    (if (not (member node approved-exemplars))
		(progn
		  (push (list node strength nil) *hypotheses*)
		  (if *trace-hypotheses*
		      (format t "~%   Direct reminding to this exemplar:~
                           ~%~5T~A~26T~4,2F"
			      (getname node) strength))))
	    
	    ;; Else this is a category node, so get the N best exemplars (where N
	    ;; is controlled by the strength of the reminding to the category).
	    (if (not (member node approved-categories))
		;; If this category has no exemplars ...
		(if (null (category-exemplars node))
		    ;; Then optionally print trace message
		    (if *trace-hypotheses*
			(format t "~%   Reminding to ~A, but it has no exemplars."
				(getname node)))
		    ;; Else select some exemplars as hypotheses.
		    (let* ((exemplars             (category-exemplars node))
			   (prototype             (car exemplars))
			   (prototype-typicality  (exemplar-typicality prototype))
			   (limit-of-exemplars    (max-exemplars strength high-rem-strength)))
		      
		      (if *trace-hypotheses*
			  (format t "~%   Reminding to ~A may select up to ~A exemplars:"
				  (getname node) limit-of-exemplars))
		      

		      (do* ((exemps exemplars (cdr exemps))
			    (i 0 (+ i 1)))
			   ((or (= i limit-of-exemplars) 
				(endp exemps)))
			(let ((e-strength (* strength 
					     (/ (exemplar-typicality (car exemps))
						prototype-typicality))))
			  (push (list (car exemps) e-strength nil) *hypotheses*)
			  (if *trace-hypotheses*
			      (format t "~%~5T~A~26T~4,2F"
				      (getname (car exemps)) e-strength))))))))))
    
    ;; If we still don't have any hypotheses,
    (if (null *hypotheses*)
	;; but we do have previously approved matches,
	(if approved-matches
	    ;; then return,
	    (return-from build-hypotheses nil)
	    ;; else try one last thing.
	    (build-hypotheses2 remindings approved-categories)))
    
    ;; Sort the hypotheses in INCREASING order of strength.  This ordering
    ;; allows us to (1) delete the weakest duplicate hypotheses, if any,
    ;; and (2) mark the last/weakest hypothesis in each category.
    (setq *hypotheses* (sort  *hypotheses*  #'<=  :key #'cadr))
    
    ;; Delete the weakest duplicate hypotheses, if any.
    (setq *hypotheses* (delete-duplicates *hypotheses* :key #'car))
    
    ;; Reverse the list and mark the last exemplar in each category/class.
    (setq *hypotheses* (mark-hypotheses *hypotheses*))
    
    (if *trace-hypotheses*
	(progn
	  (format t "~%   Sorted list of hypotheses (exemplars):")
	  (dolist (hypothesis *hypotheses*)
	    (format t "~%~5T~A~26T~4,2F"
		    (getname (first hypothesis)) (second hypothesis)))))
    
    (return-from build-hypotheses *hypotheses*)))



;;;----------------------------------------------------------------------------
;;;  Function:  (build-hypotheses2  remindings approved-categories)
;;;
;;;  Purpose:   This function is called by build-hypotheses only when there
;;;             are no combined remindings to exemplars or exemplar-containing
;;;             categories.  This can happen if all the remindings are to
;;;             non-exemplar-containing categories.  In this situation we will
;;;             take as hypotheses the prototype (i.e., the strongest exem-
;;;             plar) of each of the subordinate exemplar-containing categories,
;;;             skipping any category that is one of 'approved-categories'.
;;;
;;;  Design:    The job to be done here is very similar to the function
;;;             "strengthen-categories" and "search-homogeneous".  Namely,
;;;             we want the most specific categories to inherit reminding
;;;             strength from any reminded superordinates, and then we will
;;;             take prototypes ONLY from the subordinate exemplar-containing
;;;             categories of these most specific categories.  For example,
;;;             if we had remindings to animals, vertebrates, and canines,
;;;             then canines would inherit strength from vertebrates and
;;;             animals (since they are both generalizations of canine) and
;;;             then we would go get the prototype of dogs and wolves (the
;;;             exemplar-containing categories subordinate to canines).
;;;----------------------------------------------------------------------------

(defun build-hypotheses2 (remindings approved-categories)
  (let ((myrems (copy-alist remindings)))
    (declare (special myrems))    ; make visible to strengthen-categories2
                                  ; and search-homogeneous2.

    (if *trace-hypotheses*
	(format t "~%   No remindings to exemplar-containing categories;~
                   ~%   now moving down to subordinate categories."))

    ;; Strengthen the most specific categories and delete all the remindings
    ;; to the more general categories.
    (strengthen-categories2)

    ;; For each surviving reminding (which is to a non-exemplar-containing
    ;; category) descend the hierarchy until the exemplar-containing
    ;; categories are found, then add the prototype of each such category.
    (dolist (rem myrems)
      (add-prototype (car rem) (cdr rem) approved-categories))))



;;;----------------------------------------------------------------------------
;;;  Function:  (add-prototype  category strength approved-categories)
;;;
;;;  Purpose:   Given a starting category and a reminding strength, this
;;;             function recursively descends the category network, adding to
;;;             the hypothesis list the prototype of each exemplar-containing
;;;             category subordinate to the given category.
;;;             'approved-categories' is a list of categories that should be
;;;             skipped since a successful match (of a lower-level concept)
;;;             has already been made to those categories.
;;;
;;;  Called by:  build-hypotheses2
;;;----------------------------------------------------------------------------

(defun add-prototype (category strength approved-categories)

  (let (prototype)
    (if (not (member category approved-categories))
	;; If this category contains an exemplar ...
	(if (setq prototype (first (category-exemplars category)))
	    ;; then add the strongest one to the hypotheses list ...
	    (progn
	      (if *trace-hypotheses*
		  (format t "~%   Adding prototype of ~A (~A)."
			  (getname category)
			  (getname prototype)))
	      (push (list prototype strength nil) *hypotheses*))
	    ;; else descend to any subordinate categories it may have.
	    (dolist (rel (node-relations category))
	      (if (eq *verb-hasTypicalSpec* (relation-verb rel))
		  (add-prototype (car (relation-to-nodes rel)) strength approved-categories)))))))


	  
;;;---------------------------------------------------------------------------
;;;  Function:	(strengthen-categories2)
;;;
;;;  Given:	myrems, the combined remindings (special variable);
;;;
;;;  Purpose:	This function strengthens remindings to 
;;;		categories by inheriting the additional strength of any
;;;		remindings to superordinate categories related through
;;;		homogeneous paths of generalization, causal, or functional
;;;		relations.
;;;
;;;  Called by:  build-hypotheses2
;;;---------------------------------------------------------------------------

(defun strengthen-categories2 ()
  (declare (special myrems))
  
  (if *trace-remindings*
      (format t "~%Strengthen categories ..."))
  
  ;; For each of the merged remindings ...
  (dolist (rem myrems)
    (let ((category (car rem)))
      
      ;; Then for each relational link emanating from this category ...
      (dolist (rel (node-relations category))
	(let ((verb (relation-verb rel)))
	  
	  ;; If it's of type generalization, causal, or functional ...
	  (if (member verb (list *verb-hasTypicalGen*
				 *verb-causedBy*
				 *verb-hasFunction*))
	      
	      ;; Then for each target node of the relation ...
	      (dolist (to-node (relation-to-nodes rel))
		
		;; Go process that node (and continue recursively).
		(search-homogeneous2 to-node verb rem nil))))))))



;;;---------------------------------------------------------------------------
;;;  Function:	(search-homogeneous2  node  rel-type  reminding  closed)
;;;
;;;  Given:	-- node, a category node,
;;;		-- rel-type, a relation type of generalization, functional,
;;;		   or causal,
;;;		-- reminding, the merged reminding to an exemplar-containing
;;;		   category that we are trying to strengthen,
;;;		-- closed, a list of nodes we have already visited.
;;;
;;;  Purpose:	This function first checks to see if there is a reminding to
;;;		the given node.  If so, it adds the strength of that reminding
;;;		to the given reminding.  It then continues recursively by
;;;		pursuing relational links of the given type from the given
;;;		node.  This function exists to assist the function
;;;		'strengthen-categories'; to understand why this function does
;;;		what it does, go read the prologue for 'strengthen-categories'.
;;;
;;;  Caller:	strengthen-categories2
;;;---------------------------------------------------------------------------

(defun search-homogeneous2 (node  verb  reminding  closed)
  (declare (special myrems))    ;; myrems from caller strengthen-categories2
  
  ;; See if there is a reminding to this node.
  (let ((rem (assoc node myrems)))
    
    ;; If so, then add its strength to that of the original reminding.
    (if rem
	(progn
	  (incf (cdr reminding) (cdr rem))
	  (if *trace-remindings*
	      (format t "~%   ~A inherits strength ~4,2F from ~A"
		      (getname (car reminding))  (cdr rem)  (getname node)))
	  (setq myrems (delete rem myrems))))
    
    ;; Recursively pursue homogeneous relations to other nodes.
    (dolist (rel (node-relations node))
      
      ;; If this is the same type of relation ...
      (if (eql verb (relation-verb rel))
	  
	  ;; Then for each target node of this relation ...
	  (dolist (to-node (relation-to-nodes rel))
	    
	    ;; If this node hasn't already been visited ...
	    (if (not (member to-node closed))
		
		;; Then continue recursive search.
		(search-homogeneous to-node verb reminding (cons node closed))))))))



;;;----------------------------------------------------------------------------
;;;  Function:  (max-exemplars  strength  highest-reminding-strength)
;;;
;;;  Given:    -- the strength of the combined reminding to a category,
;;;            -- the highest strength of all the combined remindings, and
;;;            -- *n-best-exemplars*, the maximum number of exemplars of a
;;;               given category that may be added to the hypotheses list.
;;;
;;;  Returns:  The maximum number of exemplars of the current category that
;;;            may be added to the hypothesis list.  This number will be 
;;;            at least 1 and at most *n-best-exemplars*.
;;;
;;;  Caller:   build-hypotheses       
;;;-----------------------------------------------------------------------------

(defun max-exemplars (strength divisor)
  (declare (special *n-best-exemplars*))
  (max 1 (round (* (/ strength divisor) *n-best-exemplars*))))

;;;----------------------------------------------------------------------------
;;;  Function:  (mark-hypotheses  hypotheses)
;;;
;;;  Given:     hypotheses, a list of hypotheses sorted in INCREASING order
;;;		of strength, with each entry of the form
;;;		(exemplar strength lastp);
;;;
;;;  Returns:   the same list of hypotheses modified in two ways:
;;;		1.  The entries are sorted in DECREASING order of strength
;;;		    since that is the order that the hypotheses are to be
;;;		    considered during classification.
;;;		2.  The entry containing the last exemplar of each category
;;;		    in the list is marked by setting lastp to non-nil.  This
;;;		    is used later by 'test-hypotheses2' to reassess remindings
;;;		    to a particular category when it has rejected all the 
;;;		    prototypes of that category.
;;;
;;;  Called by: build-hypotheses
;;;----------------------------------------------------------------------------


(defun mark-hypotheses (hypotheses)
  (let ((categories nil)		; List of categories found so far.
	(marked-hypotheses nil))	; New list, to be built.
    
    ;; For each hypothesis on the (reverse-ordered) list of hypotheses ...
    (dolist (hypothesis hypotheses)
      (let ((category (exemplar-category (car hypothesis))))
	;; If this exemplar introduces a new category not seen before ...
	(if (not (member category categories))
	    ;; then mark this hypothesis as being the last for this category.
	    (progn
	      (push category categories)
	      (setf (caddr hypothesis) t)))
	;; Now, push this exemplar onto the marked hypotheses list.
	(push hypothesis marked-hypotheses)))

    ;; Return the list of marked hypotheses.
    marked-hypotheses))

;;;----------------------------------------------------------------------------
;;;  Function:  (test-hypotheses  newcase  hypotheses  approved-matches)
;;;
;;;  Given:	-- newcase, a new case to be classified;
;;;		-- hypotheses, a list of the form (exemplar strength lastp)
;;;		   containing exemplars to be compared to newcase;
;;;             -- approved-matches, a list of matches that have already
;;;                been approved (either by the program or the teacher).
;;;
;;;  Returns:	2 values:  action and matches:
;;;             -- action, a value of either 'done or 'retry;
;;;             -- matches, a list of matches, possibly nil.
;;;
;;;             a 'match' structure containing details of comparing an
;;;		exemplar to newcase.  Note that the matched exemplar might not
;;;		be from the hypotheses list -- it might be from examining a
;;;		difference link or it might be that newcase has become a new
;;;		exemplar of the category.  'Test-hypotheses' returns nil if
;;;             there is no match structure to return, as in the case where a
;;;             new exemplar gets created or when the case simply cannot be
;;;             classified.
;;;
;;;  Called by: classify
;;;
;;;  Design:	Starting with the first exemplar on the hypotheses list,
;;;		this function compares the exemplar to newcase.  If it's a
;;;		poor match, it is rejected and the next hypothesized exemplar
;;;		is compared.  If the match is strong enough, it tries to
;;;		improve the match through difference links.  The selection of
;;;		the best match depends on what mode Protos is in, as described
;;;             below.
;;;
;;;		If Protos is in "maximum learning mode" it will present each
;;;		match to the teacher for approval, and as soon as the teacher
;;;		approves one, that's what is returned.  If Protos is in
;;;		"minimum learning mode" it will first compute matches for up
;;;		to *n-best-matches*, then sort them by similarity, then
;;;		present them to the teacher in that order for approval.
;;;		If Protos is in performance mode (i.e., NOT in learning mode)
;;;		it simply returns the sorted list of matches (without ever
;;;             asking the teacher any questions).
;;;
;;;  FUTURE:	Currently, the "best" match is determined by the raw similarity
;;;		of the match.  However, maybe it should take into account the
;;;		relative prototypicality of the exemplar.  For example, given
;;;		a strong match to a very untypical exemplar and a moderate
;;;		match to the best prototype of a very common category, we
;;;		might want to prefer the latter match.  This says that matches
;;;		should be ranked by some combination of similarity and
;;;		prototypicality.  Consider the following proposed formula:
;;;
;;;		   X = "similarity versus typicality" tuning factor;
;;;		   Y = "absolute versus relative typicality" tuning factor;
;;;
;;;		   Typicality = exemplar_prototypicality /
;;;				[(Y * prototype_prototypicality) + (1.0 - Y)]
;;;		   Rank = [X * similarity * {(1.0 - Y) * typicality + Y}] +
;;;			   [(1.0 - X) * typicality]
;;;
;;;		X and Y are user-settable tuning factors in the range 0 to 1.0.
;;;		If X is close to 1.0 then similarity dominates the ranking.
;;;		If X is close to 0.0 then prototypicality dominates the ranking.
;;;		If Y = 1.0 then we use relative prototypicality; if Y = 0.0
;;;		then we use absolute prototypicality.  Naturally, the user
;;;		may set X and Y to any value between 0 and 1.
;;;----------------------------------------------------------------------------


(defparameter *no-hypotheses-menu* (make-menu
    :label nil
    :items '((#\C . ("Create an exemplar from this case."   return  create-exemplar))
	     (#\U . ("give Unfocused instruction, then retry the case." return give-unfocused))
	     (#\A . ("Abandon this case."  return  abandon)))))

(defparameter *no-hypotheses-menu2* (make-menu
    :label nil
    :items '((#\U . ("give Unfocused instruction, then retry the case." return give-unfocused))
	     (#\A . ("All done with this case."  return  abandon)))))

(defparameter *failure-menu2* (make-menu
  :label  "~%~%What do you want to do with this case now?~
             ~%------------------------------------------"
  :items  '((#\C . ("Create an exemplar from this case"   return  create))
	    (#\R . ("Retry finding a match for this case" return  retry))
	    (#\A . ("Abandon this case entirely"          return  abandon)))))

(defparameter *failure-menu3* (make-menu
  :label  "~%~%What do you want to do with this case now?~
             ~%------------------------------------------"
  :items  '((#\R . ("Retry finding a match for this case" return  retry))
	    (#\A . ("All done with this case."            return  abandon)))))



(defun test-hypotheses (newcase hypotheses approved-matches)
  (declare (special *unfocused-instruction-menu*))

  ;; Reset any remembered near misses.
  (setq *near-misses* nil)

  ;; If there are any hypotheses ...
  (if hypotheses
      ;; Then try to classify this case,
      (test-hypotheses2 newcase hypotheses approved-matches)
      ;; Else if in performance mode ...
      (if (not *learning-mode*)
	  ;; then just return saying we're all done,
	  (values 'done nil)
	  ;; else ask teacher what to do.
	  (let (menu1 menu2)
	    (if (null approved-matches)
		(progn
		  (format t "~%Protos was unable to form any hypotheses about this case.~
                             ~%What do you want to do?")
		  (setq menu1 *no-hypotheses-menu*
			menu2 *failure-menu2*))
		(progn
		  (format t "~%Protos was unable to form any additional hypotheses~
                             ~%based on the last match.  What do you want to do?")
		  (setq menu1 *no-hypotheses-menu2*
			menu2 *failure-menu3*)))
	    
	    (case (menu-select menu1)
	      ;; Create an exemplar from this case.
	      (create-exemplar   (values 'done (add-new-exemplar newcase) nil))
	      
	      ;; Give unfocused instruction before retrying to classify this case.
	      (give-unfocused    (menu-select *unfocused-instruction-menu*)
				 (case (menu-select menu2)
				   (create   (values 'done  (add-new-exemplar newcase)))
				   (retry    (values 'retry nil))
				   (abandon  (values 'done  nil))))
	      
	      ;; Abandon this case.
	      (abandon           (values 'done nil)))))))



;;;----------------------------------------------------------------------------
;;;  Function:  (test-hypotheses2  newcase  hypotheses  approved-matches)
;;;
;;;  Purpose:   This is a subordinate function of 'test-hypotheses'.
;;;----------------------------------------------------------------------------

(defun test-hypotheses2 (newcase hypotheses approved-matches)
  (let ((matches            nil)	 ; list of matches computed so far
	(closed-categories  nil)	 ; list of known wrong categories
	(closed-exemplars   nil))	 ; list of known wrong exemplars

    ;; The following two "closed" lists are declared special so that they
    ;; will be visible to the called functions where it becomes known that
    ;; the teacher has rejected a given exemplar or category.
    (declare (special closed-categories closed-exemplars))

    (dolist (hypothesis hypotheses)
      ;; Note: a hypothesis is a list of the form (exemplar strength lastp).

      ;; Early exit from dolist if limit reached on number of matches.
      (if (>= (length matches) *n-best-matches*)
	  (return (values)))
      
      (prog* ((exemplar   (car hypothesis))
	      (category   (exemplar-category exemplar))
	      match)

	     ;; Skip this hypothesis if the teacher has already rejected
	     ;; this category or this specific exemplar.  Also, skip this
	     ;; hypothesis if it is of the same category as an already-
	     ;; approved match (this can happen with featural exemplars).
	     (if (or (member exemplar closed-exemplars)
		     (member category closed-categories)
		     (member category approved-matches
			     :key #'(lambda (x) (exemplar-category (match-exemplar x)))))
		 (go NEXT-HYPOTHESIS))
	     
	  RETRY
	     ;; Determine degree-of-match from exemplar to new case.
	     (setq match (compare-new-case exemplar newcase))
	     
	     ;; If match is poor or has been prevented by a mutual-exclusion relation,
	     ;; then if this is the last exemplar on the hypotheses list with that
	     ;; category, then reassess remindings and go on to the next hypothesis.
	     
	     (if (< (match-nth-root-of-similarity match) *minimum-match*)
		 (progn
		   (if *trace-case-matching*
		       (progn
			 (format t "~%The match with exemplar ~A is below the minimum acceptable threshold.~
                                    ~%Here is that match:"
				 (getname exemplar))
			 (print-match match)))
		   (if (caddr hypothesis)
		       (reassess-remindings match))
		   (go NEXT-HYPOTHESIS)))
	     
	     ;; If there is a reasonable match then examine the difference links.
	     (if (>= (match-similarity match) *difference-threshold*)
		 (setq match (explore-differences match closed-exemplars)))
	     
	     ;; If Protos is in "maximum learning mode" then it must present every
	     ;; reasonable match to the teacher (as a learning opportunity).
	     
	     (if (and *learning-mode* *maximize-mode*)
		 
		 ;; then present match to teacher.  If accepted, return.
		 (let (action d-match category)
		   (multiple-value-setq (action d-match) (discuss-match match))
		   
		   (case action
		     (done  (return-from test-hypotheses2 (values 'done (if d-match
									    (list d-match)
									    nil))))
		     (newex (return-from test-hypotheses2 (values 'newex d-match)))
		     (retry (return-from test-hypotheses2 (values 'retry nil)))
		     (redo  (setq exemplar (match-exemplar d-match))
			    (setq newcase  (match-newcase d-match))
			    (go RETRY))
		     (next  (setq category (exemplar-category (match-exemplar d-match)))
			    (pushnew category closed-categories)
			    (go NEXT-HYPOTHESIS))
		     (otherwise (format t "~%Test-hypotheses: unknown action ~A."
					action))))
		 
		 ;; else save the results of each match for later.
		 (push match matches))
	     
	  NEXT-HYPOTHESIS   ; this is a tag at the end of the hypothesis loop.
	     ))

      ;; Sort matches in decreasing order of similarity.
      (setq matches (sort matches  #'>=  :key #'match-nth-root-of-similarity))
      
      ;; If performance mode then return the sorted list.
      (if (not *learning-mode*)
	  (return-from test-hypotheses2 (values 'done matches)))
      
      ;; This must be minimum learning mode, so present each match to teacher,
      ;;   starting with the best (highest similarity).
      (if (not *maximize-mode*)
	  (dolist (match matches)
	    (multiple-value-bind (action d-match)
		(discuss-match match)
	      (case action
		(done   (return-from test-hypotheses2 (values 'done (list d-match))))
		(newex  (return-from test-hypotheses2 (values 'newex d-match)))
		(next   nil)
		((redo retry) (format t "~%Redo/retry not legal in minimum learning mode."))))))
      
      ;; The only way to get here is to be in learning-mode and the teacher
      ;;    has rejected all proposed matches.
      (if (prompt "~%You have rejected all proposed exemplars for matching the new case.~
                   ~%Do you wish to make an exemplar of the new case? "
		  nil 'y-or-n nil nil)
	  (values 'newex (add-new-exemplar newcase))
	  (progn
	    (format t "~&Then this case is abandoned, unclassified.~%")
	    (values 'done nil)))))


;;;----------------------------------------------------------------------------
;;;  Function:  (compare-new-case  exemplar  newcase)
;;;
;;;  Given:	exemplar, an exemplar to match to the new case; and
;;;		newcase, a case to be matched against;
;;;
;;;  Returns:	a 'match' structure containing the results of the match.
;;;
;;;  Called by: test-hypotheses2
;;;
;;;  Design:	Each feature of the exemplar is considered in order of
;;;		importance (with spurious features ignored).  An attempt is
;;;		made to find a direct or explained match from the exemplar
;;;		feature to any feature of newcase.  The result is stored as an
;;;		entry in the 'results' slot of the match structure.  The overall
;;;		similarity of the match (a value in the range 0 to 1.0) is
;;;		computed by multiplying together the similarities of the
;;;		individual featural matches (however, if *switch-nth-root*
;;;		is t, then compute nth root of similarity, where n is the
;;;		number of exemplar features).
;;;
;;;  Note:	It is helpful to remember that the matching goes FROM the
;;;		exemplar TO the new case, not the other way around.  Thus,
;;;		unmatched features of the exemplar weaken the overall match
;;;		whereas unmatched features of newcase do not.
;;;
;;;  FUTURE:	-- The amount of effort to spend on a match should be
;;;		   influenced by the relative prototypicality of the exemplar.
;;;		   We should spend more effort trying to match the prototype
;;;		   of a category than an outlying exemplar.
;;;		-- Bruce Porter mentioned that there should be some way to 
;;;		   "normalize" degree-of-match across categories so that a
;;;		   match of, say, 0.27 from one category means about the same
;;;		   thing as a match of 0.27 from any other category.
;;;----------------------------------------------------------------------------

(defun compare-new-case (exemplar newcase)
  
  (let ((importances   (get-sorted-importances exemplar))
	 (features     (exemplar-features exemplar))
	 (similarity   1.0)
	 (exclusion-strength  0)
	 (unmatched    (copy-list (case-features newcase)))
	 (match        (make-match :exemplar exemplar :newcase newcase))
	 (n            0)
	 nth-root-of-similarity)

	;; Delete entries that are not features of this exemplar.
	(setq importances
	      (delete-if-not #'(lambda (x) (member x features)) importances :key #'car))

	;; Let kbpm know what the fault variables are, if any.
	(setq *fault-variables* (category-faultvars (exemplar-category exemplar)))
	
	;; For each feature in the exemplar, try to match it to the new case.
	(dolist (importance importances)
	  (let* ((feature (car importance))
		 (weight  (cdr importance))
		 (result  (compare-feature feature weight exemplar newcase)))

	    ;; Add this result to the list of match results.
	    (push result (match-results match))

	    ;; Depending on the type of result, adjust the similarity and
	    ;; the list of unmatched case features.
	    (ecase (result-type result)
	      (identical
		(setq unmatched (delete feature unmatched)))
	      (explained
		(setq unmatched (nset-difference
				  unmatched (explanation-from-terms (result-explanation result))))
		(setq similarity (* similarity (result-quality result))))
	      (unmatched
		(setq similarity (* similarity (result-quality result))))
	      (spurious
		;; Spurious explanations are ignored here since we don't
		;; want to modify either the similarity of the match or
		;; the list of unmatched case features.
		nil)
	      (excluded
		(setq unmatched (nset-difference
				  unmatched (explanation-from-terms (result-explanation result))))
                ;; The following line excluded as per Ray's dissertation
                ;;(setq similarity (* similarity (result-quality result)))
		(incf exclusion-strength (result-quality result))))

	    ;; Abandon the match if we've dropped below the minimum acceptable
	    ;;   strength for an explanation (based on the average strength of
	    ;;   explanations so far.
	    (if (< (expt similarity (/ 1 (incf n))) *minimum-match*)
		(progn
                  (if *trace-case-matching*
		    (format t "~%This match has dropped below minimum threshold."))
                  (setf (match-nth-root-of-similarity match) (expt similarity (/ 1 n)))
                  (setf (match-similarity match) similarity)
	          (setf (match-results match) (nreverse (match-results match)))
	          (setf (match-unmatched match)  unmatched)
                  (return-from compare-new-case match)))
	    )
	  )

	(setq nth-root-of-similarity   (expt similarity (/ 1 (length importances))))
	(setf (match-nth-root-of-similarity match) nth-root-of-similarity)
	(setf (match-similarity match) similarity)
	(setf (match-results match)    (nreverse (match-results match)))
	(setf (match-unmatched match)  unmatched)

	;; If mutual exclusion is sufficiently strong, disallow the match.
	(if (or (>= exclusion-strength (quantifier-strength *quant-strongly*))
		(and (>= exclusion-strength (quantifier-strength *quant-usually*))
		     (< nth-root-of-similarity *perfect-match*))
		(and (>= exclusion-strength (quantifier-strength *quant-sometimes*))
		     (< nth-root-of-similarity *strong-match*))
		(and (>= exclusion-strength (quantifier-strength *quant-occasionally*))
		     (<= nth-root-of-similarity *weak-match*)))
	    (disallow match))
	
	(return-from compare-new-case match)))



;;;-----------------------------------------------------------------------------
;;;  Function:	(disallow  match)
;;;
;;;  Purpose:	Given a match between an exemplar and the new case, this
;;;             function simply sets the similarity values to zero inside the
;;;             'match' structure, effectively excluding this match from any
;;;             further consideration.
;;;-----------------------------------------------------------------------------

(defun disallow (match)
  (setf (match-similarity match) 0
	(match-nth-root-of-similarity match) 0)
  (if *trace-case-matching*
      (format t "~%Match to exemplar ~A disallowed by a mutual exclusion."
	      (getname (match-exemplar match)))))



;;;-----------------------------------------------------------------------------
;;;  Function:	(get-sorted-importances  exemplar)
;;;
;;;  Given:	an exemplar
;;;
;;;  Returns:	an alist of (feature . importance) sorted in decreasing order
;;;		of importance, for every feature of the exemplar which has an
;;;             importance.
;;;
;;;  Caller:	compare-new-case
;;;
;;;  Purpose:	When matching an exemplar to a new case, matching begins with
;;;		the most important feature and proceeds to the least important.
;;;		Features generally have importance only to categories, but
;;;		occasionally a feature has idiosyncratic importance to a
;;;		particular exemplar.  In such a case, the strength of the
;;;		idiosyncratic importance should override any importance that
;;;		the same feature has to the category.  The purpose of this
;;;		function is to combine the two types of importances.
;;;
;;;  Algorithm:
;;;		If no idiosyncratic importances,
;;;		   then return the already-sorted importances of the category
;;;		   else For each idiosyncratic importance
;;;			    do
;;;			      If same feature appears in category's importances
;;;				 then override it with idiosyncratic strength
;;;				 else add it to alist of importances.
;;;			    done
;;;			Sort alist in decreasing order of importance.
;;;-----------------------------------------------------------------------------

(defun get-sorted-importances (exemplar)
  (let* ((features       (exemplar-features exemplar))
	 (category       (exemplar-category exemplar))
       ;;(e-importances  (exemplar-importances exemplar))
	 (c-importances  (category-importances category))
	 (importances    nil)
	 imp)

    (dolist (feature features)
      (cond
      ;;((setq imp (assoc feature e-importances)) (push imp importances))
	((setq imp (assoc feature c-importances)) (push imp importances))
	((setq imp (get-importance feature category))
	                                          (push imp importances))
	(t   (push (cons feature 0.0) importances)
	     (if *trace-importances*
		 (format t "~%get-sorted-importances: no importance found for ~A~
                            ~%Protos will assume importance value of 0.0"
		        (getname feature))))))
	  

    ;; Sort the combined importances in decreasing order of strength.
    (sort importances  #'> :key  #'cdr)))


;;;----------------------------------------------------------------------------
;;;  Function:  (get-importance  feature  node)
;;;
;;;  Given:     a feature and a node (a category or exemplar)
;;;
;;;  Returns:   two values:
;;;             -- the importance cons, i.e. (feature . strength); and
;;;             -- the node where the importance was found.
;;;             If no importance can be found, it returns nil, nil.
;;;
;;;  Design:    If the given node is an exemplar, then this function looks 
;;;             for the importance first in the exemplar and, if not there,
;;;             looks in the exemplar's category.  If the given node is a
;;;             category then it looks in that category.  If an importance
;;;             cannot be found by in the exemplar or category, then this
;;;             functions searches "up" the generalization hierarchy in order
;;;             to inherit an importance value from a more general category.
;;;             Thus, with this simple inheritance mechanism for importances,
;;;             the most specific importance is found and used, overriding any
;;;             more-general importances.
;;;
;;;  FUTURE:    This function implements a very simple inheritance mechanism
;;;             which could be made more sophisticated if desired.
;;;             For example, should it also search up the part-to-whole,
;;;             causal, and functional hierarchies?  Should it weaken or
;;;             strengthen the retrieved importance based on the relations it
;;;             has passed through?  Should it do a breadth-first search up
;;;             through the different types of hierarchies or should it do
;;;             depth-first search on the generalization hierarchy, then
;;;             part-to-whole, then ...?
;;;----------------------------------------------------------------------------

(defun get-importance (feature node)
  (let (imp category)

    (if (my-exemplar-p node)
	;; Look for importance in the exemplar's category
	(progn
	  (setq category (exemplar-category node))
	  (if (setq imp (assoc feature (category-importances category)))
	      (return-from get-importance (values imp category))))

	;; Look for importance in this category.
	(progn
	  (setq category node)
	  (if (setq imp (assoc feature (category-importances category)))
	      (return-from get-importance (values imp category)))))

    ;; Importance of this feature was not found in the given category
    ;; (or the given exemplar's category), so we now look for the
    ;; importance in a more general category.

    ;; For each relation emanating from that category ...
    (dolist (rel (node-relations category))
      ;; If it points to a more general category of a hierarchy ...
      (if (eq (relation-verb rel) *verb-hasTypicalGen*)
	  (multiple-value-bind (imp cat)
	      (get-importance feature (first (relation-to-nodes rel)))
	    (if imp
		(return-from get-importance (values imp cat))))))

    ;; If no importance can be found, return nil, nil.
    (values nil nil)))

;;;----------------------------------------------------------------------------
;;;  Function:  (compare-feature  feature  importance  exemplar  newcase)
;;;
;;;  Given:	feature,     a feature of the exemplar
;;;		importance,  the importance of the feature
;;;		exemplar,    for exemplar-to-newcase matching
;;;		newcase,     for exemplar-to-newcase matching
;;;
;;;  Returns:   An instance of the 'result' structure, which contains the
;;;             following slots:
;;;
;;;		  (feature  importance  type  similarity  [explanation])
;;;
;;;		where "type" can be any of the following:
;;;		  'identical	the two features are identical.
;;;		  'explained	the two features are related by explanation.
;;;		  'spurious	this feature is spurious (in this case).
;;;		  'excluded	this feature has a mutual exclusion to newcase.
;;;		  'unmatched	this feature doesn't match anything in newcase.
;;;
;;;		and "similarity" is a floating-point value in the range 0 to 1.0
;;;		representing the amount by which the overall match should be
;;;		multiplicatively weakened.  For example, 'identical features
;;;		and 'spurious features have similarity 1.0, 'unmatched features
;;;		have similarity [1.0 - importance], 'excluded features have
;;;		similarity 0.0, and 'explained features have a similarity
;;;		equal to the strength of the explanation.
;;;
;;;  Callers:	compare-new-case
;;;
;;;  Design:	An explanation of the equivalence of a feature of the exemplar
;;;		to a feature of the new case is found by doing a uniform-cost
;;;		search of the category network.  The amount of search expended
;;;		is limited by the importance of the feature, i.e., the "cost"
;;;		spent on a search can be up to, but not exceeding, the import-
;;;		ance of the feature.
;;;
;;;  Note:	In the worst case, uniform-cost graph search is exponentially
;;;		explosive in the number of nodes to be searched as the depth
;;;		of the search increases.  A possible future performance
;;;		improvement here is to build an explanation-cache for fast
;;;		retrieval of previously found explanations.  This could make
;;;		a substantial performance improvement, but care must be
;;;             exercised in deciding when to update the cache.
;;;----------------------------------------------------------------------------

(defun compare-feature (feature importance exemplar newcase)

  ;; Knowledge-Based Pattern Matcher does all the work now.
  (kbpm 'FtoF feature importance exemplar (case-features newcase)))

;;;----------------------------------------------------------------------------
;;;  Function:	(explore-differences  match  closed)
;;;
;;;  Given:	-- match, the current exemplar-to-newcase match, and
;;;		-- closed, a list of exemplars that have already been explored.
;;;
;;;  Returns:	the best match found, possibly the original match or a better
;;;		match found by traversing difference links.
;;;
;;;  Called by:	test-hypotheses2
;;;
;;;  Notes:	This function recursively explores difference links in a
;;;		hill-climbing search for an improved match.  If it cannot
;;;		improve the match, it returns the original match.
;;;
;;;  Algorithm:
;;;		For all difference links from exemplar do:
;;;		  If difference link has features matching newcase
;;;		     then calculate total importance of matched features,
;;;			  add target exemplar to list of candidates.
;;;
;;;		While list of candidates is not empty
;;;		  Match new case to candidate.
;;;		  If similarity > current similarity
;;;		     then recursively improve match to candidate.
;;;
;;;		Return current best match.
;;;----------------------------------------------------------------------------

(defun explore-differences (match closed)
  (let* ((exemplar	(match-exemplar match))
	 (newcase       (match-newcase match))
	 (unmatched	(match-unmatched match))
	 (differences	(exemplar-differences exemplar))
	 (candidates	nil)
	 imp)

    ;; If no difference links or unmatched newcase features, just return.
    (if (or (null differences) (null unmatched))
	(return-from explore-differences match))

    ;; Consider each difference link from the exemplar.
    (dolist (diff differences)
      ;; Skip this link if it points to a node we've already visited.
      (if (not (member (difference-node diff) closed))
	  ;; Compute intersection of unmatched features and difference features.
	  (let ((features (intersection unmatched (difference-features diff))))
	    (setf (difference-sum diff) 0.0)
	    ;; Proceed only if the intersection is non-nil.
	    (if features
		(progn
		  (push diff candidates)
		  (dolist (feature features)
		    (setq imp (get-importance feature (difference-node diff)))
		    (if imp
			(incf (difference-sum diff) (cdr imp)))))))))

    ;; Sort the difference link candidates by decreasing importance sums.
    (setq candidates (sort candidates  #'>=  :key #'difference-sum))

    ;; Do hill-climbing search through the difference links.
    (dolist (diff candidates)
      (let ((newmatch (compare-new-case (difference-node diff) newcase)))
	(if (> (match-similarity newmatch) (match-similarity match))
	    (return-from explore-differences
		(explore-differences newmatch (cons exemplar closed))))))

    ;; No improvement was found, so return the original match.
    (return-from explore-differences match)))





