;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;                 "R E M I N D I N G"   F U N C T I O N S
;;;   -------------------------------------------------------------------------
;;;
;;;  Overview:    This file contains all functions associated with processing
;;;               remindings and censors in CL-Protos.
;;;
;;;  Definitions: -- A "raw" reminding is conceptually a triple:
;;;                  [feature, category or exemplar, strength].
;;;               -- A censor is a reminding having a negative strength.
;;;
;;;  Functions:   combine-remindings
;;;               get-new-remindings
;;;               get-raw-remindings
;;;               merge-remindings
;;;		  reassess-remindings
;;;               search-homogeneous
;;;               strengthen-categories
;;;               strengthen-exemplars
;;;               strengthen-subordinates
;;;=============================================================================





;;;----------------------------------------------------------------------------
;;;  Function:	(combine-remindings  raw-reminding)
;;;
;;;  Purpose:	Given a a set of raw remindings, this function returns a list
;;;		of remindings to categories and/or exemplars suggested by the
;;;		presence of features of the new case.  These are not simply raw
;;;		remindings; they have been combined in ways detailed below.
;;;		The combined remindings generated by this function are used as
;;;		input to the function 'build-hypotheses'.
;;;
;;;  Caller:	classify
;;;
;;;  Design:	1.  For each reminded node, sum all of its remindings and
;;;		    censors.
;;;		2.  Remove any target nodes that have an absolute censor.
;;;		3.  Remove any target node whose combined reminding < 0.
;;;		4.  Strengthen remindings to exemplar-containing categories
;;;		    by inheriting the reminding strength of superordinate
;;;		    categories related through homogeneous generalization,
;;;		    causal, or functional relation paths.
;;;             5.  Generate remindings for any category that is a common
;;;                 specialization of 2 or more reminded categories.
;;;		6.  Strengthen remindings to exemplars by inheriting the
;;;		    strength of the combined reminding to the exemplar's
;;;		    category.  (Now subsumed by new step 7).
;;;             7.  For each reminded term, strengthen it by inheriting any
;;;                 reminding to terms that are superior in the sense of
;;;                 "is exemplar of" or "is feature of" relations.
;;;                 This was added for featural exemplars.
;;;             8.  Sort the remindings in decreasing order of strength.
;;;
;;;  Notes:	-- In the category network, remindings and censors are stored
;;;		   in the 'remindings' slot of each feature node.  This slot
;;;		   contains an alist with entries of the form (node . strength).
;;;             -- Steps 4,5,6,7 have the effect of promoting matching to
;;;                the most likely and most specific term.
;;;-----------------------------------------------------------------------------

(defun combine-remindings (raw-remindings)
  (let (rems)

    ;; Step 1 -- Merge remindings to same node by adding their strengths together.
    (setq rems (merge-remindings raw-remindings))

    ;; Steps 2 & 3 -- Remove any remindings having strength < 0.  This removes all
    ;; absolutely censored nodes since those have a large negative value.
    (setq *remindings* (remove-if #'(lambda (rem) (minusp (cdr rem))) rems))
    (setq *initial-remindings* (copy-alist *remindings*))

    ;; Step 4 -- Strenghten remindings to exemplar-containing categories.
    (strengthen-categories)

    ;; Step 5 -- Generate remindings for common-specializations.
    (common-specializations)

    ;; Step 6 -- Strengthen remindings to reminded exemplars.
    (strengthen-exemplars)

    ;; Step 7 -- Strengthen remindings to features of reminded terms.
    (strengthen-features)

    ;; Step 8 -- Sort the remindings in decreasing order of strength.
    (setq *remindings* (sort *remindings* #'>= :key #'cdr))

    ;; Optionally print the final combined remindings, sorted by strength.
    (if *trace-remindings*
	(progn
	  (format t "~%~%Final sorted remindings:~%   ")
	  (print-alist *remindings* t 1))))
  *remindings*)



;;;-----------------------------------------------------------------------------
;;;  Function:	(get-raw-remindings  features)
;;;
;;;  Given:	a list of features (which may evoke remindings)
;;;
;;;  Returns:	an alist of 'reminding' pairs of the form (node . strength).
;;;		Note that a feature may have none, one, or several remindings.
;;;-----------------------------------------------------------------------------

(defun get-raw-remindings (features)
  
  (if *trace-remindings*
      (format t "~%~%Raw remindings evoked:"))
  
  (let ((raw-rems nil)
	rems)
    
    (dolist (feature features)
      (setq rems (copy-alist (feature-remindings feature)))
      ;; copy-alist is used above to create a fresh copy of the
      ;;  remindings since we will ultimately be combining (and
      ;;  thus modifying) the strength values in the alist.
      
      (if (and *trace-remindings* rems)
	  (progn
	    (format t "~%   ~(~22,1,1,'-A~)>  " (getname feature))
	    (print-alist rems t 1)))
      
      (setq raw-rems (nconc rems raw-rems)))
      
    raw-rems))



;;;-----------------------------------------------------------------------------
;;;  Function: 	 (get-new-remindings  matches)
;;;
;;;  Given:	 a list of matches;
;;;
;;;  Returns:	 an alist of 'reminding' pairs of the form (node . strength)
;;;              associated with the matched exemplars and their categories.
;;;
;;;  Called by:  classify
;;;
;;;  Note:       This function is called by classify after a successful match
;;;              so that if featural exemplars are involved, this will get
;;;              remindings to higher-level concepts.
;;;-----------------------------------------------------------------------------

(defun get-new-remindings (matches)
  
  (if *trace-remindings*
      (format t "~%~%New remindings evoked:"))
  
  (let ((new-rems nil))
    (dolist (match matches new-rems)
      (let* ((exemplar  (match-exemplar match))
	     (category  (exemplar-category exemplar)))
	
	(dolist (node (list exemplar category))
	  (let ((rems (copy-alist (feature-remindings node))))
	    ;; copy-alist is used above to create a fresh copy of the
	    ;;  remindings since we will ultimately be combining (and
	    ;;  thus modifying) the strength values in the alist.
	    
	    (if (and *trace-remindings* rems)
		(progn
		  (format t "~%   ~(~22,1,1,'-A~)>  " (getname node))
		  (print-alist rems t 1)))
	    
	    (setq new-rems (nconc rems new-rems))))))))




;;;---------------------------------------------------------------------------
;;;  Function:	(merge-remindings  remindings)
;;;
;;;  Given:	an alist of evoked remindings
;;;
;;;  Returns:	a new alist of remindings with exactly one entry for each
;;;		target of a reminding, with the combined strength (by addition)
;;;		of all the evoked remindings to that target node.  In other
;;;		words, if two or more remindings point to the same node, this
;;;		function replaces them with a single stronger reminding.
;;;---------------------------------------------------------------------------

(defun merge-remindings (remindings)

  (let* ((merged-rems '())
	 pair)

    (dolist (rem remindings)
      (setq pair (assoc (car rem) merged-rems))
      (if pair				; If this node already seen
	  (incf (cdr pair) (cdr rem))	;   Then add strengths together
	  (push rem merged-rems)))	;   Else push 'new' reminding
    
    (if *trace-remindings*
      (progn
        (format t "~%~%Merged remindings:~%   ")
	(print-alist merged-rems t 1)
	(terpri)))

    merged-rems))


;;;---------------------------------------------------------------------------
;;;  Function:	(strengthen-subordinates rems)
;;;
;;;  Given:	the merged remindings in "rems",
;;;
;;;  Returns:   enlarged set of merged remindings.
;;;
;;;  Purpose:	This function strengthens remindings to subordinate nodes.
;;;             For example, if node X has a combined reminding of strength S,
;;;             then new remindings of strength S are generated for every
;;;             subordinate node of X.  "Subordinate" refers to any node
;;;             related to X by has-typical-specialization, causes, or
;;;		is-function-of.  (The newly generated remindings are 
;;;		combined in the last step).
;;;
;;;             The effects of this "trickle down" of remindings are:
;;;             1.  Exemplar-containing categories inherit the strength of
;;;                 all of their superordinate categories.
;;;             2.  Common-specialization categories inherit the strength of
;;;                 their multiple parent categories (actually, ALL ancestor
;;;                 categories).
;;;             3.  Remindings are generated for all appropriate exemplar-
;;;                 containing categories, even if all of the raw remindings
;;;                 were to non-exemplar-containing categories.
;;;
;;;  Note: 	Although this function works as advertised, it has been
;;;		abandoned because it would be inefficient for large category
;;;		networks.  
;;;---------------------------------------------------------------------------

;(defun strengthen-subordinates (rems)
;  
;  (let ((newrems nil))
;    (declare (special newrems))	    ; make visible to strengthen-subordinates2
;    
;    (if *trace-remindings*
;	(format t "~%Strengthen subordinates ..."))
;    
;    ;; For each of the merged remindings ...
;    (dolist (rem rems)
;      ;; [ignore exemplars here]
;      (if (category-p (car rem))
;	  (progn
;	    (if *trace-remindings*
;		(format t "~%   ~A's reminding of ~4,2F added to:"
;			(node-name (car rem)) (cdr rem)))
;	    (strengthen-subordinates2 (car rem) (cdr rem) nil))))
;    
;    ;; Combine the new remindings with the "old" combined remindings.
;    (setq rems (merge-remindings (nconc rems newrems)))))
;
;
;
;(defun strengthen-subordinates2 (node strength add)
;  (declare (special newrems))
;  
;  ;; If this is a category (i.e., not an exemplar) ...
;  (if (category-p node)
;      
;      (progn
;	;; If "add" flag true, then add new reminding for this category.
;	(if add
;	    (progn
;	      (if *trace-remindings*
;		  (format t "~%      ~A" (node-name node)))
;	      (push (cons node strength) newrems)))
;	
;	;; Then for each relational link emanating from this category ...
;	(dolist (rel (node-relations node))
;	  (let ((verb (relation-verb rel)))
;	    
;	    ;; If it's of type generalization, causal, or functional ...
;	    (if (member verb (list *verb-hasTypicalSpec*
;				   *verb-causes*
;				   *verb-isFunctionOf*))
;		
;		;; Then for each target node of the relation ...
;		(dolist (to-node (relation-to-nodes rel))
;		  
;		  ;; Go process that node (and continue recursively).
;		  (strengthen-subordinates2 to-node strength t))))))))


;;;---------------------------------------------------------------------------
;;;  Function:	(strengthen-categories)
;;;
;;;  Given:	the merged remindings in the global variable *remindings*
;;;
;;;  Purpose:	This function strengthens remindings to exemplar-containing
;;;		categories by inheriting the additional strength of any
;;;		remindings to superordinate categories related through
;;;		homogeneous paths of generalization, causal, or functional
;;;		relations.
;;;---------------------------------------------------------------------------


(defun strengthen-categories ()
  
  (if *trace-remindings*
      (format t "~%Strengthen categories ..."))
  
  ;; For each of the merged remindings ...
  (dolist (rem *remindings*)
    (let ((category (car rem)))

      ;; If this is an exemplar-containing category ...
      (if (category-exemplars category)
	  
	  ;; Then for each relational link emanating from this category ...
	  (dolist (rel (node-relations category))
	    (let ((verb (relation-verb rel)))

	      ;; If it's of type generalization, causal, or functional ...
	      (if (member verb (list *verb-hasTypicalGen*
				     *verb-causedBy*
				     *verb-hasFunction*))
		  
		  ;; Then for each target node of the relation ...
		  (dolist (to-node (relation-to-nodes rel))

		    ;; Go process that node (and continue recursively).
		    (search-homogeneous to-node verb rem nil)))))))))



;;;---------------------------------------------------------------------------
;;;  Function:	(search-homogeneous  node  rel-type  reminding  closed)
;;;
;;;  Given:	-- node, a category node,
;;;		-- rel-type, a relation type of generalization, functional,
;;;		   or causal,
;;;		-- reminding, the merged reminding to an exemplar-containing
;;;		   category that we are trying to strengthen,
;;;		-- closed, a list of nodes we have already visited.
;;;
;;;  Purpose:	This function first checks to see if there is a reminding to
;;;		the given node.  If so, it adds the strength of that reminding
;;;		to the given reminding.  It then continues recursively by
;;;		pursuing relational links of the given type from the given
;;;		node.  This function exists to assist the function
;;;		'strengthen-categories'; to understand why this function does
;;;		what it does, go read the prologue for 'strengthen-categories'.
;;;
;;;  Caller:	strengthen-categories
;;;---------------------------------------------------------------------------

(defun search-homogeneous (node  verb  reminding  closed)

  ;; See if there is a reminding to this node.
  (let ((rem (assoc node *initial-remindings*)))
    
    ;; If so, then add its strength to that of the original reminding.
    (if rem
	(progn
	  (incf (cdr reminding) (cdr rem))
	  (if *trace-remindings*
	      (format t "~%   ~A inherits strength ~4,2F from ~A"
		      (getname (car reminding))  (cdr rem)  (getname node)))))
    
    ;; Recursively pursue homogeneous relations to other nodes.
    (dolist (rel (node-relations node))
      
      ;; If this is the same type of relation ...
      (if (eql verb (relation-verb rel))
	  
	  ;; Then for each target node of this relation ...
	  (dolist (to-node (relation-to-nodes rel))
	    
	    ;; If this node hasn't already been visited ...
	    (if (not (member to-node closed))
		
		;; Then continue recursive search.
		(search-homogeneous to-node verb reminding (cons node closed))))))))



;;;---------------------------------------------------------------------------
;;;  Function:	(common-specialization)
;;;
;;;  Given:	the merged remindings in the global variable *remindings*
;;;
;;;  Purpose:	Generate remindings to any category that is a common-special-
;;;		ization of two (or more) other reminded categories.
;;;		For example, consider the situation in audiology where there
;;;		are two distinct categories "cochlear_age" and 
;;;		"cochlear_noise", and they have a common specialization
;;;		"cochlear_age_and_noise".  The problem is that Protos will
;;;		learn remindings to the two exemplar-containing superordinate
;;;		categories (cochlear_age and cochlear_noise), but not to
;;;		their common specialization (cochlear_age_and_noise).
;;;		This is as it should be since the expert should want any
;;;		age-related features to remind to cochlear_age rather than to
;;;		the more specific cochlear_age_and_noise.  However, if Protos
;;;		has gotten remindings to 2 categories that share a common
;;;		specialization, then the common specialization should inherit
;;;		both remindings and thus be tried first.
;;;
;;;  Design:	For each pair of reminded categories, see if they share a
;;;		1-step specialization.  If so, generate a reminding to that
;;;		specialized category having the strength of its 2 "parents".
;;;
;;;  FUTURE:	This function should probably be extended to consider ALL
;;;		common-specializations, i.e., deeper than 1-step.
;;;             Also, this function has a minor bug that probably isn't worth
;;;             fixing: if a category is a common-specialization of 3 or more
;;;             reminded categories, it will inherit more strength than it
;;;             should because it will inherit from a category more than once.
;;;
;;;  Original Author: Dan Dvorak
;;;  Rewritten By:    Erik Eilerts
;;;
;;;---------------------------------------------------------------------------


(defun common-specializations ()
  (if *trace-remindings*
      (format t "~%Common-specializations ..."))

  (let ((common-rem-list nil)		; list of common-specialization remindings
        spec-list                       ; list of all specializations
	pair)
    
    (do* ((rem-list  *initial-remindings* (cdr rem-list))
	  (rem       (first rem-list) (first rem-list))
	  (cat       (car rem)        (car rem)))
	 ((endp rem-list))

      ;; Collect all 1-step specializations of all categories.
      ;; Makes a list with nodes of this type: (name strength parent)
      (dolist (rel (node-relations cat))
	(if (eql *verb-hasTypicalSpec* (relation-verb rel))
	    (push (list (car (relation-to-nodes rel)) (cdr rem) cat) spec-list))))


    ;; Combine the strengths of all categories that appear more than once in the list.
    ;; Makes a list with nodes of this type: (name strength (list of parents))
    (do* ((common-spec (first spec-list) (first spec-list))
          (common-spec-name (first common-spec) (first common-spec))
          (common-rem  (list (second common-spec) (list (third common-spec)))
                       (list (second common-spec) (list (third common-spec)))))

         ((endp common-spec))

       (setq spec-list (cdr spec-list))

       (do* ((num 1 (1+ num))
             (new-spec (assoc common-spec-name spec-list) (assoc common-spec-name spec-list)))
            ((endp new-spec)
             (if (> num 1) (push (cons common-spec-name common-rem) common-rem-list)))
         (setq common-rem (list (+ (first common-rem) (second new-spec)) 
                                (cons (third new-spec) (second common-rem))))
         (setq spec-list (remove new-spec spec-list :count 1))
      ))
    

    ;; For each common-specialization, if it is a new category that is
    ;; already in *remindings*, then it is ignored, since it will already have
    ;; received reminding values from its parents in strengthen-categories.
    ;; Otherwise, add it to *remindings*.
    (dolist (new-rem common-rem-list)
      (if (null (assoc (car new-rem) *initial-remindings*))       ;; doesn't exist, so add it
          (let* ((rem-name (first new-rem))
                 (rem-strength (second new-rem))
                 (rem-from (third new-rem)))

	    (if *trace-remindings*
                (progn
		   (format t "~%   ~A inherits ~4,2F from ~A"
			(getname rem-name) rem-strength (getname (car rem-from)))
                   (dolist (from-node (cdr rem-from)) (format t " and ~A" (getname from-node)))))

            (if (setq pair (assoc rem-name *remindings*))
	      (incf (cdr pair) rem-strength)                     ;; from old code, not executed
	      (push (cons rem-name rem-strength) *remindings*)))))))



;;;---------------------------------------------------------------------------
;;;  Function:	(strengthen-exemplars)
;;;
;;;  Given:	the merged remindings in the global variable *remindings*
;;;
;;;  Purpose:	Strengthen remindings to exemplars by inheriting the strength
;;;		of the merged remindings (if any) to the exemplar's category.
;;;
;;;  Design:	FOR ALL reminded nodes
;;;		  IF the node is an exemplar AND its category (class) is also
;;;		     a reminded node
;;;		  THEN increase the strength of the exemplar's reminding by 
;;;		     the strength of the category's reminding.
;;;
;;;  Note:	In the source code, "e-rem" and "c-rem" distinguish between
;;;		the remindings to exemplars and to categories, respectively.
;;;---------------------------------------------------------------------------

(defun strengthen-exemplars ()

  (if *trace-remindings*
      (format t "~%Strengthen exemplars ..."))

  (dolist (e-rem *remindings*)
    (let ((node (car e-rem))
          c-rem)
      (and (my-exemplar-p node)
           (setq c-rem (assoc (exemplar-category node) *remindings*))
           (incf (cdr e-rem) (cdr c-rem))
           (if *trace-remindings*
               (format t "~%   ~A inherits strength ~4,2F from ~A"
                            (getname node)
                            (cdr c-rem)
                            (getname (exemplar-category node))))))))


;;;---------------------------------------------------------------------------
;;;  Function:	(strengthen-features)
;;;
;;;  Given:	the merged remindings in the global variable *remindings*
;;;
;;;  Purpose:	Strengthen remindings to terms that are features of reminded
;;;             terms.  This is done so that the most specific terms of a
;;;             hierarchically-defined concept will be matched first before
;;;             proceeding to the more general terms.
;;;
;;;  Example:   If there is a reminding to "engine", and "engine" is a
;;;             feature of "my_car", and "my_car" is an exemplar of "car",
;;;             and there is a reminding to "car", then the reminding to
;;;             "engine" should inherit the strength of the reminding to
;;;             "car".
;;;
;;;  Design:    For each reminded term (term1), search "up" the category
;;;             network for other reminded terms (term2) that are related by
;;;             "feature-of" or "exemplar-of" links.  When found, create a
;;;             a new reminding to term1 having the strength of the reminding
;;;             to term2, and save this temporarily on a separate list.
;;;             Then, after creating all the new remindings, merge them with
;;;             the existing remindings.
;;;
;;;             The reason for not simply immediately adjusting the strength
;;;             of the existing reminding to term1 is that if there were
;;;             remindings to terms at more than two levels, then the result-
;;;             ing strengths would depend on the order in which the terms
;;;             were processed.
;;;---------------------------------------------------------------------------

(defun strengthen-features ()
  (let ((new-remindings nil)
	term1)
    (declare (special new-remindings term1))	; make visible to strengthen-features2
    
    (if *trace-remindings*
	(format t "~%Strengthen features ..."))
    
    ;; For each reminded term ...
    (dolist (rem *remindings*)
      (setq term1 (car rem))
      ;; search "up" through feature-of and exemplar-of links to other
      ;; reminded terms.
      (strengthen-features2 term1))
    
    ;; If any new remindings generated, then merge them all together.
    (if new-remindings
	(setq *remindings* (merge-remindings (nconc new-remindings *remindings*))))))


(defun strengthen-features2 (start)
  (declare (special new-remindings term1))     ; defined in strengthen-features

  ;; Collect into 'nodes' the list of exemplars of which term1 is a feature
  ;; plus the category of term1 (if it's an exemplar).
  (let ((category  (exemplar-category start))
	(nodes     (feature-of-exemplars start))
	rem)
    (if category
	(push category nodes))

    ;; For each of these related nodes, inherit any reminding to the nodes
    ;; and/or their "superior" nodes.
    (dolist (term2 nodes)
      (if (setq rem (assoc term2 *remindings*))
	  (progn
	    (format t "~%   ~(~A~) inherits ~4,2F from ~(~A~)"
		    (getname term1) (cdr rem) (getname term2))
	    (push (cons term1 (cdr rem)) new-remindings)))
      (strengthen-features2 term2))))
      

;;;-----------------------------------------------------------------------------
;;;  Function:	(reassess-remindings  match)
;;;
;;;  Given:	match, the results of an incorrect or unsuccessful match.
;;;
;;;  Purpose:	This function is called when Protos has been reminded of a
;;;		category which the teacher has rejected or within which Protos
;;;		could not find an exemplar matching the new case.  The purpose
;;;		of this function is to reassess the remindings that led to this
;;;		category, lowering or even removing the remindings.
;;;
;;;  Design:	Protos has no way of assigning blame to particular remindings,
;;;		so all remindings which led it to try the category are
;;;		reassessed.  The following steps are taken:
;;;
;;;		1.  Collect all the remindings to the category that were evoked
;;;		    by features of the new case.
;;;
;;;		2.  Attempt to regenerate an explanation for each of these
;;;		    remindings.  If no explanation can be found, remove the
;;;		    reminding, else replace the reminding's strength with the
;;;		    explanation's strength.
;;;
;;;		3.  If there are moderate or strong remindings from this same
;;;		    feature to categories which are not known to be subordinates
;;;		    or superordinates of the target category, then decrease the
;;;		    strength of the reminding to the target category (to let the
;;;		    other remindings compete).
;;;
;;;		4.  If neither step 2 nor step 3 results in a weakening of the
;;;		    reminding, then decrease the reminding's strength slightly.
;;;                 If the reminding becomes very weak, it is removed.
;;;
;;;		5.  If a feature reminds Protos of at least 2 of 3 categories
;;;		    that have a common superordinate, the reminding is moved
;;;		    to the superordinate and given the strength of the strongest
;;;		    reminding to a subordinate.
;;;
;;;		6.  If a general category with at least 3 subordinate categories
;;;		    shares a reminding with only one of the subordinates, the
;;;		    reminding to the general category is removed.
;;;-----------------------------------------------------------------------------

(defun reassess-remindings (match)
  
  (let* ((*rem-category*  (exemplar-category (match-exemplar match)))
	 (features        (case-features     (match-newcase  match)))
	 (remindings      nil)
	 (remfeatures     nil))
    (declare (special *rem-category*))   ; make *rem-category* visible to reassessor.

    (if *trace-remindings*
	(format t "~%Reassessing remindings that led to ~A: "
		(getname *rem-category*)))
    
    ;; Collect all remindings to the category evoked by features of newcase.
    (dolist (feature features)
      (let ((rem (assoc *rem-category* (feature-remindings feature))))
	(if rem
	    (progn
	      (push feature remfeatures)
	      (push rem remindings)))))
    
    (if remindings
	;; For each feature that evoked a reminding, perform steps 2-6.
	(mapc #'reassessor remfeatures remindings))))



(defun reassessor (feature reminding)
  (prog ((fremindings (feature-remindings feature))
	 (rstrength   (cdr reminding))
	 estrength
	 result)
	(declare (special *rem-category*))

	(format t "~%   Trying to regenerate reminding from ~A to ~A"
		(getname feature) (getname *rem-category*))

	;; Step 2 -- attempt to regenerate explanation for each reminding.
	;; MAYBE SHOULD SKIP THIS IF rstrength < *reminding-weak* (Ray did).
	;;(if (< rstrength *reminding-weak*)  (go step4))
        (setq *ignore-conditions*  T)  ;;ignore conditions when searching for remindings
	(setq result  (kbpm 'FtoT feature *importance-big* *rem-category* nil))
        (setq *ignore-conditions* nil) ;;reset

	(case (result-type result)
	  (unmatched  (setf (feature-remindings feature) (delete reminding fremindings))
		      (if *trace-remindings*
			  (format t "~%   No match.  Deleting reminding from ~A"
				  (getname feature)))
		      (go step5))

	  (spurious   (setf (feature-remindings feature) (delete reminding fremindings))
		      (if *trace-remindings*
			  (format t "~%   Spurious.  Deleting reminding from ~A"
				  (getname feature)))
		      (go step5))

	  (explained  (setq estrength (explanation-strength (result-explanation result)))
		      (rplacd reminding estrength)
		      (if (and *trace-remindings* (not (= estrength rstrength)))
			  (format t "~%   Adjusting reminding from ~A"
				  (getname feature)))
		      (if (< estrength rstrength)
			  (go step5)
			  (setq rstrength estrength)))

	  (excluded   (setq estrength (explanation-strength (result-explanation result)))
		      (rplacd reminding (- estrength))
		      (if *trace-remindings*
			  (format t "~%   Creating censor from ~A"
				  (getname feature)))
		      (go step5))

	  (otherwise  (format t "~%ERROR: reassessor: bad result type!~%")))

	;; Step 3 -- adjust for competing remindings.
	;; MAYBE SKIP THIS STEP IF REMINDING IS WEAK (as shown below).
	;;(if (< rstrength *reminding-weak*)  (go step4))
	(let (relatives competitors)
	  (setq relatives (collect-relatives *rem-category*
					     (list *verb-hasTypicalGen*
						   *verb-hasTypicalSpec*
						   *verb-hasFunction*
						   *verb-isFunctionOf*
						   *verb-causes*
						   *verb-causedBy*
						   *verb-hasPart*
						   *verb-partOf*)
					     nil))
	  (dolist (rem fremindings)
	    (if (and (> (cdr rem) *reminding-weak*)
		     (not (member (car rem) relatives)))
		(push rem competitors)))
	  
	  (if competitors
	      (let (strength-sum)
		(setq competitors (sort competitors #'> :key #'cdr))
		(setq strength-sum (cdr (first competitors)))
		(if (second competitors)
		    (incf strength-sum (cdr (second competitors))))
		(decf (cdr reminding) (* strength-sum (/ (cdr reminding) 3)))
		(if *trace-remindings*
		    (format t "~%   Adjusting for competing remindings from ~A"
			    (getname feature)))
		(go step5))))

	;; Step 4 -- If no weakening from steps 2 or 3, weaken it slightly.
	step4
	(decf (cdr reminding) 0.025)
	(if *trace-remindings*
	    (format t "~%   Slightly weakening reminding from ~A"
		    (getname feature)))
	(if (< (cdr reminding) *reminding-very-weak*)
	    (progn
	      (setf (feature-remindings feature) (delete reminding fremindings))
	      (if *trace-remindings*
		  (format t "~%   Deleting very weak reminding from ~A"
			  (getname feature)))))

	;; Step 5 -- possibly move reminding to superordinate.
	step5
	(multiple-value-bind (parent siblings)
	    (collect-siblings *rem-category* *verb-hasTypicalGen*)
	  (if (and siblings (>= (length siblings) 3))
	      (let (remcategories remsiblings proportion)
		(setq remcategories (mapcar #'car fremindings))
		(setq remsiblings   (intersection remcategories siblings))
		(setq proportion    (/ (length remsiblings) (length siblings)))
		(if (>= proportion (/ 2 3))
		    (let (best-strength)
		      (setq remsiblings (sort remsiblings #'> :key #'cdr))
		      (setq best-strength (cdar remsiblings))
		      (if (assoc parent fremindings)
			  (rplacd (assoc parent fremindings) best-strength)
			  (push (cons parent best-strength) (feature-remindings feature)))
		      (setf (feature-remindings feature) (delete reminding fremindings))
		      (if *trace-remindings*
			  (format t "~%   Moving reminding from ~A to ~A"
				  (getname feature) (getname parent))))))))
	
	;; Step 6 -- possibly remove reminding from a general category.
	(let ((remcategories (mapcar #'car fremindings))
	      (children (collect-children *rem-category* *verb-hasTypicalSpec*)))
	  (if (and children (>= (length children) 3))
	      (let ((remchildren (nintersection children remcategories)))
		;; If only one child is getting a reminding ...
		(if (and remchildren (null (cdr remchildren)))
		    (progn
		      ;; then delete the reminding to the parent.
		      (setf (feature-remindings feature)
			    (delete reminding (feature-remindings feature)))
		      (if *trace-remindings*
			  (format t "~%   Deleting reminding to ~A"
				  (getname feature))))))))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (collect-relatives  node  verbs  relatives)
;;;
;;;  Given:     -- a node whose relatives are to be found;
;;;             -- verbs, a list of verbs specifying the types of relations
;;;                  allowed in this collection; and
;;;             -- relatives, a list of related nodes already found.
;;;
;;;  Returns:   a list of nodes that are related to the given node through
;;;             the verbs specified (including everything in 'relatives').
;;;-----------------------------------------------------------------------------

(defun collect-relatives (node1 verbs relatives)
  ;; For each relation emanating from the given node ...
  (dolist (rel (node-relations node1) relatives)
    ;; If the relation is one of the allowed types ...
    (if (member (relation-verb rel) verbs)
	;; then add its targets to relatives, and recurse.
	(dolist (node (relation-to-nodes rel))
	  (if (not (member node relatives))
	      (progn
		(push node relatives)
		(setq relatives (collect-relatives node verbs relatives))))))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (collect-siblings  node  verb)
;;;
;;;  Given:     -- a node whose siblings are to be found; and
;;;             -- a verb specifying the relation to the parent node;
;;;
;;;  Returns:   two values are returned:
;;;             -- the parent node, if found, otherwise nil;
;;;             -- the sibling nodes of the given node.
;;;
;;;  Bugs:      If a node has more than one parent related by the given verb
;;;-----------------------------------------------------------------------------

(defun collect-siblings  (node1  verb)
  (let ((inverse-verb (verb-inverse verb))
	(parent       nil)
	(siblings     nil))
    (dolist (rel (node-relations node1))
      (if (eq verb (relation-verb rel))
	  (progn
	    (setq parent (car (relation-to-nodes rel)))
	    (dolist (rel2 (node-relations parent))
	      (if (and (eq inverse-verb (relation-verb rel2))
		       (not (eq node1 (car (relation-to-nodes rel2)))))
		  (push (car (relation-to-nodes rel2)) siblings)))
	    (if siblings
		(return-from collect-siblings (values parent siblings))))))
    (values nil nil)))

	
;;;-----------------------------------------------------------------------------
;;;  Function:  (collect-children  parent  verb)
;;;
;;;  Given:     -- a parent node whose child nodes are to be found; and
;;;             -- a verb specifying the relation from parent to children;
;;;
;;;  Returns:   a list of child nodes (possibly nil).
;;;-----------------------------------------------------------------------------

(defun collect-children  (parent  verb)
  (let ((children nil))
    (dolist (rel (node-relations parent))
      (if (eq verb (relation-verb rel))
	  (dolist (node (relation-to-nodes rel))
	    (push node children))))
    children))



;;;-----------------------------------------------------------------------------
;;;  Function:  (trim-remindings  feature)
;;;
;;;  Purpose:    Given a feature, this function checks to see if the number of
;;;              remindings emanating from the feature exceeds a "reasonable"
;;;              limit.  If so, the weakest remindings in excess of the limit
;;;              are deleted from the feature's list of remindings.
;;;
;;;  FUTURE:     This function should probably be modified to NOT delete any
;;;              strong remindings, even if they are in excess of the limit.
;;;
;;;  Motivation: If any feature evokes a large number of remindings, then 
;;;              whenever that feature appears in a case, it generates a large
;;;              number of hypotheses, which in turn causes Protos to attempt
;;;              matches to a large number of exemplars.  Since knowledge-
;;;              based pattern matching is computationally expensive, Protos
;;;              is slowed down if it permits large numbers of remindings.
;;;
;;;              Also, this function serves as a safeguard against teachers
;;;              who (innocently) create a lot of remindings when they feel
;;;              compelled to explain the relation between every feature and
;;;              its category, even when the presence of the feature doesn't
;;;              notably increase their belief in the category.
;;;
;;;  Note:       The current upper limit on the number of remindings allowed
;;;              per feature is 25% of the number of exemplar-containing
;;;              categories, or 3, whichever is larger.  These values are
;;;              defined in global.lisp and can, of course, be modified.
;;;-----------------------------------------------------------------------------

(defun trim-remindings (feature)
  (let* ((limit1  (round (*  *percent-of-ec-categories*  *number-of-ec-categories*)))
	 (limit2  (max  *low-reminding-limit*  limit1))
	 (rems    (feature-remindings feature))
	 (count   (length rems)))

    ;; If the number of remindings exceeds the limit ...
    (if (> count limit2)
	;; then keep only the strongest N remindings (where N is the limit).
	(let ((sorted-rems   (sort rems  #'>=  :key #'cdr))
	      (delete-count  (- count limit2)))
	  (if *trace-remindings*
	      (progn
		(format t "~%Reminding limit hit for ~A -- removing " (getname feature))
		(print-alist (nthcdr limit2 (feature-remindings feature)) t 1)))
	  (setf (feature-remindings feature) (nbutlast sorted-rems delete-count))))))

