;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak

(in-package 'protos)



;;;=============================================================================
;;;
;;;          S A V E   /   L O A D    K N O W L E D G E    B A S E
;;;  -------------------------------------------------------------------------
;;;
;;;  Contents:   This file contains all functions for saving a knowledge-base
;;;              to a file and for loading a knowledge-base from a file.  The
;;;              file system interface functions of Common Lisp are used, thus
;;;              making this software portable.
;;;
;;;  Format:     A KB file must be a character file (not a binary file) so that
;;;              knowledge bases can be ported to other types of Common Lisp
;;;              systems.  Also, the KB file format defined by these functions
;;;              is intended to be easily read and edited by humans.  Although
;;;              most of the development of a knowledge base will be 
;;;              accomplished by interacting with the CL-Protos program, some
;;;              preparation and editing may be easier done with a full-screen
;;;              editor.
;;;
;;;  NOTE:       1.  If you make changes to any of the structures used in the
;;;                  category network, that may necessitate changes in one or
;;;                  more of these KB save and load functions.
;;;
;;;              2.  Note that these "save-xxx" functions are particularly easy
;;;                  to test/inspect by simply calling the function with its
;;;                  'stream' argument set to 'T'.  This will cause it to write
;;;                  its output to the terminal rather than a file.
;;;
;;;  Functions:  forget-kb, forget-kb2, forget-name, forget-node
;;;              save-kb, save-kb2, save-case, save-predicate, save-term
;;;              load-kb, read-kb, load-pass2, load-term, load-note,
;;;              load-predicate, load-term2
;;;              save-transformations
;;;=============================================================================


(defparameter *kb-filename* "junk-kb.lisp")     ; contains last filename given


;;;----------------------------------------------------------------------------
;;;  Function:  (forget-kb  proceed)
;;;
;;;  Purpose:   This function causes all elements of the current knowledge
;;;             base to be "forgotten" from the Protos package.  Specifically,
;;;             all of the symbols that were created as part of this
;;;             knowledge base (i.e., named terms and predicates) are made to
;;;             be unbound.  Thus, when a new knowledge base is loaded or
;;;             created interactively, terms or predicates having the same 
;;;             name as something in the old KB will not accidentally be
;;;             already bound to an old object.  Got that?
;;;
;;;             If the "proceed" argument is NIL, this function will first
;;;             ask the user to confirm before forgetting the KB.
;;;----------------------------------------------------------------------------
  
(defun forget-kb (proceed)

  ;; If there is no KB to forget ...
  (if (and (null *history*)
	   (null *transformations*)
	   (null *uninstantiated-predicates*))

      (progn
	;; then issue optional message  ...
	(if (null proceed)
	    (format t "~%The knowledge base was already empty.~%"))
	(return-from forget-kb (values))))

  ;; Check with user before proceeding.
  (if (not proceed)
      (if (not (prompt "~%Are you SURE it's OK to forget the current KB? "
		       nil 'y-or-n nil nil))
	  (progn
	    (format t "~%Very well, the knowledge base remains unchanged.")
	    (return-from forget-kb (values)))))
  
  ;; FORGET EACH TERM -- predicates as well as propositions
  (dolist (item *history*)
    (typecase item
      (node       (forget-node item))
      (case       (forget-name (case-name item)))
      (string     nil)
      (otherwise  (format t "~%Unrecognized type ~A in history."
			  (type-of item)))))

  (setq *history* nil)

  (setq *number-of-ec-categories* 0)

  ;; FORGET EACH TRANSFORMATION.
  (dolist (xform *transformations*)
    (makunbound (transformation-name xform)))

  (setq  *transformations* nil)

  ;; FORGET EACH UNINSTANTIATED PREDICATES
  (dolist (pred *uninstantiated-predicates*)
    (makunbound (predicate-name pred)))

  (setq  *uninstantiated-predicates* nil) 
  
  (format t "~%The knowledge base is now empty."))

;;;----------------------------------------------------------------------------

(defun forget-name (name)
  (and name
       (symbolp    name)
       (boundp     name)
       (makunbound name)))

;;;----------------------------------------------------------------------------

(defun forget-node (node)
  (forget-name (node-name node))
  (forget-name (node-abbrev node))
  (mapc #'forget-name (node-synonyms node)))



;;;-----------------------------------------------------------------------------
;;;
;;;  Function:   save-kb
;;;
;;;  Given:      A complete knowledge base which includes nodes,
;;;              terms, exemplars, features, and categories, and links
;;;              between them. 
;;;              A stream, out-stream, to a file in which the knowledge will 
;;;              be saved.
;;;
;;;  Purpose:    This top level function constructs a pathname to the file
;;;              in which structures will be saved. It then calls the functions
;;;              which save the various knowledge structures in the knowledge 
;;;              base.
;;;
;;;-----------------------------------------------------------------------------
  
(defun save-kb ()

  (let* ((prompt1 (format nil "~%~%Please enter file name for storing KB~
                                 ~%(default \"~A\", or \"q\" to quit) ---> "
                                 *kb-filename*))
	 (filename (prompt prompt1 nil 'string nil nil)))
    (if (or (null filename) (string= "" filename))
	(setq filename *kb-filename*))
    (if (string= "q" filename)
	(return-from save-kb (values)))

    (format t "~%Opening ~A ..." filename)
    (if (probe-file filename)
	(if (prompt "~%This file already exists.  OK to overwrite? "
		    nil 'y-or-n nil nil)
	    (save-kb2 filename)
	    (format t "~%Then KB save is cancelled."))
	(save-kb2 filename))))

;;;-----------------------------------------------------------------------------

(defun save-kb2 (filename)
  (let ((print-length *print-length*))
    ;; Temporarily reset *print-length* to NIL during KB save.
    (setq *print-length* nil)
    
    ;; Open the file to save the knowledge base into.
    (with-open-file 
      (stream filename
	      :direction :output
	      :if-exists :new-version
	      :if-does-not-exist :create)

      (setq *kb-filename* filename)

      (format t "~%Saving knowledge base ...")
      (princ ";;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; -*-" stream)
      (terpri stream)

      ;; Save every object on the *history* list.
      (let ((rhistory (reverse *history*))
	    (count  0))
	(dolist (item rhistory)
	  (if (= 0 (mod (incf count 1) 10))
	      (progn
		(write-char #\. t)
		(force-output t)))
	  (typecase item
	    (term       (save-term      item stream))
	    (predicate  (save-predicate item stream))
	    (case       (save-case      item stream))
	    (string     (save-note      item stream))
	    (otherwise  (format t "~%History item type ~A unknown!"
				(type-of item))))))

      ;; Save transformations.
      ;; (save-transformations stream)
      )

    ;; Restore original value of *print-length*
    (setq *print-length* print-length)

    (format t "~%Knowledge base saved.")))



;;;-----------------------------------------------------------------------------
;;;  Function:  (falist  alist  stream)
;;;
;;;  Purpose:   This function [which stands for "format alist"] writes a human-
;;;             readable representation of an alist of "remindings" or
;;;             "importances" to the specified stream.  The format of this
;;;             output must be compatible with the function build-alist.
;;;-----------------------------------------------------------------------------

(defun falist (stream alist column)
  (let ((n 0))
    (format stream "((~(~A~) ~4,2F)" (getname (caar alist)) (cdar alist))
    (dolist (pair (cdr alist))
      (if (= 0 (mod (incf n) 3))
	  (format stream "~%~VT" column))
      (format stream " (~(~A~) ~4,2F)" (getname (car pair)) (cdr pair)))
    (write-char #\) stream)))

(defun build-alist (alist)
  (let ((alist2 nil))
    (dolist (pair alist)
      (push (cons (car pair) (cdr pair)) alist2))
    (nreverse alist2)))

(defun fnlist (stream namelist column)
  (let ((n 0))
    (format stream "(~(~A~)" (getname (car namelist)))
    (dolist (term (cdr namelist))
      (if (= 0 (mod (incf n) 4))
	  (format stream "~%~VT" column))
      (format stream " ~(~A~)" (getname term)))
    (write-char #\) stream)))

(defun fdiff (stream diff)
  (format stream "(~(~A~) <-- " (getname (difference-node diff)))
  (fnlist stream (difference-features diff) 19)
  (write-char #\) stream))



(defun save-predicate (pred stream)
  (format stream   "~%~%(predicate :name        ~(~A~)" (node-name pred))
  (if (node-synonyms pred)
      (format stream "~%           :synonyms    ~(~A~)" (node-synonyms pred)))
  (if (node-abbrev pred)
      (format stream "~%           :abbrev      ~(~A~)" (node-abbrev pred)))
  (if (node-comment pred)
      (format stream "~%           :comment     \"~A\"" (node-comment pred)))
  (if (predicate-relations pred)
      (progn
	(format stream   "~%           :relations   (")
	(format-relation stream (first (node-relations pred)))
	(dolist (rel (rest (node-relations pred)))
	  (format stream "~%                         ")
	  (format-relation stream rel))
	(write-char #\) stream)))
  (write-char #\) stream))


    
(defun save-term (term stream)
  (format stream   "~%~%(term :name        ~(~A~)" (term-name term))
  (if (term-synonyms term)
      (format stream "~%      :synonyms    ~(~A~)" (term-synonyms term)))
  (if (term-abbrev term)
      (format stream "~%      :abbrev      ~(~A~)" (term-abbrev term)))
  (if (term-comment term)
      (format stream "~%      :comment     \"~A\"" (term-comment term)))
  (if (term-type term)
      (format stream "~%      :type        ~(~A~)" (term-type term)))
  (if (feature-remindings term)
      (progn
	(format stream "~%      :remindings  ")
	(falist stream (feature-remindings term) 19)))
  (if (category-importances term)
      (progn
	(format stream "~%      :importances ")
	(falist stream (category-importances term) 19)))
  (if (category-exemplars term)
      (progn
	(format stream "~%      :exemplars   ")
	(fnlist stream (category-exemplars term) 19)))
  (if (category-faultvars term)
      (progn
	(format stream "~%      :faultvars   ")
	(fnlist stream (category-faultvars term) 19)))
  (if (exemplar-category term)
      (format stream "~%      :category    ~(~A~)"
	      (node-name (exemplar-category term))))
  (if (exemplar-features term)
      (progn
	(format stream "~%      :features    ")
	(fnlist stream (exemplar-features term) 19)))
  (if (exemplar-typicality term)
      (format stream "~%      :typicality  ~4,2F" (exemplar-typicality term)))
  (if (exemplar-differences term)
      (progn
	(format stream "~%      :differences (")
	(fdiff stream (first (exemplar-differences term)))
	(dolist (diff (rest (exemplar-differences term)))
	  (princ "                    " stream)
	  (fdiff stream diff))
	(write-char #\) stream)))
  (if (node-relations term)
      (let ((flag nil))
	(dolist (rel (node-relations term))
	  ;; Don't write out the "has-exemplar" and "is-exemplar-of"
	  ;; relations since these can regenerated at load-time.
	  ;; (It's always bad to have two versions of the truth, so this
	  ;; helps to minimize the problem).
	  (if (not (member (relation-verb rel)
			   (list *verb-hasExemplar* *verb-isExemplarOf*)))
	      (progn
		(if (not flag)
		    (progn
		      	(format stream "~%      :relations   (")
			(setq flag t))
		    (format stream "~%                    "))
		(format-relation stream rel))))
	(if flag
	    (write-char #\) stream))))
  ;; Last parenthesis to close the "term" expression.
  (write-char #\) stream))




(defun format-relation (stream rel)
  (let ((from-nodes  (relation-from-nodes rel))
	(to-nodes    (relation-to-nodes rel))
	(condition   (relation-condition rel))
	(quantifiers (relation-quantifiers rel)))
 ;(print-relation rel t 1)
    (princ "((" stream)

    (if condition
	(print-condition condition stream 1))
 ;(print-relation rel t 1)

    (print-node-names from-nodes stream " and ")
 ;(print-relation rel t 1)

    (if quantifiers
	(dolist (quant quantifiers)
	  (format stream " ~(~A~)" (getname quant))))
 ;(print-relation rel t 1)

    ;; Print the verb.
    (format stream " ~(~A~) " (getname (relation-verb rel)))

 ;(print-relation rel t 1)
    ;; to-nodes may be nil if verb = "spurious"
    (if to-nodes
	(print-node-names to-nodes stream " and "))
 ;(print-relation rel t 1)

    (write-char #\) stream)

    ;; Print the user name and comment field.
    (write-char #\space stream)
 ;(print-relation rel t 1)
    (prin1 (relation-creator rel) stream)
    (write-char #\space stream)
 ;(format t "ALMOST DONE.")
    (prin1 (relation-comment rel) stream)
    (write-char #\) stream)
 ;(format t "~%ALL DONE in format-relation")
  ))



(defun save-case (case stream)
  (format stream "~%~%(case :name        ~(~A~)" (case-name case))
  (if (case-category case)
      (format stream "~%      :category    ~A"
	      (node-name (case-category case))))
  (if (case-preclassify case)
      (format stream "~%      :preclassify T"))
  (if (case-features case)
      (progn
	(format stream "~%      :features    ")
	(fnlist stream (case-features case) 19)))
  (if (case-disposition case)
      (format stream "~%      :disposition (~A ~A)"
	      (first  (case-disposition case))
	      (second (case-disposition case))))
  (if (case-comment case)
      (format stream "~%      :comment     \"~A\"" (case-comment case)))
  (if (case-creator case)
      (format stream "~%      :creator     ~A" (case-creator case)))
  (write-char #\) stream))
  
(defun save-note (note stream)
  (terpri stream)
  (terpri stream)
  (princ  "(note " stream)
  (prin1  note     stream)
  (write-char #\)  stream))



;;;----------------------------------------------------------------------------
;;;  Function:  load-kb
;;;
;;;  Purpose:   This function prompts the user for a file name and then loads
;;;             the knowledge base contained in that file.  Loading is a
;;;             two-pass process -- the first pass creates all the symbols of
;;;             the KB and the second pass then evaluates all the things that
;;;             refer to those symbols (e.g., remindings, importances,
;;;             relations, differences, etc.).
;;;-----------------------------------------------------------------------------

(defparameter *relation-forms* nil)
(defparameter *loading-kb* nil)                 ; only used when kb is loaded


(defun load-kb ()
  (let* ((prompt1 (format nil "~%~%Please enter file name for loading KB~
                                 ~%(default \"~A\", or \"q\" to quit) ---> "
                                 *kb-filename*))
	 filename)

    (if *history*
	(if (prompt "~%~%There is an existing knowledge base.~
                       ~%Is it OK to forget this KB? "
		    nil 'y-or-n nil nil)
	    (forget-kb t)
	    (if (prompt "~%Then do you wish to save this KB to a file? "
			nil 'y-or-n nil nil)
		(progn
		  (save-kb)
		  (format t "~%Now proceeding with KB load."))
		(progn
		  (format t "~%Then this load operation is cancelled.")
		  (return-from load-kb (values))))))

    (setq *history*        nil
	  *relation-forms* nil)

    (setq filename (prompt prompt1 nil 'string nil nil))
    (if (or (null filename) (string= "" filename))
	(setq filename *kb-filename*))
    (if (string= "q" filename)
	(return-from load-kb (values)))

    (format t "~%   Opening ~A ..." filename)
    (setq *kb-filename* filename)
    
    (with-open-file (stream filename :direction :input :if-does-not-exist nil)
      (if (null stream)
	  (format t "~%File ~A does not exist." filename)
	  (read-kb stream)))
    (format t "~%   Pass 2 on KB ...")
    (load-pass2)
    (format t "~%   Done.")))



(defun read-kb (stream)
  (format t "~%   Reading KB ...")
  (let ((count 0)
	form)
    (loop
      (setq form (read stream nil 'eof))
      (if (eql 'eof form)
	  (return (values)))
      (if (= 0 (mod (incf count 1) 10))
	  (progn
	    (write-char #\. t)
	    (force-output t)))
      (if (not (listp form))
	  (format t "~%Input form ~A not a list.  Skipping it." form)
	  (case (car form)
	    (term      (load-term form))
	    (predicate (load-predicate form))
	    (string    (load-note form))
	    (note      (load-note (second form)))
	    (case      (format t "~%Note: cases currently ignored."))
	    (otherwise (format t "Warning: unrecognized type '~A' in ~A"
			       (car form) form)))))))



(defun load-pass2 ()
  (declare (special *only-warn-on-conflict*)
           (special *loading-kb*))

  ;; Run a second pass over each term in *history*.
  (dolist (item *history*)
    (etypecase item
      (string     nil)
      (term       (load-term2 item))
      (predicate  nil)
      ))
  ;; Parse each relation form and install the relation.
  (if (null *relation-forms*)
      (format t "~%Note: There are no relations in this KB.")
      (unwind-protect
	  (progn
	    (setq *only-warn-on-conflict* t)
            (setq *loading-kb* t)
            (let ((count 0))
	      (dolist (rel-form *relation-forms*)
                (if (= 0 (mod (incf count 1) 20))
	          (progn
	            (write-char #\. t)
	            (force-output t)))
	        ;;;(format t "~%load-pass2:  ~A" (car rel-form))
	        (multiple-value-bind (expl rel)
	  	   (get-explanation (car rel-form))
		 (declare (ignore expl))
		 (if rel
		    (progn
		      (setf (relation-creator rel) (second rel-form))
		      (setf (relation-comment rel) (third  rel-form)))
		    (format t "~%Warning: couldn't parse ~A" rel-form))))))
	(setq *only-warn-on-conflict* nil)
        (setq *loading-kb* nil)))
  )


(defun load-term (form)
  (let (term)
    (do* ((rform   (cdr form)   (cddr rform))
	  (keyword (car rform)  (car rform))
	  (value   (cadr rform) (cadr rform)))
	 ((endp rform))
    
      ;;;(format t "~%Keyword: ~S   Value: ~A" keyword value)

      (case keyword
	(:name        (setq term (check-term-name value 'create)))

	(:abbrev      (setf (node-abbrev term) value)
	 	      (set value term))

	(:synonyms    (if (not (listp value))
			  (setq value (list value)))
	              (setf (node-synonyms term) value)
		      (dolist (name value)
			(set name term)))

	(:comment     (setf (node-comment term) value))

	(:type        (setf (term-type term) value))

	(:remindings  (setf (feature-remindings term) value))

	(:importances (setf (category-importances term) value))

	(:exemplars   (setf (category-exemplars term) value))

	(:faultvars   (setf (category-faultvars term) value))

	(:category    (setf (exemplar-category term) value))

	(:features    (setf (exemplar-features term) value))

	(:typicality  (setf (exemplar-typicality term) value))

	(:differences (setf (exemplar-differences term) value))

	(:relations   (dolist (rel-form value)
			;;;(format t "~%rel-form: ~A" rel-form)
			(push rel-form *relation-forms*)))

	(otherwise    (format t "~%UNRECOGNIZED: ~(~S~) ~(~S~)" keyword value))))))




(defun load-note (form)
  (if (stringp form)
      (push form *history*)
      (format t "~%Warning: string missing in: ~A" form)))

(defun load-predicate (form)
  (let (pred)
    (do* ((rform   (cdr form)   (cddr rform))
	  (keyword (car rform)  (car rform))
	  (value   (cadr rform) (cadr rform)))
	 ((endp rform))
    
      ;;(format t "~%Keyword: ~S   Value: ~A" keyword value)

      (case keyword
	(:name       (if (boundp value)
			 (progn
			   (setq pred (eval value))
			   (if (predicate-p pred)
			       (progn
				 (push pred *history*)
				 (format t "~%Overwriting existing instance of ~A" value))
			       (progn
				 (format t "~%ERROR: ~A is already in use as a ~A~
                                            ~%so it can't be used as a predicate!"
					 value (type-of (eval value)))
				 (return-from load-predicate (values)))))
			 (progn
			   (setq pred (make-predicate :name value))
			   (set value pred)
			   (push pred *history*))))

	(:abbrev     (setf (node-abbrev pred) value)
		     (set value pred))

	(:synonyms    (if (not (listp value))
			  (setq value (list value)))
	             (setf (node-synonyms pred) value)
		     (dolist (name value)
		       (set name pred)))

	(:comment    (setf (node-comment pred) value))

	(:relations   ;; Handle old version of relation syntax that didn't
	              ;; have creator and comment.
	              (dolist (rel-form value)
			(if (stringp (second rel-form))
			    (push rel-form *relation-forms*)
			    (push (list rel-form "" "") *relation-forms*))))

	(otherwise   (format t "~%UNRECOGNIZED: ~(~S~) ~(~S~)" keyword value))))))
  


(defun load-term2 (term)

  ;; DO REMINDINGS OF FEATURE
  (if (feature-remindings term)
      (let ((remindings nil))
	(dolist (pair (feature-remindings term))
	  (let* ((name     (car pair))
		 (strength (cadr pair))
		 (rem-term (check-term-name name 'fail)))
	    (if (and rem-term (category-p rem-term))
		(push (cons rem-term strength) remindings)
		(format t "~%Error: term ~A not bound or not a term." name))))
	(setf (feature-remindings term) (nreverse remindings))))

  ;; DO IMPORTANCES OF CATEGORY
  (if (category-importances term)
      (let ((importances nil))
	(dolist (pair (category-importances term))
	  (let* ((name     (car pair))
		 (strength (cadr pair))
		 (imp-term (check-term-name name 'fail)))
	    (if (and imp-term (feature-p imp-term))
		(push (cons imp-term strength) importances)
		(format t "~%Error: term ~A not bound or not a term." name))))
	(setf (category-importances term) (nreverse importances))))

  ;; DO EXEMPLARS OF CATEGORY
  (if (category-exemplars term)
      (let ((exemplars nil))
	(dolist (name (category-exemplars term))
	  (let ((exemplar (check-term-name name 'fail)))
	    (if (and exemplar (exemplar-p exemplar))
		(push exemplar exemplars)
		(format t "~%Error: term ~A not bound or not an exemplar." name))))
	(setf (category-exemplars term) (nreverse exemplars))))

  ;; DO FAULTVARS OF CATEGORY
  (if (category-faultvars term)
      (let ((faultvars nil))
	(dolist (name (category-faultvars term))
	  (let ((faultvar (check-pred-name name 'fail)))
	    (if (and faultvar (predicate-p faultvar))
		(push faultvar faultvars)
		(format t "~%Error: term ~A not bound or not a predicate." name))))
	(setf (category-faultvars term) (nreverse faultvars))))

  ;; DO CATEGORY OF EXEMPLAR
  (if (exemplar-category term)
      (let* ((name     (exemplar-category term))
	     (category (check-term-name name 'fail)))
	(if (and category (category-p category))
	    (setf (exemplar-category term) category)
	    (format t "~%Error: term ~A not bound or not a category." name))
	;; Automatically create the "has-exemplar" and "is-exemplar-of"
	;; relations.
	(install-relation (list category) (list term) nil *verb-hasExemplar* nil)))

  ;; DO FEATURES OF EXEMPLAR
  (if (exemplar-features term)
      (let ((features nil))
	(dolist (name (exemplar-features term))
	  (let ((feature (check-term-name name 'create)))
	    (if (and feature (feature-p feature))
		(push feature features)
		(format t "~%Error: term ~A not bound or not a feature." name))))
	(setf (exemplar-features term) (nreverse features))))

  ;; DO DIFFERENCE LINKS
  (if (exemplar-differences term)
      (let ((differences nil))
	(dolist (diff (exemplar-differences term))
	  (let* ((name     (first diff))
		 (exemplar (check-term-name name 'fail)))
	    (if (and exemplar (exemplar-p exemplar))
		(let ((features nil)
		      ndiff
		      feature)
		  (setq ndiff (make-difference :node exemplar))
		  (push ndiff differences)
		  (dolist (fname (third diff))
		    (if (setq feature (check-term-name fname 'fail))
			(push feature features)
			(format t "~%Error: term ~A not bound or not a feature." fname)))
		  (setf (difference-features ndiff) (nreverse features)))
		(format t "~%Error: term ~A not bound or not an exemplar." name))))
	(setf (exemplar-differences term) (nreverse differences))))

)




;;;-----------------------------------------------------------------------------
;;;  Function:   (save-transformations (stream))
;;;
;;;  Given:      "stream", an output stream.
;;; 
;;;  Purpose:    To save all transformations tructures on the *trasformations*
;;;              global list. The predicate slots referring to transformations
;;;              for a predicate are also saved.
;;;------------------------------------------------------------------------------

(defun save-transformations (stream)
  (let ((old-printlevel *print-level*)
	(defined-preds nil))

    ;; Need to set the parenthesis level deep enough in order to print out the
    ;; lisp-function of a transformation to file:
    (setq *print-level* 99)
    (dolist (xform-struct *transformations*)
	   (format stream "~%(setf ~A (make-transformation :name '~A ~
                           ~%   :creator '~A ~
                           ~%   :type '~A ~
                           ~%   :in-predicates '~A ~
                           ~%   :out-predicate '~A ~
                           ~%   :in-features '~A ~
                           ~%   :out-feature '~A ~
                           ~%   :description ~S ~
                           ~%   :lisp-function '~A))"
		   (transformation-name xform-struct)
		   (transformation-name xform-struct)
		   (transformation-creator xform-struct)
		   (transformation-type xform-struct)
		   (transformation-in-predicates xform-struct)
		   (transformation-out-predicate xform-struct)
		   (transformation-in-features xform-struct)
		   (transformation-out-feature xform-struct)
		   (transformation-description xform-struct)
		   (transformation-lisp-function xform-struct))
	   (format stream "~%(push ~A  *transformations*)" (transformation-name xform-struct))

	   
	   ;; Save the transformations on the appropriate predicate slots. Defines
           ;; the predicates in case they are not defined before. In that case, put the
           ;; predicate on the *uninstantiated-predicates* list.

	   (let ((out-pred (transformation-out-predicate xform-struct)))

	     (cond ((and (not (member out-pred defined-preds))
			 (member (eval out-pred) *uninstantiated-predicates*))
		    (format stream "~%(setq ~A (make-predicate :name '~A))" 
			    out-pred out-pred)
		    (push out-pred defined-preds)
		    (format stream "~%(push ~A *uninstantiated-predicates*)" out-pred)))
	     (format stream "~%(setf (predicate-output-xform ~A) '~A)"			       
		     (transformation-out-predicate xform-struct) (transformation-name xform-struct)))
	   (dolist (in-pred (transformation-in-predicates xform-struct))
	     (cond ((and (not (member in-pred defined-preds))
			 (member (eval in-pred) *uninstantiated-predicates*))
		    (format stream "~%(setq ~A (make-predicate :name '~A))"
			    in-pred in-pred)
		    (push in-pred defined-preds)
		    (format stream "~%(push ~A *uninstantiated-predicates*)" in-pred)))
	     (format stream "~%(push '~A (predicate-input-xforms ~A))"
		     (transformation-name xform-struct) in-pred)))

    ;; Resets print level
    (setq *print-level* old-printlevel)))

