(* The following global variables allow you to alter certain aspects of
   the behaviour of the Classical Refinement method.

   CRprint := true  specifies that certain information should be printed out
                    on the screen as the function is executing.
   CRisPO  := true  specifies that this module should simulate Path Opportunism,
                    rather than being non-opportunistic.
   CRstop  := false specifies that all of the neighbours of the current node
                    (the one just de-queued) should be enqueued.
                    The alternative (true) specifies that you should
                    immediately stop enqueuing if you encounter a neighbour
                    that is in the next class.
                      CRstop := true tends to find longer paths, traverse fewer
                                     edges, and create more opporutnities for
                                     opportunism than CRstop := false.
                    As path-length is our biggest concern, false is the default.
*)

val currentstep = ref 0 ;
val CRprint = ref false ; (*  default: no printing     *)
val CRisPO = ref false ;  (*  default: NO opportunism  *)
val CRstop = ref false ;  (*  default: don't stop enqueuing the instant you
                                       find something in the next class *)


functor ClassicRefFUN (structure Graph : GRAPH):  BFS=  
struct
 structure Graph = Graph
local open Graph Priority_Queue in

 (* Classical Refinement. *)
 (* Init: mark nodes on the abstract path with their distance from
          the start state, if searching in the same direction at the
          current level and the abstract level, otherwise with their distance
          from the goal.
    Classical Refinement means that we must go move abstract class I to
    abstract class I+1, we cannot consider states in any other abstract
    class, whether ahead (greater than I+1) or behind (less than I), nor
    states that are off the abstract path.
       Classical refinement, as just defined, assumes the abstract path is
    refinable. What if it is not ?  There is not really a "classical" answer
    to this, although Knoblock and others handle this contingency by
    backtracking to the abstract space.  What we do here is relax the
    constraints of classical refinement: the nodes that were reached during
    search but were unusable within the classical refinement strategy
    are stored on a separate queue from the ones used for classical refinement.
    When the CR queue becomes empty (a result of lack of refinability)
    this "Backup" becomes the CR queue and a new (empty) Backup queue is begun.
       It is possible for a node to be put onto both the Backup Queue and
    the CR queue: this happens if the node is first reached but not usable for
    classical refinement, and so is put on the BAckup Queue, and later is
    reached again from a place that permits the ndoe's use (so it gets put
    on the CR queue).  If the Backup Queue is ever consulted, there will
    obviously be wasted effort, as this node will be explored twice.
    However, no node can be enqueued more than twice, and the children of a
    node explored once will not be enqueued again, so the overhead is
    expected to be negligible.  If it turns out to be otherwise, we may have
    to write code that will remove a node form a queue.

      The backtrack-when-unrefinable part of this code has never been tested
    so I included a print statement issuing a warning whenever
    backtracking occurs

   *)

     fun loop [] _ = ()
       | loop ((node as Node{flag,...})::nodes) i  =
               let val mark = Step i
               in (allNodes := node :: (!allNodes);
                   inc markCost; flag:=mark ;
                   loop  nodes (i+1)
                  )
                end
     fun init absPath sameDir _ = if (sameDir) then loop absPath 1 
					       else loop (rev absPath) 1

    fun look_up_step (node as Node{flagP,...}) =
        (case flagP of
           ref(ref(Step j)) => j
         |   _              => 0  (* step# 0 means "Other" everywhere *)
        )


    fun belongs_on_CRqueue current_class node_class =
               (!CRisPO)
        orelse (current_class = 0)
        orelse (      (node_class > 0)
                andalso (        (node_class = current_class)
                          orelse (node_class = current_class + 1)
                        )
                )


    fun bfs neighbours absPathLength start =
       let val initial_CRqueue =
		        (revfold (fn (node,queue) => enqueue((1,node),1,queue)) 
		                  start (mkEmpty (1+ absPathLength)))

           fun  doQueue CRqueue BackupQueue =
              	if isEmpty(CRqueue)
                then if isEmpty(BackupQueue)
                     then NONE
                     else (if (!CRprint)
                           then print "\n\n**** Classical Refinement: BACKTRACK !!! \n\n"
                           else () ;
                           doQueue BackupQueue CRqueue )
                else
       		let val (classno,node)  = dequeue(CRqueue)
                    val _ =( if (!CRprint) andalso (classno > 1+(!currentstep))
                            then (print ("\ncurrent step = "^makestring(!currentstep)) ;
                                 print ("\n NEW    step = "^makestring(classno)^"\n"))
                            else () ;
                           currentstep := classno )

           	in doSuccs (neighbours node) (Back (1,node)) classno
                            CRqueue BackupQueue
                end

    	 and doSuccs [] _ _ CRqueue BackupQueue  = doQueue CRqueue BackupQueue
            | doSuccs ((node as Node{flag,...})::nodes) back current_class
                      CRqueue BackupQueue =
	 	(inc edgeCount ;
         	(case (!flag,look_up_step node) of
            	  (Stop, _ )      => (flag:=back ;
                                      SOME node)
          	| (Other, j) => (flag:=back; 
                                 allNodes := node :: (!allNodes) ; inc markCost;
                                 if belongs_on_CRqueue current_class j
                                 then   enqueue((j,node),j,CRqueue)
                                 else   enqueue((j,node),j,BackupQueue) ;
                                 if (       (!CRstop)
                                    andalso (j > current_class)
                                    andalso (belongs_on_CRqueue current_class j))
                                 then  doQueue CRqueue BackupQueue
                                 else  doSuccs nodes back current_class
                                         CRqueue BackupQueue )
          	| (Back (_,pred), 0) =>  doSuccs nodes back current_class
                                                 CRqueue BackupQueue
          	| (Back (_,pred), j) =>
                      let val predclass =  look_up_step pred
                      in if         (belongs_on_CRqueue current_class j)
                            andalso (not (belongs_on_CRqueue predclass j))
                         then (flag := back ; 
			       allNodes := node :: (!allNodes) ; inc markCost; 
			       enqueue((j,node),j,CRqueue) ;
                               doSuccs nodes back current_class
                                         CRqueue BackupQueue )
                         else doSuccs nodes back current_class
                                         CRqueue BackupQueue
                      end
          	| _      =>  doSuccs nodes back current_class
                                     CRqueue BackupQueue ))
         in 
            ( currentstep := 1 ;
              doQueue initial_CRqueue (mkEmpty (1+ absPathLength))
            )
         end
end
end
