structure Nim : GENERIC_PUZZLE =
struct
local open Bits Random  (* NJML needs this  *)
in
 (* The puzzle generator for the nim game turned into a one man puzzle. The game has
    been slightly changed for simplicity. It has a number of stacks of counters, normally
    three, represnted by integers which have the starting values of three consecutive
    primes. One of three consecutive small prime numbers of counters can be removed 
    from any one stack until all stacks are empty *)

 type internal_state = int
 type external_state = int list
 type puzzle_parameter = int

 val  sameState = (op =):(internal_state*internal_state)->bool
 fun  hashState n = n

 (* Tests if a number is prime *)
 fun isprime n =
    let val ndiv2 = n div 2
        fun isp(m) = (m > ndiv2) 
                      orelse (((n mod m) <> 0) andalso isp(m+1))
    in isp(2) end

 (* Generates the next n primes greater or equal to the argument "above" *)
 fun genPrimes n above = 
    let fun genP next primes =
         if (length primes = n) 
         then primes
         else genP (next+1) (if (isprime next)
              			then ( next::primes)
 				else primes)
     in genP above  [] end

 (* Merges two lists of integers to form a list of integer pairs. Used
    to form a pair representing which stack plus how many counters
    to remove *)
  fun merge [] _  newList = newList
    | merge (h::t) list newList =
          merge t list(fold (fn (x,y) => (h,x)::y) list newList)

 (* A state is an integer.  *)
 fun get (state,stack) = andb(rshift(state,stack*8),255)
 fun clear (state,stack)  = andb(state,notb(lshift(255,stack*8)))
 fun put (state,stack,to) = orb(clear(state,stack),lshift(to,stack*8))
 fun compress state =
     let fun repair ~1 _  newState = newState
           | repair stack  oldState newState =
	     let val stackVal = get(oldState,stack)
	     in repair (stack-1) oldState
 	      (if (stackVal = 0)  
		then  newState
	        else orb ((lshift (newState,8)),stackVal)) end

     in repair 3  state 0 end

 (* Convert the integer state representation into an integer list*)
 fun decode state  = 
  let fun dec 0 intList  = intList 
	| dec state intList  = 
	    dec (rshift(state,8)) ((andb(state,255))::intList) 
  in dec state []  end

 (* Convert an integer list into the integer state representation *)
 fun encode intList = 
  let fun enc [] state = state
	| enc (h::t) state =
	     enc t  (orb (lshift(state,8),h))
  in enc intList 0 end


 fun nim_generator state (stack,subs)  = 
    let val stackVal = get (state,stack) 
    in  if (subs > stackVal) 
		then NONE
 (* else if (subs = stackVal) then SOME (compress (clear(state,stack)))
     creates multiple arcs to the same state *)
		else SOME (put (state,stack,(stackVal-subs))) end


  (* construct the nim generatorsby merging a list of stack numbers
     with a list of valid counter removals *)

  fun mkGenerator subs state = 
    let fun cleanup (NONE  ::l) =      cleanup l
          | cleanup (SOME s::l) = s :: cleanup l
          | cleanup []          = []
         val states = map (nim_generator state)  
			(merge [0,1,2] (genPrimes 3 subs) []);
    in cleanup states end

  fun mkInstance subs =
     { decode   = decode,
       encode   = encode,
       successor= mkGenerator subs,
       initialStates = [encode (genPrimes 3 16)]
      }

 (* Generate a random path for testing  by randomly selecting a start
    state in the upper part of the graph. The goal is then randomly selected
    to be in the lower half of nodes that are below the start *)

 fun randomPath () = 
    let 
    fun randomState high (value,list) =
	let val new = (newrandom (value div 2)) +
			(if high then (value div 2) else 0)
        in if (new = 0) then list else new::list end
    val start  = fold (randomState true) (genPrimes 3 16) []
    val goal  = fold (randomState false) start []
    in (start,goal) end 

 end (* local open Bits *)
end
