


structure HiQ : GENERIC_PUZZLE =
struct
local open Random Bits
in


(* This is a small version of the Hi-Q puzzle
     (I don't know how many of the states are reachable).
     The positions are numbered

                                  18 19 20                                   
                                  21 22 23                                   
                                  24 25 26                                   
                          0  1  2  3  4  5
                          6  7  8  9 10 11
                         12 13 14 15 16 17

   Externally, a state is represented by the list of positions containing pegs.
   Internally, a state is represented by the 27 low-order bits of an integer,
   the bit is "set" if there is a peg in the corresponding position and
   "clear" if there i no peg there.

   Moves are jumps involving three adjacent positions, A-B-C:
   two adajcent positions must have pegs in them (say, A and B).
   The two peg-containing positions become empty, and the third position
   gets a peg in it.

   We will represent the moves by a list of 3 positions -- the move is
   applicable if the first two positions have pegs and the third does not.

*)



 val moves = [
[0,1,2], [0,6,12], [1,7,13], [1,2,3], [2,1,0], [2,3,4], [2,8,14],
[3,2,1], [3,4,5],  [3,9,15], [3,24,21], [4,3,2], [4,10,16], [4,25,22],
[5,4,3], [5,11,17], [5,26,23], [6,7,8], [7,8,9], [8,9,10], [8,7,6],
[9,8,7], [9,10,11], [9,3,24], [10,9,8], [10,4,25], [11,10,9], [11,5,26],
[12,6,0], [12,13,14], [13,7,1], [13,14,15], [14,13,12], [14,15,16], [14,8,2],
[15,14,13], [15,16,17], [15,9,3], [16,15,14], [16,10,4], [17,16,15], [17,11,5],
[18,19,20], [18,21,24], [19,22,25], [20,19,18], [20,23,26],
[21,22,23], [21,24,3], [22,25,4], [23,22,21], [23,26,5],
[24,25,26], [24,21,18], [24,3,9], [25,22,19], [25,4,10],
[26,25,24], [26,23,20], [26,5,11]
             ] ;

 type external_state = int list
 type internal_state = int
 type puzzle_parameter = external_state list   (* the initial states *)

 val  sameState = (op =):(internal_state*internal_state)->bool
 fun  hashState n = n


val all_clear = 0 ;
val all_set   = notb all_clear ;


 fun clear (state,position) =
     let val mask = xorb(all_set,lshift(1,position))
     in  andb (state,mask)
     end

 fun set   (state,position) = orb(state,lshift(1,position))
 fun get state position  = rshift(andb(state,lshift(1,position)),position)

fun encode [] = 0
 | encode (h::t) = set(encode t,h)

fun decoder (start,stop,s) =
    if start > stop
    then []
    else if andb(s,1)=1
         then start::(decoder (start+1,stop,rshift(s,1)))
         else (decoder (start+1,stop,rshift(s,1)))

fun decode s = decoder (0,26,s)

 fun make_move state (move as [p1,p2,p3]) =
 let val check = map (get state) move
 in  if check = [1,1,0]
     then SOME  (set( clear( clear(state,p1), p2), p3))
     else NONE
 end

 fun successors state =
 let fun cleanup (NONE  ::l) =      cleanup l
       | cleanup (SOME s::l) = s :: cleanup l
       | cleanup []          = []
 in cleanup (map (make_move state) moves)
 end


fun random_element L =
    let val pos = (newrandom (length L))-1
    in  (nth (L,pos))
    end


(* the following takes a state and returns a random successor of that state,
   either a terminal state (one with no successors), or one that is distance
   D from the initially given state.

   (D is really only needed to rpevent infinite loops in cyclic spaces.
    I have set it large enough here that it will never cause termination)
*)

 fun randState 0 state = (decode state)
  |  randState D state =
      case (successors state)
      of  [] => (decode state)
       |  s  => randState (D-1) (random_element s)

 fun mkInstance istates  =
 let val estates = map encode istates
 in
     { decode   =  decode,
       encode   = encode,
       successor= successors,
       initialStates = estates ,
       randomStatePair = (fn () =>
                           let val xstate =  random_element estates
                            in (decode xstate, randState 30 xstate)
                           end )
      }
 end
end
end

(* an initial states that generates a very big space is...
   (the heap got to 37 megs so I quit)

                                  18 19 20                                   
                                  21 22 23                                   
                                  24 25 26                                   
                          0  1  2  3  4  5
                          6  7  8  9  -  -
                         12 13 14 15 16 17

*)

val istateBIG = [ [
                18,19,20,
                21,22,23,
                24,25,26,
        0, 1, 2, 3, 4, 5,
        6, 7, 8, 9, (* 10, 11, *)
       12,13,14,15,16,17
       ]  ] ;

(* the following generates a graph with 2325 states... *)

val istate2 = [ [
           (*     18,19,20, *)
           (*     21,22,23, *)
           (*     24,25,26, *)
        0, 1, 2, 3, 4, 5,
        6, 7, 8, 9, (* 10, 11, *)
       12,13,14,15,16,17
       ]  ] ;


(* the following generates a graph with 6682 states... 

                                                                   -  -  - 
                                                                   -  -  - 
                                                                  24  -  - 
                                                          0  1  2  3  4  5
                                                          6  7  8  9  -  -
                                                         12 13 14 15 16 17
*)

val istate6 = [ [
           (*     18,19,20, *)
           (*     21,22,23, *)
                24, (* 25,26,   *)
        0, 1, 2, 3, 4, 5,
        6, 7, 8, 9, (* 10, 11, *)
       12,13,14,15,16,17
       ]  ] ;



