;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;          G L O B A L    P A R A M E T E R S    O F    P R O T O S
;;;   -------------------------------------------------------------------------
;;;
;;;  Contents:  This file contains global constants, variables, and parameters
;;;             of Protos.  In keeping with Lisp conventions, the names of
;;;		these global terms begin and end with '*' (e.g. *history*).
;;;=============================================================================





;;;---------------------------------------------------------------------------
;;;                        F I L E S   &   S T R E A M S
;;;
;;;  All fixed names that are used in constructing a pathname/filename in
;;;  CL-Protos are defined below.  Although the Common Lisp pathname functions
;;;  used in this implementation help promote portability, they do not
;;;  guarantee it.  Thus, some sites may have to alter some of these names
;;;  to conform with their file system.
;;;
;;;  If *log-fullname* is nil, then Protos will put the log file in the
;;;  user's home directory on the home host.  If *log-fullname* is not nil,
;;;  Protos expects it to contain a string specifying the full name of the
;;;  file, starting from the top-level directory, as in the UNIX example
;;;  "/usr/grad/dvorak/cl-protos/protos-log".  If *log-host* is nil then
;;;  *log-fullname* is assumed to be on the home host; otherwise it should
;;;  contain a string giving the host name, as in "athena".
;;;---------------------------------------------------------------------------

(defparameter  *log-fullname*   nil)
(defparameter  *log-host*       nil)

;; This is the default name of the log file in the user's home directory.
(defparameter  *log-filename*  "protos-log")

(defparameter  *logstream*  nil)        ; stream for the log file



;;;---------------------------------------------------------------------------
;;;                         "A N C H O R S"
;;;
;;;  Certain tasks, such as writing out the whole knowledge base to a file,
;;;  requires a way of finding all the components of the KB that have been
;;;  created.  The lists below provide this.  Naturally, any functions that
;;;  add (or delete) things to the KB must take care to update these lists.
;;;---------------------------------------------------------------------------

(defparameter *history*         nil)	; List of created terms, predicates, etc.
(defparameter *transformations* nil)    ; List of pointers to all transformations.
(defparameter *uninstantiated-predicates* nil)  ; List of pointers to predicates
;;    that are defined but not part of instantiated features. Used in order to
;;    be able to forget them. These predicates are created when a transformation
;;    is defined involving predicates that so far have not been used in features.



;;;---------------------------------------------------------------------------
;;;                M O D E S   O F   O P E R A T I O N 
;;;
;;;  PROTOS can operate in one of two major modes:
;;;  
;;;  -- In "learning mode" Protos interacts with a teacher in the process
;;;     of classifying new cases.  It acquires knowledge from teacher-provided
;;;     explanations and through retention of exemplars whose classification
;;;     has been verified by the teacher.
;;;
;;;  -- In "performance mode" Protos performs classification without any
;;;     consultation with the teacher.  In this mode it does not acquire
;;;     any new knowledge; it just uses what it has.
;;;
;;;  When Protos is in "learning mode" it operates in one of two sub-modes:
;;;
;;;  -- In "maximize mode" Protos maximizes its opportunities for learning
;;;     from the teacher.  Rather than spending much effort looking for
;;;     possibly better matches, Protos presents the first match to the
;;;     teacher.  This gives the teacher more opportunity to correct any
;;;     faulty explanations constructed by Protos.
;;;
;;;  -- In "minimize mode" Protos spends considerably more effort to find
;;;     the best match before asking the teacher for verification.
;;;
;;;  These modes and sub-modes are settable from the Control Menu.
;;;---------------------------------------------------------------------------

(defparameter *learning-mode*  t)     ; if nil, then performance mode.
(defparameter *maximize-mode*  t)     ; if nil, then minimize mode.
(defparameter *heuristic-mode* t)     ; if nil, then do not use heuristics 
                                      ; when searching for an explanation.
(defparameter *abbreviation-mode* nil) ; use abbreviations on output


;;;---------------------------------------------------------------------------
;;; 		     D I A G N O S T I C    M O D E L
;;;
;;;  CL-Protos contains an improvement over the original Protos in that it
;;;  can apply knowledge contained in a diagnostic model in order to find
;;;  better explanations.  If this parameter is non-nil then CL-Protos 
;;;  distinguishes among different types of terms (lab data, patient history,
;;;  findings, etc.) in formulating strong diagnostic explanations.
;;;  If this parameter is nil then CL-Protos performs domain-independent
;;;  knowledge-based pattern matching using a uniform-cost graph-search.
;;;
;;;  NOTE:  This was never fully implemented.
;;;---------------------------------------------------------------------------

(defparameter *diagnostic-model* nil)



;;;---------------------------------------------------------------------------
;;;                    T R A C E   S W I T C H E S
;;;
;;;  There are a variety of trace switches intended for turning on or off
;;;  various messages tracing the internal workings of CL-Protos.  These
;;;  messages are useful for debugging as well as examining the detailed
;;;  operation of CL-Protos.  These are settable from the Control Menu.
;;;---------------------------------------------------------------------------

(defparameter *trace-remindings*        t) ; evoked remindings and combining
(defparameter *trace-importances*       t) ; importance values
(defparameter *trace-hypotheses*        t) ; building of hypotheses list
(defparameter *trace-kbpm*            nil) ; knowledge-based pattern-matching
(defparameter *trace-kbpm2*           nil) ; kbpm, shows relations traversed
(defparameter *trace-heuristics*      nil) ; application of heuristics in kbpm
(defparameter *trace-case-matching*     t) ; case-to-exemplar matching
(defparameter *trace-new-exemplar*      t) ; new exemplar creation
(defparameter *trace-new-remindings*    t) ; newly created remindings
(defparameter *trace-transformations*   t) ; numeric-to-symbolic xformations
(defparameter *switch-critiques*      nil) ; critique explanations
(defparameter *switch-nth-root*       nil) ; take nth-root of similarity



;;;---------------------------------------------------------------------------
;;;                         C O N S T A N T S
;;;
;;;  CL-Protos contains several constant values, most typically as limit or
;;;  threshold values.  The purpose of this section is to gather these
;;;  constants into one place where they can be seen (as opposed to having
;;;  them scattered throughout the source code).  This should make it easier
;;;  for researchers who wish to "tune" CL-Protos for some application or
;;;  who wish to evaluate the sensitivity of the Protos algorithm to the
;;;  various numeric values.
;;;---------------------------------------------------------------------------

(defconstant  *protos-version* 0.7)	; Version of Protos software.

(defconstant  *weak-relation* 0.25)    ; used by truncate-explanation2

;; Strength thresholds for exemplar-to-newcase matches.
(defconstant  *perfect-match* 1.0)     ; perfect == strength =  1.0
(defconstant  *strong-match*  0.7)     ; strong  == strength >= 0.7
(defconstant  *weak-match*    0.4)     ; weak    == strength <= 0.4

;; Used by test-hypotheses2 and compare-new-case.  Any match below this
;; level of similarity will be discarded.
(defconstant  *minimum-match* 0.15) 

;; This is used by test-hypotheses2.  If a match is at least this strong,
;; then pursue the exemplar's difference links to try to improve the match.
(defconstant  *difference-threshold* 0.35)  ; used by test-hypotheses2

;; Used by discuss-exemplars to decide if an incorrect exemplar-to-newcase
;; match is strong enough to be considered a "near miss".  If this value
;; is too high, then Protos will fail to be "discriminating" and won't 
;; have many difference links.  If the value is too low then Protos may
;; annoy the teacher by too often asking the teacher what features
;; discriminate among weakly related exemplars.
(defconstant  *near-miss-threshold* 0.5)

;; Strengths of reminding thresholds:
(defconstant  *reminding-absolute*   1.00)
(defconstant  *reminding-strong*     0.75)
(defconstant  *reminding-moderate*   0.25)
(defconstant  *reminding-weak*	     0.10)
(defconstant  *reminding-very-weak*  0.05)  ; Any reminding weaker than this
                                            ; is removed by reassess-remindings

;; Censor values.
(defconstant  *absolute-censor* -9999)
(defconstant  *censor-cutoff*   -999)


;; Strengths of importance thresholds:
(defconstant  *importance-absolute*   1.00)
(defconstant  *importance-necessary*  0.90)
(defconstant  *importance-high*       0.65)
(defconstant  *importance-moderate*   0.35)
(defconstant  *importance-low*        0.10)

;; This is used by discuss-differences.  If the teacher says that X is
;; a discriminating feature of some exemplar, then if the importance of
;; the feature is below this threshold, Protos will ask the teacher to
;; reassess this importance (usually to raise it).
(defconstant  *reassess-importance-threshold* 0.35)

;; Importance value used when trying to find a match between two terms,
;; but not as part of an exemplar-to-case match.
(defconstant  *importance-big*        0.8)


;;;----------------------------------------------------------------------------
;;;  Constant:	*n-best-exemplars*
;;;
;;;  Purpose:	Given a reminding to a category, this is the maximum number of
;;;		exemplars of that category that Protos will put onto the
;;;		hypotheses list (remindings to specific exemplars are not
;;;		limited).
;;;
;;;  Used by:	build-hypotheses
;;;----------------------------------------------------------------------------

(defconstant *n-best-exemplars* 5)



;;;----------------------------------------------------------------------------
;;;  Constant:	*n-best-matches*
;;;
;;;  Purpose:	This is the maximum number of matches that Protos will
;;;		compute in trying to find the correct match for a new case.
;;;
;;;  Used by:	classify
;;;----------------------------------------------------------------------------

(defconstant  *n-best-matches*  10)



;;;----------------------------------------------------------------------------
;;;  Parameters:  *number-of-ec-categories*
;;;               *percent-of-ec-categories*
;;;               *low-reminding-limit*
;;;
;;;  Purpose:     These parameters all exist for use in the 'trim-remindings'
;;;               function.  Collectively, they control the maximum number of
;;;               remindings that a given feature may possess, which is
;;;               dependent upon the current number of exemplar-containing
;;;               categories.  This maximum number is simply the maximum of
;;;               *low-reminding-limit* and (*number-of-ec-categories* times
;;;               *percent-of-ec-categories*).
;;;----------------------------------------------------------------------------

(defparameter  *number-of-ec-categories*   0)     ; update by add-new-exemplar
(defconstant   *low-reminding-limit*       3)
(defconstant   *percent-of-ec-categories*  0.25)



;;;----------------------------------------------------------------------------
;;;		   T E M P O R A R Y    S T O R A G E
;;;
;;;  The following parameters are globals simply because their values are
;;;  accessed and modified from several functions.
;;;----------------------------------------------------------------------------

(defparameter *username*    nil)    ; Name of current user/teacher
(defparameter *near-misses* nil)    ; possible candidate for difference links.
(defparameter *remindings*  nil)    ; List of current remindings from new case.
(defparameter *initial-remindings* nil) ; List of initial values for the remindings
                                        ; to the new case
